#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <inttypes.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <xs.h>
#include <xen/io/xenbus.h>

#include "shared.h"
#include "list.h"
#include "daemon.h"
#include "xenbackd.h"

/* ------------------------------------------------------------- */

struct vfbdev {
    struct xendev       xendev;  /* must be first */
    char                *type;
    char                *domain;
    pid_t               pid;
};

/* ------------------------------------------------------------- */

static void vfb_start_backend(struct vfbdev *vfbdev)
{
    char domid[16];
    char title[64];
    char *sdl_argv[] = {
	"/usr/" LIB "/xen/bin/xen-sdlfb",
	"--domid", domid,
	"--title", title,
	NULL
    };
    char *vnc_argv[] = {
	"/usr/" LIB "/xen/bin/xen-vncfb",
	"--unused",
	"--domid", domid,
	"--title", title,
	NULL
    };
    char *qemu_argv[] = {
	"/usr/" LIB "/xen/bin/qemu-dm",
	"-M", "xenpv",
	"-vnc", "127.0.0.1:0" /* ",passwd" */,
	"-vncunused",
	"-d", domid,
	"-domain-name", title,
	NULL
    };
    char **argv = NULL;
    pid_t pid;

    vfbdev->type   = read_be_str(&vfbdev->xendev, "type");
    vfbdev->domain = read_be_str(&vfbdev->xendev, "domain");
    d1printf("%s: type %s, name \"%s\"\n", __FUNCTION__,
	     vfbdev->type   ?: "unset",
	     vfbdev->domain ?: "unset");

    if (!vfbdev->type || !vfbdev->domain)
	return;
    if (0 == strcasecmp(vfbdev->type, "sdl"))
	argv = sdl_argv;
    if (0 == strcasecmp(vfbdev->type, "vnc"))
	argv = vnc_argv;
    if (0 == strcasecmp(vfbdev->type, "qemu"))
	argv = qemu_argv;
    if (!argv)
	return;

    d1printf("%s: starting %s\n", __FUNCTION__, argv[0]);
    snprintf(domid, sizeof(domid), "%d", vfbdev->xendev.dom);
    snprintf(title, sizeof(title), "%s", vfbdev->domain);
    pid = fork();
    switch (pid) {
    case -1:
	/* error */
	d1printf("fork: %s\n", strerror(errno));
	break;
    case 0:
	/* child */
	execv(argv[0], argv);
	d1printf("execv(%s): %s\n", argv[0], strerror(errno));
	exit(1);
    default:
	/* parent */
	vfbdev->pid = pid;
	break;
    }
    return;
}

static int vfb_setup(struct xendev *xendev)
{
    struct vfbdev *vfbdev = container_of(xendev, struct vfbdev, xendev);

    d1printf("%s: %p\n", __FUNCTION__, vfbdev);

    if (vfbdev->pid)
	return 0;
    vfb_start_backend(vfbdev);
    return 0;
}

static int vfb_backend(struct xendev *xendev, char *node, char *val)
{
    struct vfbdev *vfbdev = container_of(xendev, struct vfbdev, xendev);

    d1printf("%s: %p: %s = \"%s\"\n", __FUNCTION__, vfbdev, node, val);

    if (vfbdev->pid)
	return 0;
    vfb_start_backend(vfbdev);
    return 0;
}

static int vfb_free(struct xendev *xendev)
{
    struct vfbdev *vfbdev = container_of(xendev, struct vfbdev, xendev);
    int status;

    if (!vfbdev->pid)
	return 0;

    d1printf("%s: %p\n", __FUNCTION__, vfbdev);
    kill(vfbdev->pid, SIGTERM);
    waitpid(vfbdev->pid, &status, 0);
    vfbdev->pid = 0;
    return 0;
}

static struct devops vfbdev_ops = {
    .size  = sizeof(struct vfbdev),
    .setup = vfb_setup,
    .xs_be = vfb_backend,
    .free  = vfb_free,
};

/* ------------------------------------------------------------- */

static void prepare_environment(char *argv0)
{
    char *path[] = {
	".",
	".",
	"/usr/" LIB "/xenner",
	"/usr/local/" LIB "/xenner",
    };
    char cwd[256];
    char exe[256];
    char buf[256];
    char *h;
    int i;

    getcwd(cwd, sizeof(cwd));
    path[0] = cwd;

    strncpy(exe, argv0, sizeof(exe));
    h = strrchr(exe,'/');
    if (h)
	*h = 0;
    path[1] = exe;

    for (i = 0; i < sizeof(path)/sizeof(path[0]); i++) {
	snprintf(buf, sizeof(buf), "%s/%s", path[i], "libxenctrl.so.3.0");
	if (0 != access(buf, R_OK))
	    continue;
	setenv("LD_LIBRARY_PATH", path[i], 1);
	d1printf("%s: LD_LIBRARY_PATH=%s\n", __FUNCTION__, path[i]);
	break;
    }
}

static void usage(FILE *fp)
{
    fprintf(fp,
	    "\n"
	    "vfbbackd --  xenner vfb backend daemon\n"
	    "\n"
	    "usage: vfbbackd [options]\n"
	    "options:\n"
	    "   -h            print this text\n"
	    "   -d            increase debuglevel\n"
	    "   -p <file>     specify pidfile\n"
	    "\n"
	    "-- \n"
	    "(c) 2007 Gerd Hoffmann <kraxel@redhat.com>\n"
	    "\n");
	    
}

int main(int argc, char *argv[])
{
    int c;

    for (;;) {
        if (-1 == (c = getopt(argc, argv, "hdp:")))
            break;
        switch (c) {
        case 'd':
	    debug++;
	    break;
	case 'p':
	    pidfile = optarg;
	    break;

        case 'h':
            usage(stdout);
            exit(0);
        default:
            usage(stderr);
            exit(1);
        }
    }

    prepare_environment(argv[0]);
    return mainloop(&vfbdev_ops, "vfb", 0);
}
