/* Support for MMS Protocol.
   Copyright (C) 2003
   Shane Caple (with code from mms_client 0.0.3 and asfrecorder 1.1).

This file is part of WgetPro.

WgetPro is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

WgetPro is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

The author of WgetPro gives permission to link the code of this release
with the OpenSSL project's "OpenSSL" library (or with modified versions
of it that use the same license as the "OpenSSL" library), and distribute
the linked executables.  You must obey the GNU General Public License
in all respects for all of the code used other than "OpenSSL".  If you
modify this file, you may extend this exception to your version of the
file, but you are not obligated to do so.  If you do not wish to do
so, delete this exception statement from your version.  */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#ifdef HAVE_STRING_H
# include <string.h>
#else
# include <strings.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#include <assert.h>
#include <errno.h>
#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
# else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#include "wget.h"
#include "utils.h"
#include "mms.h"
#include "retr.h"
#include "progress.h"
#include "url.h"
#include "recur.h"
#include "ftp.h"
#include "host.h"
#include "connect.h"
#include "hash.h"

#ifdef HAVE_SSL
# include "gen_sslfunc.h"
#endif /* HAVE_SSL */
#include "cookies.h"
#ifdef USE_DIGEST
# include "gen-md5.h"
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

extern char *version_string;

#ifndef errno
extern int errno;
#endif

extern LARGE_INT total_downloaded_bytes;

/*
 * COLLABORATIVE CODING -- START code from ASFRecorder 1.1
 * http://sourceforge.net/projects/asfrecorder/
 */

/* The function formerly known as randomize_guid() -
   Create a randomized GUID string to avoid filtering wgetpro's
   stream requests by means of detecting any hardcoded GUIDs. */

void
mms_random_guid (unsigned char *buf)
{
  time_t curtime;
  int digit, dig;

  *buf++ = '{';
  time(&curtime);
  srand(curtime);
  for (digit = 0; digit < 32; digit++)
    {
      if (digit ==  8 || digit == 12 ||
          digit == 16 || digit == 20)
        *buf++ = '-';
      dig = rand() % 0xf;
      if (dig < 10)
        *buf++ = '0' + dig;
      else
        *buf++ = 'A' + (dig - 10);
    }
  *buf++ = '}';
  *buf++ = '\0';
}

/*
 * COLLABORATIVE CODING -- ENDOF code from ASFRecorder 1.1
 * http://sourceforge.net/projects/asfrecorder/
 */

#if 0
void
mms_random_sig (unsigned char *buf)
{
  time_t curtime;
  int digit, dig;

  time(&curtime);
  srand(curtime);
  for (digit = 0; digit < 4; digit++)
    {
      dig = rand() % 0x100;
      buf[digit] = 'x';
    }
  /* scaple: Only seems to work with "b00bface" */
  buf[3] = 0xb0;
  buf[2] = 0x0b;
  buf[1] = 0xfa;
  buf[0] = 0xce;
}
#endif

void
mms_init (struct url *u, struct mms_stat *msp)
{
  msp->u		= u;
  *(msp->url)		= '\0';
  msp->local_fn		= url_file_name(u);
  msp->local_fn0	= msp->local_fn;

  mms_random_guid((char *)&(msp->guid));

  /* b00bface ?!? */
  msp->sig[3]		= 0xb0;
  msp->sig[2]		= 0x0b;
  msp->sig[1]		= 0xfa;
  msp->sig[0]		= 0xce;

  msp->sock		= -1;
  msp->fh		= (FILE *)NULL;

  msp->head_len		= 0;
  msp->mpad_len		= 0;
  msp->mpad_total	= 0;

  msp->mpkt_len		= 0;
  msp->mpkt_to_get	= 0;
  msp->mpkt_to_pad	= 0;

  msp->asf_dl		= 0;
  msp->asf_dltime	= 0;
  msp->asf_total	= 0;

  msp->seq_num		= 0;
  msp->num_stream_ids	= 0;

  msp->status		= MMSOK;
}

int
mms_connect (struct mms_stat *msp)
{
  struct url *u = msp->u;
  struct address_list *al;

  al = lookup_host (u->host, 0);
  if (!al)
    {
      msp->status = MMSERR;
      return HOSTERR;
    }
  set_connection_host_name (u->host);
  msp->sock = connect_to_many (al, u->port, 0);
  set_connection_host_name (NULL);
  address_list_release (al);
  if (msp->sock < 0)
    {
      msp->status = MMSERR;
      return errno == ECONNREFUSED ? CONREFUSED : CONERROR;
    }
  msp->status = MMSOK;
  return 0;
}

int
mms_open_file (struct mms_stat *msp)
{
  msp->fh = fopen (msp->local_fn, "wb");
  if (!msp->fh)
    {
      msp->status = MMSERR;
      return FOPENERR;
    }
  return 0;
}

int
mms_cleanup (struct mms_stat *msp)
{
  /* Only free msp->local_fn if the mms functions
     created it - do not free it if it was copied
     from somewhere else */
  if (msp->local_fn == msp->local_fn0)
    xfree(msp->local_fn);
  msp->local_fn  = (char *)NULL;
  msp->local_fn0 = (char *)NULL;

  if (msp->sock > -1)
    close(msp->sock);
  msp->sock = -1; 

  if (msp->fh)
    fclose(msp->fh);
  msp->fh = (FILE *)NULL;
  return 0;
}

/*
 * COLLABORATIVE CODING -- START code from mms_client 0.0.3
 * http://www.geocities.com/majormms/
 */

static void
put_32 (command_t *cmd, uint32_t value)
{
  cmd->buf[cmd->num_bytes] = value % 256;
  value = value >> 8;
  cmd->buf[cmd->num_bytes + 1] = value % 256 ;
  value = value >> 8;
  cmd->buf[cmd->num_bytes + 2] = value % 256 ;
  value = value >> 8;
  cmd->buf[cmd->num_bytes + 3] = value % 256 ;

  cmd->num_bytes += 4;
}

static uint32_t
get_32 (unsigned char *cmd, int offset)
{
  uint32_t ret;

  ret  = cmd[offset] ;
  ret |= cmd[offset + 1] <<8 ;
  ret |= cmd[offset + 2]<<16 ;
  ret |= cmd[offset + 3]<<24 ;
  return ret;
}

static int
mms_send_command (struct mms_stat *msp, int command,
    uint32_t switches, uint32_t extra, int length, char *data)
{
  command_t cmd;
  int len8, i;

  /* Maybe should abort() instead? */
  if (length + 48 > MBUF_SIZE)
    {
      msp->status = MMSSENDERR;
      return msp->status;
    }

  len8 = (length + (length % 8)) / 8;
  cmd.num_bytes = 0;

  put_32 (&cmd, 0x00000001); /* start sequence */
  /* put_32 (&cmd, 0xB00BFACE); */
  put_32 (&cmd, get_32(msp->sig, 0));
  put_32 (&cmd, length + 32);
  put_32 (&cmd, 0x20534d4d); /* protocol type "MMS " */
  put_32 (&cmd, len8 + 4);
  put_32 (&cmd, msp->seq_num);
  msp->seq_num++;
  put_32 (&cmd, 0x0);        /* unknown */
  put_32 (&cmd, 0x0);
  put_32 (&cmd, len8 + 2);
  put_32 (&cmd, 0x00030000 | command); /* dir | command */
  put_32 (&cmd, switches);
  put_32 (&cmd, extra);

  memcpy (&cmd.buf[48], data, length);
  if ((i = iwrite (msp->sock, cmd.buf, length + 48)) == -1)
    {
      DEBUGP(("write error (%s)\n", strerror(errno)));
      msp->status = MMSSENDERR;
      return msp->status;
    }
#ifdef DEBUG
  DEBUGP (("\n***************************************************\n" \
      "command sent, %d bytes\n", length + 48));
  DEBUGP (("start sequence %08x\n", get_32 (cmd.buf,  0)));
  DEBUGP (("command id     %08x\n", get_32 (cmd.buf,  4)));
  DEBUGP (("length         %8x \n", get_32 (cmd.buf,  8)));
  DEBUGP (("len8           %8x \n", get_32 (cmd.buf, 16)));
  DEBUGP (("sequence #     %08x\n", get_32 (cmd.buf, 20)));
  DEBUGP (("len8  (II)     %8x \n", get_32 (cmd.buf, 32)));
  DEBUGP (("dir | comm     %08x\n", get_32 (cmd.buf, 36)));
  DEBUGP (("switches       %08x\n", get_32 (cmd.buf, 40)));

  DEBUGP (("ascii contents>"));
  for (i = 48; i < (length + 48); i += 2)
    {
      unsigned char c = cmd.buf[i];

      if ((c >= 32) && (c <= 128))
        DEBUGP (("%c", c));
      else
        DEBUGP (("."));
    }
  DEBUGP (("\n"));
  DEBUGP (("complete hexdump of package follows:\n"));
  for (i = 0; i < (length + 48); i++)
    {
      unsigned char c = cmd.buf[i];

      DEBUGP (("%02x", c));
      if ((i % 16) == 15)
        DEBUGP (("\n"));
      if ((i % 2) == 1)
        DEBUGP ((" "));
    }
  DEBUGP (("\n"));
#endif /* DEBUG */
  return 0;
}

static void
string_utf16 (char *dest, char *src, int len)
{
  int i;

  memset (dest, 0, 1000);
  for (i = 0; i < len; i++)
    {
      dest[i * 2] = src[i];
      dest[i * 2 + 1] = 0;
    }
  dest[i * 2] = 0;
  dest[i * 2 + 1] = 0;
}

#ifdef DEBUG
static void
mms_print_answer (char *data, int len)
{
  int i;

  DEBUGP (("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<\n" \
      "answer received, %d bytes\n", len));
  DEBUGP (("start sequence %08x\n", get_32 (data, 0)));
  DEBUGP (("command id     %08x\n", get_32 (data, 4)));
  DEBUGP (("length         %8x \n", get_32 (data, 8)));
  DEBUGP (("len8           %8x \n", get_32 (data, 16)));
  DEBUGP (("sequence #     %08x\n", get_32 (data, 20)));
  DEBUGP (("len8  (II)     %8x \n", get_32 (data, 32)));
  DEBUGP (("dir | comm     %08x\n", get_32 (data, 36)));
  DEBUGP (("switches       %08x\n", get_32 (data, 40)));

  for (i = 48; i < len; i += 2)
    {
      unsigned char c = data[i];
    
      if ((c >= 32) && (c < 128))
        DEBUGP (("%c", c));
      else
        DEBUGP ((" %02x ", c));
    }
  DEBUGP (("\n"));
}  
#endif

static int
mms_get_answer (struct mms_stat *msp)
{
  char data[MBUF_SIZE];
  int command = 0x1b, err = 0;
  size_t len;

  while (command == 0x1b)
    {
      len = iread (msp->sock, data, MBUF_SIZE);
      if (len == -1)
        return MMSEOF;
#ifdef DEBUG
      mms_print_answer (data, len);
#endif
      command = get_32 (data, 36) & 0xFFFF;
      if (command == 0x1b) 
        {
          err = mms_send_command (msp, 0x1b, 0, 0, 0, data);
          if (err)
            return err;
        }
    }
  return 0;
}

static int
mms_read_sock (int sock, char *buf, size_t count, size_t padded)
{
  size_t total = 0;

  while (total < count)
    {
      size_t len;

      len = iread (sock, &buf[total], count - total);
      if (len < 0) {
        DEBUGP (("read error (%s)\n", strerror(errno)));
        return READERR;
      }
      total += len;
      if (len != 0)
        {
          DEBUGP (("[%d/%d]", (int)total, (int)count));
          fflush (stdout);
        }
    }
  /* scaple: Pad to end of buf. This makes debugging easier,
     as it allows saved streams to be binary compared */ 
  while (total < padded)
    buf[total++] = '\000';
  return 0;
}

int
mms_get_header (struct mms_stat *msp)
{
  unsigned char	pre_header[8];
  char data[MBUF_SIZE], *p;
  int this_pktlen, command;
  int err = 0, i = 0;

  msp->status = MMSHEADERR;
  while (1)
    {
      err = mms_read_sock (msp->sock, (char *)pre_header, 8, 0);
      if (err)
        return err;
#ifdef DEBUG    
      for (i = 0; i < 8; i++)
        DEBUGP (("pre_header[%d] = %02x (%d)\n",
            i, pre_header[i], pre_header[i]));
#endif
      if (pre_header[4] == 0x02)
        {
          this_pktlen = (pre_header[7] << 8 | pre_header[6]) - 8;
          DEBUGP (("asf header packet detected, len=%d\n", this_pktlen));
          if (msp->head_len + this_pktlen > MBUF_SIZE)
            return MMSHEADOFLOW;

          p = &(msp->head_buf[msp->head_len]);
          err = mms_read_sock(msp->sock, p, this_pktlen, 0);
          if (err)
            return err;

          msp->head_len += this_pktlen;
          p = &(msp->head_buf[msp->head_len]);
          if ((*(p - 1) == 1) && (*(p - 2) == 1))
            {
              DEBUGP (("get header packet finished\n"));
              break;
            }
        }
      else
        {
          char bytes[4];

          err = mms_read_sock (msp->sock, (char *)&bytes, 4, 0);
          if (err)
            return err;

          this_pktlen = get_32 (bytes, 0) + 4;
          DEBUGP (("command packet detected, len=%d\n", this_pktlen));

          err = mms_read_sock (msp->sock, data, this_pktlen, 0);
          if (err)
            {
	      DEBUGP (("command data read failed\n"));
	      return MMSHEADERR;
            }

          command = get_32 (data, 24) & 0xFFFF;
          DEBUGP (("command: %02x\n", command));

          if (command == 0x1b) 
            {
              err = mms_send_command (msp, 0x1b, 0, 0, 0, data);
              if (err)
                return err;
            }
        }
      DEBUGP (("get header packet succ\n"));
    }

  err = interp_header (msp);
  if (err)
    return err;

  /* 0x33 */

  memset (data, 0, 40);
  for (i = 1; i < msp->num_stream_ids; i++) {
    data [(i - 1) * 6 + 2 ] = 0xFF;
    data [(i - 1) * 6 + 3 ] = 0xFF;
    data [(i - 1) * 6 + 4 ] = msp->stream_ids[i];
    data [(i - 1) * 6 + 5 ] = 0x00;
  }
  err = mms_send_command (msp, 0x33,
      msp->num_stream_ids, 0xFFFF | msp->stream_ids[0] << 16,
      (msp->num_stream_ids-1) * 6 + 2, data);
  if (err)
    return err;
  err = mms_get_answer (msp);
  if (err)
    return err;

  /* 0x07 */

  memset (data, 0, 40);
  for (i = 8; i < 16; i++)
    data[i] = 0xFF;
  data[20] = 0x04;
  err = mms_send_command (msp, 0x07,
      1, 0xFFFF | msp->stream_ids[0] << 16, 24, data);
  if (err)
    return err;

  msp->status = MMSOK;
  return 0;
}

int
interp_header (struct mms_stat *msp)
{
  uint8_t *header;  
  int head_len, i;

  header   = msp->head_buf;
  head_len = msp->head_len;

  i = 30;
  while (i < head_len)
    {
      uint64_t guid_1, guid_2, length;

      guid_2 = (uint64_t)header[i] | ((uint64_t)header[i + 1] << 8) 
        | ((uint64_t)header[i + 2] << 16) | ((uint64_t)header[i + 3] << 24)
        | ((uint64_t)header[i + 4] << 32) | ((uint64_t)header[i + 5] << 40)
        | ((uint64_t)header[i + 6] << 48) | ((uint64_t)header[i + 7] << 56);
      i += 8;

      guid_1 = (uint64_t)header[i] | ((uint64_t)header[i + 1] << 8) 
        | ((uint64_t)header[i + 2] << 16) | ((uint64_t)header[i + 3] << 24)
        | ((uint64_t)header[i + 4] << 32) | ((uint64_t)header[i + 5] << 40)
        | ((uint64_t)header[i + 6] << 48) | ((uint64_t)header[i + 7] << 56);
      i += 8;
      DEBUGP (("guid found: %016llx%016llx\n",
          (long long)guid_1, (long long)guid_2));

      length = (uint64_t)header[i] | ((uint64_t)header[i + 1]<<8) 
        | ((uint64_t)header[i + 2] << 16) | ((uint64_t)header[i + 3] << 24)
        | ((uint64_t)header[i + 4] << 32) | ((uint64_t)header[i + 5] << 40)
        | ((uint64_t)header[i + 6] << 48) | ((uint64_t)header[i + 7] << 56);
      i += 8;

      if ((guid_1 == 0x6cce6200aa00d9a6) &&
          (guid_2 == 0x11cf668e75b22630))
        {
          DEBUGP (("header object\n"));
        }
      else if ((guid_1 == 0x6cce6200aa00d9a6) &&
          (guid_2 == 0x11cf668e75b22636))
        {
          DEBUGP (("data object\n"));
        }
      else if ((guid_1 == 0x6553200cc000e48e) &&
          (guid_2 == 0x11cfa9478cabdca1))
        {
          msp->mpad_len = get_32(header, i + 92 - 24);
          DEBUGP (("file object, packet length = %ld (%d)\n",
  	      msp->mpad_len, get_32(header, i + 96 - 24)));
        }
      else if ((guid_1 == 0x6553200cc000e68e) &&
          (guid_2 == 0x11cfa9b7b7dc0791))
        {
          int stream_id;

          stream_id = header[i + 48] | header[i + 49] << 8;
          DEBUGP (("stream object, stream id: %d\n", stream_id));

          if (msp->num_stream_ids >= MSTREAM_SIZE)
            {
              msp->status = MMSSTREAMERR;
              return msp->status;
            }
          msp->stream_ids[msp->num_stream_ids] = stream_id;
          msp->num_stream_ids++;
        }
      else
        DEBUGP (("unknown object\n"));
      DEBUGP (("length    : %lld\n", (long long)length));

      /* scaple: Does the final length always gives media
         content length + 50? */
      msp->mpad_total = length - 50;

      i += length - 24;
    }
  return 0;
}

/*
 * START functions to deal with large media packets
 */

/* mms_get_mpkt_a() knows about MBUF_SIZE, maybe this should be passed in
   as an argument instead */
void mms_get_mpkt_a(struct mms_stat *msp, int *len, int *pad)
{
  *len = 0;
  *pad = 0;

#ifdef DEBUG
  DEBUGP(("\nmms_get_mpkt_a TOP: mpktl:%ld *l:%d *p:%d mpadl:%ld (%d)\n", 
      msp->mpkt_len, *len, *pad, msp->mpad_len, MBUF_SIZE));
#endif
  msp->mpkt_to_get = msp->mpkt_len;
  msp->mpkt_to_pad = msp->mpad_len - msp->mpkt_len;
  if (msp->mpkt_len > MBUF_SIZE)
    {
      /* Deal with large packets */
      msp->mpkt_to_get = msp->mpkt_len - MBUF_SIZE;
      *len = MBUF_SIZE;
      *pad = 0;
    }
  else
   {
     *len = msp->mpkt_len;
     *pad = msp->mpkt_to_pad;
     if (*len + *pad > MBUF_SIZE)
       {
         int i;

         i = *len + *pad - MBUF_SIZE;
         *pad -= i;
       }
     msp->mpkt_to_get -= *len;
     msp->mpkt_to_pad -= *pad;
     /* Now make *pad >= *len */
     *pad += *len;
   }
#ifdef DEBUG
  DEBUGP(("mms_get_mpkt_a BOT: mpktl:%ld *l:%d *p:%d mpadl:%ld (%d)\n", 
      msp->mpkt_len, *len, *pad, msp->mpad_len, MBUF_SIZE));
#endif
}

/* mms_get_mpkt_b() knows about MBUF_SIZE, maybe this should be passed in
   as an argument instead */
void
mms_get_mpkt_b(struct mms_stat *msp, int *len, int *pad)
{
  *len = 0;
  *pad = 0;

#ifdef DEBUG
  DEBUGP (("\nmms_get_mpkt_b TOP mptg:%ld mptp:%ld *l:%d *p:%d mptl:%ld\n", 
      msp->mpkt_to_get, msp->mpkt_to_pad, *len, *pad, msp->mpkt_len));
#endif
  if ((msp->mpkt_to_get > 0) || (msp->mpkt_to_pad > 0))
    {
      *len = msp->mpkt_to_get;
      *pad = msp->mpkt_to_pad;
      if (*len > MBUF_SIZE)
        *len = MBUF_SIZE;
      if (*len + *pad > MBUF_SIZE)
        {
          int i;

          i = *len + *pad - MBUF_SIZE;
          *pad -= i;
        }
      msp->mpkt_to_get -= *len;
      msp->mpkt_to_pad -= *pad;
      /* Now make *pad >= *len */
      *pad += *len;
    }
   else
    {
      msp->mpkt_len = 0;
      msp->mpkt_to_get = 0;
      msp->mpkt_to_pad = 0;
    }
#ifdef DEBUG
  DEBUGP (("mms_get_mpkt_b BOT mptg:%ld mptp:%ld *l:%d *p:%d mptl:%ld\n", 
      msp->mpkt_to_get, msp->mpkt_to_pad, *len, *pad, msp->mpkt_len));
#endif
}

/*
 * ENDOF functions to deal with large media packets
 */

int
mms_get_mpkt_buf (struct mms_stat *msp)
{
  unsigned char pre_header[8];
  char data[MBUF_SIZE];
  int this_pktlen, command;
  int len = 0, pad = 0, err = 0;
#ifdef DEBUG
  int i = 0;
#endif

  msp->status = MMSGETERR;

  /* First get remaining bytes, if any, from previous media packet */
  mms_get_mpkt_b(msp, &len, &pad);
  if (len > 0 || pad > 0)
    {
      /* Get some more bytes from a large media packet */
      err = mms_read_sock (msp->sock, (char *)&(msp->mpkt_buf), len, pad);
      if (err)
        return err;
      msp->mpkt_len = (pad > len) ? pad : len; 
      msp->status = MMSOK;
      return 0; 
    }

  /* Get the start of the next packet */
  err = mms_read_sock (msp->sock, pre_header, 8, 0);
  if (err)
    {
      DEBUGP (("pre-header read failed\n"));
      return err;
    }
#ifdef DEBUG
  for (i = 0; i < 8; i++)
    DEBUGP (("pre_header[%d] = %02x (%d)\n",
        i, pre_header[i], pre_header[i]));
#endif
  if (pre_header[4] == 0x04)
    {
      this_pktlen = (pre_header[7] << 8 | pre_header[6]) - 8;
      DEBUGP (("asf media packet detected, len=%d\n", this_pktlen));

      if ((this_pktlen < 0) || (this_pktlen > msp->mpad_len))
        return MMSGETERR;
      msp->mpkt_len = this_pktlen;
      mms_get_mpkt_a (msp, &len, &pad);
      err = mms_read_sock (msp->sock, (char *)&(msp->mpkt_buf), len, pad);
      if (err)
        {
          DEBUGP (("media data read failed\n"));
          return err;
        }
      msp->mpkt_len = (pad > len) ? pad : len;
      msp->status = MMSPACKETOK;
      return 0;
    }
  else
    {
      char bytes[4];

      err = mms_read_sock (msp->sock, (char *)&bytes, 4, 0);
      if (err)
        {
          DEBUGP (("thispacketlen read failed\n"));
          return err;
        }
      this_pktlen = get_32 (bytes, 0) + 4;
      DEBUGP (("command packet detected, len=%d\n", this_pktlen));

      if (this_pktlen > MBUF_SIZE)
        {
          logprintf (LOG_NOTQUIET,
              _("MMS command packet too large (%d bytes).\n"), this_pktlen);
          msp->status = MMSERR;
          return MMSERR;
        }

      err = mms_read_sock (msp->sock, data, this_pktlen, 0);
      if (err)
        {
          DEBUGP (("command data read failed\n"));
          return err;
        }
      if ((pre_header[7] != msp->sig[3]) || (pre_header[6] != msp->sig[2]) ||
          (pre_header[5] != msp->sig[1]) || (pre_header[4] != msp->sig[0]))
        {
          DEBUGP (("missing signature\n"));
          return MMSGETERR; 
        }

      command = get_32 (data, 24) & 0xFFFF;
      DEBUGP (("command: %02x\n", command));

      if (command == 0x1b)
        {
          /* Answer the keepalive */
          err = mms_send_command (msp, 0x1b, 0, 0, 0, data);
          if (err)
            return err;
        }
      else if (command == 0x1e)
        {
          DEBUGP (("everything done.\n"));
          msp->status = MMSFINISHED;
          return 0;
        }
      else if (command != 0x05)
        {
          DEBUGP (("unknown command %02x\n", command));
          return MMSGETERR;
        }
    }
  msp->status = MMSOK;
  return 0;
}

int 
mms_init_stream (struct mms_stat *msp)
{
  struct url *u = msp->u;
  char str[1024], data[1024];
  int err;

  msp->status = MMSERR;
  /* cmd1 */
  
  sprintf (str, "\034\003NSPlayer/7.0.0.1956; %s; Host:%s",
      msp->guid, u->host);
  string_utf16 (data, str, strlen(str) + 2);
  err = mms_send_command (msp, 1, 0, 0x0004000b, strlen(str) * 2 + 8, data);
  if (err)
    return err;
  err = mms_get_answer (msp);
  if (err)
    return err;

  /* cmd2 */

  string_utf16 (&data[8], "\002\000\\\\192.168.0.129\\TCP\\1037\0000", 28);
  memset (data, 0, 8);
  err = mms_send_command (msp, 2, 0, 0, 28*2+8, data);
  if (err)
    return err;
  err = mms_get_answer (msp);
  if (err)
    return err;

  /* 0x5 */

  string_utf16 (&data[8], msp->u->path, strlen(u->path));
  memset (data, 0, 8);
  err = mms_send_command (msp, 5, 0, 0, strlen(u->path) * 2 + 12, data);
  if (err)
    return err;
  err = mms_get_answer (msp);
  if (err)
    return err;

  /* 0x15 */

  memset (data, 0, 40);
  data[32] = 2;
  err = mms_send_command (msp, 0x15, 1, 0, 40, data);
  if (err)
    return err;
  msp->num_stream_ids = 0;

  msp->status = MMSOK;
  return 0;
}

/*
 * COLLABORATIVE CODING -- ENDOF code from mms_client 0.0.3
 * http://www.geocities.com/majormms/
 */

int
getmms (struct url *u, struct mms_stat *msp)
{
  int nbytes = 0, err = 0;
  long last_dltime = 0;
  void *progress = (void *)NULL;
  struct wget_timer *timer;

  timer = wtimer_allocate();
  wtimer_reset (timer);

  /* Connect */
  err = mms_connect (msp);
  if (err)
    return err;

  /* Send the MMS request */
  err = mms_init_stream (msp);
  if (err)
    return err;
  logprintf (LOG_VERBOSE, ("MMS request sent, awaiting response... "));

  /* Get the MMS header */
  err = mms_get_header (msp);
  if (err)
    return err;
  logprintf (LOG_VERBOSE, ("OK\n"));
  msp->asf_total = msp->head_len + msp->mpad_total;

  /* For broadcasts with opt.limit_broadcast,
     adjust download size so that the last media packet is complete */
  if ((msp->mpad_total == 0) &&
      (opt.limit_broadcast > msp->head_len))
    {
      msp->asf_total = opt.limit_broadcast;
      if (msp->mpad_len > 0)
        {
          msp->asf_total = (msp->asf_total - msp->head_len)
              / msp->mpad_len;
          msp->asf_total = (msp->asf_total * msp->mpad_len)
              + msp->head_len;
        }
      else
        msp->asf_total = 0;
    }

  if ((msp->mpad_total == 0) &&
      (opt.limit_broadcast > 0) &&
      (opt.limit_broadcast < msp->head_len))
    msp->asf_total = 0;

  if (opt.verbose)
    {
      if (msp->mpad_total > 0)
        logprintf (LOG_VERBOSE,
            _("Length: %s [Microsoft ASF]"), legible (msp->asf_total));
      else
        {
          logprintf (LOG_VERBOSE, _("Length: unspecified [Microsoft ASF]"));
          if (opt.limit_broadcast > 0)
            logprintf (LOG_VERBOSE,
                _("; option --limit-broadcast=%s"),
                legible_large_int(opt.limit_broadcast));
          else
            logprintf (LOG_VERBOSE,
                _("; option --limit-broadcast=inf"));
        }
    }

  /* Open the local file for writing */
  err = mms_open_file(msp);
  if (err)
    return FOPENERR;

  /* For broadcasts with opt.limit_broadcast,
     we are already done if the header is larger than this limit */
  if ((msp->mpad_total == 0) && (opt.limit_broadcast > 0) &&
      (msp->head_len > msp->asf_total))
    {
      msp->asf_total = 0;
      if (opt.verbose)
        {
          /* opt.limit_broadcast must be set _really_ small! */
          progress = (void *)progress_create (0, msp->asf_total);
          progress_update (progress, 0, msp->asf_dltime);
          progress_finish (progress, msp->asf_dltime);
        }
        return RETROK;
    }

  /* Write the header to the local file */
  if (fwrite (msp->head_buf, 1, msp->head_len, msp->fh) !=
      msp->head_len) 
    return FWRITEERR;
  nbytes = 0;
  msp->asf_dl = msp->head_len;
 
  if (opt.verbose) {
    progress = (void *)progress_create (0, msp->asf_total);
    progress_update (progress, msp->head_len, msp->asf_dltime);
  }

  if (opt.limit_rate)
    limit_bandwidth_reset ();

  while (1)
    {
      /* scaple: If we break here, should we rewrite the header for
         proper playback of the downloaded stream? */
      if (msp->asf_dl >= msp->asf_total)
        {
          if (msp->mpad_total > 0)
            break;
          else if (opt.limit_broadcast > 0)
            break;
        }

      /* mms_get_mpkt_buf() becomes a frontend for mms_get_mpkt_b()
         when packets are larger than MBUF_SIZE */
      err = mms_get_mpkt_buf(msp);
      if (err)
        return err;
      if (msp->status == MMSFINISHED)
        break;
      if (msp->status != MMSPACKETOK && msp->status != MMSOK)
        continue;

      /* Write some more media packet bytes */
      if (fwrite (msp->mpkt_buf, 1, msp->mpkt_len, msp->fh)
          != msp->mpkt_len) 
        return FWRITEERR;
      nbytes += msp->mpkt_len;
      msp->asf_dl += msp->mpkt_len;

      if (opt.verbose && nbytes > MPROGRESS_SIZE)
        {
          msp->asf_dltime = wtimer_elapsed (timer);
          progress_update (progress, nbytes, msp->asf_dltime);
          nbytes = 0;
        }
      else
        msp->asf_dltime = wtimer_elapsed (timer);

      if (opt.limit_rate)
        {
          /* scaple: Does this work? */
          limit_bandwidth (msp->mpkt_len, msp->asf_dltime - last_dltime);
          msp->asf_dltime = wtimer_elapsed (timer);
          last_dltime = msp->asf_dltime;
        }
    }
  if (opt.verbose)
    {
      msp->asf_dltime = wtimer_elapsed (timer);
      progress_update (progress, nbytes, msp->asf_dltime);
      progress_finish (progress, msp->asf_dltime);
    }
  return RETROK;
}

/* The genuine copy of the genuine HTTP loop!
   This does the MMS retreival */
int
mms_loop (struct url *u, char **local_file)
{
  char *locf;
  struct mms_stat ms, *msp = &ms;
  int count = 0, err = 0;

  /* the following options are silently ignored:
     opt.timestamping, opt.server_response, "continue" */

  /* Warn on (likely bogus) wildcard usage in MMS. */
  if (strchr (u->url, '*'))
    logputs (LOG_VERBOSE, _("Warning: wildcards not supported in MMS.\n"));

  /* Prepare MMS structure for use */
  mms_init(u, msp);

  /* The local_file argument can override what mms_init chose */
  if (local_file && *local_file)
    {
      xfree(msp->local_fn);
      msp->local_fn = *local_file;
    }

  if (!opt.output_document)
    {
      int len;

      locf = msp->local_fn;
      len  = strlen (locf);
      if ((len >= 10) && (strcmp (locf + len - 10, "index.html") == 0))
        msp->local_fn = "index.asf";
    }
  else
    msp->local_fn = opt.output_document;
  locf = xstrdup(msp->local_fn);

  /* If opt.noclobber is turned on and file already exists,
     do not retrieve the file */
  if (opt.noclobber && file_exists_p (locf))
    {
      logprintf (LOG_VERBOSE,
          _("File `%s' already there, will not retrieve.\n"), locf);
      /* If the file is there, we suppose it's retrieved OK.  */
      return RETROK;
    }

  /* Create local directories corresponding to the URL path */
  mkalldirs (locf);
  if (opt.backups)
    rotate_backups(locf);

  /* THE loop */
  do
    {
      char *tms;

      /* Increment the pass counter.  */
      ++count;
      sleep_between_retrievals (count);
      tms = time_str (NULL);

      /* Print fetch message, if opt.verbose.  */
      if (opt.verbose)
        {
          char *hurl = url_string (u, 1);
          char tmp[15];
          strcpy (tmp, "        ");
          if (count > 1)
            sprintf (tmp, _("(try:%2d)"), count);
          logprintf (LOG_VERBOSE, "--%s--  %s\n  %s => `%s'\n",
              tms, hurl, tmp, locf);
#ifdef WINDOWS
          ws_changetitle (hurl, 1);
#endif
          xfree (hurl);
        }

      /* Attempt the download */
      err = getmms(u, msp);

      switch (err)
        {
        case CONSOCKERR: case CONCLOSED:
        case CONERROR: case READERR: case WRITEFAILED:
        case MMSPACKETERR:
        case MMSGETERR: case MMSSENDERR: case MMSEOF:
          /* Non-fatal errors continue executing the loop, which will
             bring them to "while" statement at the end, to judge
             whether the number of tries was exceeded.  */
          if (err == MMSSENDERR)
            logputs (LOG_NOTQUIET,
                _("\nAn error occurred while sending an MMS command.\n")); 
          mms_cleanup(msp);
          mms_init(u, msp);
          msp->local_fn = locf; 
          printwhat (count, opt.ntry);
          break;
        case HOSTERR: case CONREFUSED:
        case MMSSTREAMERR: case MMSHEADERR: case MMSERR: 
        case MMSHEADOFLOW: 
          /* Fatal errors just return from the function. */
          if (err == MMSHEADOFLOW)
            logprintf (LOG_NOTQUIET,
                _("\nMMS header overflow occurred (>= %d bytes).\n"),
               MBUF_SIZE); 
          mms_cleanup(msp);
          return err;
          break;
	case FWRITEERR: case FOPENERR:
          /* Another fatal error. */ 
          logputs (LOG_VERBOSE, "\n");
          logprintf (LOG_NOTQUIET,
              ("Cannot write to `%s' (%s).\n"), locf, strerror (errno));
          mms_cleanup(msp);
          return err;
          break;
        case RETRFINISHED: case MMSFINISHED: case RETROK:
          /* Deal with you later.  */
          break;
        default:
          /* All possibilities should have been exhausted.  */
          abort ();
        }

      /* Download was successful */
      if (err == RETROK)
        {
          char *tmrate;

          /* scaple: Don't forget to subtract restval from this
             when "continue" is implemented someday */
          tmrate = retr_rate (msp->asf_dl, msp->asf_dltime, 0);
          tms = time_str (NULL);

          logprintf (LOG_VERBOSE,
              _("%s (%s) - `%s' saved [%ld/%ld]\n\n"), tms,
              tmrate, msp->local_fn, msp->asf_dl, msp->asf_total);
          logprintf (LOG_NONVERBOSE,
              "%s URL:%s [%ld/%ld] -> \"%s\" [%d]\n", tms,
              u->url, msp->asf_dl, msp->asf_total, msp->local_fn, count);

          /* scaple: No need to remember that we downloaded the
             file for later ".orig" code (as in gethttp) - this
             is only required for html link rewriting */

          ++opt.numurls;
          /* downloaded_increase (msp->asf_total); */
	  total_downloaded_bytes += msp->asf_total;
          break;
        }
      DEBUGP (("err %d (%s)", errno, strerror (errno)));
    }
  while (!opt.ntry || (count < opt.ntry));

  /* Cleanup */
  mms_cleanup(msp);
  return err;
}
