/*
 * playlistwin.c
 * Thomas Nemeth, le 15.10.2003
 *
 *   Copyright (C) 1999  Thomas Nemeth
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include "defines.h"
#include <ncurses.h>
#include "terminal.h"
#include "tmms.h"
#include "xmem.h"
#include "playlist.h"
#include "playerwin.h"
#include "playeractions.h"
#include "playlistwin.h"
#include "printlog.h"


static int mv_up();
static int mv_dn();
static int mv_pu();
static int mv_pd();
static int mv_home();
static int mv_end();
static int mv_done();

#define CHECK_PLAYLIST(what) \
        if (playlist_max() == 0) \
        { \
                return CONT; \
        } \
        if (! playlist_display) \
        { \
                printlog(2, "--> No playlist window...\n"); \
                return what; \
        }

#define TITLE "Playlist :"

#define PAGE_LEN (y - 7)


int            playlist_display = TRUE;
static int     mv_mode = FALSE;
static WINDOW *playlist_win = NULL;
static ListID  start = 0, cursor = 0;



static Cmdslst move_cmds[] = {
        {KEY_UP,     mv_up},
        {KEY_DOWN,   mv_dn},
        {KEY_PPAGE,  mv_pu},
        {KEY_NPAGE,  mv_pd},
        {KEY_HOME,   mv_home},
        {KEY_END,    mv_end},
        {KEY_ENTER,  mv_done},
        {10,         mv_done},
        {KEY_RESIZE, redisp},
        {0,          NULL}
};


static char *get_playlist_title(int x)
{
        char *name = get_playlist_name() ? get_playlist_name() : "none";
        char *title;
        int   len = strlen(TITLE);

        if ((int)strlen(name) + len + 7 > x)
        {
                name = strrchr(get_playlist_name(), '/');
                if (name)
                {
                        name++;
                }
                if ((name && (((int)strlen(name) + len + 7 > (int)COLS))) ||
                    (! name))
                {
                        name = "...";
                }
        }
        title = xmalloc(len + strlen(name) + 2);
        strcpy(title, TITLE" ");
        strcat(title, name);
        return title;
}


void display_playlist()
{
        int x, y;
        char *playlist_name = NULL;

        CHECK_DISPLAY(playlist_display);
        cursor = 0;
        start = 0;
        if (playlist_win)
        {
                destroy_win(playlist_win);
        }
        getmaxyx(stdscr, y, x);
        playlist_name = get_playlist_title(x);
        playlist_win = create_newwin(playlist_name, y - 4, x, 4, 0);
        free(playlist_name);
        printlog(1, "Taille cran : %dLx%dC\n", y, x);
        cursor = get_current();
        display_list();
}


static void display_entry(ListID i, ListID current,
                char *text, char *spaces, char *length)
{
        int switch_hl = FALSE;

        if (cursor == i)
        {
                wattron(playlist_win, A_REVERSE);
        }
        if (current == i)
        {
                switch_hl = TRUE;
        }
        set_color(playlist_win, NumberColor, switch_hl);
        wmove(playlist_win, i - start + 1, 1);
        if (current == i)
        {
                wprintw(playlist_win, ">%3d. ",
                        playlist_get_sel(i) != 0 ? playlist_get_sel(i) : i);
        }
        else
        {
                wprintw(playlist_win, " %3d. ",
                        playlist_get_sel(i) != 0 ? playlist_get_sel(i) : i);
        }
        if (mv_mode && (cursor == i))
        {
                set_color(playlist_win, MoveCursorColor, switch_hl);
        }
        else
        {
                if (playlist_get_sel(i) != 0)
                {
                        set_color(playlist_win, SelectedColor, switch_hl);
                }
                else
                {
                        set_color(playlist_win, ListSongColor, switch_hl);
                }
        }
        wmove(playlist_win, i - start + 1, 7);
        wprintw(playlist_win, "%s%s%s",
                        text,
                        spaces ? spaces : "",
                        length ? length : "");
        if (cursor == i)
        {
                wattroff(playlist_win, A_REVERSE);
        }
}


static void clean_lastlines(ListID last, int x, int y)
{
        ListID i;
        for (i = last - start + 2 ; i < y - 5 ; i++)
        {
                int j;
                char *spaces = xmalloc(x - 1);

                for (j = 0 ; j < x - 2; j++)
                {
                        spaces[j] = ' ';
                }
                spaces[j] = 0;
                wmove(playlist_win, i, 1);
                wprintw(playlist_win, "%s", spaces);
                free(spaces);
        }
}


static void get_display_entry(ListID i, int x,
                char **entry, char **spaces, char **length)
{
        char *text = strrchr(playlist_get(i), '/');
        char *l = playlist_get_length(i);
        char *s = NULL, *e = NULL;
        int   j, free_space = x - 8 - (l ? strlen(l) + 1 : 0);

        if ((! text) || (text[0] == 0))
        {
                text = playlist_get(i);
        }
        else
        {
                text++;
                if (text[0] == 0)
                {
                        text = playlist_get(i);
                }
        }

        if ((int) strlen(text) > free_space)
        {
                e = xmalloc(free_space + 1);
                memset(e, 0, free_space + 1);
                strncpy(e, text, free_space);
                if (l)
                {
                        s = xstrdup(" ");
                }
        }
        else
        {
                int nb_spaces = free_space - (int)strlen(text);
                if (l)
                {
                        nb_spaces++;
                }
                e = xstrdup(text);
                s = xmalloc(free_space + 1);
                memset(s, 0, free_space + 1);
                for (j = 0 ; j < nb_spaces  ; j++)
                {
                        s[j] = ' ';
                }
        }
        *entry = e;
        *spaces = s;
        *length = l;
}


void display_list()
{
        ListID i, last, current = get_current();
        int x, y;

        CHECK_DISPLAY(playlist_display);
        pthread_mutex_lock(&mutex_display);
        if (start == 0)
        {
                start = 1;
        }
        getmaxyx(stdscr, y, x);
        last = (start + y - 7 > playlist_max() ? playlist_max() : start + y - 7);

        for (i = start ; i <= last ; i++)
        {
                char *entry = NULL, *spaces = NULL, *length = NULL;

                get_display_entry(i, x, &entry, &spaces, &length);
                display_entry(i, current, entry, spaces, length);
                FREE(entry);
                FREE(spaces);
        }
        clean_lastlines(last, x, y);
        wrefresh(playlist_win);
        pthread_mutex_unlock(&mutex_display);
}


/* Common commands */

static void up()
{
        int x, y;

        getmaxyx(stdscr, y, x);
        cursor--;
        if (cursor <= 0)
        {
                cursor = playlist_max();
        }
        while (cursor < start)
        {
                start--;
        }
        while (cursor > start + PAGE_LEN)
        {
                start++;
        }
}


static void down()
{
        int x, y;

        getmaxyx(stdscr, y, x);
        cursor++;
        if (cursor > playlist_max())
        {
                cursor = 1;
        }
        while (cursor > start + PAGE_LEN)
        {
                start++;
        }
        while (cursor < start)
        {
                start--;
        }
}


static void home()
{
        cursor = 1;
        start = 1;
}


static void end()
{
        int x, y;

        getmaxyx(stdscr, y, x);
        cursor = playlist_max();
        start = cursor - PAGE_LEN;
        if (start <= 0)
        {
                start = 1;
        }
}


static void page_up()
{
        int x, y;

        getmaxyx(stdscr, y, x);
        cursor -= PAGE_LEN;
        if (cursor <= 0)
        {
                cursor = 1;
        }
        start = cursor;
}


static void page_down()
{
        int x, y;

        getmaxyx(stdscr, y, x);
        cursor += PAGE_LEN;
        if (cursor > playlist_max())
        {
                cursor = playlist_max();
        }
        start = cursor - PAGE_LEN;
        if (start <= 0)
        {
                start = 1;
        }
}


/* Commands */

int previous()
{
        CHECK_PLAYLIST(prev_file());
        up();
        display_list();
        return CONT;
}


int next()
{
        CHECK_PLAYLIST(next_file());
        down();
        display_list();
        return CONT;
}


int first()
{
        CHECK_PLAYLIST(CONT);
        home();
        display_list();
        return CONT;
}


int last()
{
        CHECK_PLAYLIST(CONT);
        end();
        display_list();
        return CONT;
}


int pgup()
{
        CHECK_PLAYLIST(CONT);
        page_up();
        display_list();
        return CONT;
}


int pgdn()
{
        CHECK_PLAYLIST(CONT);
        page_down();
        display_list();
        return CONT;
}


int enter()
{
        int cont;
        int nbs = playlist_nb_sel();
        int sel = playlist_get_sel(cursor);
        int title = cursor;

        CHECK_PLAYLIST(play_pause());
        if ((nbs != 0) && (sel == 0))
        {
                title = playlist_first_sel();
        }
        cont = stop();
        set_current(title);
        return play_pause();
}


int remove_entry()
{
        CHECK_PLAYLIST(CONT);
        playlist_del(cursor);
        up();
        down();
        display_list();
        return CONT;
}


int select_entry()
{
        CHECK_PLAYLIST(CONT);
        playlist_tsel(cursor);
        return next();
}


int sel_all()
{
        ListID i;

        CHECK_PLAYLIST(CONT);
        for (i = 1 ; i <= playlist_max() ; i++)
        {
                playlist_sel(i, TRUE);
        }
        display_list();
        return CONT;
}


int unsel_all()
{
        ListID i;

        CHECK_PLAYLIST(CONT);
        for (i = 1 ; i <= playlist_max() ; i++)
        {
                playlist_sel(i, FALSE);
        }
        display_list();
        return CONT;
}


int invert_sel()
{
        ListID i;

        CHECK_PLAYLIST(CONT);
        for (i = 1 ; i <= playlist_max() ; i++)
        {
                playlist_tsel(i);
        }
        display_list();
        return CONT;
}


int move_entry()
{
        int cont = CONT;

        CHECK_PLAYLIST(CONT);
        mv_mode = TRUE;
        display_list();
        while (cont == CONT)
        {
                cont = command_loop(move_cmds);
        }

        return CONT;
}


static int mv_up()
{
        int pos = cursor;

        up();
        playlist_move(pos, cursor);
        display_list();
        return CONT;
}


static int mv_dn()
{
        int pos = cursor;

        down();
        playlist_move(pos, cursor);
        display_list();
        return CONT;
}


static int mv_pu()
{
        int pos = cursor;

        page_up();
        playlist_move(pos, cursor);
        display_list();
        return CONT;
}


static int mv_pd()
{
        int pos = cursor;

        page_down();
        playlist_move(pos, cursor);
        display_list();
        return CONT;
}


static int mv_home()
{
        int pos = cursor;

        home();
        playlist_move(pos, cursor);
        display_list();
        return CONT;
}


static int mv_end()
{
        int pos = cursor;

        end();
        playlist_move(pos, cursor);
        display_list();
        return CONT;
}


static int mv_done()
{
        mv_mode = FALSE;
        display_list();
        return STOP;
}


int get_cursor()
{
        return cursor;
}


void set_cursor(int new_pos)
{
        int x, y;

        getmaxyx(stdscr, y, x);
        if (new_pos > start + PAGE_LEN)
        {
                start = new_pos - PAGE_LEN; 
        }
        else if (new_pos < start)
        {
                start = new_pos;
        }
        cursor = new_pos;
        display_list();
}

