/***************************************************************************
    smb4ksynchronizer  -  This is the synchronizer of Smb4K.
                             -------------------
    begin                : Mo Jul 4 2005
    copyright            : (C) 2005-2008 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

#ifndef SMB4KSYNCHRONIZER_H
#define SMB4KSYNCHRONIZER_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// KDE includes
#include <kprocess.h>

// Qt includes
#include <QObject>
#include <QString>

// KDE includes
#include <kdemacros.h>

// forward declarations
class Smb4KShare;
class Smb4KSynchronizationInfo;


/**
 * This is a core class of Smb4K. It manages the synchronization of remote
 * shares with a local copy (and vice versa).
 *
 * @author Alexander Reinholdt <dustpuppy@users.berlios.de>
 */


class KDE_EXPORT Smb4KSynchronizer : public QObject
{
  Q_OBJECT

  public:
    /**
     * The constructor of the synchronizer.
     *
     * @param parent      The parent of this object
     */
    Smb4KSynchronizer( QObject *parent = 0 );

    /**
     * The destructor.
     */
    ~Smb4KSynchronizer();

    /**
     * This function synchronizes a destination with the source.
     *
     * @param source      The source location
     *
     * @param destination The destination
     */
    void synchronize( const QString &source, const QString &destination );

    /**
     * This function reports if the synchronizer is running or not.
     *
     * @returns           TRUE if the synchronizer is running an FALSE otherwise.
     */
    bool isRunning() { return m_working; }

  public slots:
    /**
     * This function aborts the synchronization, i.e. the sync process is killed.
     */
    void abort();

  signals:
    /**
     * This signal emits the run state.
     *
     * @param state       The so-called run state. There are several defined
     *                    in the smb4kdefs.h header file.
     */
    void state( int state );

    /**
     * This signal is emitted just before the synchronization process is
     * started.
     */
    void start();

    /**
     * This signal is emitted when the synchronization process finished.
     */
    void finished();

    /**
     * Emit information about the progress of current synchronization process.
     * The information that's available may only be partial, i.e. that maybe
     * the file name or the rate or somthing else is missing. That's because
     * of the way the output of rsync is processed.
     *
     * @param info        Information about progress of the synchronization
     * process
     */
    void progress( Smb4KSynchronizationInfo *info );

  protected slots:
    /**
     * This slot is called, when the process exited.
     *
     * @param exitCode           The exit code of the process
     *
     * @param exitStatus         The exit status of the process (@see QProcess::ExitStatus)
     */
    void slotProcessFinished( int exitCode, QProcess::ExitStatus exitStatus );

    /**
     * This slot is invoked when an error occurred with the process.
     *
     * @param errorCode       The error code returned by the process
     */
    void slotProcessError( QProcess::ProcessError errorCode );

    /**
     * This slot reads from stdout and processes the output that is received. It emits
     * the process() signal when done.
     */
    void slotReadStandardOutput();


    /**
     * This slot reads from stderr and processes the output that is received. It
     * reports an error if necessary.
     */
    void slotReadStandardError();


    /**
     * This slot is connected to QApplication::aboutToQuit() signal.
     * It aborts the running KProcess if necessary.
     */
    void slotAboutToQuit();

  private:
    /**
     * The process object for this class.
     */
    KProcess *m_proc;

    /**
     * This booian is TRUE if the synchronizer is working and FALSE otherwise.
     */
    bool m_working;

    /**
     * This function reads the options, that the user chose to use with rsync.
     *
     * @returns     an option string
     */
    const QString readRsyncOptions();

    /**
     * Total number of files to transfer
     */
//     QString m_total_files;

    /**
     * If the process was aborted, this property is TRUE and FALSE
     * otherwise
     */
    bool m_aborted;

    /**
     * The process error code
     */
    QProcess::ProcessError m_process_error;
};

#endif
