/***************************************************************************
*   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
***************************************************************************/
/** @file
* This file defines classes skgmainpanel.
*
* @author Stephane MANKOWSKI / Guillaume DE BURE
*/
#include "skgmainpanel.h"
#include "skgtraces.h"
#include "skgservices.h"
#include "skginterfaceplugin.h"
#include "skgtransactionmng.h"
#include "skgzoomselector.h"
#include "skgtreeview.h"

#include <KTemporaryFile>
#include <ktip.h>
#include <ksplashscreen.h>
#include <kconfigdialog.h>
#include <kstatusbar.h>
#include <ktoolbar.h>
#include <kmenubar.h>
#include <kpushbutton.h>
#include <KDE/KLocale>
#include <kglobal.h>
#include <kparts/componentfactory.h>
#include <kaction.h>
#include <kactionmenu.h>
#include <kactioncollection.h>
#include <kmessagebox.h>
#include <ktabwidget.h>
#include <kservice.h>
#include <kxmlguifactory.h>
#include <knotification.h>
#include <kfiledialog.h>
#include <kmenu.h>
#include <kcombobox.h>
#include <kselectaction.h>
#include <kstandarddirs.h>
#include <kstringhandler.h>
#include <ktoolbarpopupaction.h>
#include <kaboutdata.h>
#include <kcmdlineargs.h>

#include <QtGui/QProgressBar>
#include <QtGui/QHeaderView>
#include <QApplication>
#include <QDir>
#include <QResizeEvent>

bool SKGMainPanel::currentActionCanceled=false;
SKGMainPanel* SKGMainPanel::mainPanel=NULL;
SKGMainPanel::SKGMainPanel ( KSplashScreen* iSplashScreen, SKGDocument* iDocument, const QStringList& iArguments )
        : KParts::MainWindow(),
        tabWidget ( new SKGTabWidget ( this ) ),
        spScreen ( iSplashScreen ),
        currentBankDocument ( iDocument ),
        tipOfDayAction ( NULL ),closePageAction ( NULL ), closeAllPagesAction ( NULL ), closeAllOtherPagesAction ( NULL ),
        kSystemTrayIcon ( NULL ), kNormalMessage ( NULL ), previousAction ( NULL ), nextAction ( NULL ),
        saveOnClose(false)
{
    SKGTRACEIN ( 1, "SKGMainPanel::SKGMainPanel" );

    //Set main panel
    mainPanel=this;

    //Initialize settings
    KSharedConfigPtr config=KSharedConfig::openConfig ();
    KConfigGroup prefskg=config->group ( "Main Panel" );
    if ( prefskg.readEntry ( "update_modified_contexts", 100 ) ==100 )
    {
        //First call, we set default values
        prefskg.writeEntry ( "update_modified_contexts", 2 );

        KConfigGroup pref=config->group ( "Notification Messages" );
        pref.writeEntry ( "updateContextOnClose", false );
    }

    //Search plugins
    KService::List offers=KServiceTypeTrader::self()->query ( QLatin1String ( "SKG GUI/Plugin" ) );

    //Create temporary file for tips
    tipsFile.setSuffix ( ".tips" );

    bool tipsFileOpened=tipsFile.open();
    bool tipsFound=false;

    QTextStream stream ( &tipsFile );

    //Load plugins
    KAboutData* about=(KAboutData*) KGlobal::mainComponent().aboutData();
    int nb=offers.count();
    SKGTRACEL ( 1 ) << nb << " plugins found" << endl;
    if ( spScreen ) spScreen->showMessage ( i18nc("Splash screen message", "Loading plugins..." ), Qt::AlignLeft, QColor ( 221, 130, 8 ) ); // krazy:exclude=qmethods
    SKGError err;
    for ( int i=0; i<nb; ++i )
    {
        KService::Ptr service = offers.at ( i );
        QString name=service->name();
        QString version=service->property ( "X-KDE-PluginInfo-Version", QVariant::String ).toString();
        QString email=service->property ( "X-KDE-PluginInfo-Email", QVariant::String ).toString();
        QString id=service->property ( "X-Krunner-ID", QVariant::String ).toString();
        QString msg=i18nc("Splash screen message", "Loading plugin %1/%2: %3...", i+1, nb, name );
        SKGTRACEL ( 1 ) << msg << endl;
        if ( spScreen ) spScreen->showMessage ( msg, Qt::AlignLeft, QColor ( 221, 130, 8 ) ); // krazy:exclude=qmethods
        KPluginLoader loader ( service->library() );
        if (version.isEmpty() || SKGServices::stringToDouble(version)<1)
        {
            SKGTRACE << "WARNING: plugin [" << name << "] not loaded because of version too old (<1)" << endl;
        }
        else
        {
            KPluginFactory *factory =loader.factory();
            if ( factory )
            {
                SKGInterfacePlugin *pluginInterface = factory->create<SKGInterfacePlugin> ( this );
                if ( pluginInterface && pluginInterface->isEnabled() && pluginInterface->setupActions ( currentBankDocument, iArguments ) )
                {
                    //Add plugin in about
                    if (about)
                    {
                        QString author= service->property ( "X-KDE-PluginInfo-Author", QVariant::String ).toString();
                        if (!author.isEmpty())
                            about->addCredit(ki18n("%1").subs(author.toAscii().data()),
                                             ki18n("Developer of plugin '%1'").subs(name.toAscii().data()),
                                             email.toAscii() );
                    }

                    //Store plugin
                    int nbplugin=pluginsList.count();
                    int pos=nbplugin;
                    for ( int j=nbplugin-1; j>=0; --j )
                    {
                        if ( pluginInterface->getOrder() <pluginsList.at ( j )->getOrder() )
                        {
                            pos=j;
                        }
                    }

                    pluginsList.insert ( pos, pluginInterface );
                    pluginInterface->setObjectName ( id );

                    //Build temporary tips file
                    if ( tipsFileOpened )
                    {
                        QStringList tipsList=pluginInterface->tips();
                        int nbtips=tipsList.count();
                        for ( int t=0; t<nbtips; ++t )
                        {
                            tipsFound=true;
                            stream << "<tip category=\""
                            << id
                            << "\"><html>"<< SKGServices::stringToHtml ( tipsList.at ( t ) )
                            << "</html></tip>"
                            << endl;
                        }
                    }
                }
            }
            else
            {
                currentBankDocument->sendMessage(i18nc("An information message",  "Loading plugin %1 failed because the factory could not be found in %2", id, service->library() ) );
            }
        }
    }

    spScreen=NULL;

    // accept dnd
    setAcceptDrops ( true );

    ui.setupUi ( this );

    // tell the KXmlGuiWindow that this is indeed the main widget
    setCentralWidget ( tabWidget );

    refreshTabPosition();
    tabWidget->setTabReorderingEnabled ( true );
    tabWidget->setTabsClosable ( true );
    tabWidget->setAutomaticResizeTabs ( true );
    tabWidget->setTabBarHidden ( false );
    tabWidget->setUsesScrollButtons ( true );

    tabWidget->show();

    //System tray
    if ( KSystemTrayIcon::isSystemTrayAvailable() )
    {
        kSystemTrayIcon=new KSystemTrayIcon ( qApp->windowIcon() );
        KMenu* systemTrayMenu = new KMenu ( this );
        QAction* quit = systemTrayMenu->addAction ( KIcon ( "application-exit" ),  i18nc ("Verb", "Quit" ) );
        connect ( quit, SIGNAL ( triggered ( bool ) ), this, SLOT ( onQuitAction() ) );

        kSystemTrayIcon->setContextMenu ( systemTrayMenu );
        connect ( kSystemTrayIcon, SIGNAL ( activated ( QSystemTrayIcon::ActivationReason ) ), SLOT ( onTrayActivated ( QSystemTrayIcon::ActivationReason ) ) );
        if ( iconInSystemtray() ) kSystemTrayIcon->show();
    }

    // then, setup our actions
    setupActions();

    displayErrorMessage ( err );

    if ( tipsFileOpened )
    {
        tipsFile.close();
    }

    //Add a status bar
    statusBar()->show();

    //Add action to control hide / display of Context Chooser
    ui.kDockContext->toggleViewAction()->setText ( i18nc ("Noun", "Context Chooser" ) );
    ui.kDockContext->toggleViewAction()->setShortcut ( Qt::SHIFT+Qt::Key_F9 );
    actionCollection()->addAction ( "view_context", ui.kDockContext->toggleViewAction() );

    KSelectAction* contextMenu = new KSelectAction ( KIcon ( "tab-new" ), i18nc ("Noun", "Context" ), this );
    actionCollection()->addAction ( "view_contextmenu" , contextMenu );

    //Add plugin in client in right order
    int shortCutIndex=0;
    QString shortCutPrefix="Ctrl+";
    int nbplugin=pluginsList.count();
    for ( int j=0; j<nbplugin; ++j )
    {
        SKGInterfacePlugin *pluginInterface=pluginsList.at ( j );
        if ( pluginInterface )
        {
            //Creation of the item
            QString title=pluginInterface->title();
            SKGTRACEL ( 1 ) << "Add plugin (" << pluginInterface->getOrder() << ") : " << title << endl;
            if ( title.length() >0 )
            {
                //Add menu item with shortcuts
                if ( pluginInterface->isInContext() )
                {
                    KIcon icon ( pluginInterface->icon() );

                    QListWidgetItem *contextItem = new QListWidgetItem ( icon, title );
                    contextItem->setStatusTip ( pluginInterface->statusTip() );
                    contextItem->setToolTip ( pluginInterface->toolTip() );
                    contextItem->setData ( 12, j );	//context item ==> plugin
                    int page=ui.kContextList->count();
                    pluginInterface->setProperty("contextItem", page); //plugin ==> context item

                    ui.kContextList->addItem ( contextItem );

                    KAction* action=contextMenu->addAction ( icon, title );
                    if ( action )
                    {
                        action->setCheckable ( false );
                        action->setData ( page );
                        if ( !shortCutPrefix.isEmpty() )
                        {
                            ++shortCutIndex;
                            if ( shortCutIndex==10 )
                            {
                                shortCutIndex=0;
                                if ( shortCutPrefix=="Ctrl+" ) shortCutPrefix+="Alt+";
                                else shortCutPrefix="";
                            }
                            if ( !shortCutPrefix.isEmpty() ) action->setShortcut ( shortCutPrefix+SKGServices::intToString ( shortCutIndex ) );
                        }
                        connect ( action, SIGNAL ( triggered ( bool ) ), this, SLOT ( onOpenContext() ) );
                    }
                }

                //Create dock if needed
                QDockWidget* dockWidget=pluginInterface->getDockWidget();
                if ( dockWidget )  this->addDockWidget ( Qt::LeftDockWidgetArea, dockWidget );
            }
        }
    }

    // a call to KXmlGuiWindow::setupGUI() populates the GUI
    // with actions, using KXMLGUI.
    // It also applies the saved mainwindow settings, if any, and ask the
    // mainwindow to automatically save settings if changed: window size,
    // toolbar position, icon size, etc.
    setupGUI ( Default, "../skg/skgmainpanel.rc" );

    for ( int j=0; j<nbplugin; ++j )
    {
        SKGInterfacePlugin *pluginInterface=pluginsList.at ( j );
        if ( pluginInterface )
        {
            QString title=pluginInterface->title();
            SKGTRACEL ( 1 ) << "Add plugin client (" << pluginInterface->getOrder() << ") : " << title << endl;
            guiFactory()->addClient ( pluginInterface );
        }
    }

    //Set status bar
    kNormalMessage=new QLabel ( this );

    QProgressBar *kProgressBar=new QProgressBar ( this );
    kProgressBar->setObjectName ( QString::fromUtf8 ( "kProgressBar" ) );
    kProgressBar->setValue ( 0 );
    kProgressBar->setFixedSize ( 100, 20 );
    QSizePolicy sizePolicy ( QSizePolicy::Fixed, QSizePolicy::Expanding );
    kProgressBar->setSizePolicy ( sizePolicy );
    kProgressBar->setToolTip ( i18nc ("Widget description", "Progress of the current action" ) );
    kProgressBar->setStatusTip ( i18nc ("Widget description", "Progress of the current action" ) );

    zoomSelector=new SKGZoomSelector(this);
    zoomSelector->setSizePolicy ( sizePolicy );
    connect ( zoomSelector, SIGNAL ( changed(int)), this, SLOT ( onZoomChanged() ) );

    statusBar()->addWidget ( kNormalMessage );
    statusBar()->addPermanentWidget ( zoomSelector );
    statusBar()->addPermanentWidget ( kProgressBar );

    //Set Cancel button
    kCancelButton=new KPushButton();
    kCancelButton->setObjectName ( QString::fromUtf8 ( "kCancelButton" ) );
    kCancelButton->setIcon ( KIcon ( "media-playback-stop" ) );
    kCancelButton->setEnabled ( false );
    kCancelButton->setToolTip ( i18nc ("Widget description", "Cancel the current action" ) );
    kCancelButton->setStatusTip ( i18nc ("Widget description", "Cancel the current action" ) );
    connect ( kCancelButton, SIGNAL ( clicked() ), this, SLOT ( onCancelCurrentAction() ) );

    statusBar()->addPermanentWidget ( kCancelButton );

    //Set progress bar call back
    if ( currentBankDocument )
    {
        progressObjects.p1= ( void* ) kProgressBar;
        progressObjects.p2= ( void* ) kCancelButton;
        currentBankDocument->setProgressCallback ( &SKGMainPanel::progressBarCallBack, ( void* ) &progressObjects );
    }

    //Connection
    if ( currentBankDocument )
    {
        connect ( ( const QObject* ) currentBankDocument, SIGNAL ( transactionSuccessfullyEnded ( int ) ), this, SLOT ( refresh() ), Qt::QueuedConnection );
        connect ( ( const QObject* ) currentBankDocument, SIGNAL ( transactionSuccessfullyEnded ( int ) ), this, SLOT ( notify ( int ) ), Qt::QueuedConnection );
    }
    connect ( ui.kContextList, SIGNAL ( itemPressed ( QListWidgetItem* ) ), this, SLOT ( onBeforeOpenContext () ) );
    connect ( ui.kContextList, SIGNAL ( itemClicked ( QListWidgetItem* ) ), this, SLOT ( onOpenContext () ) );
    connect ( tabWidget, SIGNAL ( closeRequest ( QWidget* ) ), this, SLOT ( closePage ( QWidget* ) ) );
    connect ( tabWidget, SIGNAL ( mouseMiddleClick ( QWidget* ) ), this, SLOT ( closePage ( QWidget* ) ) );
    connect ( tabWidget, SIGNAL ( mouseDoubleClick() ), this, SLOT ( addTab() ) );
    connect ( tabWidget, SIGNAL ( currentChanged ( int ) ), this, SIGNAL ( currentPageChanged() ) );
    connect ( this, SIGNAL ( currentPageChanged () ), this, SLOT ( refresh() ) );

    //Refresh
    refresh();

    //Tip of day
    if ( tipsFound )
    {
        //Tip of day is available
        KSharedConfigPtr config=KSharedConfig::openConfig ();
        KConfigGroup pref=config->group ( "TipOfDay" );
        if ( pref.readEntry ( "RunOnStart", true ) ) onTipOfDay();
    }
    else
    {
        //Remove useless menu
        delete tipOfDayAction;
        tipOfDayAction=NULL;
    }

    ui.kContextList->installEventFilter ( this );

    notify(); //Due to sendMessage not in a transaction
}

SKGMainPanel::~SKGMainPanel()
{
    SKGTRACEIN ( 1, "SKGMainPanel::~SKGMainPanel" );
    disconnect ( ( const QObject* ) currentBankDocument, 0, this, 0 );

    //close plugins
    int nb=pluginsList.count();
    for ( int i = 0; i < nb; ++i )
    {
        getPluginByIndex ( i )->close();
    }

    if ( currentBankDocument )
    {
        currentBankDocument->close();
    }
    kNormalMessage=NULL;
}

void SKGMainPanel::processArguments ( const QStringList& iArgument )
{
    foreach ( SKGInterfacePlugin * plugin, pluginsList )
    {
        if ( plugin ) plugin->processArguments ( iArgument );
    }
}

void SKGMainPanel::registedGlobalAction ( const QString& iIdentifier, KAction* iAction )
{
    registeredGlogalAction[iIdentifier]=iAction;
}

KAction* SKGMainPanel::getGlobalAction ( const QString& iIdentifier )
{
    return registeredGlogalAction[iIdentifier];
}

SKGInterfacePlugin * SKGMainPanel::getPluginByIndex ( int iIndex )
{
    SKGInterfacePlugin * output=NULL;
    if ( iIndex>=0 && iIndex<pluginsList.count() )
    {
        output=pluginsList[iIndex];
    }

    return output;
}

SKGInterfacePlugin * SKGMainPanel::getPluginByName ( const QString& iName )
{
    SKGInterfacePlugin * output=NULL;
    int nbplugin=pluginsList.count();
    for ( int j=0; !output && j<nbplugin; ++j )
    {
        if ( pluginsList[j]->objectName() ==iName )
        {
            output=pluginsList[j];
        }
    }

    return output;
}

void SKGMainPanel::setupActions()
{
    SKGTRACEIN ( 1, "SKGMainPanel::setupActions" );

    //Std File
    KStandardAction::quit ( this, SLOT ( onQuitAction() ), actionCollection() );

    //Preferences
    KStandardAction::preferences ( this, SLOT ( optionsPreferences() ), actionCollection() );

    // New Tab
    addTabAction = new KAction ( KIcon ( "tab-new-background" ), i18nc ("Noun, user action", "New Tab" ), this );
    actionCollection()->addAction ( QLatin1String ( "new_tab" ), addTabAction );
    addTabAction->setShortcut ( Qt::CTRL + Qt::SHIFT + Qt::Key_W );
    connect ( addTabAction,SIGNAL ( triggered() ), this, SLOT ( addTab() ) );

    registedGlobalAction ( "tab-new-background", addTabAction );

    //
    closePageAction = new KAction ( KIcon ( "window-close" ), i18nc ("Noun, user action", "Close" ), this );
    actionCollection()->addAction ( QLatin1String ( "tab_close" ), closePageAction );
    closePageAction->setShortcut ( Qt::CTRL + Qt::Key_W );
    connect ( closePageAction,SIGNAL ( triggered() ), this, SLOT ( closePage() ) );

    registedGlobalAction ( "tab_close", closePageAction );

    //
    closeAllPagesAction = new KAction ( KIcon ( "window-close" ), i18nc ("Noun, user action", "Close All" ), this );
    actionCollection()->addAction ( QLatin1String ( "tab_closeall" ), closeAllPagesAction );
    closeAllPagesAction->setShortcut ( Qt::ALT +Qt::Key_W );
    connect ( closeAllPagesAction,SIGNAL ( triggered() ), this, SLOT ( closeAllPages() ) );

    registedGlobalAction ( "tab_closeall", closeAllPagesAction );

    //
    closeAllOtherPagesAction = new KAction ( KIcon ( "window-close" ), i18nc ("Noun, user action", "Close All Other" ), this );
    actionCollection()->addAction ( QLatin1String ( "tab_closeallother" ), closeAllOtherPagesAction );
    closeAllOtherPagesAction->setShortcut ( Qt::CTRL + Qt::ALT +Qt::Key_W );
    connect ( closeAllOtherPagesAction,SIGNAL ( triggered() ), this, SLOT ( closeAllOtherPages() ) );

    registedGlobalAction ( "tab_closeallother", closeAllOtherPagesAction );

    //
    saveDefaultStateAction = new KAction ( KIcon ( "document-save" ), i18nc ("Noun, user action", "Save context state" ), this );
    actionCollection()->addAction ( QLatin1String ( "tab_savedefaultstate" ), saveDefaultStateAction );
    saveDefaultStateAction->setShortcut ( Qt::CTRL + Qt::ALT +Qt::Key_S );
    connect ( saveDefaultStateAction,SIGNAL ( triggered() ), this, SLOT ( saveDefaultState() ) );

    registedGlobalAction ( "tab_savedefaultstate", saveDefaultStateAction );

    //
    resetDefaultStateAction = new KAction ( KIcon ( "edit-clear" ), i18nc ("Noun, user action", "Reset context state" ), this );
    actionCollection()->addAction ( QLatin1String ( "tab_resetdefaultstate" ), resetDefaultStateAction );
    resetDefaultStateAction->setShortcut ( Qt::CTRL + Qt::ALT +Qt::Key_R );
    connect ( resetDefaultStateAction,SIGNAL ( triggered() ), this, SLOT ( resetDefaultState() ) );

    registedGlobalAction ( "tab_resetdefaultstate", resetDefaultStateAction );

    //
    QStringList overlay;
    overlay.push_back ( "rating" );
    overwriteBookmarkStateAction = new KAction ( KIcon ( "document-save", NULL, overlay ), i18nc ("Noun, user action", "Overwrite bookmark state" ), this );
    actionCollection()->addAction ( QLatin1String ( "tab_overwritebookmark" ), overwriteBookmarkStateAction );
    connect ( overwriteBookmarkStateAction,SIGNAL ( triggered() ), this, SLOT ( overwriteBookmarkState() ) );
    overwriteBookmarkStateAction->setShortcut ( Qt::CTRL + Qt::SHIFT +Qt::Key_S );

    registedGlobalAction ( "tab_overwritebookmark", overwriteBookmarkStateAction );

    //
    configureAction = new KAction ( KIcon ( "configure" ), i18nc ("Noun, user action", "Configure..." ), this );
    actionCollection()->addAction ( QLatin1String ( "tab_configure" ), configureAction );
    connect ( configureAction, SIGNAL ( triggered() ), this, SLOT ( optionsPreferences() ) );

    registedGlobalAction ( "tab_configure", configureAction );

    tipOfDayAction=KStandardAction::tipOfDay ( this, SLOT ( onTipOfDay() ), actionCollection() );

    //
    previousAction = new KToolBarPopupAction ( KIcon ( "go-previous" ), i18nc ("Noun, user action", "Previous" ), this );
    connect ( previousAction, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionPrevious() ) );
    actionCollection()->addAction ( QLatin1String ( "go_previous" ), previousAction );
    previousAction->setShortcut ( Qt::ALT +Qt::Key_Left );

    previousMenu= static_cast<KMenu*>(previousAction->menu());
    connect ( previousMenu , SIGNAL ( aboutToShow() ), this, SLOT ( onShowPreviousMenu() ) );

    previousAction->setStickyMenu ( false );
    previousAction->setData ( 0 );

    registedGlobalAction ( "go_previous", previousAction );

    //
    nextAction = new KToolBarPopupAction ( KIcon ( "go-next" ), i18nc ("Noun, user action", "Next" ), this );
    connect ( nextAction, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionNext() ) );
    actionCollection()->addAction ( QLatin1String ( "go_next" ), nextAction );
    nextAction->setShortcut ( Qt::ALT +Qt::Key_Right );

    nextMenu= static_cast<KMenu*>(nextAction->menu());
    connect ( nextMenu , SIGNAL ( aboutToShow() ), this, SLOT ( onShowNextMenu() ) );

    nextAction->setStickyMenu ( false );
    nextAction->setData ( 0 );

    registedGlobalAction ( "go_next", nextAction );

    //
    enableEditorAction = new KAction ( KIcon ( "appointment-new" ), i18nc ("Noun, user action", "Enable editor" ), this );
    actionCollection()->addAction ( QLatin1String ( "enableeditor" ), enableEditorAction );
    enableEditorAction->setShortcut ( Qt::CTRL + Qt::Key_Insert );
    connect ( enableEditorAction,SIGNAL ( triggered() ), this, SLOT ( enableEditor() ) );

    //Contextual menu
    tabWidget->setContextMenuPolicy ( Qt::ActionsContextMenu );
    tabWidget->insertAction ( 0, getGlobalAction ( "tab-new-background" ) );
    tabWidget->insertAction ( 0, getGlobalAction ( "tab_close" ) );
    tabWidget->insertAction ( 0, getGlobalAction ( "tab_closeall" ) );
    tabWidget->insertAction ( 0, getGlobalAction ( "tab_closeallother" ) );
    QAction* sep=new QAction ( this );
    sep->setSeparator ( true );
    tabWidget->insertAction ( 0, sep );
    tabWidget->insertAction ( 0, getGlobalAction ( "tab_resetdefaultstate" ) );
    tabWidget->insertAction ( 0, getGlobalAction ( "tab_savedefaultstate" ) );
    tabWidget->insertAction ( 0, getGlobalAction ( "tab_overwritebookmark" ) );
    tabWidget->insertAction ( 0, getGlobalAction ( "tab_configure" ) );
}

void SKGMainPanel::enableEditor()
{
    SKGTabPage* cPage=currentPage();
    if ( cPage ) cPage->activateEditor();
}

void SKGMainPanel::actionPrevious()
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGMainPanel::actionPrevious",err );

    SKGTabPage* cPage=currentPage();
    if ( cPage )
    {
        //Get index in history of page to refresh
        int pos= ( ( QAction* ) sender() )->data().toInt();

        //Get lists
        QList<SKGTabPage::SKGPageHistoryItem> listPrevious=cPage->getPreviousPages();
        QList<SKGTabPage::SKGPageHistoryItem> listNext=cPage->getNextPages();
        SKGTabPage::SKGPageHistoryItem current=currentPageHistoryItem();

        //Get item to refresh
        SKGTabPage::SKGPageHistoryItem item=listPrevious.at ( pos );

        //Open page
        cPage=openPage ( getPluginByName ( item.plugin ), currentPageIndex(), item.state, item.name, false );
        if (cPage)
        {
            cPage->setBookmarkID ( item.bookmarkID );

            //Update lists
            listNext.insert ( 0, current );
            listPrevious.removeAt ( pos );
            for ( int i=0; i<pos; ++i )
            {
                SKGTabPage::SKGPageHistoryItem item=listPrevious.at ( 0 ); //Because the list is modified
                listNext.insert ( 0, item );
                listPrevious.removeAt ( 0 );
            }

            //Set lists
            cPage->setPreviousPages ( listPrevious );
            cPage->setNextPages ( listNext );
        }

        refresh();
    }
}

void SKGMainPanel::onShowPreviousMenu()
{
    if ( previousMenu )
    {
        previousMenu->clear();

        SKGTabPage* cPage=currentPage();
        if ( cPage )
        {
            QList<SKGTabPage::SKGPageHistoryItem> list=cPage->getPreviousPages();
            int nb=list.count();
            for ( int i=0; i<nb; ++i )
            {
                QAction* act=previousMenu->addAction ( KIcon ( list.at ( i ).icon ), list.at ( i ).name );
                if ( act )
                {
                    act->setData ( i );
                    connect ( act, SIGNAL ( triggered() ), this, SLOT ( actionPrevious() ) );
                }
            }
        }
    }
}

void SKGMainPanel::actionNext()
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGMainPanel::actionNext",err );

    SKGTabPage* cPage=currentPage();
    if ( cPage )
    {
        //Get index in history of page to refresh
        int pos= ( ( QAction* ) sender() )->data().toInt();

        //Get lists
        QList<SKGTabPage::SKGPageHistoryItem> listPrevious=cPage->getPreviousPages();
        QList<SKGTabPage::SKGPageHistoryItem> listNext=cPage->getNextPages();
        SKGTabPage::SKGPageHistoryItem current=currentPageHistoryItem();

        //Get item to refresh
        SKGTabPage::SKGPageHistoryItem item=listNext.at ( pos );

        //Open page
        cPage=openPage ( getPluginByName ( item.plugin ), currentPageIndex(), item.state, item.name, false);
        if (cPage)
        {
            cPage->setBookmarkID ( item.bookmarkID );

            //Update lists
            listPrevious.insert ( 0, current );
            listNext.removeAt ( pos );
            for ( int i=0; i<pos; ++i )
            {
                SKGTabPage::SKGPageHistoryItem item=listNext.at ( 0 ); //Because the list is modified
                listPrevious.insert ( 0, item );
                listNext.removeAt ( 0 );
            }

            //Set lists
            cPage->setPreviousPages ( listPrevious );
            cPage->setNextPages ( listNext );
        }

        refresh();
    }
}

void SKGMainPanel::onShowNextMenu()
{
    if ( nextMenu )
    {
        nextMenu->clear();

        SKGTabPage* cPage=currentPage();
        if ( cPage )
        {
            QList<SKGTabPage::SKGPageHistoryItem> list=cPage->getNextPages();
            int nb=list.count();
            for ( int i=0; i<nb; ++i )
            {
                QAction* act=nextMenu->addAction ( KIcon ( list.at ( i ).icon ), list.at ( i ).name );
                if ( act )
                {
                    act->setData ( i );
                    connect ( act, SIGNAL ( triggered() ), this, SLOT ( actionNext() ) );
                }
            }
        }
    }
}

bool SKGMainPanel::queryClose()
{
    SKGTRACEIN ( 1, "SKGMainPanel::queryClose" );
    //Bug 2777697: To be sure that all page modifications are closed
    closeAllPages();
    //Bug 2777697:

    bool output=queryFileClose();

    if ( output && kSystemTrayIcon )
    {
        kSystemTrayIcon->hide();
        delete kSystemTrayIcon;
        kSystemTrayIcon=NULL;
    }

    return output;
}

void SKGMainPanel::setSaveOnClose(bool iSaveOnClose)
{
    saveOnClose=iSaveOnClose;
}

bool SKGMainPanel::queryFileClose()
{
    SKGTRACEIN ( 1, "SKGMainPanel::queryFileClose" );
    bool output=true;
    if ( currentBankDocument->getCurrentTransaction() )
    {
        QApplication::setOverrideCursor ( QCursor ( Qt::ArrowCursor ) );
        KMessageBox::information ( this,i18nc("Question", "The application cannot be closed when an operation is running." ) );
        QApplication::restoreOverrideCursor();
        output=false;
    }
    else if ( currentBankDocument->isFileModified() )
    {
        QApplication::setOverrideCursor ( QCursor ( Qt::ArrowCursor ) );
        int code=KMessageBox::Yes;
        QString fileName=currentBankDocument->getCurrentFileName();
        KAction* save=getGlobalAction ( fileName.isEmpty() ? "file_save_as" : "file_save" );
        if ( save )
        {
            if (!saveOnClose) code=KMessageBox::questionYesNoCancel ( this, i18nc("Question", "The document has been modified.\nDo you want to save it before closing?" ),
                                       QString(),
                                       KGuiItem ( fileName.isEmpty() ?  i18nc("Question", "Save as" ) : i18nc("Question", "Save" ),
                                                  KIcon ( fileName.isEmpty() ?  "document-save-as" : "document-save" ) ),
                                       KGuiItem ( i18nc("Question", "Do not save" ) )
                                                                        );
            if ( code==KMessageBox::Yes ) save->trigger();
            output= ( code==KMessageBox::No || code==KMessageBox::Yes );
        }
        else
        {
            code=KMessageBox::questionYesNo ( this,i18nc("Question", "Current modifications will not be saved.\nDo you want to continue ?" ) );
            output= ( code==KMessageBox::Yes );
        }
        QApplication::restoreOverrideCursor();
    }

    return output;
}

SKGObjectBase::SKGListSKGObjectBase SKGMainPanel::getSelectedObjects()
{
    SKGObjectBase::SKGListSKGObjectBase selection;

    SKGTabPage* cPage= currentPage();
    if ( cPage )
    {
        selection=cPage->getSelectedObjects();
    }
    return selection;
}

int SKGMainPanel::getNbSelectedObjects()
{
    int nb=0;

    SKGTabPage* cPage= currentPage();
    if ( cPage )
    {
        nb=cPage->getNbSelectedObjects();
    }
    return nb;
}

KConfigGroup SKGMainPanel::getMainConfigGroup()
{
    KSharedConfigPtr config=KSharedConfig::openConfig ();
    return config->group ( "Main Panel" );
}

void SKGMainPanel::optionsPreferences()
{
    SKGTRACEIN ( 1, "SKGMainPanel::optionsPreferences" );
    //Synchronize setting with confirmation panel
    {
        KMessageBox::ButtonCode confirm;
        int ask=KMessageBox::shouldBeShownYesNo ( "updateBookmarkOnClose", confirm );

        KConfigGroup pref=getMainConfigGroup();
        if ( ask )
        {
            pref.writeEntry ( "update_modified_bookmarks", 0 );
        }
        else if ( confirm==KMessageBox::Yes )
        {
            pref.writeEntry ( "update_modified_bookmarks", 1 );
        }
        else
        {
            pref.writeEntry ( "update_modified_bookmarks", 2 );
        }
    }
    {
        KMessageBox::ButtonCode confirm;
        int ask=KMessageBox::shouldBeShownYesNo ( "updateContextOnClose", confirm );

        KConfigGroup pref=getMainConfigGroup();
        if ( ask )
        {
            pref.writeEntry ( "update_modified_contexts", 0 );
        }
        else if ( confirm==KMessageBox::Yes )
        {
            pref.writeEntry ( "update_modified_contexts", 1 );
        }
        else
        {
            pref.writeEntry ( "update_modified_contexts", 2 );
        }
    }
    skgbasegui_settings::self()->readConfig();


    if ( KConfigDialog::showDialog ( "settings" ) ) return;

    KConfigDialog *dialog = new KConfigDialog ( this, "settings", skgbasegui_settings::self() );

    //Add main
    QWidget* widget=new QWidget();
    uipref.setupUi ( widget );

    uipref.kcfg_date_format->addItem(i18nc("Date format", "Short date (%1, %2)",
                                           KGlobal::locale()->formatDate (QDate::currentDate(), KLocale::ShortDate),
                                           KGlobal::locale()->formatDate (QDate::currentDate().addDays(-10), KLocale::ShortDate)));
    uipref.kcfg_date_format->addItem(i18nc("Date format", "Long date (%1, %2)",
                                           KGlobal::locale()->formatDate (QDate::currentDate(), KLocale::LongDate),
                                           KGlobal::locale()->formatDate (QDate::currentDate().addDays(-10), KLocale::LongDate)));
    uipref.kcfg_date_format->addItem(i18nc("Date format", "Fancy short date (%1, %2)",
                                           KGlobal::locale()->formatDate (QDate::currentDate(), KLocale::FancyShortDate),
                                           KGlobal::locale()->formatDate (QDate::currentDate().addDays(-10), KLocale::FancyShortDate)));
    uipref.kcfg_date_format->addItem(i18nc("Date format", "Fancy long date (%1, %2)",
                                           KGlobal::locale()->formatDate (QDate::currentDate(), KLocale::FancyLongDate),
                                           KGlobal::locale()->formatDate (QDate::currentDate().addDays(-10), KLocale::FancyLongDate)));
    uipref.kcfg_date_format->addItem(i18nc("Date format", "ISO date (%1, %2)",
                                           KGlobal::locale()->formatDate (QDate::currentDate(), KLocale::IsoDate),
                                           KGlobal::locale()->formatDate (QDate::currentDate().addDays(-10), KLocale::IsoDate)));
    uipref.kcfg_date_format->addItem(i18nc("Date format", "ISO week date (%1, %2)",
                                           KGlobal::locale()->formatDate (QDate::currentDate(), KLocale::IsoWeekDate),
                                           KGlobal::locale()->formatDate (QDate::currentDate().addDays(-10), KLocale::IsoWeekDate)));
    uipref.kcfg_date_format->addItem(i18nc("Date format", "ISO ordinal date (%1, %2)",
                                           KGlobal::locale()->formatDate (QDate::currentDate(), KLocale::IsoOrdinalDate),
                                           KGlobal::locale()->formatDate (QDate::currentDate().addDays(-10), KLocale::IsoOrdinalDate)));;

    dialog->addPage ( widget, skgbasegui_settings::self(), i18nc ("Noun", "General" ), "preferences-other" );

    //Add plugin in client in right order
    int nbplugin=pluginsList.count();
    for ( int j=0; j<nbplugin; ++j )
    {
        SKGInterfacePlugin *pluginInterface=getPluginByIndex ( j );
        if ( pluginInterface )
        {
            QWidget* w=pluginInterface->getPreferenceWidget();
            if ( w )
            {
                KPageWidgetItem * p=dialog->addPage ( w, pluginInterface->getPreferenceSkeleton(), pluginInterface->title(), pluginInterface->icon() );
                if (p && sender()==configureAction && this->currentPage() && this->currentPage()->objectName()==pluginInterface->objectName())
                {
                    dialog->setCurrentPage(p);
                }
            }
        }
    }

    connect ( dialog, SIGNAL ( settingsChanged ( const QString& ) ), this, SLOT ( onSettingsChanged() ) );

    dialog->setAttribute ( Qt::WA_DeleteOnClose );
    dialog->show();

    //Refresh
    refresh();
}

void SKGMainPanel::refreshTabPosition()
{
    tabWidget->setTabPosition ( ( QTabWidget::TabPosition ) skgbasegui_settings::main_tabs_position() );
}

void SKGMainPanel::onSettingsChanged()
{
    SKGError err;
    SKGTRACEINRC ( 1, "SKGMainPanel::onSettingsChanged", err );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    {
        int nb=pluginsList.count();
        SKGBEGINPROGRESSTRANSACTION(*currentBankDocument, i18nc("Noun, name of the user action", "Save settings" ), err, nb );

        //Refresh plugins
        for ( int i = 0; err.isSucceeded() && i < nb; ++i )
        {
            err=getPluginByIndex ( i )->savePreferences();
            if ( err.isSucceeded() ) err=currentBankDocument->stepForward ( i+1 );
        }

        //Setting for icon
        if (kSystemTrayIcon)
        {
            if ( iconInSystemtray() ) kSystemTrayIcon->show();
            else kSystemTrayIcon->hide();
        }

        //Setting for tab position
        refreshTabPosition();

        //Setting for bookmarks modification
        {
            int option=skgbasegui_settings::update_modified_bookmarks();
            if ( option==0 )
            {
                //ASK: remove following setting
                KMessageBox::enableMessage("updateBookmarkOnClose");
            }
            else if ( option==1 )
            {
                //ALWAYS: set following setting
                KMessageBox::saveDontShowAgainYesNo("updateBookmarkOnClose", KMessageBox::Yes);
            }
            else
            {
                //NEVER: set following setting
                KMessageBox::saveDontShowAgainYesNo("updateBookmarkOnClose", KMessageBox::No);
            }
        }
        {
            int option=skgbasegui_settings::update_modified_contexts();
            if ( option==0 )
            {
                //ASK: remove following setting
                KMessageBox::enableMessage("updateContextOnClose");
            }
            else if ( option==1 )
            {
                //ALWAYS: set following setting
                KMessageBox::saveDontShowAgainYesNo("updateContextOnClose", KMessageBox::Yes);
            }
            else
            {
                //NEVER: set following setting
                KMessageBox::saveDontShowAgainYesNo("updateContextOnClose", KMessageBox::No);
            }
        }
        skgbasegui_settings::self()->readConfig();
    }
    QApplication::restoreOverrideCursor();

    //Display error
    displayErrorMessage ( err );
}

void SKGMainPanel::refresh()
{
    SKGTRACEIN ( 1, "SKGMainPanel::refresh" );

    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

    //Refresh plugins
    int nb=pluginsList.count();
    for ( int i = 0; i < nb; ++i )
    {
        getPluginByIndex ( i )->refresh();
    }

    //Enable addTabeAction
    bool atLeastOnePageOpened= ( tabWidget->count() >0 );
    addTabAction->setEnabled ( atLeastOnePageOpened );
    closePageAction->setEnabled ( atLeastOnePageOpened );
    closeAllPagesAction->setEnabled ( atLeastOnePageOpened );
    closeAllOtherPagesAction->setEnabled ( tabWidget->count() >1 );

    SKGTabPage* toSave= currentPage();
    if ( saveDefaultStateAction ) saveDefaultStateAction->setEnabled ( toSave && !toSave->getDefaultStateAttribute().isEmpty() );
    if ( resetDefaultStateAction ) resetDefaultStateAction->setEnabled ( toSave && !toSave->getDefaultStateAttribute().isEmpty() );
    if ( overwriteBookmarkStateAction ) overwriteBookmarkStateAction->setEnabled ( toSave && !toSave->getBookmarkID().isEmpty() );
    if ( enableEditorAction ) enableEditorAction->setEnabled ( toSave && toSave->isEditor());
    if ( zoomSelector )
    {
        zoomSelector->setVisible ( toSave && toSave->isZoomable());
        if (toSave)
        {
            zoomSelector->setValue(toSave->zoomPosition());
            QWidget* widget=toSave->zoomableWidget();
            SKGTreeView* treeView=dynamic_cast<SKGTreeView*>(widget);
            if (treeView)
            {
                disconnect ( treeView, SIGNAL ( zoomChanged(int)), zoomSelector, SLOT ( setValue(int)) );
                connect ( treeView, SIGNAL ( zoomChanged(int)), zoomSelector, SLOT ( setValue(int)) );
            }
        }
    }

    if ( previousAction )
    {
        QList<SKGTabPage::SKGPageHistoryItem> list;
        if ( toSave ) list=toSave->getPreviousPages();
        previousAction->setEnabled ( list.count() );
    }
    if ( nextAction )
    {
        QList<SKGTabPage::SKGPageHistoryItem> list;
        if ( toSave ) list=toSave->getNextPages();
        nextAction->setEnabled ( list.count() );
    }

    //Set current selection of context
    ui.kContextList->clearSelection();
    if (toSave)
    {
        //Get plugin of current page
        SKGInterfacePlugin* plugin=getPluginByName(toSave->objectName());
        int index=(plugin ? plugin->property("contextItem").toInt() : -1);
        if (index!=-1) ui.kContextList->setCurrentItem ( ui.kContextList->item(index) );
    }

    //Set window title
    QString modified;
    if ( currentBankDocument->isFileModified() ) modified=i18nc ("Noun, indicate that current document is modified", " [modified]" );
    QString fileName=currentBankDocument->getCurrentFileName();
    if ( fileName.isEmpty() ) fileName=i18nc ("Noun, default name for a new document", "Untitled" );
    setWindowTitle ( i18nc ("Title of the main window", "%1%2 - %3", fileName, modified, KGlobal::activeComponent ().componentName () ) );

    QApplication::restoreOverrideCursor();
}

SKGTabPage::SKGPageHistoryItem SKGMainPanel::currentPageHistoryItem() const
{
    SKGTabPage::SKGPageHistoryItem page;
    int currentIndex=currentPageIndex();
    SKGTabPage* cPage= currentPage();
    if ( currentIndex>=0 && cPage )
    {
        page.plugin=cPage->objectName();
        SKGInterfacePlugin* plugin=SKGMainPanel::getMainPanel()->getPluginByName ( page.plugin );
        if ( plugin )
        {
            page.name=tabWidget->tabText ( currentIndex );
            page.icon=plugin->icon();
        }
        page.state=cPage->getState();
        page.bookmarkID=cPage->getBookmarkID();
    }

    return page;
}

SKGTabPage* SKGMainPanel::openPage ( SKGInterfacePlugin* plugin, int index, const QString& parameters, const QString& title, const QString& iID )
{
    SKGTRACEIN ( 1, "SKGMainPanel::openPage" );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    bool previous=tabWidget->blockSignals(true);
    SKGTabPage* w=NULL;

    QList<SKGTabPage::SKGPageHistoryItem> previousPages;

    if ( index!=-1 )
    {
        int currentIndex=currentPageIndex();
        SKGTabPage* cPage= currentPage();
        if ( currentIndex>=0 && cPage )
        {
            previousPages=cPage->getPreviousPages();
            previousPages.insert ( 0, currentPageHistoryItem() );

            tabWidget->removeTab ( currentIndex );
            closePage ( cPage );
        }
    }

    if ( plugin )
    {
        w=plugin->getWidget();
        if ( w )
        {
            //Title
            QString title2= ( title.isEmpty() ? plugin->title() : title );
            w->setObjectName ( plugin->objectName() );
            if (!iID.isEmpty()) w->setBookmarkID(iID);

            QString param=parameters;
            if ( param.isEmpty() )
            {
                QString def=w->getDefaultStateAttribute();
                if ( !def.isEmpty() )
                {
                    param=currentBankDocument->getParameter ( def );
                }
            }
            SKGTRACEL(10) << "state=[" << param << "]" << endl;
            w->setState ( param );
            connect ( w, SIGNAL ( selectionChanged() ), SLOT ( refresh() ) );

            if ( index==-1 )
            {
                SKGTRACEIN ( 20, "SKGMainPanel::openPage-addTab" );
                tabWidget->addTab ( w, KIcon ( plugin->icon() ), title2 );
                tabWidget->setCurrentWidget ( w );
            }
            else
            {
                SKGTRACEIN ( 20, "SKGMainPanel::openPage-insertTab" );
                tabWidget->insertTab ( index, w, KIcon ( plugin->icon() ), title2 );
                tabWidget->setCurrentWidget ( w );

                w->setPreviousPages ( previousPages );
                QList<SKGTabPage::SKGPageHistoryItem> empty;
                w->setNextPages ( empty );
            }
            SKGTRACEL ( 1 ) << "opening plugin [" << plugin->objectName () << ']' << endl;
        }
    }
    else
    {
        currentBankDocument->sendMessage(i18nc("An information message",  "Impossible to open the page because the plugin was not found" ) );
        notify(); //Due to sendMessage not in a transaction
    }
    tabWidget->blockSignals(previous);
    this->emit currentPageChanged();
    QApplication::restoreOverrideCursor();
    return w;
}

int SKGMainPanel::currentPageIndex() const
{
    return tabWidget->currentIndex();
}

SKGTabPage* SKGMainPanel::currentPage() const
{
    return ( SKGTabPage* ) tabWidget->currentWidget();
}

KSplashScreen* SKGMainPanel::splashScreen() const
{
    return spScreen;
}

int SKGMainPanel::countPages() const
{
    return tabWidget->count();
}

void SKGMainPanel::setCurrentPage ( int iIndex )
{
    tabWidget->setCurrentIndex ( iIndex );
}

void SKGMainPanel::onBeforeOpenContext ()
{
    middleClick=QApplication::mouseButtons () &Qt::MidButton;
}

void SKGMainPanel::openPage (int iPage, bool iNewPage)
{
    SKGTRACEIN ( 1, "SKGMainPanel::openPage" );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    SKGTRACEL(1) << "iPage=" << iPage << endl;
    int index=ui.kContextList->item(iPage)->data ( 12 ).toInt();
    openPage ( getPluginByIndex ( index ), iNewPage ? -1 : currentPageIndex() );
    QApplication::restoreOverrideCursor();
}

void SKGMainPanel::onOpenContext ()
{
    SKGTRACEIN ( 1, "SKGMainPanel::onOpenContext" );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    int page=-1;
    QAction* sender=dynamic_cast<QAction*> ( this->sender() );
    if ( sender ) page=sender->data().toInt();
    else page=ui.kContextList->currentRow ();
    if ( page!=-1 ) openPage(page, sender || QApplication::keyboardModifiers() &Qt::ControlModifier || middleClick);
    QApplication::restoreOverrideCursor();
}

void SKGMainPanel::closePage ( QWidget* iWidget )
{
    SKGTRACEIN ( 1, "SKGMainPanel::closePage" );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    QWidget* toRemove=iWidget;
    if ( toRemove==NULL )  toRemove=tabWidget->currentWidget();
    if ( toRemove )
    {
        ( ( SKGTabPage* ) toRemove )->close();
        delete toRemove;
    }
    QApplication::restoreOverrideCursor();
}

void SKGMainPanel::closeAllPages()
{
    SKGTRACEIN ( 1, "SKGMainPanel::closeAllPages" );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    bool previous=tabWidget->blockSignals(true);
    int nb=tabWidget->count();
    for ( int i=nb-1; i>=0; --i )
    {
        closePage ( tabWidget->widget ( i ) );
    }
    tabWidget->blockSignals(previous);
    emit currentPageChanged();
    QApplication::restoreOverrideCursor();
}

void SKGMainPanel::closeAllOtherPages ( QWidget* iWidget )
{
    SKGTRACEIN ( 1, "SKGMainPanel::closeAllOtherPages" );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    bool previous=tabWidget->blockSignals(true);
    QWidget* toKeep=iWidget;
    if ( toKeep==NULL )  toKeep=tabWidget->currentWidget();

    int nb=tabWidget->count();
    for ( int i=nb-1; i>=0; --i )
    {
        QWidget* w=tabWidget->widget ( i );
        if ( w!=toKeep ) closePage ( w );
    }
    tabWidget->blockSignals(previous);
    emit currentPageChanged();
    QApplication::restoreOverrideCursor();
}

void SKGMainPanel::saveDefaultState()
{
    SKGTRACEIN ( 1, "SKGMainPanel::saveDefaultState" );
    SKGError err;

    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    SKGTabPage* toSave= ( SKGTabPage* ) tabWidget->currentWidget();
    if ( toSave )
    {
        //Get bookmarks uuid
        QString uuid=toSave->getBookmarkID();

        //Reset bookmarks uuid to overwrite context state
        toSave->setBookmarkID ( "" );

        //Overwrite
        toSave->overwrite ( false );

        //Set original bookmarks uuid
        toSave->setBookmarkID ( uuid );
    }
    QApplication::restoreOverrideCursor();
}

void SKGMainPanel::overwriteBookmarkState()
{
    SKGTRACEIN ( 1, "SKGMainPanel::overwriteBookmarkState" );
    SKGError err;

    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    SKGTabPage* toSave= ( SKGTabPage* ) tabWidget->currentWidget();
    if ( toSave )
    {
        //Get bookmarks uuid
        QString uuid=toSave->getBookmarkID();
        if ( !uuid.isEmpty() )
        {
            //Overwrite
            toSave->overwrite ( false );
        }
    }
    QApplication::restoreOverrideCursor();
}

void SKGMainPanel::resetDefaultState()
{
    SKGTRACEIN ( 1, "SKGMainPanel::resetDefaultState" );
    SKGError err;

    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    SKGTabPage* toSave= ( SKGTabPage* ) tabWidget->currentWidget();
    if ( toSave )
    {
        QString name=toSave->getDefaultStateAttribute();
        if ( !name.isEmpty() )
        {
            SKGBEGINTRANSACTION ( *currentBankDocument, i18nc("Noun, name of the user action", "Reset default state" ), err );
            err=currentBankDocument->setParameter ( name, "" );
        }
    }
    QApplication::restoreOverrideCursor();

    //status bar
    if ( err.isSucceeded() ) err=SKGError(0, i18nc("Successful message after an user action", "Default state has been reset" ) );
    displayErrorMessage ( err );
}

void SKGMainPanel::addTab()
{
    SKGTRACEIN ( 1, "SKGMainPanel::addTab" );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    SKGTabPage* cPage= currentPage();
    if ( cPage )
    {
        openPage ( getPluginByName ( cPage->objectName() ) );
    }
    QApplication::restoreOverrideCursor();
}

bool SKGMainPanel::eventFilter ( QObject *object, QEvent *event )
{
    if ( event && object )
    {
        QResizeEvent *resizeEvent = dynamic_cast<QResizeEvent*> ( event );
        if ( resizeEvent )
        {
            QSize newSize=resizeEvent->size ();

            //Compute icon size
            int s=qMax ( qMin ( newSize.width() /5,64 ),16 );
            ui.kContextList->setIconSize ( QSize ( s, s ) );
        }
    }
    return false;
}

QString SKGMainPanel::getTipOfDayFileName()
{
    return tipsFile.fileName();
}

void SKGMainPanel::onTipOfDay()
{
    SKGTRACEIN ( 1, "SKGMainPanel::onTipOfDay" );
    KTipDatabase* tipDatabase=new KTipDatabase(getTipOfDayFileName()); //This pointer is deleted by destructor of KTipDialog
    KTipDialog* tipsDialog=new KTipDialog ( tipDatabase, this );
    tipsDialog->setAttribute ( Qt::WA_DeleteOnClose );
    tipsDialog->show();
}

void SKGMainPanel::onTrayActivated ( QSystemTrayIcon::ActivationReason reason )
{
    SKGTRACEIN ( 1, "SKGMainPanel::onTrayActivated" );
    if ( reason==KSystemTrayIcon::Trigger )
    {
        setVisible ( !isVisible() );
    }
}

void SKGMainPanel::notify ( int iTransaction )
{
    SKGTRACEIN ( 1, "SKGMainPanel::notify" );
    SKGTRACEL ( 1 ) << "iTransaction=" << iTransaction << endl;

    //Notify
    SKGObjectBase transaction ( currentBankDocument, "doctransaction", iTransaction );
    if ( iTransaction==0 || transaction.getAttribute ( "t_mode" ) !="R" )
    {
        QStringList msg;
        currentBankDocument->getMessages ( iTransaction, msg, false );
        int nbMessages=msg.count();
        if ( nbMessages )
        {
            //Build message
            QString message;
            for ( int i=0; i<nbMessages; ++i )
            {
                if ( i!=0 ) message+="<br>";
                message+=msg.at ( i );
            }

            if (nbMessages<20)
            {
                //Display a simple notification
                KNotification *notify = new KNotification ( KCmdLineArgs::appName()+"_info_event" , this );
                notify->setText ( message );
                notify->sendEvent();
            }
            else
            {
                //Too many message, display a warning panel
                KMessageBox::information ( SKGMainPanel::getMainPanel(), message, i18nc ("Noun",  "Notification" ) );
            }
        }
    }
}

void SKGMainPanel::changeEvent ( QEvent * e )
{
    KParts::MainWindow::changeEvent ( e );
}

bool SKGMainPanel::iconInSystemtray()
{
    return skgbasegui_settings::icon_in_system_tray();
}

QLabel* SKGMainPanel::statusNormalMessage() const
{
    return kNormalMessage;
}

void SKGMainPanel::displayErrorMessage ( const SKGError& iError )
{
    SKGTRACEIN ( 1, "SKGMainPanel::displayErrorMessage" );
    QWidget* parent=static_cast<SKGMainPanel*>(QApplication::activeWindow());

    if ( iError.isFailed() )
    {
        QApplication::setOverrideCursor ( QCursor ( Qt::ArrowCursor ) );
        int rc=KMessageBox::Cancel;
        if ( iError.getHistoricalSize() ) rc=KMessageBox::warningContinueCancel ( parent, iError.getFullMessage(),
                                                 i18nc ("Question",  "Error" ),
                                                 KStandardGuiItem::cont(),
                                                 KGuiItem ( i18nc ("Noun", "History" ), KIcon ( "dialog-information" ) ) );
        if ( rc==KMessageBox::Cancel )
        {
            KMessageBox::error ( parent, iError.getFullMessageWithHistorical(), i18nc ("Noun", "Error" ) );
        }
        QApplication::restoreOverrideCursor();
    }
    else
    {
        //Status bar
        SKGMainPanel *parent2 = qobject_cast<SKGMainPanel *> ( parent );
        if ( parent2 )
        {
            QLabel* label=parent2->statusNormalMessage();
            QString message=iError.getMessage();
            if ( label && !message.isEmpty() ) label->setText ( message );
        }
    }
}

void SKGMainPanel::onCancelCurrentAction()
{
    SKGMainPanel::currentActionCanceled=true;
}

void SKGMainPanel::onQuitAction()
{
    //Bug 2777697: To be sure that all page modifications are closed
    closeAllPages();
    //Bug 2777697:

    qApp->closeAllWindows();
}

int SKGMainPanel::progressBarCallBack ( int iPos, void* iProgressBar )
{
    QProgressBar* progressBar=NULL;
    QPushButton* button=NULL;
    doublePointer* pointers= ( doublePointer* ) iProgressBar;
    if ( pointers )
    {
        progressBar= static_cast<QProgressBar*>(pointers->p1);
        button= static_cast<QPushButton*>(pointers->p2);
    }

    if ( progressBar ) progressBar->setValue ( iPos );
    if ( button ) button->setEnabled ( iPos>0 && iPos<100 );

    SKGMainPanel::currentActionCanceled=false;
    if (iPos!=0 && iPos!=100) qApp->processEvents(QEventLoop::AllEvents, 500);
    return ( SKGMainPanel::currentActionCanceled ? 1 : 0 );
}

QString SKGMainPanel::getSaveFileName ( const KUrl & startDir, const QString & filter, QWidget * parent, const QString & caption )
{
    QString fileName=KFileDialog::getSaveFileName ( startDir, filter , parent, caption );
    if ( fileName.isEmpty() ) return "";
    QFile f ( fileName );
    if ( f.exists() && KMessageBox::warningContinueCancel ( parent,
            i18nc("Question", "File <b>%1</b> already exists. Do you really want to overwrite it?", fileName ),
            i18nc("Question", "Warning" ),
            KGuiItem ( i18nc ("Verb",  "Save" ), KIcon ( "document-save" ) ) ) !=KMessageBox::Continue ) return "";

    return fileName;

}

void SKGMainPanel::fillWithDistinctValue (
    QWidget* iWidget,
    SKGDocument* iDoc,
    const QString& iTable,
    const QString& iAttribut,
    const QString& iWhereClause )
{
    SKGTRACEIN ( 10, "SKGMainPanel::fillWithDistinctValue" );

    if ( iWidget )
    {
        //Get list
        QStringList list;
        SKGServices::getDistinctValues ( iDoc, iTable, iAttribut, iWhereClause, list );
        if ( list.count() && !list.at ( 0 ).isEmpty() ) list.insert ( 0, "" );

        //Fill completion
        KComboBox *kcmb = qobject_cast<KComboBox *> ( iWidget );
        if ( kcmb )
        {
            //Sorting list
            //Correction bug 202341 vvv
            qSort ( list.begin(), list.end(), naturalLessThan );
            //Correction bug 202341 ^^^

            //Fill combo
            kcmb->clear();
            kcmb->addItems ( list );

            KCompletion *comp = kcmb->completionObject();
            if ( comp )
            {
                comp->setIgnoreCase ( true );
                comp->setSoundsEnabled ( true );
                comp->clear ();
                comp->insertItems ( list );
            }
        }
    }
}

bool SKGMainPanel::naturalLessThan ( const QString &s1, const QString &s2 )
{
    return ( KStringHandler::naturalCompare ( s1, s2, Qt::CaseInsensitive ) <0 );
}

SKGMainPanel* SKGMainPanel::getMainPanel()
{
    return mainPanel;
}

void SKGMainPanel::onZoomChanged()
{
    SKGTabPage* toSave= currentPage();
    if (toSave)
    {
        toSave->setZoomPosition(zoomSelector->value());
        zoomSelector->setValue(toSave->zoomPosition()); //In case of a limit is reached
    }
}

#include "skgmainpanel.moc"
