/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A date editor with more features.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgdatepickerpopup.h"

#include <KDatePicker>
#include <KLocale>

#include <QtCore/QDateTime>
#include <QtGui/QWidgetAction>

/**
 * The date picker action
 */
class SKGDatePickerAction : public QWidgetAction
{
public:
    /**
     * Constructor
     * @param widget the date picker
     * @param parent the parent
     */
    SKGDatePickerAction( KDatePicker *widget, QObject *parent )
            : QWidgetAction( parent ),
            mDatePicker( widget ), mOriginalParent( widget->parentWidget() ) {
    }

protected:
    /**
     * This function is called whenever the action is added to a container widget that supports custom widgets.
     * If you don't want a custom widget to be used as representation of the action in the specified parent widget then 0 should be returned
     * @param parent the parent
     * @return the date picker
     */
    QWidget *createWidget( QWidget *parent ) {
        mDatePicker->setParent( parent );
        return mDatePicker;
    }

    /**
     * This function is called whenever the action is removed from a container widget that displays the action using a custom widget previously created using createWidget().
     * The default implementation hides the widget and schedules it for deletion using QObject::deleteLater().
     * @param widget the widget
     */
    void deleteWidget( QWidget *widget ) {
        if ( widget != mDatePicker ) {
            return;
        }

        mDatePicker->setParent( mOriginalParent );
    }

private:
    KDatePicker *mDatePicker;
    QWidget *mOriginalParent;
};

SKGDatePickerPopup::SKGDatePickerPopup( Items items, const QDate &date, QWidget *parent )
        : QMenu( parent )
{
    mItems = items;

    mDatePicker = new KDatePicker( this );
    mDatePicker->setCloseButton( false );

    connect( mDatePicker, SIGNAL( dateEntered( const QDate& ) ), SLOT( slotDateChanged( const QDate& ) ) );
    connect( mDatePicker, SIGNAL( dateSelected( const QDate& ) ), SLOT( slotDateChanged( const QDate& ) ) );
    mDatePicker->setDate( date );
    buildMenu();
}

void SKGDatePickerPopup::buildMenu()
{
    if ( isVisible() ) return;
    clear();

    if ( mItems & DatePicker ) {
        addAction( new SKGDatePickerAction( mDatePicker, this ) );

        if ( ( mItems & NoDate ) || ( mItems & Words ) ) {
            addSeparator();
        }
    }

    if ( mItems & Words ) {
        addAction( i18nc( "@option today", "Today" ), this, SLOT( slotToday() ) );
        addAction( i18nc( "@option tomorrow", "Tomorrow" ), this, SLOT( slotTomorrow() ) );
        addAction( i18nc( "@option next week", "Next Week" ), this, SLOT( slotNextWeek() ) );
        addAction( i18nc( "@option next month", "Next Month" ), this, SLOT( slotNextMonth() ) );

        if ( mItems & NoDate ) {
            addSeparator();
        }
    }

    if ( mItems & NoDate ) {
        addAction( i18nc( "@option do not specify a date", "No Date" ), this, SLOT( slotNoDate() ) );
    }
}

KDatePicker *SKGDatePickerPopup::datePicker() const
{
    return mDatePicker;
}

void SKGDatePickerPopup::setDate( const QDate &date )
{
    if (mDatePicker) mDatePicker->setDate( date );
}

void SKGDatePickerPopup::slotDateChanged( const QDate &date )
{
    emit dateChanged( date );
    hide();
}

void SKGDatePickerPopup::slotToday()
{
    emit dateChanged( QDate::currentDate() );
}

void SKGDatePickerPopup::slotTomorrow()
{
    emit dateChanged( QDate::currentDate().addDays( 1 ) );
}

void SKGDatePickerPopup::slotNoDate()
{
    emit dateChanged( QDate() );
}

void SKGDatePickerPopup::slotNextWeek()
{
    emit dateChanged( QDate::currentDate().addDays( 7 ) );
}

void SKGDatePickerPopup::slotNextMonth()
{
    emit dateChanged( QDate::currentDate().addMonths( 1 ) );
}

#include "skgdatepickerpopup.moc"
