# Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
#
# This file is part of Rast.
# See the file COPYING for redistribution information.
#

require "test/unit"
require "tempfile"

require "rast"
require "rast/filter"
require "rast/database-generatable"

module Rast
  class ApplicationTarTest < Test::Unit::TestCase
    include DatabaseGeneratable

    def test_invoke
      invoke_test_simple
      invoke_test_unsupported_magic_file
    end

    def invoke_test_simple
      create_options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, create_options)

      LocalDB.open(db_name, Rast::DB::RDWR) do |db|
        doc = db.create_document
        chain = FilterChain.new(doc)
        brigade = Brigade.new
        filename = File.join(MakeVariable::TOP_SRCDIR, "tests", "data",
                             "filter", "ja", "utf-8.txt.tar")
        File.open(filename) do |f|
          brigade.insert_tail(FileBucket.new(f))
          brigade.insert_tail(EOSBucket.new)
          chain.invoke(brigade, "application/x-tar")
          doc.commit
        end
      end
      LocalDB.open(db_name) do |db|
        assert_match(/tar テスト これは tar ファイルの1つ目です/,
                     db.get_text(1))
        assert_match(/tar テスト これは tar ファイルの2つ目です/,
                     db.get_text(1))
      end
    end

    def invoke_test_unsupported_magic_file
      create_options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, create_options)

      LocalDB.open(db_name, Rast::DB::RDWR) do |db|
        doc = db.create_document
        chain = FilterChain.new(doc)
        brigade = Brigade.new
        filename = File.join("tests", "data",
                             "filter", "ja", "open-office-org.sxw")
        tmp_file = Tempfile.new("ooo-tar")
        begin
          tmp_file.close(false)
          system("cd #{MakeVariable::TOP_SRCDIR}; tar cf #{tmp_file.path} #{filename}")
          File.open(tmp_file.path) do |f|
            brigade.insert_tail(FileBucket.new(f))
            brigade.insert_tail(EOSBucket.new)
            chain.invoke(brigade, "application/x-tar")
            doc.commit
          end
        ensure
          tmp_file.close(true)
        end
      end
      LocalDB.open(db_name) do |db|
        assert_match(/これはWriterファイルです/, db.get_text(1))
      end
    end

    def brigade_to_bucket_array(brigade)
      buckets = []
      brigade.each do |bucket|
        buckets.push(bucket)
      end
      return buckets
    end
  end
end
