/***************************************************************************
 *   Copyright (C) 2005 by Niklas Knutsson                                 *
 *   nq@altern.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "qalculateeditnamesdialog.h"
#include "qalculate_kde_utils.h"

#include <klocale.h>
#include <klistview.h>
#include <kpushbutton.h>
#include <qlayout.h>
#include <klineedit.h>
#include <qlabel.h>
#include <qapplication.h>
#include <kstdguiitem.h>

QalculateEditNamesDialog::QalculateEditNamesDialog(int item_type, QWidget *parent, const char *name) : KDialog(parent, name, true) {

	setCaption(i18n("Edit Names"));

	i_type = item_type;

	QVBoxLayout *layout = new QVBoxLayout(this, marginHint(), spacingHint());

	namesView = new KListView(this);
	namesView->addColumn(i18n("Name"));
	namesView->addColumn(i18n("Abbreviation"));
	namesView->setColumnAlignment(1, Qt::AlignCenter);
	namesView->addColumn(i18n("Plural"));
	namesView->setColumnAlignment(2, Qt::AlignCenter);
	namesView->addColumn(i18n("Reference"));
	namesView->setColumnAlignment(3, Qt::AlignCenter);
	namesView->addColumn(i18n("Avoid Input"));
	namesView->setColumnAlignment(4, Qt::AlignCenter);
	namesView->addColumn(i18n("Unicode"));
	namesView->setColumnAlignment(5, Qt::AlignCenter);
	namesView->addColumn(i18n("Suffix"));
	namesView->setColumnAlignment(6, Qt::AlignCenter);
	namesView->addColumn(i18n("Case Sensitive"));
	namesView->setColumnAlignment(7, Qt::AlignCenter);
	namesView->setSorting(-1);
	namesView->setItemsRenameable(true);
	namesView->setItemsMovable(true);
	namesView->setDragEnabled(true);
	namesView->setAcceptDrops(true);
	layout->addWidget(namesView);

	QHBoxLayout *buttonLayout = new QHBoxLayout(layout, spacingHint());
	newButton = new QPushButton(i18n("New"), this);
	buttonLayout->addWidget(newButton);
	deleteButton = new QPushButton(i18n("Delete"), this);
	deleteButton->setEnabled(false);
	buttonLayout->addWidget(deleteButton);
	buttonLayout->addItem(new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Minimum));
	buttonClose = new KPushButton(KStdGuiItem::close(), this);
	buttonClose->setAutoDefault(true);
	buttonClose->setDefault(true);
	buttonLayout->addWidget(buttonClose);

	connect(namesView, SIGNAL(selectionChanged()), this, SLOT(nameSelectionChanged()));
	connect(buttonClose, SIGNAL(clicked()), this, SLOT(accept()));
	connect(newButton, SIGNAL(clicked()), this, SLOT(newName()));
	connect(deleteButton, SIGNAL(clicked()), this, SLOT(deleteName()));
	connect(namesView, SIGNAL(clicked(QListViewItem*, const QPoint&, int)), this, SLOT(itemClicked(QListViewItem*, const QPoint&, int)));
	connect(namesView->renameLineEdit(), SIGNAL(textChanged(const QString&)), this, SLOT(nameChanged(const QString&)));

}

QalculateEditNamesDialog::~QalculateEditNamesDialog() {}

void QalculateEditNamesDialog::newName() {
	ExpressionName name;
	QListViewItem *i = new KListViewItem(namesView, namesView->lastChild(), "", i18n("No"), i18n("No"), i18n("No"), i18n("No"), i18n("No"), i18n("No"), i18n("No"));
	namesView->setSelected(i, true);
	qApp->processEvents();
	namesView->rename(i, 0);
}

void QalculateEditNamesDialog::setNames(ExpressionItem *item, const QString &namestr, bool names_edited) {

	if(!names_edited) {
		deleteButton->setEnabled(false);
		namesView->clear();
	}
	if(!names_edited && item && item->countNames() > 0) {
		QListViewItem *i_prev = NULL;
		for(size_t index = 1; index <= item->countNames(); index++) {
			const ExpressionName *ename = &item->getName(index);
			QListViewItem *i = new KListViewItem(namesView, i_prev);
			i_prev = i;
			if(ename->abbreviation)
				i->setText(1, i18n("Yes"));
			else
				i->setText(1, i18n("No"));
			if(ename->plural)
				i->setText(2, i18n("Yes"));
			else
				i->setText(2, i18n("No"));
			if(ename->reference)
				i->setText(3, i18n("Yes"));
			else
				i->setText(3, i18n("No"));
			if(ename->avoid_input)
				i->setText(4, i18n("Yes"));
			else
				i->setText(4, i18n("No"));
			if(ename->unicode)
				i->setText(5, i18n("Yes"));
			else
				i->setText(5, i18n("No"));
			if(ename->suffix)
				i->setText(6, i18n("Yes"));
			else
				i->setText(6, i18n("No"));
			if(ename->case_sensitive)
				i->setText(7, i18n("Yes"));
			else
				i->setText(7, i18n("No"));
			if(index == 1 && !namestr.isEmpty()) {
				i->setText(0, namestr);
			} else {
				i->setText(0, ename->name.c_str());
			}
		}
	} else if(!namestr.isEmpty()) {
		if(names_edited) {
			QListViewItemIterator it(namesView);
			if(it.current())
				it.current()->setText(0, namestr);
		} else {
			ExpressionName ename(namestr.ascii());
			QListViewItem *i = new KListViewItem(namesView, ename.name.c_str());
			if(ename.abbreviation)
				i->setText(1, i18n("Yes"));
			else
				i->setText(1, i18n("No"));
			if(ename.plural)
				i->setText(2, i18n("Yes"));
			else
				i->setText(2, i18n("No"));
			if(ename.reference)
				i->setText(3, i18n("Yes"));
			else
				i->setText(3, i18n("No"));
			if(ename.avoid_input)
				i->setText(4, i18n("Yes"));
			else
				i->setText(4, i18n("No"));
			if(ename.unicode)
				i->setText(5, i18n("Yes"));
			else
				i->setText(5, i18n("No"));
			if(ename.suffix)
				i->setText(6, i18n("Yes"));
			else
				i->setText(6, i18n("No"));
			if(ename.case_sensitive)
				i->setText(7, i18n("Yes"));
			else
				i->setText(7, i18n("No"));
		}
	}
}
void QalculateEditNamesDialog::saveNames(ExpressionItem *item, const QString &str) {
	item->clearNames();
	QListViewItemIterator it(namesView);
	if(it.current()) {
		bool b = false;
		while(it.current()) {
			ExpressionName ename;
			if(b || str.isEmpty())
				ename.name = it.current()->text(0).ascii();
			else
				ename.name = str.ascii();
			ename.abbreviation = (it.current()->text(1) == i18n("Yes"));
			ename.plural = (it.current()->text(2) == i18n("Yes"));
			ename.reference = (it.current()->text(3) == i18n("Yes"));
			ename.avoid_input = (it.current()->text(4) == i18n("Yes"));
			ename.unicode = (it.current()->text(5) == i18n("Yes"));
			ename.suffix = (it.current()->text(6) == i18n("Yes"));
			ename.case_sensitive = (it.current()->text(7) == i18n("Yes"));
			item->addName(ename);
			++it;
			b = true;
		}
	} else {
		item->addName(str.ascii());
	}
}
void QalculateEditNamesDialog::setNamesLE(QLineEdit *entry, QLabel *label) {
	int index = 0;
	QString str;
	QListViewItemIterator it(namesView);
	while(it.current()) {
		if(index == 0) {
			entry->setText(it.current()->text(0));
		} else {
			if(index == 1) {
				str = "+ ";
			} else {
				str += ", ";
			}
			str += it.current()->text(0);
		}
		++it;
		index++;
	}
	label->setText(str);
}
bool QalculateEditNamesDialog::isEmpty() {
	return namesView->childCount() <= 0;
}

void QalculateEditNamesDialog::deleteName() {
	namesView->removeItem(namesView->selectedItem());
}

void QalculateEditNamesDialog::itemClicked(QListViewItem *i, const QPoint &, int c) {
	if(i && c > 0) {
		if(i->text(c) == i18n("Yes"))
			i->setText(c, i18n("No"));
		else
			i->setText(c, i18n("Yes"));
		if(c == 1)
			i->setText(7, i->text(c));
	}
}

void QalculateEditNamesDialog::nameSelectionChanged() {
	deleteButton->setEnabled(namesView->selectedItem() != NULL);
}

void QalculateEditNamesDialog::nameChanged(const QString &name) {
	if(name.isNull()) return;
	namesView->renameLineEdit()->blockSignals(true);
	switch(i_type) {
		case TYPE_VARIABLE: {
			if(!CALCULATOR->variableNameIsValid(name.ascii())) {
				namesView->renameLineEdit()->setText(CALCULATOR->convertToValidVariableName(name.ascii()).c_str());
			}
			break;
		}
		case TYPE_UNIT: {
			if(!CALCULATOR->unitNameIsValid(name.ascii())) {
				namesView->renameLineEdit()->setText(CALCULATOR->convertToValidUnitName(name.ascii()).c_str());
			}
			break;
		}
		case TYPE_FUNCTION: {
			if(!CALCULATOR->functionNameIsValid(name.ascii())) {
				namesView->renameLineEdit()->setText(CALCULATOR->convertToValidFunctionName(name.ascii()).c_str());
			}
			break;
		}
	}
	namesView->renameLineEdit()->blockSignals(false);
}


#include "qalculateeditnamesdialog.moc"
