/*************************************************************************
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * Copyright 2000, 2010 Oracle and/or its affiliates.
 *
 * OpenOffice.org - a multi-platform office productivity suite
 *
 * This file is part of OpenOffice.org.
 *
 * OpenOffice.org is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * OpenOffice.org is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with OpenOffice.org.  If not, see
 * <http://www.openoffice.org/license.html>
 * for a copy of the LGPLv3 License.
 *
 ************************************************************************/

#ifndef _BGFX_RANGE_B3DRANGE_HXX
#define _BGFX_RANGE_B3DRANGE_HXX

#include <basegfx/vector/b3dvector.hxx>
#include <basegfx/point/b3dpoint.hxx>
#include <basegfx/tuple/b3dtuple.hxx>
#include <basegfx/range/basicrange.hxx>

namespace basegfx
{
    // predeclarations
    class B3IRange;
    class B3DHomMatrix;

    class B3DRange
    {
        typedef ::basegfx::BasicRange< double, DoubleTraits >	MyBasicRange;

        MyBasicRange			maRangeX;
        MyBasicRange			maRangeY;
        MyBasicRange			maRangeZ;

    public:		
        B3DRange() 
        {
        }
        
        explicit B3DRange(const B3DTuple& rTuple)
        :	maRangeX(rTuple.getX()),
            maRangeY(rTuple.getY()),
            maRangeZ(rTuple.getZ())
        {
        }
        
        B3DRange(double x1,
                 double y1,
                 double z1,
                 double x2,
                 double y2,
                 double z2)
        :	maRangeX(x1),
            maRangeY(y1),
            maRangeZ(z1)
        {
            maRangeX.expand(x2);
            maRangeY.expand(y2);
            maRangeZ.expand(z2);
        }

        B3DRange(const B3DTuple& rTuple1,
                 const B3DTuple& rTuple2)
        :	maRangeX(rTuple1.getX()),
            maRangeY(rTuple1.getY()),
            maRangeZ(rTuple1.getZ())
        {
            expand(rTuple2);
        }
        
        B3DRange(const B3DRange& rRange)
        :	maRangeX(rRange.maRangeX),
            maRangeY(rRange.maRangeY),
            maRangeZ(rRange.maRangeZ)
        {
        }

        explicit B3DRange(const B3IRange& rRange);

        bool isEmpty() const 
        {
            return (
                maRangeX.isEmpty() 
                || maRangeY.isEmpty() 
                || maRangeZ.isEmpty()
                );
        }

        void reset() 
        { 
            maRangeX.reset(); 
            maRangeY.reset(); 
            maRangeZ.reset(); 
        }
        
        bool operator==( const B3DRange& rRange ) const 
        { 
            return (maRangeX == rRange.maRangeX 
                && maRangeY == rRange.maRangeY
                && maRangeZ == rRange.maRangeZ); 
        }

        bool operator!=( const B3DRange& rRange ) const 
        { 
            return (maRangeX != rRange.maRangeX 
                || maRangeY != rRange.maRangeY
                || maRangeZ != rRange.maRangeZ); 
        }

        B3DRange& operator=(const B3DRange& rRange) 
        { 
            maRangeX = rRange.maRangeX; 
            maRangeY = rRange.maRangeY; 
            maRangeZ = rRange.maRangeZ; 
            return *this; 
        }

        bool equal(const B3DRange& rRange) const
        {
            return (maRangeX.equal(rRange.maRangeX) 
                    && maRangeY.equal(rRange.maRangeY)
                    && maRangeZ.equal(rRange.maRangeZ)); 
        }

        double getMinX() const
        {
            return maRangeX.getMinimum();
        }

        double getMinY() const
        {
            return maRangeY.getMinimum();
        }

        double getMinZ() const
        {
            return maRangeZ.getMinimum();
        }

        double getMaxX() const
        {
            return maRangeX.getMaximum();
        }

        double getMaxY() const
        {
            return maRangeY.getMaximum();
        }

        double getMaxZ() const
        {
            return maRangeZ.getMaximum();
        }

        double getWidth() const
        {
            return maRangeX.getRange();
        }

        double getHeight() const
        {
            return maRangeY.getRange();
        }

        double getDepth() const
        {
            return maRangeZ.getRange();
        }

        B3DPoint getMinimum() const
        {
            return B3DPoint(
                maRangeX.getMinimum(),
                maRangeY.getMinimum(),
                maRangeZ.getMinimum()
                );
        }
        
        B3DPoint getMaximum() const
        {
            return B3DPoint(
                maRangeX.getMaximum(),
                maRangeY.getMaximum(),
                maRangeZ.getMaximum()
                );
        }

        B3DVector getRange() const
        {
            return B3DVector(
                maRangeX.getRange(),
                maRangeY.getRange(),
                maRangeZ.getRange()
                );
        }
    
        B3DPoint getCenter() const
        {
            return B3DPoint(
                maRangeX.getCenter(),
                maRangeY.getCenter(),
                maRangeZ.getCenter()
                );
        }
    
        double getCenterX() const
        {
            return maRangeX.getCenter();
        }
    
        double getCenterY() const
        {
            return maRangeY.getCenter();
        }
    
        double getCenterZ() const
        {
            return maRangeZ.getCenter();
        }
    
        bool isInside(const B3DTuple& rTuple) const
        {
            return (
                maRangeX.isInside(rTuple.getX()) 
                && maRangeY.isInside(rTuple.getY())
                && maRangeZ.isInside(rTuple.getZ())
                );
        }

        bool isInside(const B3DRange& rRange) const
        {
            return (
                maRangeX.isInside(rRange.maRangeX) 
                && maRangeY.isInside(rRange.maRangeY)
                && maRangeZ.isInside(rRange.maRangeZ)
                );
        }

        bool overlaps(const B3DRange& rRange) const
        {
            return (
                maRangeX.overlaps(rRange.maRangeX) 
                && maRangeY.overlaps(rRange.maRangeY)
                && maRangeZ.overlaps(rRange.maRangeZ)
                );
        }

        void expand(const B3DTuple& rTuple)
        {
            maRangeX.expand(rTuple.getX());
            maRangeY.expand(rTuple.getY());
            maRangeZ.expand(rTuple.getZ());
        }

        void expand(const B3DRange& rRange)
        {
            maRangeX.expand(rRange.maRangeX); 
            maRangeY.expand(rRange.maRangeY); 
            maRangeZ.expand(rRange.maRangeZ); 
        }

        void intersect(const B3DRange& rRange)
        {
            maRangeX.intersect(rRange.maRangeX); 
            maRangeY.intersect(rRange.maRangeY); 
            maRangeZ.intersect(rRange.maRangeZ); 
        }

        void grow(double fValue)
        {
            maRangeX.grow(fValue); 
            maRangeY.grow(fValue); 
            maRangeZ.grow(fValue); 
        }

        void transform(const B3DHomMatrix& rMatrix);
    };

    /** Round double to nearest integer for 3D range

        @return the nearest integer for this range
    */
    B3IRange fround(const B3DRange& rRange);
} // end of namespace basegfx


#endif /* _BGFX_RANGE_B3DRANGE_HXX */
