/*
 * Stuff pertaining to bouncing through socks proxies.
 *
 * written by Joshua J. Drake
 * on Nov. 4, 1998
 * 
 * fixed bugs, nov 14 99
 * added passwd auth, sep 26 00
 */

#include "irc.h"
#include "output.h"
#include "socks.h"
#include "vars.h"


#define SOCKS_DEBUG

u_char *
socks4_error(int cd)
{
   switch (cd)
     {
        case 91:
           return "rejected or failed";
           break;
        case 92:
           return "no identd";
           break;
        case 93:
           return "identd response != username";
           break;
        default:
           return "Unknown error";
     }
}

u_char *
socks5_error(int cd)
{
   switch (cd)
     {
        case SOCKS5_FAIL:
           return "Rejected or failed";
           break;
        case SOCKS5_AUTHORIZE:
           return "Connection not allowed by ruleset";
           break;
        case SOCKS5_NETUNREACH:
           return "Network unreachable";
           break;
        case SOCKS5_HOSTUNREACH:
           return "Host unreachable";
           break;
        case SOCKS5_CONNREF:
           return "Connection refused";
           break;
        case SOCKS5_TTLEXP:
           return "Time to live expired";
           break;
        case SOCKS5_BADCMND:
           return "Bad command";
           break;
        case SOCKS5_BADADDR:
           return "Bad address";
           break;
        default:
           return "Unknown error";
     }
}

/*
 * try to negotiate a SOCKS4 connection.
 *
 */
int
socks4_connect(s, server)
   int s;
   struct sockaddr_in server;
{
   u_char socksreq[512], *ptr;
   int red;

   ptr = socksreq;
   *(ptr++) = SOCKS4_VERSION;
   *(ptr++) = SOCKS_CONNECT;
   memcpy(ptr, &server.sin_port, 2);
   ptr += 2;
   memcpy(ptr, &server.sin_addr.s_addr, 4);
   ptr += 4;
   strcpy(ptr, username);
   ptr += strlen(username);
   *ptr = '\0';
   red = write(s, socksreq, 9 + strlen(username));
   if (red == -1)
     {
	put_error("Cannot write to socks proxy: %s", strerror(errno));
	return 0;
     }
   red = read(s, socksreq, 8);
   if (red == -1)
     {
	put_error("Cannot read from socks proxy: %s", strerror(errno));
	return 0;
     }
   if (socksreq[1] != 90)
     {
        put_error("Cannot connect to SOCKS4 proxy: %s", socks4_error(socksreq[1]));
        return 0;
     }
   return 1;
}






/*
 * send a socks auth request
 */
int
socks5_send_auth_req(s)
   int s;
{
   u_char req[8], *p;
   int rl, wl;
   
#ifdef SOCKS_DEBUG
   put_info("SOCKS5: sending supported auth type request");
#endif
   p = req;
   *p++ = SOCKS5_VERSION;
   *p++ = 2;
   *p++ = AUTH_NONE;
   *p++ = AUTH_PASSWD;
   
   rl = (int)(p - req);
   if ((wl = write(s, req, rl)) != rl)
     {
	put_error("SOCKS5: short write %d on authentication request: %s",
		  wl, strerror(errno));
	return 0;
     }
   return 1;
}

/*
 * read/analyze a socks response
 */
int
socks5_recv_auth_rep(s)
   int s;
{
   char rep[128];
   int rl;

#ifdef SOCKS_DEBUG
   put_info("SOCKS5: reading supported auth type response");
#endif
   memset(rep, 0, sizeof(rep));
   if ((rl = read(s, rep, sizeof(rep))) < 1)
     {
	put_error("SOCKS5: read failed during auth: %s", strerror(errno));
        return 0;
     }
#ifdef SOCKS_DEBUG
   put_info("SOCKS5: selected auth type: %x", rep[1]);
#endif

   /* report server desired authentication (if not none) */
   switch (rep[1])
     {
      case AUTH_NONE:
	return 1;
	/* all done heheh */
      case AUTH_PASSWD:
	return 2;
	/* :) */
      default:
	put_error("Cannot use SOCKS proxy, server wants type %x authentication.", rep[1]);
	return 0;
	/* :( */
     }
   /* not reached... */
   return 0;
}

/*
 * send the username/password
 */
int
socks5_send_userpass_req(s)
   int s;
{
   char req[512], *p;
   char *user, *pass;
   int rl, wl;
   
   /* try username and password */
   user = get_string_var(SOCKS_USERNAME_VAR);
   pass = get_string_var(SOCKS_PASSWORD_VAR);
   if (!user || !*user
       || !pass || !*pass)
     {
	put_error("SOCKS5: no username/passsword specified.");
	return 0;
     }
#ifdef SOCKS_DEBUG
   put_info("SOCKS5: sending user/pass request: %s/%s", user,pass);
#endif
   
   memset(req, 0, sizeof(req));
   p = req;
   *(p++) = 0x01;
   *(p++) = (char)strlen(user);
   strncpy(p, user, sizeof(req) - 1 - (p - req));
   p += strlen(user);
   *(p++) = strlen(pass);
   strncpy(p, pass, sizeof(req) - 1 - (p - req));
   p += strlen(pass);
   rl = (int)(p - req);
   
   if ((wl = write(s, req, rl)) != rl)
     {
	put_error("SOCKS5: short write %d during USER/PASS negotiation: %s", 
		  wl, strerror(errno));
	return 0;
     }
   return 1;
}

/*
 * read the user/pass response
 */
int
socks5_recv_userpass_rep(s)
   int s;
{
   char rep[128];
   int rl;
   
#ifdef SOCKS_DEBUG
   put_info("SOCKS5: reading user/pass response");
#endif
   memset(rep, 0, sizeof(rep));
   if ((rl = read(s, rep, sizeof(rep))) == -1)
     {
	put_error("SOCKS5: read failed during USER/PASS auth: %s", strerror(errno));
	return 0;
     }
   if (rep[1] != 0)
     {
	put_error("SOCKS5: USER or PASS incorrect.");
	return 0;
     }
#ifdef SOCKS_DEBUG
   put_info("SOCKS5: user/pass accepted");
#endif
   return 1;
}


/*
 * send a socks5 connect request...
 */
int
socks5_send_connect_req(s, server)
   int s;
   struct sockaddr_in server;
{
   char req[128], *p;
   int wl, rl;
   
   p = req;
   *p++ = SOCKS5_VERSION;
   *p++ = SOCKS_CONNECT;
   *p++ = 0;
   *p++ = SOCKS5_IPV4ADDR;
   memcpy(p, &(server.sin_addr.s_addr), 4);
   p += 4;
   memcpy(p, &(server.sin_port), 2);
   p += 2;
   
   put_info("SOCKS5: Connecting through proxy to: %s:%u...",
	    inet_ntoa(server.sin_addr), ntohs(server.sin_port));
   
   rl = (int)(p - req);
   if ((wl = write(s, req, rl)) != 10)
     {
	put_error("SOCKS5: short write %d on connect: %s", wl, strerror(errno));
	return 0;
     }
   return 1;
}


/*
 * receive a socks5 connect response
 */
int
socks5_recv_connect_rep(s)
   int s;
{
   char req[128], tb[256];
   int rl;
   struct sockaddr_in sin;
   unsigned short ts;
   
#ifdef SOCKS_DEBUG
   put_info("SOCKS5: reading connect response");
#endif
   if ((rl = read(s, req, sizeof(req))) < 1)
     {
	put_error("SOCKS5: read failed during bounce: %s", strerror(errno));
	return 0;
     }
   if (req[0] != SOCKS5_VERSION)
     {
	put_error("SOCKS5: This is not a SOCKS version 5 proxy.");
	return 0;
     }
   if (req[1] != SOCKS5_NOERR)
     {
	put_error("SOCKS5: server failed: %s", socks5_error(req[1]));
	return 0;
     }
   /*
    * parse the rest of the response..
    */
   switch (req[3])
     {
      case 1:
	memcpy(&(sin.sin_addr.s_addr), req+4, sizeof(sin.sin_addr.s_addr));
	memcpy(&(sin.sin_port), req+8, sizeof(sin.sin_port));
	put_info("SOCKS5: bounce successful, your address will be: %s:%d", 
		 inet_ntoa(sin.sin_addr), ntohs(sin.sin_port));
	break;
      case 3:
	ts = (u_short)req[4];
	memcpy(tb, req+5, MIN(ts, sizeof(tb)-1));
	tb[MIN(ts, sizeof(tb)-1)] = '\0';
	memcpy(&(sin.sin_port), req + ts + 5, sizeof(sin.sin_port));
	put_info("SOCKS5: bounce successful, your address will be: %s:%d",
		 tb, ntohs(sin.sin_port));
	break;
      case 4:
	/* don't report address of ipv6 addresses. */
	put_info("SOCKS bounce successful.");
	break;
      default:
	put_error("SOCKS5: Unknown address type: %x", req[3]);
	return 0;
     }
   return 1;
}


/*
 * try to negotiate a SOCKS5 connection. (with the socket/username, to the server)
 */
int
socks5_connect(s, server)
   int s;
   struct sockaddr_in server;
{
   /* propose desired authentication */
   if (!socks5_send_auth_req(s))
     return 0;

   /* get response */
   switch (socks5_recv_auth_rep(s))
     {
      case 1:
	/* no-auth */
	break;
      case 2:
	if (!socks5_send_userpass_req(s))
	  return 0;
	if (!socks5_recv_userpass_rep(s))
	  return 0;
	break;
      default:
	return 0;
     }
   
   /* try to bounce to target */
   if (!socks5_send_connect_req(s, server))
     return 0;
   
   /* read the response... */
   if (!socks5_recv_connect_rep(s))
     return 0;
   return 1;
}
