/*
 * Ninja IRC unique code.
 * 
 * written by Kraig Amador and Joshua J. Drake
 * 
 */

#include "irc.h"

#include "dma.h"
#include "ninja.h"
#include "info.h"
#include "friends.h"
#include "enemies.h"
#include "ckey.h"
#include "hosts.h"
#include "tabkey.h"
#include "orignick.h"
#include "how_many.h"

#include "notify.h"
#include "vars.h"
#include "dcc.h"
#include "output.h"
#include "server.h"
#include "ircaux.h"
#include "newio.h"

#include "nchk_up.h"

#ifdef HAVE_DIRENT_H
# include <dirent.h>
#endif

#if defined(ISC30) && defined(_POSIX_SOURCE)
# undef _POSIX_SOURCE
#include <sys/stat.h>
# define _POSIX_SOURCE
#else
# include <sys/stat.h>
#endif /* ICS30 || _POSIX_SOURCE */

/*
 * perhaps this stuff would be usefull elsewhere?
 */
#define ONE_SECOND	1
#define ONE_MINUTE	60
#define ONE_HOUR	ONE_MINUTE * 60
#define ONE_DAY		ONE_HOUR * 24
#define ONE_WEEK	ONE_DAY * 7
#define ONE_MONTH	ONE_DAY * 30	/* just an average */
#define ONE_YEAR	ONE_DAY * 365	/* not exact either */
/* this one has a minor adjustment, 8.5 days off.. */
#define ONE_DECADE	(ONE_YEAR * 10) - (ONE_DAY * 8)
/* unix epoch won't hold century, millenium, etc */
#define NUM_ONES	8
typedef struct __etime_stru 
{
   u_long sec;
   char *unit;
} etVal;
etVal et_stuff[NUM_ONES] = 
{
     { ONE_SECOND,	"s" },
     { ONE_MINUTE,	"m" },
     { ONE_HOUR,	"h" },
     { ONE_DAY,		"d" },
     { ONE_WEEK,	"w" },
     { ONE_MONTH,	"mo" },
     { ONE_YEAR,	"yr" },
     { ONE_DECADE,	"D" }, /* i doubt this will be used */
};


/* external routines */
extern	void	away _((u_char *, u_char *, u_char *));

/* static routines.. */
static	void	nchk_idleaway _((time_t));
static	void	nchk_dccs _((time_t));
static	void	nchk_delayop _((time_t));
static	void	free_delayop _((DelayOp *));

/* these should be gone soon... */
static char tmpbuf1[512]; /* make_nice(), un_nice() */
extern time_t last_server_check;

/* globals */
FILE *awayfile = NULL;
FILE *helpfile;
DelayOp *delayop_list = (DelayOp *) NULL;

int checked_for_update = 0;	/* did we already check? */

/*
 * return a pointer to a static string that
 * holds a float followed by the units
 */
u_char *
ninja_size(bytes)
   unsigned long bytes;
{
   double tbn_size;
   static u_char tmptxt[64];
   
   memset(tmptxt, 0, sizeof(tmptxt));
   if (bytes > 1048576)
     {
	tbn_size = (double)bytes / (double)1048576.0;
	sprintf(CP(tmptxt), "%2.4gMB", tbn_size); /* safe */
     }
   else if (bytes > 1024)
     {
	tbn_size = (double)bytes / (double)1024.0;
	sprintf(CP(tmptxt), "%2.4gKB", tbn_size); /* safe */
     }
   else
     sprintf(CP(tmptxt), "%luB", bytes); /* safe */
   return tmptxt;
}



/* this is a little helper routine for ninja_etime */
void
netime_helper(sec, divi, unit, str, sl)
   u_long *sec, divi;
   u_char *unit, *str;
   int sl;
{
   u_char ltb[16];
   u_long tl;
   
   if (*sec < divi || divi == 0)
     return;
   tl = *sec / divi;
   *sec %= divi;
   snprintf(ltb, sizeof(ltb)-1, "%lu%s", tl,
	    unit ? unit : empty_string);
   ltb[sizeof(ltb)-1] = '\0';
   my_strmcat(str, ltb, sl-1);
}
/*
 * ninja_etime() returns a pointer to the static string contains the seconds
 * ascii elapased time representation, defined as so:
 * XdXhXmXs where X represents the # of days, hours, minutes, and seconds
 * respectively.  If any of these X's are 0, that section is ommitted.
 */
u_char *
ninja_etime(seconds)
   unsigned long seconds;
{
   static u_char tmptxt[64];
   u_long ts = seconds;
   int i;

   tmptxt[0] = '\0';
   /* called with 0 */
   if (seconds <= 0)
     {
	my_strcpy(tmptxt, "0s"); /* safe */
	return tmptxt;
     }
   /* see how long! */
   for (i = NUM_ONES-1; i >= 0; i--)
     {
	if (ts >= et_stuff[i].sec)
	  netime_helper(&ts, 
			et_stuff[i].sec, et_stuff[i].unit,
			tmptxt, sizeof(tmptxt));
	if (ts == 0)
	  break;
     }
   return tmptxt;
}


/*
 * checks hostnames (resolves)
 * if you have the RESOLVE_HOSTS_VAR off, it won't resolve it...
 * if you have DOUBLE_CHECK_ADDRESSES variable on and host->ip != ip->host
 * then a warning message is "shout"ed and we use the IP...
 */
extern u_char *
ninja_host(numaddr)
   struct in_addr numaddr;
{
// #ifdef INET6
// # error "ninja_host() is not IPv6 ready... "
// #endif
   u_char *nums_n_dots;
   struct hostent *hp;
   static u_char ninjahost[257];	/* MAXDNAME + 1 */

   bzero(ninjahost, sizeof(ninjahost));
   nums_n_dots = inet_ntoa(numaddr);
   my_strncpy(ninjahost, nums_n_dots, sizeof(ninjahost)-1);
   
   if (!get_int_var(RESOLVE_VAR))
     return ninjahost;
   
   /* if we can't get the name from the ip, return the ip */
   if ((hp = gethostbyaddr((char *)&numaddr, sizeof(long), AF_INET)) == NULL)
     return ninjahost;
   my_strncpy(ninjahost, hp->h_name, sizeof(ninjahost)-1);
   
   /* if we should double check... do it */
   if (get_int_var(DOUBLE_CHECK_ADDRESSES_VAR))
     {
	struct in_addr chktmp;
	
	/* if they don't match, warn about it */
	if (!resolve_host(&chktmp, ninjahost) || chktmp.s_addr != numaddr.s_addr)
	  {
	     put_info("WARNING %s != %s, using the IP address.", ninjahost, nums_n_dots);
	     my_strncpy(ninjahost, nums_n_dots, sizeof(ninjahost)-1);
	  }
     }
   return ninjahost;
}

/*
 * returns something like:
 * 
 * converts caltime to localtime (timeval) and uses strftime with the passed
 * time format..
 * 
 */
u_char *
ninja_strftime(t, fmt)
   time_t *t;
   u_char *fmt;
{
   static u_char nfstr[64];
   struct tm *bdt;

   bzero(nfstr, sizeof(nfstr));
   bdt = localtime(t);
   strftime(nfstr, sizeof(nfstr)-1, fmt, bdt);
   return nfstr;
}

/*
 * returns something like:
 * Fri, Sep 28
 */
u_char *
ninja_date(t)
   time_t *t;
{
   return ninja_strftime(t, "%a, %b %d");
}

/*
 * returns something like:
 * 05:04AM
 */
u_char *
ninja_ctime(t)
   time_t *t;
{
   return ninja_strftime(t, "%I:%M%p");
}

/*
 * load save files...
 * 
 * if the NINJA_DIR doesn't exist, create it
 */
void
ninja_load_saves()
{
   u_char *fn = NINJA_DIR, *ffn;
   int i;
   
   /* check out the NINJA_DIR */
   if (*fn == '~')
     ffn = expand_twiddle(fn);
   else
     ffn = fn;

   /* if it's not there, then make it! */
   if (access(ffn, F_OK) != 0)
      mkdir(ffn, S_IWUSR | S_IRUSR | S_IXUSR);	/* private modes! */

   /* free if it needs it */
   if (ffn != fn)
     dma_Free(&ffn);
   
   /* load the NINJA_SAVE file */
   fn = NINJA_SAVE_FILE;
   if (*fn == '~')
     ffn = expand_twiddle(fn);
   else
     ffn = fn;
   put_info("loading saved information from %s...", ffn);
   load(empty_string, ffn, empty_string);
   if (ffn != fn)
     dma_Free(&ffn);
   
   /* load everything else */
   i = load_friends(0);
   if (i > 0)
     put_info("loaded %u friend%s...", i, PLURAL(i));
   i = load_enemies(1);
   if (i > 0)
     put_info("loaded %u enem%s...", i, Y_PLURAL(i));
   load_ckeys();
}



/*
 * print the usage for a Ninja IRC command
 * 
 * overhauled 5/9/1999 jjd
 * tweaked 9/28/01 jjd
 */
void
usage(cmd, arg)
   u_char *cmd, *arg;
{
   u_char ltb[64];
   extern int help_fp;
   
   /* copy and lowercase the command */
   my_strncpy(ltb, cmd, sizeof(ltb)-1);
   ltb[sizeof(ltb)-1] = '\0';
   lower(ltb);
   
   /* tell them the usage */
   put_info("usage: /%s %s", ltb, arg);
   
   /* tell them how to get details, unless we are in help stuff */
   if (!helpfile && !help_fp)
      put_info("for more information, try /help %s", ltb);
   else if (my_strcmp(ltb, "index") != 0)
     put_info("for a list of commands, try /help index");
}


void
do_server_lag_check(i, current)
   int i;
   time_t current;
{
   int of_server = from_server;
   
   from_server = i;
   server_list[i].lag_time = current;
#ifdef DEBUG_SERVER_LAG
   put_info("checking lag on %s at %s (was %u)...", server_list[i].itsname, 
	    ninja_ctime(&server_list[i].lag_time),
	    server_list[i].lag);
#endif
   /* ping or privmsg?! */
   if (get_int_var(ANTI_IDLE_VAR))
     send_to_server("PRIVMSG %s :LAG CHECK %d", server_list[i].nickname, server_list[i].lag_time);
   else
     send_to_server("PING LAG:%d %s", server_list[i].lag_time, get_server_itsname(i));
   server_list[i].in_ping = 1;
   from_server = of_server;
}

/*
 * check lag on all servers using the desired method (PING or PRIVMSG)
 * the method is configured via the ANTI_IDLE variable...
 * on: privmsg, off: ping
 */
static void
nchk_server_lag(current)
   time_t current;
{
   int i = 0;
   static time_t last_lag_check = 0;
   time_t lag_limit = get_int_var(LAG_LIMIT_VAR);
   
   /* no checking if there's nothing to check! */
   if (number_of_servers < 1)
     return;
   
#ifndef NFREQ_SERVER_LAG_CHECK
# define NFREQ_SERVER_LAG_CHECK 30
#endif
   /* if we're getting here too fast, do nothing till it's time */
   if ((current - last_lag_check) < NFREQ_SERVER_LAG_CHECK)
     return;
   last_lag_check = current;
   
   /* check 'em all! */
   for (i = 0; i < number_of_servers; i++)
     {
	/* well if they're connected and..
	 * not already checking that is..
	 */
	if (is_server_connected(i))
	  {
	     /* check to see if we surpassed the lag limit */
	     if (number_of_servers > 1 
		 && server_list[i].in_ping)
	       {
		  if (lag_limit > 0)
		    {
		       time_t tdiff = time(NULL) - server_list[i].lag_time;
		       
		       if (tdiff > lag_limit)
			 {
			    close_server(i, "too much lag..");
			    put_info("Server lag too high, jumping to next server..");
			    server_list[i].in_ping = 0;
			    get_connected(i + 1);
			    continue;
			 }
		    }
	       }
	     else if (!server_list[i].in_ping) 	     /* check lag... */
	       do_server_lag_check(i, current);
	  }
     }
}

/*
 * checks if we're connected to a server, if not, we connect to one (or at least try)
 */
static void
nchk_server_connection(current)
   time_t current;
{
   int found = 0, oldserver;
   
#ifndef NFREQ_SERVER_CONNECT
# define NFREQ_SERVER_CONNECT 300
#endif
   /* only check so often.. */
   if ((current - last_server_check) < NFREQ_SERVER_CONNECT)
     return;
   last_server_check = current;
   
   /* we have to have a server to connect to one! */
   if (number_of_servers < 1)
     {
	put_info("You have no servers!  Why is this program even running?!");
	return;
     }
   
   /* connect to a server! */
   for (oldserver = 0; oldserver < number_of_servers; oldserver++)
     {
	if (!found || found == -1)
	  found = server_list[oldserver].read;
     }
   if (!found || found == -1)
     {
	put_info("None of the servers are connected, attempting to connect a server..");
	servercmd(NULL, "+", NULL);
     }
}

/*
 * this routine is now just a routine
 * for dispatching other ninja check routines...
 */
void
ninja_check(void)
{
   time_t current;	/* passed to most nchk_* routines so they don't have
			 * to keep calling time()
			 */
   static time_t last_check = 0;

   /* get the time.. */
   time(&current);
   
   /* check for an updated release! */
#ifndef CHECK_UPDATE_AS_OPTION
   if (!checked_for_update
       && (current - start_time) > 60)
     {
	checked_for_update = 1;
	ninja_check_update(0);
     }
#endif
# ifdef NON_BLOCKING_CONNECTS
   if (checked_for_update
       && !update_check_finished)
     ninja_update_chk_connect();
# endif     

   /* check server lag.. */
   nchk_server_lag(current);
   
   /* check tabkey expirations */
   nchk_tabkeys(current);
   
   /* check our idle time for auto away */
   nchk_idleaway(current);

#ifndef NFREQ_ORIGNICK
# define NFREQ_ORIGNICK 30
#endif
   /* check orignick */
   if ((current - last_check) > NFREQ_ORIGNICK)
     {
	last_check = current;
	nchk_orignick(NULL);
     }
   
   /* check server connection.. this should be last */
   nchk_server_connection(current);
   
   /* check for idle/too slow DCC's */
   nchk_dccs(current);
   
   /* check for delayop execution */
   nchk_delayop(current);
}

/*
 * closes idle/too slow DCCs
 */
static void
nchk_dccs(current)
   time_t current;
{
   DCC_list *dccp;
   extern char *dcc_types[];
   
   /* go through the list and check.. */
   for (dccp = ClientList; dccp; dccp = dccp->next)
     {
	/* only if its not being deleted.. */
	if (!(dccp->flags & DCC_DELETE))
	  {
	     /* ..first for idle un-connected dcc's idling past DCC_IDLE_LIMIT... */
	     if (!(dccp->flags & DCC_ACTIVE))
	       {
		  /* its not connected, is it a file offer? */
		  if ((dccp->flags & DCC_WAIT)
		      && (current - dccp->lasttime) >= get_int_var(DCC_IDLE_LIMIT_VAR))
		    {
		       u_char *desc;
		       
		       if ((dccp->flags & DCC_TYPES) == DCC_FILEOFFER)
			 desc = strip_path(dccp->description);
		       else
			 desc = dccp->description;
		       if (!desc)
			 desc = "<none>";
		       
		       put_info("DCC: Closing idle %s:%s with %s.",
				dcc_types[dccp->flags & DCC_TYPES],
				desc, dccp->user);
		       
		       if (my_stricmp(dccp->user, "-NinjaIRC-") != 0)
			 send_to_server("NOTICE %s :DCC %s %s was unconnected too long, removing.",
					dccp->user,
					dcc_types[dccp->flags & DCC_TYPES],
					desc);
		       dccp->flags |= DCC_DELETE;
		    }
	       }
	     /* then for extra slow dcc sends.. */
	     else if ((dccp->flags & DCC_TYPES) == DCC_FILEOFFER && dccp->minimum_speed >= 0.001)
	       {
		  time_t currtime = time(NULL); /* we need a more acturate time reading here */
		  double rate;
		  
		  if ((currtime -= dccp->starttime) == 0)
		    currtime = 1;
		  rate = (double)((dccp->bytes_sent - dccp->resume_offset) / currtime) / 1024.0;
		  if (rate < dccp->minimum_speed)
		    {
		       put_info("DCC: Closing slow %s %s to %s (%.2f kbytes/s, %.2f kbytes/s minimum)",
				dcc_types[dccp->flags & DCC_TYPES], strip_path(dccp->description),
				dccp->user, rate, dccp->minimum_speed);
		       send_to_server("NOTICE %s :DCC %s %s is too slow, (%.2f kbytes/s, %.2f kbytes/s minimum)",
				      dccp->user, dcc_types[dccp->flags & DCC_TYPES], strip_path(dccp->description),
				      rate, dccp->minimum_speed);
		       dccp->flags |= DCC_DELETE;
		       continue;
		    }
	       }
	  }
     }
}

/*
 * checks the delay mode list to see if we need to do something
 */
static void
nchk_delayop(current)
   time_t current;
{
   int oldserver, mcount = 0;
   DelayOp *delayop, *next;
   u_char *ptr, *dumb = NULL, *dumb_free = NULL;
   u_char mstr[5];
   u_char nbuf[1024];
   Channel *chan;
   Nick *n;
   
   for (delayop = delayop_list; delayop; delayop = next)
     {
	next = delayop->next;
	if (!delayop->nick /* i hope this never happens */
	    || (current - delayop->time) < delayop->delay)
	  continue;
	
	/* remove it from the list! */
	delayop = (DelayOp *) remove_from_list((List **)&delayop_list, delayop->nick);
	  
	/* check for the channel */
	chan = lookup_channel(delayop->channel, delayop->server, CHAN_NOUNLINK);
	if (!chan
	    || !(chan->status & CHAN_CHOP))
	  {
	     /* we'll go ahead and free it now */
	     free_delayop(delayop);
	     continue;
	  }
	
	/* we are on the channel and opped! */
	memset(nbuf, 0, sizeof(nbuf));
	memset(mstr, 0, sizeof(mstr));
	mcount = 0;
	dma_strcpy(&dumb, delayop->nick);	/* copy it */
	dumb_free = dumb;			/* save a pointer to free it */
	while ((ptr = my_strsep(&dumb, " ")))
	  {
	     /* op someone? */
	     if (*ptr == '@')
	       {
		  ptr++;
		  if ((n = find_nick(ptr, UNULL, chan->server, chan))
		      && !(n->status & NICK_CHOP))
		    {
		       if (*nbuf)
			 my_strmcat(nbuf, " ", sizeof(nbuf)-1);
		       my_strmcat(nbuf, n->nick, sizeof(nbuf)-1);
		       mstr[mcount++] = 'o';
		    }
	       }
	     else if (*ptr == '+')	/* voice someone? */
	       {
		  ptr++;
		  if ((n = find_nick(ptr, UNULL, chan->server, chan))
		      && !(n->status & NICK_VOICE))
		    {
		       if (*nbuf)
			 my_strmcat(nbuf, " ", sizeof(nbuf)-1);
		       my_strmcat(nbuf, n->nick, sizeof(nbuf)-1);
		       mstr[mcount++] = 'v';
		    }
	       }
	     if (mcount == 4)
	       {
		  oldserver = from_server;
		  from_server = delayop->server;
		  send_to_server("MODE %s +%s %s", chan->channel,
				 mstr, nbuf);
		  from_server = oldserver;
		  mcount = 0;
		  memset(mstr, 0, sizeof(mstr));
		  memset(nbuf, 0, sizeof(nbuf));
	       }
	  }
	if (mcount > 0
	    && *nbuf
	    && *mstr)
	  {
	     oldserver = from_server;
	     from_server = delayop->server;
	     send_to_server("MODE %s +%s %s", chan->channel,
			    mstr, nbuf);
	     from_server = oldserver;
	  }
	dma_Free(&dumb_free);
	free_delayop(delayop);
     }
}

static void
free_delayop(d)
   DelayOp *d;
{
   dma_Free(&d->nick);
   dma_Free(&d->channel);
   dma_Free(&d);
}
   

/*
   Add a line to the away log, w/time stamp
 */
void
#ifdef HAVE_STDARG_H
add_to_awaylog(char *format,...)
{
   va_list vl;
#else
add_to_awaylog(format, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10)
   char *format;
   char *arg1, *arg2, *arg3, *arg4, *arg5, *arg6, *arg7, *arg8, *arg9,
    *arg10;
{
#endif

   char log_this[BIG_BUFFER_SIZE + 1];
   time_t tootie;

#ifdef HAVE_STDARG_H
   va_start(vl, format);
   vsnprintf(log_this, sizeof(log_this), format, vl);
   va_end(vl);
#else
   snprintf(log_this, sizeof(log_this), format, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10);
#endif

   tootie = time((time_t *) 0);

   fprintf(awayfile, "[%.20s] %s\n", ninja_ctime(&tootie), log_this);
   if (from_server >= 0 && from_server < number_of_servers)
     server_list[from_server].away_count++;
   fflush(awayfile);
}

/*
 * close the away log.. add a time stamp of when it was closed..
 */
void
close_away_log(reason)
   u_char *reason;
{
   time_t right_now = time(NULL);
   int i, do_close = 1;

   if (awayfile == NULL)
     return;
   for (i = 0; i < number_of_servers; i++)
      if (server_list[i].away_set != 0)
	 do_close = 0;
   if (do_close)
     {
	fprintf(awayfile, "-+- Ninja IRC Away log closed at %s: %s\n",
		ninja_strftime(&right_now, "%I:%M%p on %a, %b %d"),
		reason);
	fclose(awayfile);
	awayfile = (FILE *) NULL;
     }
/*
 else
      put_info("Not closing the away logfile, it's in use still.");
 */
}

/*
 * open the away log and add a time stamp and reason why it was opened.. 
 */
void
open_away_log(reason)
   u_char *reason;
{
   char *ptr = NULL, afstr[512];
   time_t now = time(NULL);

   if (awayfile == NULL)
     {
	ptr = NINJA_AWAY_FILE;
	if (*ptr == '~')
	  ptr = expand_twiddle(ptr);
	strmcpy(afstr, ptr, sizeof(afstr)-1);
	awayfile = fopen(ptr, "a");
	if (ptr != NINJA_AWAY_FILE)
	  dma_Free(&ptr);
	if (awayfile == NULL)
	  {
	     put_info("Cannot open %s, msgs will not be logged..", afstr);
	     return;
	  }
	put_info("Saving logged messages to %s...", NINJA_AWAY_FILE);
	fprintf(awayfile, "-+- Ninja IRC Away log opened at %s: %s\n",
		ninja_strftime(&now, "%I:%M%p on %a, %b %d"),
		reason);
     }
/*   
   else
      put_info("Log file already open..");
 */
}


/*
 * the following two functions are used with ndcc/dcc stuff
 * 
 * UHH WTF?
 * 
 * translates : and \ to \: and \\
 */
char *
make_nice(char *line)
{
   char *ptr, ltb[5];
   
   ptr = line;
   *tmpbuf1 = '\0';
   while (*ptr)
     {
	if (*ptr == ':' || *ptr == '\\')
	  snprintf(ltb, sizeof(ltb), "\\%c", *ptr);
	else
	  snprintf(ltb, sizeof(ltb), "%c", *ptr);
	strmcat(tmpbuf1, ltb, sizeof(tmpbuf1)-1);
	ptr++;
     }
   strcpy(line, tmpbuf1);
   return (line);
}

/*
 * strips all \
 */
char *
un_nice(char *line)
{
   char *ptr = line, *tb = tmpbuf1;
   
   while (ptr && *ptr && ptr - line < sizeof(tmpbuf1) - 1)
     {
	if (*ptr == '\\')
	   ptr++;
	else
	  *tb++ = *ptr++;
     }
   *tb = '\0';
   strcpy(line, tmpbuf1);
   return (line);
}

/*
 * returns up to 250 characters (c)
 * usually it returns a string consisting of 
 * num c(s)
 */
char *
strfill(char c, int num)
{
   int i = 0;
   static char buf[256];

   if (num > sizeof(buf))
     num = sizeof(buf)-1;
   for (i = 0; i < num; i++)
     buf[i] = c;
   buf[num] = '\0';
   return buf;
}

/*
 * do a swap-a-roo on ansi graphics characters..
 * turn them into ascii equivalents
 */
char *
strip_8bit(str)
   char *str;
{
   char *tmpptr;
   int ndx = 0;
   static char bit_stripped[4096];

   if (!str)
      return empty_string;
   tmpptr = str;
   bzero(bit_stripped, sizeof(bit_stripped));
   while (*tmpptr)
     {
	switch (*tmpptr)
	  {
	   case '':
	     bit_stripped[ndx] = '-';
	     break;
	   case '':
	     bit_stripped[ndx] = '=';
	     break;
	   case '':
	   case '':
	   case '':
	   case '':
	     bit_stripped[ndx] = '+';
	     break;
	   case '':
	   case '':
	   case '':
	     bit_stripped[ndx] = '|';
	     break;
	   case '':
	   case '':
	     bit_stripped[ndx] = '*';
	     break;
	   case '':
	     bit_stripped[ndx] = '#';
	     break;
	   case '':
	     bit_stripped[ndx] = '~';
	     break;
	   case 0x1b:
	     bit_stripped[ndx] = *tmpptr;
	     break;
	   default:
	     bit_stripped[ndx] = (*tmpptr & 0x7f); /* mask out non-ascii */
	     break;
	  }
	ndx++;
	tmpptr++;
     }
   bit_stripped[ndx] = '\0';
   return bit_stripped;
}

/*
 * strip ansi sequences from a string
 */
char *
strip_ansi(char *str)
{
   char *tmpptr;
   int ndx = 0, in_ansi = 0, meat = 0;
   static char ansi_stripped[4096];

   if (!str)
      return empty_string;
   tmpptr = str;
   bzero(ansi_stripped, sizeof(ansi_stripped));
   while (*tmpptr)
     {
	if (*tmpptr == '')
	  {
	     if (*(tmpptr + 1) == '[')
	       {
		  tmpptr += 2;
		  meat = 1;
		  in_ansi = 1;
		  while (*tmpptr && in_ansi)
		    {
		       if (*tmpptr == 'm' || *tmpptr == 'J' || *tmpptr == 'H'
			   || *tmpptr == 'h' || *tmpptr == 'K')
			 {
			    if (meat)
			      {
				 meat = 0;
				 in_ansi = 0;
			      }
			    else
			      break;
			 }
		       if ((*tmpptr >= '0' && *tmpptr <= '9') || *tmpptr == ';')
			 meat = 1;
		       else
			 meat = 0;
		       tmpptr++;
		    }
	       }
	     else
	       tmpptr++;
	  }
	else
	  {
	     ansi_stripped[ndx] = *tmpptr;
	     ndx++;
	     tmpptr++;
	  }
     }
   ansi_stripped[ndx] = '\0';
   return ansi_stripped;
}

/*
 * strips the path from an explicit filename
 * just returns a pointer that points past the last /
 */
char *
strip_path(char *str)
{
   char *ptr;

   ptr = strrchr(str, '/');
   if (ptr == NULL)
      return str;
   else
      return ptr + 1;
}

/*
 * add a nickname to the delayed op/voice list
 */
void
adddelayop(u_char *channel, u_char *nick, int server, int voice, int delay)
{
   DelayOp *delayop;
   u_char *tnick, *tspace;
   
   /* if we have an existing line for the same
    * server/channel/time, then append it to that line 
    */
   for (delayop = delayop_list; delayop; delayop = delayop->next)
     {
	/* got a matching delay/chan/server? */
	if (delay == delayop->delay
	    && !my_stricmp(channel, delayop->channel)
	    && server == delayop->server)
	  {
	     /* not if the same nick/type is alread here! */
	     tnick = delayop->nick;
	     while (tnick)
	       {
		  /* if there's a space after, null it out for now.. */
		  tspace = strchr(tnick, ' ');
		  if (tspace)
		    *tspace = '\0';
		  
		  /* check it.. */
		  if (my_stricmp(tnick+1, nick) == 0
		      && ((voice && *tnick == '+')
			  || (!voice && *tnick == '@')))
		    {
		       /* gah! found it.. replace space and get out */
		       if (tspace)
			 *tspace = ' ';
		       return;
		    }
		  
		  /* replace the space & goto next nick.. */
		  if (tspace)
		    {
		       *tspace = ' ';
		       tnick = tspace + 1;
		    }
		  else
		    tnick = UNULL;
	       }

	     /* add it.. */
	     dma_strcat(&delayop->nick, " ");
	     if (voice)
	       dma_strcat(&delayop->nick, "+");
	     else
	       dma_strcat(&delayop->nick, "@");
	     dma_strcat(&delayop->nick, nick);
	     return;
	  }
     }
   
   /* otherwise we need to make a new entry */
   if ((delayop = (DelayOp *) dma_Malloc(sizeof(DelayOp))) != NULL)
     {
	/*
	delayop->nick = NULL;
	delayop->channel = NULL;
	delayop->next = NULL;
	 */
	if (voice)
	  dma_strcpy(&delayop->nick, "+");
	else
	  dma_strcpy(&delayop->nick, "@");
	dma_strcat(&delayop->nick, nick);
	dma_strcpy(&delayop->channel, channel);
	delayop->server = server;
	delayop->time = time((time_t *) 0);
	delayop->delay = delay;
	add_to_list((List **) & delayop_list, (List *) delayop);
     }
}

/*
 * returns if idle_limit is 0, from_server == -1, or we haven't been idling
 * long enough
 * sets servers that aren't marked away as away with the idleed away message.
 */
void
nchk_idleaway(current)
   time_t current;
{
   int old_server = from_server;
   u_char tmpbuf[1024];
   time_t been_idling = current - idle_time;
   time_t idlelimit;

   /* we have a server list to idle away */
   if (number_of_servers < 1)
     return;
   
   /* we also need to have a idle limit! */
   idlelimit = get_int_var(IDLE_LIMIT_VAR);
   if (!idlelimit)
     return;
   
   /* the idlelimit is in minutes */
   idlelimit *= 60;
   
   /* we need to have been idling longer than the limit (minutes).. */
   if (been_idling < idlelimit)
     return;
   
   /* if we have idled too long, set us away -all
    * 
    * set the away reason up..
    */
   snprintf(tmpbuf, sizeof(tmpbuf)-1, "-all Automagically away (%s idle)", ninja_etime(idlelimit));
   tmpbuf[sizeof(tmpbuf)-1] = '\0';
   /* and set us away! */
   for (from_server = 0; from_server < number_of_servers; from_server++)
     if ((server_list[from_server].away_set == 0)
	 && server_list[from_server].connected)
       away("AWAY", tmpbuf, NULL);
   from_server = old_server;
}

/*
 * this function breaks ninja irc, and apparently for no reason.
 * it causes core dumps on return from dcc_sendfiles
 int
 ninja_chkfiles(infiles, outfiles, totsize)
 u_char **infiles, *outfiles;
 unsigned long *totsize;
 {
 u_char *ptr, mask[1024], *fullname = NULL;
 struct stat stbuf;
 int selected;
 struct dirent **de;

 * loop while there are more files *
 while ((ptr = my_strsep(infiles, " ")) != NULL)
 {
 if (strlen(ptr) == 0)
 continue;
 if (*ptr == '/')
 strmcpy(mask, ptr, sizeof(mask));
 else if (*ptr == '~')
 {
 if ((fullname = expand_twiddle(ptr)) == NULL)
 continue;
 strmcpy(mask, fullname, sizeof(mask));
 dma_Free(&fullname);
 }
 else
 strmcpy(mask, ptr, sizeof(mask));
 put_info("Checking mask: %s", mask);
 if (stat_file(mask, &stbuf) == -1)
 {
 put_info("Unable to stat \"%s\": %s", mask, strerror(errno));
 continue;
 }
 if (stbuf.st_mode & S_IFDIR)
 {
 put_info("Sorry, can't send directories.  Maybe you should package it up?");
 continue;
 }
 if ((selected = scandir(mask, &de, sel, cmp)) == -1)
 {
 put_info("Unable to scan directory: %s", strerror(errno));
 continue;
 }
 snprintf(buf, sizeof(buf), "%s %s", buf, mask);
 }
 put_info("returning");
 return 0;
 }

 int
 sel(entry)
 struct dirent *entry;
 {
 return (*(entry->d_name) != '.');
 }

 int
 cmp(ent1, ent2)
 struct dirent **ent1, **ent2;
 {
 return (strcmp((*ent1)->d_name, (*ent2)->d_name));
 }

 */

/*  
   
 * end brokennes and it's couterparts
 *
 * there's more

   register_dcc_offer("NinjaIRC", "CHAT", "NinjaChat", "3472498438", "1025", NULL);
*/
