/*
 * Ninja DCC File offering
 * 
 * written by Kraig Amador and Joshua J. Drake
 * 
 * modified to comply with ircII-4.4Z code on 9/26/01
 */

#include "irc.h"

#include "ctcp.h"
#include "dma.h"
#include "dcc.h"
#include "ircaux.h"
#include "list.h"
#include "ninja.h"
#include "output.h"
#include "screen.h"
#include "server.h"
#include "friends.h"
#include "vars.h"
#include "window.h"

#include "ndcc.h"

#if defined(ISC30) && defined(_POSIX_SOURCE)
#undef _POSIX_SOURCE
#include <sys/stat.h>
#define _POSIX_SOURCE
#else
#include <sys/stat.h>
#endif /* ICS30 || _POSIX_SOURCE */

/* wait_prompt stuff */
static	void	ndcc_offer2 _((u_char *, u_char *));
static	void	ndcc_offer3 _((u_char *, u_char *));
static	void	ndcc_offer4 _((u_char *, u_char *));

/* /ndcc <X> commands */
static	void	ndcc_clear _((u_char *));
static	void	ndcc_list _((u_char *));
static	void	ndcc_load _((u_char *));
static	void	ndcc_offer _((u_char *));
static	void	ndcc_plist _((u_char *));
static	void	ndcc_remove _((u_char *));
static	void	ndcc_save _((u_char *));
static	void	ndcc_send _((u_char *));
static	void	ndcc_toggle _((u_char *));
static	void	ndcc_interval _((u_char *));

/* remote routines */
static	void	ndcc_remote_send _((u_char *, u_char *, u_char *, u_char *, u_char *));
static	void	ndcc_remote_list _((u_char *, u_char *, u_char *, u_char *, u_char *));
static	void	ndcc_remote_help _((u_char *, u_char *, u_char *, u_char *, u_char *));
static	void	ndcc_remote_version _((u_char *, u_char *, u_char *, u_char *, u_char *));

/* support routines */
static	void	make_pack _((u_char *, u_char *, float));
static	void	free_pack _(());
static	void	free_pack_file _(());
static	NDCCPack *new_pack _((u_char *, double));
static	int	ndcc_find_files _((u_char *, u_char *, u_char **, ssize_t *, NDCCPack *));
static	int	ndcc_do_filename _((u_char *, u_char *, int, u_char *, int, u_char *, int));
static	int	add_file_to_pack _((NDCCPack *, u_char *, u_char *, struct stat));
static	void	ndcc_set_pack_totals _((NDCCPack *, int, ssize_t));

/* the list head */
static NDCCPack *ndcc_packs = NULL;
static NDCCPack *ndcc_end = NULL;

/* globals for total packs/files/size */
static int total_packs = 0;
static off_t total_size = 0;
static int total_fs = 0;

/* external globals */
extern	int	in_ctcp_flag;

struct
{
   u_char *name;
   u_int uniq;
   void (*function) _((u_char *));
}
ndcc_commands[] =
{
     { UP("CLEAR"),	1, ndcc_clear },
     { UP("LIST"),	2, ndcc_list },
     { UP("LOAD"),	2, ndcc_load },
     { UP("OFFER"),	4, ndcc_offer },
     { UP("PLIST"),	1, ndcc_plist },
     { UP("REMOVE"),	1, ndcc_remove },
     { UP("SAVE"),	2, ndcc_save },
     { UP("SEND"),	2, ndcc_send },
     { UP("TOGGLE"),	1, ndcc_toggle },
     { UP("INTERVAL"),	1, ndcc_interval },
     { NULL, 0, (void (*)())NULL }
};

struct
{
   u_char *name;
   void (*function) _((u_char *, u_char *, u_char *, u_char *, u_char *));
   u_short quiet;
}
ndcc_remote[] =
{
     { UP("HELP"), ndcc_remote_help, 0 },
     { UP("LIST"), ndcc_remote_list, 0 },
     { UP("SEND"), ndcc_remote_send, 1 },
     { UP("VERSION"), ndcc_remote_version, 0 },
     { NULL, (void (*)())NULL }
};

/*
 * this routine dispatches the ndcc command based on the /ndcc argument
 */
void
process_ndcc(command, args, subargs)
   u_char *command, *args, *subargs;
{
   u_char *cmd;
   u_int i, len;
   
   if (!(cmd = next_arg(args, &args)))
     {
	usage("ndcc", "<command> [<args>]");
	return;
     }
   
   len = my_strlen(cmd);
   upper(cmd);
   for (i = 0; ndcc_commands[i].name != NULL; i++)
     {
	if (!my_strncmp(ndcc_commands[i].name, cmd, len))
	  {
	     if (len < ndcc_commands[i].uniq)
	       {
		  put_error("NDCC: Ambiguous command: %s", cmd);
		  return;
	       }
	     ndcc_commands[i].function(args);
	     return;
	  }
       }
   put_error("NDCC: Unknown command: %s", cmd);
}

/*
 * this comes in from /ctcp <me> NDCC stuff
 */
void
process_remote_ndcc(from, user, host, to, args)
   u_char *from, *user, *host, *to, *args;
{
   u_char *command, ltell[512], tb2[128];
   int i, flag, found;
   Friend *friend;
   
   /* try to get the command */
   command = next_arg(args, &args);
   upper(command);

   /* start the local tell string */
   snprintf(ltell, sizeof(ltell)-1, "NDCC: %s request received from %s (%s@%s)", command ? command : UP("Unknown"), from, user, host);
   ltell[sizeof(ltell)-1] = '\0';
   
   /* have some args? */
   if (!command)
     {
	strmcat(ltell, ", no command specified.", sizeof(ltell)-1);
	put_info("%s", ltell);
	if (!get_int_var(CLOAK_VAR))
	  send_to_server("NOTICE %s :Try /ctcp %s ndcc help", from, get_server_nickname(from_server));
	return;
     }
   
   /* to a channel? */
   if (*to == '#')
     {
	strmcat(ltell, " to ", sizeof(ltell)-1);
	strmcat(ltell, to, sizeof(ltell)-1);
	if (!get_int_var(NDCC_PUBLIC_VAR))
	  {
	     strmcat(ltell, ", ignored.", sizeof(ltell)-1);
	     put_info("%s", ltell);
	     return;
	  }
     }
   
   /* if we are using secure NDCC (friends only) */
   if (get_int_var(NDCC_SECURE_VAR))
     {
	/* check for friend */
	friend = get_friend_by_nuh(from, user, host);
	if (!friend || !(friend->modes & FL_UMODE_SECURE_NDCC))
	  {
	     snprintf(tb2, sizeof(tb2)-1, ", %s does not have access to secure ndcc.", from);
	     tb2[sizeof(tb2)-1] = '\0';
	     strmcat(ltell, tb2, sizeof(ltell)-1);
	     put_info("%s", ltell);
	     send_to_server("NOTICE %s :You do not have access to NDCC.", from);
	     return;
	  }
     }
   
   /* try to find the command */
   found = 0;
   for (i = 0; ndcc_remote[i].name != NULL; i++)
     if (!my_strcmp(ndcc_remote[i].name, command))
       {
	  found = 1;
	  break;
       }
   
   if (found)
     {
	strmcat(ltell, ", processing.", sizeof(ltell)-1);
	if (!ndcc_remote[i].quiet)
	  put_info("%s", ltell);
	flag = in_ctcp_flag;
	in_ctcp_flag = 0;
	ndcc_remote[i].function(from, user, host, to, args);
	in_ctcp_flag = flag;
     }
   else
     {
	strmcat(ltell, ", unknown command.", sizeof(ltell)-1);
	put_info("%s", ltell);
	send_to_server("NOTICE %s :Unknown NDCC command: %s", from, command);
     }
}


/*
 * the following couple of functions are used by 
 * make_pack() in conjunction with scandir()
 */
int
compar(ent1, ent2)
   const struct dirent **ent1, **ent2;
{
   return (my_strcmp((*ent1)->d_name, (*ent2)->d_name));
}
int
selectent(entry)
   SCANDIR_A3ATYPE entry;
{
   /* no dir entries beginning with dots. */
   return (*(entry->d_name) != '.');
}

/*
 * oh my god was this code ever broken...
 * re-written 3/3/2001 by jjd
 */
void
make_pack(files, desc, min_speed)
   u_char *files, *desc;
   float min_speed;
{
   /* pack and files within the pack */
   NDCCPack *newpack = NULL;
   /* count of specified files found for pack */
   int count = 0, n;
   /* total size of files found for pack */
   ssize_t sizecount = 0;
   u_char *ptr, *list = NULL;
   u_char cfn[1024], dir[1024], fname[1024];
   struct stat sb;
   
   /* try to allocate the pack first.. */
   newpack = new_pack(desc, min_speed);
   if (!newpack)
     {
	put_error("NDCC: Unable to get memory for this pack, aborting!");
	return;
     }

   /* while we still have space seperated filenames... 
    * (note: for filenames with spaces in them, use wildcards)
    */
   while ((ptr = my_strsep(&files, " ")))
     {
	/* just in case the string is empty (var "files" ends with a space) */
	if (!(*ptr))
	  continue;

	/* do the filename stuff... */
	n = ndcc_do_filename(ptr, cfn, sizeof(cfn), dir, sizeof(dir), fname, sizeof(fname));
	if (n == 0)
	  continue;
	if (n == -1)
	  {
	     free_pack(&newpack);
	     dma_Free(&list);
	  }

	/* okay, now tb1 == path and p == name
	 *
	 * all we need now is to see if it exists and readable, etc
	 */
	
	/* reset the stat buffer (remember we are looping) */
	memset(&sb, 0, sizeof(sb));
	n = stat_file(cfn, &sb);
	if (n == -1)
	  {
	     n = ndcc_find_files(dir, fname, &list, &sizecount, newpack);
	     if (n == -1)
	       {
		  /* okay, there was a problem.. */
		  put_error("NDCC: Unable to find %s in %s: %s", fname, dir, strerror(errno));
		  continue;
	       }
	     if (n > 0)
	       {
		  count += n;
	       }
	  }
	else
	  {
	     /* stat_file() succeeded, lets do further validation */
	     if (sb.st_mode & S_IFDIR)
	       {
		  put_info("NDCC: Unable to offer directory %s, skipping.", ptr);
		  continue;
	       }
	     
	     /* okay, stat found it, lets see if we can read it.. if not, skip it. */
	     if (access(cfn, R_OK))
	       {
		  put_error("NDCC: Skipping %s... Unable to access(): %s", ptr, strerror(errno));
		  continue;
	       }

	     /* All is ok, lets add this match to the pack */
	     if (!add_file_to_pack(newpack, dir, fname, sb))
	       {
		  /* if we can't allocate the pack, error, free stuff and return. */
		  put_error("NDCC: Unable to get memory for a file in this pack, aborting!");
		  free_pack(&newpack);
		  dma_Free(&list);
		  return;
	       }
	     /* increment the total pack size */
	     sizecount += sb.st_size;
	     count++;

	     /* append this filename to the pack file list */
	     if (list)
	       dma_strcat(&list, " ");
	     dma_strcat(&list, fname);
	  }
     }
   
   /* we didn't find anything! */
   if (count < 1)
     {
	put_info("NDCC: No files found, package not created.");
	free_pack(&newpack);
	dma_Free(&list);
	return;
     }

   ndcc_set_pack_totals(newpack, count, sizecount);
   /* add the new pack to the packs list */
   if (ndcc_end)
     ndcc_end->next = newpack;
   else
     ndcc_packs = newpack;
   ndcc_end = newpack;
   /* tell the user what we did (a summary from hell) */
   if (min_speed)
     put_info("NDCC: Created pack #%d: (%s) [%s/%d file%s, %.0fkbytes/s min]",
	      total_packs, list, ninja_size(sizecount), count, PLURAL(count),
	      newpack->min_speed);
   else
     put_info("NDCC: Created pack #%d: (%s) [%s/%d file%s]",
	      total_packs, list, ninja_size(sizecount), count, PLURAL(count));
   dma_Free(&list);
}

/*
 * /ndcc offer
 */
void
ndcc_offer(args)
   u_char *args;
{
   u_char *desc = NULL, *speed = NULL;
   u_char *p;
   float minspeed = 0;

   if (args && *args)
     {
	desc = my_index(args, ',');
	if (!desc)
	  {
	     usage("ndcc offer", "<file1> .. <fileN> , <description> [, <minimum speed>]");
	     return;
	  }
	/* split file list from rest */
	p = desc-1;
	*(desc++) = '\0';
	
	/* strip trailing white space from the file */
	while (p && isspace(*p))
	  *p-- = '\0';
	
	/* strip leading whitespace on description */
	while (isspace(*desc))
	  desc++;
	
	/* argh, comma but nothing after it */
	if (!(*desc))
	  {
	     usage("ndcc offer", "<file1> .. <fileN> , <description> [, <minimum speed>]");
	     return;
	  }
	
	/* try to get the speed */
	speed = my_index(desc, ',');
	if (speed)
	  {
	     /* strip comma */
	     *speed++ = '\0';
	     /* strip leading whitespace from speed */
	     while (isspace(*speed))
	       *speed++ = '\0';
	     
	     /* strip trailing whitespace from description */
	     p = &(desc[strlen(desc) - 1]);
	     while (isspace(*p))
	       *p-- = '\0';
	  }
	p = UNULL;
	if (speed && isdigit(*speed))
	  minspeed = strtod(speed, (char **)&p);
	if (p == speed)
	  minspeed = 0;
	/* make the pack! */
	make_pack(args, desc, minspeed);
     }
   else
     add_wait_prompt("Add which files to pack? ", ndcc_offer2, args, WAIT_PROMPT_LINE);
}

/*
 * this gets called with the file list they want in the pack
 */
void
ndcc_offer2(dumb, files)
   u_char *dumb, *files;
{
   if (!(files && *files))
     {
	put_info("You must specify file(s) to add to the pack!");
	return;
     }
   add_wait_prompt("Describe this pack: ", ndcc_offer3, files, WAIT_PROMPT_LINE);
}

/*
 * this gets called with the files and the pack description
 */
void
ndcc_offer3(files, desc)
   u_char *files, *desc;
{
   u_char *wtf = UNULL;
   
   if (!(desc && *desc))
     {
	put_info("You must make a pack description!");
	return;
     }
   /* this is a hack */
   dma_strcpy(&wtf, files);
   dma_strcat(&wtf, UP("\n"));
   dma_strcat(&wtf, desc);
   add_wait_prompt("Minimum speed (kbytes/s): ", ndcc_offer4, wtf, WAIT_PROMPT_LINE);
}

/*
 * this one gets the files and the descripton in f_and_d sepereated by a new line
 * 
 * also, speed is passed in
 */
void
ndcc_offer4(f_and_d, speed)
   u_char *f_and_d, *speed;
{
   double minimumspeed = 0;
   u_char *tmp;

   if (speed && isdigit(*speed))
     minimumspeed = strtod(speed, (char **)&tmp);
   if (tmp == speed)
     minimumspeed = 0;
   
   /* split the files and the description again */
   tmp = my_index(f_and_d, '\n');
   if (tmp)
     *tmp++ = '\0';
   
   make_pack(f_and_d, tmp, minimumspeed);
   dma_Free(&f_and_d);
}

/*
 * list the packs to the client user
 */
void
ndcc_list(args)
   u_char *args;
{
   NDCCPack *p;
   NDCCFile *f;
   int count = 1;
   u_char tmpbuf1[512];

   if (ndcc_packs)
     {
	put_info("NDCC: The following packs are being offered:");
	for (p = ndcc_packs; p; p = p->next, count++)
	  {
	     tmpbuf1[0] = '\0';
	     for (f = p->files; f; f = f->next)
	       {
		  if (*tmpbuf1)
		    strmcat(CP(tmpbuf1), " ", sizeof(tmpbuf1)-1);
		  strmcat(CP(tmpbuf1), f->name, sizeof(tmpbuf1)-1);
	       }
	     if (p->min_speed)
	       put_info("#%-3d (%s) %s (%s, %.2fk/s min)",
			count, ninja_size(p->total_size), p->desc, tmpbuf1, p->min_speed);
	     else
	       put_info("#%-3d (%s) %s (%s)",
			count, ninja_size(p->total_size), p->desc, tmpbuf1);
	  }
     }
   else
     put_info("NDCC: There are no packs being offered!");
}

/*
 * print the list to the current channel
 */
void
ndcc_plist(args)
   u_char *args;
{
   NDCCPack *tmp;
   int count = 0;
   u_char tb1[512], tb2[64], *chan = NULL;
   
   if (args && *args)
     chan = args;
   if (!chan)
     chan = get_channel_by_refnum(0);

   if (ndcc_packs)
     {
	snprintf(tb1, sizeof(tb1)-1, "+--( NDCC Offered files )-- /CTCP %s NDCC SEND #", get_server_nickname(from_server));
	tb1[sizeof(tb1)-1] = '\0';
	send_text(chan, tb1, NULL);
	for (tmp = ndcc_packs; tmp; tmp = tmp->next)
	  {
	     count++;
	     snprintf(tb1, sizeof(tb1)-1, "| #%-3d (%s/%d file%s) %s",
		      count, ninja_size(tmp->total_size), tmp->num_files, tmp->num_files > 1 ?
		      "s" : "", tmp->desc);
	     tb1[sizeof(tb1)-1] = '\0';
	     if (tmp->min_speed)
	       {
		  snprintf(tb2, sizeof(tb2)-1, " (min %.0fkbytes/s)", tmp->min_speed);
		  tb2[sizeof(tb2)-1] = '\0';
		  strmcat(tb1, tb2, sizeof(tb1)-1);
	       }
	     send_text(chan, tb1, NULL);
	  }
	snprintf(tb1, sizeof(tb1)-1, "+--( Total: %d pack%s %s/%d file%s )---",
		 total_packs, PLURAL(total_packs),
		 ninja_size(total_size), total_fs, PLURAL(total_fs));
	tb1[sizeof(tb1)-1] = '\0';
	send_text(chan, tb1, NULL);
     }
   else
     put_info("NDCC: There are no packs being offered!");
}

/*
 * /ndcc remove
 */
void
ndcc_remove(args)
   u_char *args;
{
   NDCCPack *tmp;
   int count = 1, number = 0;

   if (!(args && *args))
     {
	usage("ndcc remove", "<pack number>");
	return;
     }
   
   if (*args == '#')
     args++;
   number = my_atoi(args);
   
   if (number == 0)
     {
	usage("ndcc remove", "<pack number>");
	return;
     }
   if (number < 1 || number > total_packs)
     {
	put_info("NDCC: Pack #%d does not exist.", number);
	return;
     }

   for (tmp = ndcc_packs; tmp; tmp = tmp->next, count++)
     {
	if (count == number)
	  break;
     }

   if ((tmp = (NDCCPack *) remove_from_list((List **) & ndcc_packs, tmp->desc)) != NULL)
     {
	/* decrement the total # of files and size of all packs 
	 * 
	 * -- this is now handled by free_pack()
	 *
	total_size -= tmp->total_size;
	total_fs -= tmp->num_files;
	total_packs--;
	 * 
	 */
	/* free the pack and tell the user that it is done. */
	free_pack(&tmp);
	put_info("NDCC: Removed pack #%d.", number);
     }
   return;
}

/*
 * /ndcc save
 * 
 * save the packs for later loading
 */
void
ndcc_save(args)
   u_char *args;
{
   NDCCPack *tmp;
   NDCCFile *tmpfiles;
   u_char filename[512], tmpbuf1[1024];
   u_char *ptr, *ptr2;
   FILE *outfile;

   if (!(args && *args))
     {
	usage("ndcc save", "<name for list>");
	return;
     }
   ptr2 = next_arg(args, &args);
   snprintf(filename, sizeof(filename)-1, NINJA_DIR "/%s.ndcc", ptr2);
   filename[sizeof(filename)-1] = '\0';
   ptr = expand_twiddle(filename);
   outfile = fopen(ptr, "w");
   dma_Free(&ptr);
   if (outfile == NULL)
     {
	put_error("Cannot open ndcc list(%s), aborting ndcc save..", filename);
	return;
     }
   for (tmp = ndcc_packs; tmp; tmp = tmp->next)
     {
	tmpbuf1[0] = '\0';
	for (tmpfiles = tmp->files; tmpfiles; tmpfiles = tmpfiles->next)
	  {
	     if (*tmpbuf1)
		strmcat(tmpbuf1, " ", sizeof(tmpbuf1)-1);
	     strmcat(tmpbuf1, make_nice(tmpfiles->path), sizeof(tmpbuf1)-1);
	     strmcat(tmpbuf1, "/", sizeof(tmpbuf1)-1);
	     strmcat(tmpbuf1, make_nice(tmpfiles->name), sizeof(tmpbuf1)-1);
	  }
	fprintf(outfile, "%s:%.2f %s\n", tmpbuf1, tmp->min_speed, tmp->desc);
     }
   put_info("NDCC: Saved offer list as: \"%s\".", filename);
   fclose(outfile);
}

void
ndcc_load(args)
   u_char *args;
{
   u_char *ptr, *files, *desc, *ptr2;
   u_char filename[512], tmpbuf2[1024];
   FILE *infile;
   int dumb = 0;
   double mspeed;

   if (!(args && *args))
     {
	usage("ndcc load", "<name of list>");
	return;
     }
   ptr2 = next_arg(args, &args);
   snprintf(filename, sizeof(filename), "~/.ninja/%s.ndcc", ptr2);
   ptr = expand_twiddle(filename);
   infile = fopen(ptr, "r");
   dma_Free(&ptr);
   if (infile == NULL)
     {
	put_error("Cannot open ndcc list(%s), load aborted..", filename);
	return;
     }
   for (;;)
     {
	memset(tmpbuf2, 0, sizeof(tmpbuf2));
	if (fgets(tmpbuf2, sizeof(tmpbuf2)-1, infile))
	  {
	     ptr = &tmpbuf2[strlen(tmpbuf2) - 1];
	     while (*ptr == '\r' || *ptr == '\n')
	       *ptr-- = '\0';
	       
	     ptr = tmpbuf2;
	     while (*ptr)
	       {
		  if (*ptr == '\\')
		    {
		       ptr++;
		       if (*ptr == ':')
			  ptr++;
		    }
		  if (*ptr == ':')
		    {
		       *ptr = '\0';
		       ptr++;
		       desc = ptr;
		    }
		  ptr++;
	       }
	     files = tmpbuf2;
	     un_nice(files);
	     un_nice(desc);
	     ptr = my_strsep(&desc, " ");
	     
	     /* decifer minimum speed */
	     mspeed = 0;
	     mspeed = strtod(ptr, (char **)&ptr2);
	     if (ptr == ptr2)
	       mspeed = 0;
	     make_pack(files, desc, mspeed);
	     dumb = 1;
	  }
	else
	   break;
     }
   if (dumb)
      put_info("NDCC: Loaded offer list: %s.", filename);
   else
      put_error("NDCC: Couldn't load the offer list.");
   fclose(infile);
}

void
ndcc_remote_send(from, user, host, to, args)
   u_char *from, *user, *host, *to, *args;
{
   NDCCPack *tmp;
   NDCCFile *tmpfiles;
   DCC_list *tmpdcc;
   u_char *thepack, tb1[512], tb2[512];
   int packnum = 0, count = 0, sent = 0;
   double totalbytes = 0;
   unsigned int old_display;

   thepack = next_arg(args, &args);
   if (thepack)
     {
	if (*thepack == '#')
	   *thepack++ = '\0';
	packnum = atoi(thepack);
	if (!packnum || packnum > total_packs)
	  {
	     send_to_server("NOTICE %s :Invalid pack number.", from);
	     return;
	  }
	if (ndcc_packs)
	  {
	     for (tmp = ndcc_packs; tmp; tmp = tmp->next)
	       {
		  count++;
		  if (packnum == count)
		    {
		       send_to_server("NOTICE %s :Sending you pack #%d: %s", from, count, tmp->desc);
		       old_display = window_display;
		       for (tmpfiles = tmp->files; tmpfiles; tmpfiles = tmpfiles->next)
			 {
			    snprintf(tb2, sizeof(tb2)-1, "%s/%s", tmpfiles->path, tmpfiles->name);
			    snprintf(tb1, sizeof(tb1)-1, "%s %s", from, tb2);
			    window_display = 0;
			    dcc_sendfiles(tb1);
			    tmpdcc = dcc_searchlist(NULL, from, DCC_FILEOFFER, 0, tb2, 0);
			    if (tmpdcc)
			       tmpdcc->minimum_speed = tmp->min_speed;
			    window_display = old_display;
			    totalbytes += tmpfiles->size;
			    sent++;
			 }
		    }
	       }
	     if (sent)
		put_info("NDCC: Sending %s pack #%d: [%s/%d file%s]",
		    from, packnum, ninja_size(totalbytes),
		    sent, PLURAL(sent));
	  }
	else
	   send_to_server("NOTICE %s :I'm not offering anything.", from);
     }
   else
      send_to_server("NOTICE %s :Try /CTCP %s NDCC SEND <#>", from, get_server_nickname(from_server));
}

void
ndcc_remote_list(from, user, host, to, args)
   u_char *from, *user, *host, *to, *args;
{
   NDCCPack *tmp;
   int count = 0;
   u_char tb1[512];

   if (ndcc_packs)
     {
	snprintf(tb1, sizeof(tb1)-1, "+--( NDCC Offered files )-- /CTCP %s NDCC SEND <pack number>",
		 get_server_nickname(from_server));
	send_to_server("NOTICE %s :%s", from, tb1, NULL);
	for (tmp = ndcc_packs; tmp; tmp = tmp->next)
	  {
	     count++;
	     if (tmp->min_speed)
	       snprintf(tb1, sizeof(tb1)-1, "| #%d (%s/%d file%s) %s (min %.0fk/s)",
			count, ninja_size(tmp->total_size), tmp->num_files, PLURAL(tmp->num_files),
			tmp->desc, tmp->min_speed);
	     else
	       snprintf(tb1, sizeof(tb1)-1, "| #%d (%s/%d file%s) %s",
			count, ninja_size(tmp->total_size), tmp->num_files,
			PLURAL(tmp->num_files), tmp->desc);
	     send_to_server("NOTICE %s :%s", from, tb1, NULL);
	  }
	snprintf(tb1, sizeof(tb1)-1, "+--( Total: %d packs %s/%d file%s )---",
		 total_packs, ninja_size(total_size), total_fs, PLURAL(total_fs));
	send_to_server("NOTICE %s :%s", from, tb1, NULL);
     }
   else
      send_to_server("NOTICE %s :I'm not offering anything.", from);
}

void
ndcc_remote_help(from, user, host, to, args)
   u_char *from, *user, *host, *to, *args;
{
   send_to_server("NOTICE %s :/CTCP %s NDCC <command>, valid commands are: version, list, or send.",
		  from, get_server_nickname(from_server));
}

void
ndcc_remote_version(from, user, host, to, args)
   u_char *from, *user, *host, *to, *args;
{
   send_to_server("NOTICE %s :Ninja DCC version %s by kraig and Senor Pato", from, NINJA_VERSION);
}

void
ndcc_clear(unused)
   u_char *unused;
{
   NDCCPack *tmp;

   while (total_packs)
     {
	tmp = ndcc_packs;
	if ((tmp = (NDCCPack *) remove_from_list((List **) & ndcc_packs, tmp->desc)) != NULL)
	  {
	     /* reduce all-pack totals
	      * 
	      * -- this is now handled by free_pack()
	      *
	     total_size -= tmp->total_size;
	     total_fs -= tmp->num_files;
	     total_packs--;
	      *
	      * and free the pack 
	      */
	     free_pack(&tmp);
	  }
     }
   put_info("NDCC: All packs cleared.");
   return;
}

void
ndcc_send(args)
   u_char *args;
{
   int packnum, count = 0;
   u_char *to, tb1[512], tb2[512];
   NDCCPack *tmp;
   NDCCFile *tmpfiles;
   unsigned int old_display;
   DCC_list *tmpdcc;
   double totalbytes = 0;
   int sent = 0;

   if (!(args && *args) || ((to = next_arg(args, &args)) == NULL))
     {
	usage("ndcc send", "<nick> <pack number>");
	return;
     }
   if (args && *args && atoi(args))
      packnum = atoi(args);
   else if (*args == '#')
     {
	args++;
	if (args && *args && atoi(args))
	   packnum = atoi(args);
	else
	   packnum = 0;
     }
   else
      packnum = 0;
   if (!packnum || packnum == 0)
     {
	usage("ndcc send", "<nick> <pack number>");
	return;
     }
   if (packnum > total_packs)
     {
	put_error("NDCC: Pack #%d does not exist.", packnum);
	return;
     }
    for (tmp = ndcc_packs; tmp; tmp = tmp->next)
     {
	count++;
	if (count == packnum)
	  {
	     send_to_server("NOTICE %s :Sending you pack #%d: %s", to, count, tmp->desc);
	     old_display = window_display;
	     for (tmpfiles = tmp->files; tmpfiles; tmpfiles = tmpfiles->next)
	       {
		  snprintf(tb2, sizeof(tb2)-1, "%s/%s", tmpfiles->path, tmpfiles->name);
		  snprintf(tb1, sizeof(tb1)-1, "%s %s", to, tb2);
		  window_display = 0;
		  dcc_sendfiles(tb1);
		  tmpdcc = dcc_searchlist(NULL, to, DCC_FILEOFFER, 0, tb2, 0);
		  if (tmpdcc)
		     tmpdcc->minimum_speed = tmp->min_speed;
		  window_display = old_display;
		  totalbytes += tmpfiles->size;
		  sent++;
	       }
	  }
     }
   if (sent)
     put_info("NDCC: Sending %s pack #%d: [%s/%d file%s]",
	      to, packnum, ninja_size(totalbytes),
	      sent, PLURAL(sent));
}


/*
 * this code was added in March 3rd, 2001 by jduck
 * 
 * this is much needed!
 */
void
free_pack(NDCCPack **np)
{
   NDCCFile *tf1, *tf2;
   NDCCPack *end;
   
   /* if we were passed a NULL pointer, return */
   if (!(*np))
     return;
   /* reverse the total stuff... */
   total_packs--;
   total_size -= (*np)->total_size;
   total_fs -= (*np)->num_files;
   /* ok, first free the allocated stuff in the pack */
   dma_Free(&((*np)->desc));
   /* now the files.. */
   for (tf1 = (*np)->files; tf1; tf1 = tf2)
     {
	tf2 = tf1->next;
	free_pack_file(&tf1);
     }
   /* then the pack itself */
   dma_Free(np);
   /* update the end... */
   ndcc_end = NULL;
   for (end = ndcc_packs; end; end = end->next)
     ndcc_end = end;
}

/*
 * this code is handy as well
 */
void
free_pack_file(NDCCFile **tf)
{
   if (!(*tf))
     return;
   dma_Free(&((*tf)->path));
   dma_Free(&((*tf)->name));
   dma_Free(tf);
}


/*
 * toggle ndcc offering on/off
 */
void
ndcc_toggle(args)
   u_char *args;
{
   u_char *now;
   
   now = next_arg(args, &args);
   if (!now)
     set_int_var(NDCC_OFFERING_VAR, !get_int_var(NDCC_OFFERING_VAR));
   else
     {
	upper(now);
	if (my_strcmp(now, "ON"))
	  set_int_var(NDCC_OFFERING_VAR, 1);
	else if (my_strcmp(now, "OFF"))
	  set_int_var(NDCC_OFFERING_VAR, 0);
	else
	  {
	     usage("ndcc toggle", "[<on/off>]");
	     return;
	  }
     }
   put_info("NDCC: Offering is now: %s", get_int_var(NDCC_OFFERING_VAR) ? "ON" : "OFF");
   return;
}

/*
 * set NDCC offering interval
 * 
 * setting this to 0 will turn off automatic display of packs.
 */
void
ndcc_interval(args)
   u_char *args;
{
   u_int val;
   
   if (!args || !*args || !isdigit(*args))
     {
	usage("ndcc interval", "<new interval>");
	return;
     }
   val = my_atoi(args);
   set_int_var(NDCC_INTERVAL_VAR, val);
   put_info("NDCC: Offer interval is now: %u seconds", val);
}

/*
 * allocate memory for a new pack
 */
static NDCCPack *
new_pack(desc, min_speed)
   u_char *desc;
   double min_speed;
{
   NDCCPack *newpack;
   
   /* if the pack isn't allocated yet, lets allocate it */
   newpack = (NDCCPack *) dma_Malloc(sizeof(NDCCPack));
   if (!newpack)
     {
	/* if we can't allocate the pack, error, free stuff and return. */
	put_error("NDCC: Unable to allocate memory for NDCC pack, aborting!");
	return NULL;
     }
   /* newpack->description = NULL; */
   /* store the description */
   dma_strcpy(&(newpack->desc), desc);
   /* newpack->files = NULL; */
   newpack->min_speed = min_speed;
   total_packs++;
   return newpack;
}

/*
 * set pack_totals
 */
static void
ndcc_set_pack_totals(pack, files, size)
   NDCCPack *pack;
   int files;
   ssize_t size;
{
   pack->num_files = files;
   total_fs += files;
   pack->total_size = size;
   total_size += size;
}

/*
 * find files with wildcards in them...
 * 
 * dest will result in being appended with the found filenames... (eg strcat() with a space first)
 * scnt will have the sizes of the found files added to it (+=)
 * the files will be added to the specified pack..
 *
 * returns:
 * -1 scandir failed
 * >= 0 number of files matched
 */
static int
ndcc_find_files(dir, pattern, dest, scnt, pack)
   u_char *dir, *pattern;
   u_char **dest;
   ssize_t *scnt;
   NDCCPack *pack;
{
   struct dirent **dirfiles;
   int n, count = 0;
   u_char tb2[1024];
   struct stat sb;
   
   /* no wildcards?  well then scandir isn't gonna help! */
   if (my_index(pattern, '*') != 0
       && my_index(pattern, '?') != 0)
     return 0;
   
   /* well, we have wildcards, so lets attempt to match them up with existing files.. */
   n = scandir(dir, &dirfiles,
	       (int (*) _((SCANDIR_A3ATYPE))) selectent,
	       (int (*) _((const void *, const void *)))compar);
   /* scandir failed? */
   if (n == -1)
     return -1;
   /* what if we have nothing? */
   if (n == 0)
     return 0;
   /* we have something, see if it matches.. */
   while (n--)
     {
	/* if the file matches, add it to the pack.. */
	if (match(pattern, dirfiles[n]->d_name))
	  {
	     snprintf(tb2, sizeof(tb2)-1, "%s/%s", dir, dirfiles[n]->d_name);
	     tb2[sizeof(tb2)-1] = '\0';
	     /* if there's no read access, skip it */
	     if (access(tb2, R_OK) != 0)
	       {
		  put_error("NDCC: Skipping %s... Unable to access(): %s", tb2, strerror(errno));
		  continue;
	       }
	     /* try to stat the file */
	     memset(&sb, 0, sizeof(sb));
	     stat_file(tb2, &sb);
	     if (sb.st_mode & S_IFDIR)
	       {
		  put_error("NDCC: Unable to offer directory %s, skipping.", tb2);
		  continue;
	       }
	     /* All is ok, lets add this match to the pack */
	     if (!add_file_to_pack(pack, dir, dirfiles[n]->d_name, sb))
	       {
		  /* if we can't allocate the pack, error, free stuff and return. */
		  put_error("NDCC: Unable to get memory for a file in this pack, aborting!");
		  return count;
	       }
	     /* increment the total pack size */
	     *scnt += sb.st_size;
	     count++;
	     /* append this filename to the pack file list */
	     if (*dest)
	       dma_strcat(dest, " ");
	     dma_strcat(dest, dirfiles[n]->d_name);
	  }
	/* if the file doesn't match, we do nothing */
     }
   return count;
}


/*
 * add a file to a pack
 */
static int
add_file_to_pack(pack, path, fn, sb)
   NDCCPack *pack;
   u_char *path, *fn;
   struct stat sb;
{
   NDCCFile *newf;
   
   newf = (NDCCFile *) dma_Malloc(sizeof(NDCCFile));
   if (!newf)
     return 0;
   /* we succeeded, so lets fill in the pack file info */
   newf->size = sb.st_size;
   dma_strcpy(&newf->name, fn);
   dma_strcpy(&newf->path, path);
   
   /* add this pack file to the pack's files list end */
   if (pack->endfiles)
     pack->endfiles->next = newf;
   else
     pack->files = newf;
   pack->endfiles = newf;
   return 1;
}

/*
 * try to split src into a canonical filename
 * as well as a dir/name pair..
 * 
 * returns:
 * -1 fatal error
 * 0 single file error
 * 1 success!
 */
static int
ndcc_do_filename(ptr,
		 cfn, cfnl,
		 dir, dirl,
		 fn, fnl)
   u_char *ptr, *cfn;
   int cfnl;
   u_char *dir;
   int dirl;
   u_char *fn;
   int fnl;
{
   u_char *p;
     
   /* store the canonical file name in cfn */
   switch (*ptr)
     {
      case '~':
	if (!(p = expand_twiddle(ptr)))
	  {
	     put_error("NDCC: Unable to expand tilde for %s, skipping...", ptr);
	     return 0;
	  }
	my_strncpy(cfn, p, cfnl-1);
	dma_Free(&p);
	break;
      case '/':
	my_strncpy(cfn, ptr, cfnl-1);
	break;
      default:
	if (getcwd(cfn, cfnl-my_strlen(ptr)-2) == NULL)
	  {
	     put_error("NDCC: Unable to get cwd: %s", strerror(errno));
	     return -1;
	  }
	my_strmcat(cfn, "/", cfnl-1);
	my_strmcat(cfn, ptr, cfnl-1);
     }
   cfn[cfnl-1] = '\0';
	
   /* okay, now we have the full path to the suspected file stored in cfn.
    * our target is to have a path and a filename, so lets split it here
    */
   my_strncpy(dir, cfn, dirl-1);
   dir[dirl-1] = '\0';
   p = my_rindex(dir, '/');
   if (p)
     {
	*p++ = '\0';
	my_strncpy(fn, p, fnl-1);
	fn[fnl-1] = '\0';
     }
   return 1;
}
