#include "common.h"

extern "C"
{
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/types.h>
#include <fcntl.h>
#include <dirent.h>
#include <errno.h>
#include <utime.h>
#include <time.h>
#include <libgen.h>
}

eCopyOverride gCopyOverride = kNone;
eWriteProtected gWriteProtected = kWPNone;

void file_copy_override_box(char* spath, struct stat stat_, char* dfile, struct stat dfstat)
{
    const int maxx = mgetmaxx();
    const int maxy = mgetmaxy();
    
    const int y = maxy/2 - 5;
    
    mbox(y, 4, maxx-8, 11);
    
    time_t t = stat_.st_mtime;
    struct tm* tm_ = (struct tm*)localtime(&t);
    
    int year = tm_->tm_year-100;
    if(year < 0) year+=100;
    while(year > 100) year-=100;
    
    mmvprintw(y+1, 5, "source");
    char tmp[PATH_MAX];
    strcpy(tmp, spath);
    if(S_ISDIR(stat_.st_mode)) {
        strcat(tmp, "/");
    }
    else if(S_ISFIFO(stat_.st_mode)) {
        strcat(tmp, "|");
    }
    else if(S_ISSOCK(stat_.st_mode)) {
        strcat(tmp, "=");
    }
    else if(S_ISLNK(stat_.st_mode)) {
        strcat(tmp, "@");
    }
    
    
    mmvprintw(y+2, 5, "path: %s", tmp);
    mmvprintw(y+3, 5, "size: %d", stat_.st_size);
    mmvprintw(y+4, 5, "time: %02d-%02d-%02d %02d:%02d", year, tm_->tm_mon+1
           , tm_->tm_mday, tm_->tm_hour, tm_->tm_min);
    
    t = dfstat.st_mtime;
    tm_ = (struct tm*)localtime(&t);
    
    year = tm_->tm_year-100;
    if(year < 0) year+=100;
    while(year > 100) year-=100;
    
    mmvprintw(y+6, 5, "distination");
    char tmp2[PATH_MAX];
    strcpy(tmp2, dfile);
    if(S_ISDIR(dfstat.st_mode)) {
        strcat(tmp2, "/");
    }
    else if(S_ISFIFO(dfstat.st_mode)) {
        strcat(tmp, "|");
    }
    else if(S_ISSOCK(dfstat.st_mode)) {
        strcat(tmp, "=");
    }
    else if(S_ISLNK(dfstat.st_mode)) {
        strcat(tmp, "@");
    }
    
    mmvprintw(y+7, 5, "path: %s", tmp2);
    mmvprintw(y+8, 5, "size: %d", dfstat.st_size);
    mmvprintw(y+9, 5, "time: %02d-%02d-%02d %02d:%02d", year, tm_->tm_mon+1
           , tm_->tm_mday, tm_->tm_hour, tm_->tm_min);
}

static char* gSPath;
static struct stat gStat;
static char* gDFile;
static struct stat gDFStat;

void view_file_copy_override_box()
{
    file_copy_override_box(gSPath, gStat, gDFile, gDFStat);
}

bool file_copy(char* spath, char* dpath, bool move, bool preserve)
{
    /// key input? ///
    fd_set mask;

    FD_ZERO(&mask);
    FD_SET(0, &mask);

    struct timeval tv;
    tv.tv_sec = 0;
    tv.tv_usec = 0;

    select(1, &mask, NULL, NULL, &tv);
    
    if(FD_ISSET(0, &mask)) {
        int meta;
        int key = mgetch(&meta);

        if(key == 3 || key == 7 || key == 27) {    // CTRL-G and CTRL-C
            err_msg("file_copy: canceled");
            return false;
        }
    }
    
    /// new file name check ///
    char new_fname[PATH_MAX];
    strcpy(new_fname, "");
    
    struct stat dstat;
    const int dlen = strlen(dpath);

    if(stat(dpath, &dstat) < 0) {
        if(dpath[dlen -1] == '/') {
            err_msg("file_copy: destination_err(%s)", dpath);
            return false;
        }
        else {
            char* p = dpath + dlen;
            while(p != dpath && *p != '/') {
                p--;
            }
            strcpy(new_fname, p+1);

            dpath[p-dpath+1] = 0;
        }
    }
    else if(!S_ISDIR(dstat.st_mode)) {
        char* p = dpath + dlen;
        while(p != dpath && *p != '/') {
            p--;
        }
        strcpy(new_fname, p+1);

        dpath[p-dpath+1] = 0;
    }
    
    /// add / ///
    char new_dpath[PATH_MAX];
    if(dpath[strlen(dpath) -1] != '/') {
        strcpy(new_dpath, dpath);
        strcat(new_dpath, "/");

        dpath = new_dpath;
    }

    /// dpath check ///
    if(stat(dpath, &dstat) < 0) {
        err_msg("file_copy: destination_err(%s)", dpath);
        return false;
    }
    if(!S_ISDIR(dstat.st_mode)) {
        err_msg("file_copy: destination err(%s)", dpath);
        return false;
    }

    /// illegal argument check ///
    char spath2[PATH_MAX];

    strcpy(spath2, spath);
    strcat(spath2, "/");
        
    if(strstr(dpath, spath2) == dpath) {
        err_msg("file_copy: destination err(%s)", dpath);
        return false;
    }
    
    /// source stat ///
    struct stat stat_;

    if(lstat(spath, &stat_) < 0) {
        return true;
    }
    else {
        /// get destination file name ///
        char dfile[PATH_MAX];
        
        if(strcmp(new_fname, "") == 0) {
            char sfname[PATH_MAX];
            int i;
            const int len = strlen(spath);
            for(i = len-1;
                i > 0;
                i--)
            {
                if(spath[i] == '/') break;
            }
            memcpy(sfname, spath + i + 1, len - i);

            /// get destination file name ///
            strcpy(dfile, dpath);
            strcat(dfile, sfname);
        }
        else {
            strcpy(dfile, dpath);
            strcat(dfile, new_fname);
        }

        /// illegal argument check ///
        if(strcmp(spath, dfile) == 0) {
            return false;
        }

        /// dir ///
        if(S_ISDIR(stat_.st_mode)) {
            /// msg ///
            char buf[PATH_MAX+10];
            sprintf(buf, "entering %s", spath);
            msg_nonstop("%s", buf);

            /// override ///
            bool no_mkdir = false;
            bool select_newer = false;
            struct stat dfstat;
            if(access(dfile, F_OK) == 0) {
                if(lstat(dfile, &dfstat) < 0) {
                    err_msg("file_copy: lstat() err2(%s)", dfile);
                    return false;
                }
        
                /// ok ? ///
                if(S_ISDIR(dfstat.st_mode)) {
                    no_mkdir = true;
                }
                else {
                    if(gCopyOverride == kNone) {
override_select_str:
                        
                        gSPath = spath;
                        gStat = stat_;
                        gDFile = dfile;
                        gDFStat = dfstat;

                        gView = view_file_copy_override_box;

                        gView();

                        const char* str[] = {
                            "No", "Yes", "Newer", "Rename", "No(all)", "Yes(all)", "Newer(all)", "Cancel"
                        };

                        char buf[256];
                        sprintf(buf, "override? ");
                        int ret = select_str(buf, (char**)str, 8, 7);
                        
                        gView = NULL;

                        mclear();
                        view(true);
                        mrefresh();
                        
                        switch(ret) {
                        case 0:
                            return true;
                        case 1:
                            file_remove(dfile, true, false);
                            break;
                        case 2:
                            select_newer = true;
                            break;
                            
                        case 3: {
                            char result[1024];
                            
                            char dfile_tmp[PATH_MAX];
                        
                            int result2;
                            bool exist_rename_file;
                            do {
                                result2 = input_box("input name:", result, "", 0);
                                
                                if(result2 == 1) {
                                    goto override_select_str;
                                }
                                
                                strcpy(dfile_tmp, dpath);
                                strcat(dfile_tmp, result);
                                
                                exist_rename_file = access(dfile_tmp, F_OK) == 0;
                                
                                if(exist_rename_file) {
                                    msg("same name exists");
                                }
                            } while(exist_rename_file);
                            
                            strcpy(dfile,dfile_tmp);
                            
                            }
                            break;
                            
                        case 4:
                            gCopyOverride = kNoAll;
                            return true;
                        case 5:
                            if(!file_remove(dfile, true, false)) {
                                return false;
                            }
                            gCopyOverride = kYesAll;
                            break;
                        case 6:
                            gCopyOverride = kSelectNewer;
                            select_newer = true;
                            break;
                        case 7:
                            return false;
                        }
                    }
                    else if(gCopyOverride == kYesAll) {
                        if(!file_remove(dfile, true, false)) {
                            return false;
                        }
                    }
                    else if(gCopyOverride == kNoAll) {
                        return true;
                    }
                    else if(gCopyOverride == kSelectNewer) {
                        select_newer = true;
                    }
                }
            }

            if(select_newer) {
                if(stat_.st_mtime > dfstat.st_mtime) {
                    if(!file_remove(dfile, true, false)) {
                        return false;;
                    }
                }
                else {
                    return true;
                }
            }

            if(move) {
                if(rename(spath, dfile) < 0) {
                    goto dir_copy;
                }
            }
            else {
dir_copy:            
                DIR* dir = opendir(spath);
                if(dir == NULL) {
                    err_msg("file_copy: opendir err(%s)", spath);
                    return false;
                }

                if(!no_mkdir) {
                    mode_t umask_before = umask(0);
                    int result = mkdir(dfile, stat_.st_mode & (S_IRUSR|S_IWUSR|S_IXUSR
                            |S_IRGRP|S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH));
                    umask(umask_before);

                    if(result <0)
                    {
                        closedir(dir);
                        err_msg("file_copy: mkdir err(%s)", dfile);
                        return false;
                    }
                }

                struct dirent* entry;
                while(entry = readdir(dir)) {
                    if(strcmp(entry->d_name, ".") != 0
                        && strcmp(entry->d_name, "..") != 0)
                    {
                        char spath2[PATH_MAX];

                        strcpy(spath2, spath);
                        strcat(spath2, "/");
                        strcat(spath2, entry->d_name);

                        char dfile2[PATH_MAX];
        
                        strcpy(dfile2, dfile);
                        strcat(dfile2, "/");
        
                        if(!file_copy(spath2, dfile2, move, preserve)) {
                            closedir(dir);
                            return false;
                        }
                    }
                }

                if(preserve || move) {
                    if(getuid() == 0) {
                        if(chown(dfile, stat_.st_uid, stat_.st_gid) <0) {
                            err_msg("file_copy: chown() err(%s)", dfile);
                            return false;
                        }
                    }

                    struct utimbuf utb;

                    utb.actime = stat_.st_atime;
                    utb.modtime = stat_.st_mtime;

                    if(utime(dfile, &utb) < 0) {
                        err_msg("file_copy: utime() err(%s)", dfile);
                        return false;
                    }
                }

                mode_t umask_before = umask(0);
#if defined(S_ISTXT)
                if(chmod(dfile, stat_.st_mode & (S_IRUSR|S_IWUSR|S_IXUSR
                    |S_IRGRP|S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH|S_ISUID|S_ISGID|S_ISTXT)) < 0) 
                {
                    err_msg("file_copy: chmod() err(%s)", dfile);
                    return false;
                }
#else

#if defined(S_ISVTX)
                if(chmod(dfile, stat_.st_mode & (S_IRUSR|S_IWUSR|S_IXUSR
                            |S_IRGRP|S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH|S_ISUID|S_ISGID|S_ISVTX)) < 0) 
                {
                    err_msg("file_copy: chmod() err(%s)", dfile);
                    return false;
                }
#else
                if(chmod(dfile, stat_.st_mode & (S_IRUSR|S_IWUSR|S_IXUSR
                |S_IRGRP|S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH|S_ISUID|S_ISGID)) < 0) 
                {
                    err_msg("file_copy: chmod() err(%s)", dfile);
                    return false;
                }
#endif

#endif
                umask(umask_before);
        
                if(closedir(dir) < 0) {
                    err_msg("file_copy: closedir() err");
                    return false;
                }
                if(move) {
                    if(rmdir(spath) < 0) {
                        err_msg("file_copy: rmdir() err(%s)", spath);
                        return false;
                    }
                }
            }
        }
        /// file ///
        else {
            /// override ? ///
            if(access(dfile, F_OK) == 0) {
                bool select_newer = false;
                if(gCopyOverride == kNone) {
                    struct stat dfstat;
                    
                    if(lstat(dfile, &dfstat) < 0) {
                        err_msg("file_copy: lstat() err(%s)", dfile);
                        return false;
                    }
        
override_select_str2:
                    gSPath = spath;
                    gStat = stat_;
                    gDFile = dfile;
                    gDFStat = dfstat;

                    gView = view_file_copy_override_box;

                    gView();
                    
                    const char* str[] = {
                        "No", "Yes", "Newer", "Rename", "No(all)", "Yes(all)", "Newer(all)", "Cancel"
                    };
                    
                    char buf[256];
                    sprintf(buf, "override? ");
                    int ret = select_str(buf, (char**)str, 8, 7);

                    gView = NULL;
                    
                    mclear();
                    view(true);
                    mrefresh();
                    
                    switch(ret) {
                    case 0:
                        return true;
                    case 1:
                        if(!file_remove(dfile, true, false)) {
                            return false;
                        }
                        break;
                    case 2:
                        select_newer = true;
                        break;
                    case 3: {
                        char result[1024];
                        
                        int result2;
                        bool exist_rename_file;
                        
                        char dfile_tmp[PATH_MAX];
                        do {
                            result2 = input_box("input name:", result, "", 0);
                            
                            if(result2 == 1) {
                                goto override_select_str2;
                            }
                            
                            strcpy(dfile_tmp, dpath);
                            strcat(dfile_tmp, result);
                            
                            exist_rename_file = access(dfile_tmp, F_OK) == 0;
                            if(exist_rename_file) {
                                msg("same name exists");
                            }
                        } while(exist_rename_file);
                        
                        strcpy(dfile, dfile_tmp);
                        
                        }
                        break;
                        
                    case 4:
                        gCopyOverride = kNoAll;
                        return true;
                    case 5:
                        if(!file_remove(dfile, true, false)) {
                            return false;
                        }
                        gCopyOverride = kYesAll;
                        break;
                    case 6:
                        gCopyOverride = kSelectNewer;
                        select_newer = true;
                        break;
                    case 7:
                        return false;
                    }
                }
                else if(gCopyOverride == kYesAll) {
                    if(!file_remove(dfile, true, false)) {
                        return false;
                    }
                }
                else if(gCopyOverride == kNoAll) {
                    return true;
                }
                else if(gCopyOverride == kSelectNewer) {
                    select_newer = true;
                }

                if(select_newer) {
                    struct stat dfstat;

                    if(lstat(dfile, &dfstat) < 0) {
                        err_msg("file_copy: lstat() err3(%s)", dfile);
                        return false;
                    }

                    if(stat_.st_mtime > dfstat.st_mtime) {
                        if(!file_remove(dfile, true, false)) {
                            return false;
                        }
                    }
                    else {
                        return true;
                    }
                }
            }

            /// msg ///
            char buf[PATH_MAX+10];
            if(move)
                sprintf(buf, "moving %s", spath);
            else
                sprintf(buf, "copying %s", spath);
            msg_nonstop("%s", buf);

            if(move) {
                if(rename(spath, dfile) < 0) {
                    goto file_copy;
                }
            }
            else {
file_copy:            
                /// file ///
                if(S_ISREG(stat_.st_mode)) {
                    int fd = open(spath, O_RDONLY);
                    if(fd < 0) {
                        err_msg("file_copy: open(read) err(%s)", spath);
                        return false;
                    }
                    mode_t umask_before = umask(0);
                    int fd2 = open(dfile, O_WRONLY|O_TRUNC|O_CREAT
                          , stat_.st_mode & (S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP
                                 |S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH)
                        );
                    umask(umask_before);
                    if(fd2 < 0) {
                        err_msg("file_copy: open(write) err(%s)", dfile);
                        close(fd);
                        return false;
                    }
        
                    char buf[BUFSIZ];
                    while(1) {
                        /// key input? ///
                        fd_set mask;

                        FD_ZERO(&mask);
                        FD_SET(0, &mask);

                        struct timeval tv;
                        tv.tv_sec = 0;
                        tv.tv_usec = 0;

                        select(1, &mask, NULL, NULL, &tv);
                        
                        if(FD_ISSET(0, &mask)) {
                            int meta;
                            int key = mgetch(&meta);

                            if(key == 3 || key == 7 || key == 27) {    // CTRL-G and CTRL-C
                                err_msg("file_copy: canceled");
                                close(fd);
                                close(fd2);
                                return false;
                            }
                        }
                        
                        int n = read(fd, buf, BUFSIZ);
                        if(n < 0) {
                            err_msg("file_copy: read err(%s)", spath);
                            close(fd);
                            close(fd2);
                            return false;
                        }
                        
                        if(n == 0) {
                            break;
                        }
        
                        if(write(fd2, buf, n) < 0) {
                            err_msg("file_copy: write err(%s)", dfile);
                            close(fd);
                            close(fd2);
                            return false;
                        }
                    }
                
                    if(close(fd) < 0) {
                        err_msg("file_copy: close err(%s)", spath);
                        return false;
                    }
                    if(close(fd2) < 0) {
                        err_msg("file_copy: close err fd2(%s)", dfile);
                        return false;
                    }

                    
                    if(preserve || move) {
                        if(getuid() == 0) {
                            if(chown(dfile, stat_.st_uid, stat_.st_gid) < 0) {
                                err_msg("file_copy: chown() err(%s)", dfile);
                                return false;
                            }
                        }

                        struct utimbuf utb;

                        utb.actime = stat_.st_atime;
                        utb.modtime = stat_.st_mtime;

                        if(utime(dfile, &utb) < 0) {
                            err_msg("file_copy: utime() err(%s)", dfile);
                            return false;
                        }
                        
                    }
                    
                    umask_before = umask(0);
#if defined(S_ISTXT)
                    if(chmod(dfile, stat_.st_mode & (S_IRUSR|S_IWUSR|S_IXUSR
                    |S_IRGRP|S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH|S_ISUID|S_ISGID|S_ISTXT)) < 0) 
                    {
                        err_msg("file_copy: chmod() err(%s)", dfile);
                        return false;
                    }
#else

#if defined(S_ISVTX)
                    if(chmod(dfile, stat_.st_mode & (S_IRUSR|S_IWUSR|S_IXUSR
                            |S_IRGRP|S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH|S_ISUID|S_ISGID|S_ISVTX)) < 0)
                    {
                        err_msg("file_copy: chmod() err(%s)", dfile);
                        return false;
                    }
#else
                    if(chmod(dfile, stat_.st_mode & (S_IRUSR|S_IWUSR|S_IXUSR
                            |S_IRGRP|S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH|S_ISUID|S_ISGID)) < 0)
                    {
                        err_msg("file_copy: chmod() err(%s)", dfile);
                        return false;
                    }
#endif

#endif
                    umask(umask_before);
                }
                /// sym link ///
                else if(S_ISLNK(stat_.st_mode)) {
                    char link[PATH_MAX];
                    int n = readlink(spath, link, PATH_MAX);
                    if(n < 0) {
                        err_msg("file_copy: readlink err(%s)", spath);
                        return false;
                    }
                    link[n] = 0;
        
                    if(symlink(link, dfile) < 0) {
                        err_msg("file_copy: symlink err(%s)", dfile);
                        return false;
                    }

                    //chmod(dfile, stat_.st_mode);

                    if(preserve || move) {
                        if(getuid() == 0) {
                            if(lchown(dfile, stat_.st_uid, stat_.st_gid) < 0) {
                                err_msg("file_copy: lchown err(%s)", dfile);
                                return false;
                            }
                        }
/*
                        struct utimbuf utb;

                        utb.actime = stat_.st_atime;
                        utb.modtime = stat_.st_mtime;

                        utime(dfile, &utb);
*/
                    }
                }
                /// character device ///
                else if(S_ISCHR(stat_.st_mode)) {
                    int major_ = major(stat_.st_rdev);
                    int minor_ = minor(stat_.st_rdev);
                    mode_t umask_before = umask(0);
                    int result = mknod(dfile, stat_.st_mode 
                                 & (S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP
                                 |S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH) 
                                 | S_IFCHR, makedev(major_, minor_));
                    umask(umask_before);

                    if(result < 0) {
                        err_msg("making character device(%s) is err. only root can do that", dfile);
                        return false;
                    }

                    if(preserve || move) {
                        if(getuid() == 0) {
                            if(chown(dfile, stat_.st_uid, stat_.st_gid) < 0) {
                                err_msg("file_copy: chown err(%s)", dfile);
                                return false;
                            }
                        }

                        struct utimbuf utb;

                        utb.actime = stat_.st_atime;
                        utb.modtime = stat_.st_mtime;

                        if(utime(dfile, &utb) < 0) {
                            err_msg("file_copy: utime err(%s)", dfile);
                            return false;
                        }
                    }

                    umask_before = umask(0);
#if defined(S_ISTXT)
                    if(chmod(dfile, stat_.st_mode & (S_IRUSR|S_IWUSR|S_IXUSR
                    |S_IRGRP|S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH|S_ISUID|S_ISGID|S_ISTXT)) < 0) 
                    {
                        err_msg("file_copy: chmod() err(%s)", dfile);
                        return false;
                    }
#else

#if defined(S_ISVTX)
                    if(chmod(dfile, stat_.st_mode & (S_IRUSR|S_IWUSR|S_IXUSR
                            |S_IRGRP|S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH|S_ISUID|S_ISGID|S_ISVTX)) < 0)
                    {
                        err_msg("file_copy: chmod() err(%s)", dfile);
                        return false;
                    }
#else
                    if(chmod(dfile, stat_.st_mode & (S_IRUSR|S_IWUSR|S_IXUSR
                            |S_IRGRP|S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH|S_ISUID|S_ISGID)) < 0)
                    {
                        err_msg("file_copy: chmod() err(%s)", dfile);
                        return false;
                    }
#endif

#endif
                    umask(umask_before);
                    
                }
                /// block device ///
                else if(S_ISBLK(stat_.st_mode)) {
                    int major_ = major(stat_.st_rdev);
                    int minor_ = minor(stat_.st_rdev);
                    mode_t umask_before = umask(0);

                    int result = mknod(dfile, stat_.st_mode 
                                 & (S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP
                                 |S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH) 
                                 | S_IFBLK, makedev(major_, minor_));
                    umask(umask_before);
                    if(result < 0) {
                        err_msg("file_copy: making block device(%s) is err.only root can do that", dfile);
                        return false;
                    }

                    if(preserve || move) {
                        if(getuid() == 0) {
                            if(chown(dfile, stat_.st_uid, stat_.st_gid) < 0) {
                                err_msg("file_copy: chown() err (%s)", dfile);
                                return false;
                            }
                        }

                        struct utimbuf utb;

                        utb.actime = stat_.st_atime;
                        utb.modtime = stat_.st_mtime;

                        if(utime(dfile, &utb) < 0) {
                            err_msg("file_copy: utime err(%s)", dfile);
                            return false;
                        }
                    }

                    umask_before = umask(0);
#if defined(S_ISTXT)
                    if(chmod(dfile, stat_.st_mode & (S_IRUSR|S_IWUSR|S_IXUSR
                            |S_IRGRP|S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH|S_ISUID|S_ISGID|S_ISTXT)) < 0)
                    {
                        err_msg("file_copy: chmod() err(%s)", dfile);
                        return false;
                    }
#else

#if defined(S_ISVTX)
                    if(chmod(dfile, stat_.st_mode & (S_IRUSR|S_IWUSR|S_IXUSR
                            |S_IRGRP|S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH|S_ISUID|S_ISGID|S_ISVTX)) < 0)
                    {
                        err_msg("file_copy: chmod() err(%s)", dfile);
                        return false;
                    }
#else
                    if(chmod(dfile, stat_.st_mode & (S_IRUSR|S_IWUSR|S_IXUSR
                            |S_IRGRP|S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH|S_ISUID|S_ISGID)) < 0)
                    {
                        err_msg("file_copy: chmod() err(%s)", dfile);
                        return false;
                    }
#endif

#endif
                    umask(umask_before);
                    
                }
                /// fifo ///
                else if(S_ISFIFO(stat_.st_mode)) {
                    mode_t umask_before = umask(0);
                    int result = mknod(dfile, stat_.st_mode 
                                 & (S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP
                                 |S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH) 
                                 | S_IFIFO, 0);
                    umask(umask_before);
                    if(result < 0) {
                        err_msg("making fifo(%s) is err", dfile);
                        return false;
                    }

                    if(preserve || move) {
                        if(getuid() == 0) {
                            if(chown(dfile, stat_.st_uid, stat_.st_gid) < 0) {
                                err_msg("file_copy: chown() err(%s)", dfile);
                                return false;
                            }
                        }

                        struct utimbuf utb;

                        utb.actime = stat_.st_atime;
                        utb.modtime = stat_.st_mtime;

                        if(utime(dfile, &utb) < 0) 
                        {
                            err_msg("file_copy: utime err(%s)", dfile);
                            return false;
                        }
                    }

                    umask_before = umask(0);
#if defined(S_ISTXT)
                    if(chmod(dfile, stat_.st_mode & (S_IRUSR|S_IWUSR|S_IXUSR
                            |S_IRGRP|S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH|S_ISUID|S_ISGID|S_ISTXT)) < 0)
                    {
                        err_msg("file_copy: chmod() err(%s)", dfile);
                        return false;
                    }
#else

#if defined(S_ISVTX)
                    if(chmod(dfile, stat_.st_mode & (S_IRUSR|S_IWUSR|S_IXUSR
                            |S_IRGRP|S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH|S_ISUID|S_ISGID|S_ISVTX)) < 0)
                    {
                        err_msg("file_copy: chmod() err(%s)", dfile);
                        return false;
                    }
#else
                    if(chmod(dfile, stat_.st_mode & (S_IRUSR|S_IWUSR|S_IXUSR
                            |S_IRGRP|S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH|S_ISUID|S_ISGID)) < 0)
                    {
                        err_msg("file_copy: chmod() err(%s)", dfile);
                        return false;
                    }
#endif

#endif
                    umask(umask_before);
                }
                /// socket ///
                else if(S_ISSOCK(stat_.st_mode)) {
                    mode_t umask_before = umask(0);
                    int result = mknod(dfile, stat_.st_mode 
                                 & (S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP
                                 |S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH) 
                                 | S_IFSOCK, 0);
                    umask(umask_before);
                    if(result < 0) {
                        err_msg("making socket(%s) is err", dfile);
                        return false;
                    }

                    if(preserve || move) {
                        if(getuid() == 0) {
                            if(chown(dfile, stat_.st_uid, stat_.st_gid) < 0) {
                                err_msg("file_copy: chown() err(%s)", dfile);
                                return false;
                            }
                        }

                        struct utimbuf utb;

                        utb.actime = stat_.st_atime;
                        utb.modtime = stat_.st_mtime;

                        if(utime(dfile, &utb) < 0)
                        {
                            err_msg("file_copy: chmod() err(%s)", dfile);
                            return false;
                        }
                    }

                    umask_before = umask(0);
#if defined(S_ISTXT)
                    if(chmod(dfile, stat_.st_mode & (S_IRUSR|S_IWUSR|S_IXUSR
                            |S_IRGRP|S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH|S_ISUID|S_ISGID|S_ISTXT)) < 0)
                    {
                        err_msg("file_copy: chmod() err(%s)", dfile);
                        return false;
                    }
#else

#if defined(S_ISVTX)
                    if(chmod(dfile, stat_.st_mode & (S_IRUSR|S_IWUSR|S_IXUSR
                            |S_IRGRP|S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH|S_ISUID|S_ISGID|S_ISVTX)) < 0)
                    {
                        err_msg("file_copy: chmod() err(%s)", dfile);
                        return false;
                    }
#else
                    if(chmod(dfile, stat_.st_mode & (S_IRUSR|S_IWUSR|S_IXUSR
                            |S_IRGRP|S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH|S_ISUID|S_ISGID)) < 0)
                    {
                        err_msg("file_copy: chmod() err(%s)", dfile);
                        return false;
                    }
#endif

#endif
                    umask(umask_before);
                    
                }
                /// non suport file ///
                else {
                    err_msg("file_copy: sorry, non support file type(%s)", spath);
                    return false;
                }

                if(move) {
                    if(!file_remove(spath, true, false)) {
                        return false;
                    }
                }
            }
        }

        return true;
    }
}

/// don't put '/' at last

bool file_remove(char* path, bool no_ctrl_c, bool msg)
{
    /// key input ? ///
    if(!no_ctrl_c) {
        fd_set mask;

        FD_ZERO(&mask);
        FD_SET(0, &mask);

        struct timeval tv;
        tv.tv_sec = 0;
        tv.tv_usec = 0;

        select(1, &mask, NULL, NULL, &tv);
    
        if(FD_ISSET(0, &mask)) {
            int meta;
            int key = mgetch(&meta);

            if(key == 3 || key == 7 || key == 27) {    // CTRL-C and CTRL-G
                return false;
            }
        }
    }

    /// cancel ///
    struct stat stat_;

    if(lstat(path, &stat_) < 0) {
        return true;
    }
    else {
        /// file ///
        if(!S_ISDIR(stat_.st_mode)) {
            if(!(stat_.st_mode&S_IWUSR) && !(stat_.st_mode&S_IWGRP) &&
                !(stat_.st_mode&S_IWOTH) )
            {
                if(gWriteProtected == kWPNoAll) {
                    return true;
                }
                else if(gWriteProtected == kWPYesAll) {
                }
                else {
                    const char* str[] = {
                        "No", "Yes", "No(all)", "Yes(all)", "Cancel"
                    };
                    
                    char buf[PATH_MAX+20];
                    sprintf(buf, "remove write-protected file(%s)", path);
                    int ret = select_str(buf, (char**)str, 5, 4);

                    switch(ret) {
                        case 0:
                            return true;
                            break;

                        case 1:
                            break;

                        case 2:
                            gWriteProtected = kWPNoAll;
                            return true;
                            break;

                        case 3:
                            gWriteProtected = kWPYesAll;
                            break;

                        case 4:
                            return false;

                        default:
                            break;
                    }
                }
            }

            /// msg ///
            if(msg) {
                char buf[PATH_MAX+20];
                sprintf(buf, "deleting %s", path);
                msg_nonstop("%s", buf);
            }

            /// go ///
            if(unlink(path) < 0) {
                switch(errno) {
                    case EACCES:
                    case EPERM:
                        err_msg("file_remove: permission denied. (%s)", path);
                        break;

                    case EROFS:
                        err_msg("file_remove: file system is readonly. (%s)", path);
                        break;

                    default:
                        err_msg("file_remove: unlink err(%s)", path);
                        break;
                }
                return false;
            }
        }
        /// directory ///
        else {
            char buf[PATH_MAX+20];
            sprintf(buf, "entering %s", path);
            msg_nonstop("%s", buf);
            
            DIR* dir = opendir(path);
            if(dir == NULL) {
                err_msg("file_remove: opendir err(%s)", path);
                return false;
            }
            struct dirent* entry;
            while(entry = readdir(dir)) {
                if(strcmp(entry->d_name, ".") != 0
                    && strcmp(entry->d_name, "..") != 0)
                {
                    char path2[PATH_MAX];
                    strcpy(path2, path);
                    strcat(path2, "/");
                    strcat(path2, entry->d_name);

                    if(!file_remove(path2, no_ctrl_c, false)) {
                        closedir(dir);
                        return false;
                    }
                }
            }
            if(closedir(dir) < 0) {
                err_msg("file_remove: closedir() err");
                return false;
            }

            if(rmdir(path) < 0) {
                switch(errno) {
                    case EACCES:
                    case EPERM:
                        err_msg("file_remove: permission denied.(%s)", path);
                        break;

                    case EROFS:
                        err_msg("file_remove: file system is readonly. (%s)", path);
                        break;

                    default:
                        err_msg("file_remove: rmdir err(%s)", path);
                        break;
                }
                return false;
            }
        }

        
        return true;
    }
}

