/*
 *  Copyright (C) 2005 UCHINO Satoshi.  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

#include "conffile.h"
#include <iostream>
#include <fstream>

char ConfigFile::default_xdg_data_home[]   = ".local/share/";
char ConfigFile::default_xdg_config_home[] = ".config/";
char ConfigFile::default_xdg_data_dirs[]   = "/usr/local/share/:/usr/share/";
char ConfigFile::default_xdg_config_dirs[] = "/etc/xdg/";

vector<string>* ConfigFile::getXdgDataDirs()
{
  vector<string>* dirList = new vector<string>;
  char *tmp;
  // check XDG_DATA_HOME first : this is a single base directory
  tmp = getenv("XDG_DATA_HOME");
  if (tmp == NULL) {
    if ((tmp = getenv("HOME")) != NULL) {
      string item(tmp);
      item = item + "/" + default_xdg_data_home;
      dirList->push_back(item);
    }
  } else {
    string item(tmp);
    if (item[item.size() - 1] != '/')
      item = item + "/";
    dirList->push_back(item);
  }
  // then check XDG_DATA_DIRS : this is a set of base directories
  tmp = getenv("XDG_DATA_DIRS");
  if (tmp == NULL)
    tmp = default_xdg_data_dirs;
  tmp = strdup(tmp);
  for (tmp = strtok(tmp, ":"); tmp; tmp = strtok(NULL, ":")) {
    string item(tmp);
    if (item[item.size() - 1] != '/')
      item = item + "/";
    dirList->push_back(item);
  }
  free(tmp);
  return dirList;
}

vector<string>* ConfigFile::getXdgConfigDirs()
{
  vector<string>* dirList = new vector<string>;
  char *tmp;
  // check XDG_CONFIG_HOME first : this is a single base directory
  tmp = getenv("XDG_CONFIG_HOME");
  if (tmp == NULL) {
    if ((tmp = getenv("HOME")) != NULL) {
      string item(tmp);
      item = item + "/" + default_xdg_config_home;
      dirList->push_back(item);
    }
  } else {
    string item(tmp);
    if (item[item.size() - 1] != '/')
      item = item + "/";
    dirList->push_back(item);
  }
  // then check XDG_CONFIG_DIRS : this is a set of base directories
  tmp = getenv("XDG_CONFIG_DIRS");
  if (tmp == NULL)
    tmp = default_xdg_data_dirs;
  tmp = strdup(tmp);
  for (tmp = strtok(tmp, ":"); tmp; tmp = strtok(NULL, ":")) {
    string item(tmp);
    if (item[item.size() - 1] != '/')
      item = item + "/";
    dirList->push_back(item);
  }
  free(tmp);
  return dirList;
}

/* split locale into lang, country, and modifier */
void ConfigFile::splitLocale(const string& locale, string& lang, string& country, string& modifier)
{
  string::size_type pos;
  pos = locale.find('_');
  if (pos == string::npos) {
    country = "";
    lang = locale;
    pos = lang.find('@');
    if (pos == string::npos) {	/* lang */
      modifier = "";
    } else {			/* lang@modifier */
      modifier = lang.substr(pos + 1);
      lang     = lang.substr(0, pos);
    }
  } else {
    country = locale.substr(pos + 1);
    lang    = locale.substr(0, pos);
    pos = country.find('@');
    if (pos == string::npos) {	/* lang_country */
      modifier = "";
    } else {			/* lang_country@modifier */
      modifier = country.substr(pos + 1);
      country  = country.substr(0, pos);
    }
  }
}

/* read section and retrieve content corresponding to keyset
 * if keySet is empty, all keys are retrieved.
 */
ConfigFile::SpItemMap
ConfigFile::readSection(const string& filename, 
			const string& section,
			const string& locale,
			const set<string>& keySet)
{
  set<string> sectionSet;
  sectionSet.insert(section);
//cout << __FUNCTION__ << ": section = '" << section << "'" << endl;
  SpItemMapMap result = readSectionSet(filename, sectionSet, locale, keySet);
  if (result && result->find(section) != result->end())
    return result->find(section)->second;

cout << __FUNCTION__ << ": could not find section = '" << section << "'" << endl;
  return SpItemMap();
}

/* read sections and retrieve content corresponding to keyset
 * if sectionSet is empty, all sections are retrieved.
 * similarly, if keySet is empty, all keys are retrieved.
 */
ConfigFile::SpItemMapMap
ConfigFile::readSectionSet(const string& filename, 
			   const set<string>& sectionSet,
			   const string& locale,
			   const set<string>& keySet)
{
  ifstream fin(filename.c_str());
  if (!fin) {
    cerr << __FUNCTION__ << ": cannot open '" << filename << "'" << endl;
    return SpItemMapMap();
  }

  string linebuf;
  string::size_type pos;
  string lang, country, modifier, section;
  bool insection = false;
  SpItemMapMap itemMapMap = SpItemMapMap(new ItemMapMap);
  SpItemMap itemMap;

  splitLocale(locale, lang, country, modifier);

  while (getline(fin, linebuf, '\n') && !fin.eof()) {
    if (linebuf.length() < 1)
      continue;
    switch (linebuf[0]) {
    case '#':	/* comment */
      break;
    case '[':	/* beginning of a section */
      insection = false;
      if ((pos = linebuf.find(']')) == string::npos) {
//cout << "no matcing ']' could not be found" << endl;
	break;	/* break switch */
      }

      section = linebuf.substr(1, pos - 1);
      if (!sectionSet.empty() && (sectionSet.find(section) == sectionSet.end())) {
//cout << "section '" << section << "' is not in the given sectionSet" << endl;
	break;	/* break switch */
      }

      if (itemMapMap->find(section) != itemMapMap->end())
	itemMap = itemMapMap->find(section)->second;
      else {
	itemMap = SpItemMap(new ItemMap());
	itemMapMap->insert(ItemMapMap::value_type(section, itemMap));
      }
//cout << "Section : " << section << endl;

      insection = true;
      break;
    default:	/* body of a section */
      if (insection) {
	string key, keyLang, keyCountry, keyModifier;
	Item newItem;

	/* split key and body */
	pos = linebuf.find('=');
	if (pos == string::npos)
	  break;	/* break switch */

	key = linebuf.substr(0, pos);
	newItem.body = linebuf.substr(pos + 1);

	/* split key and locale */
	pos = key.find('[');
	if (pos == string::npos) {
	  newItem.hasLang = false;
	  newItem.hasCountry  = false;
	  newItem.hasModifier = false;
	} else {
	  keyLang = key.substr(pos + 1);
	  key     = key.substr(0, pos);

	  pos = keyLang.find(']');
	  if (pos == string::npos)
	    break;	/* break switch : no ']' found */

	  keyLang = keyLang.substr(0, pos);
	  newItem.locale  = keyLang;
	  newItem.hasLang = true;

	  /* split lang, country, and modifier */
	  splitLocale(keyLang, keyLang, keyCountry, keyModifier);
	  newItem.hasCountry  = (keyCountry.size()  > 0);
	  newItem.hasModifier = (keyModifier.size() > 0);
	}
//cout << "key = '" << key << "'" << endl;
//cout << "keyLang     = '" << keyLang << "'" << endl;
//cout << "keyCountry  = '" << keyCountry << "'" << endl;
//cout << "keyModifier = '" << keyModifier << "'" << endl;
//cout << "body = '" << newItem.body << "'" << endl;

	/* locale matching */
	if ((newItem.hasLang     && keyLang     != lang    ) ||
	    (newItem.hasCountry  && keyCountry  != country ) ||
	    (newItem.hasModifier && keyModifier != modifier)) {
//cout << "locale does not match" << endl;
	  break;	/* break switch : locale does not match */
	}
	/* check key */
	if (!keySet.empty() && keySet.find(key) == keySet.end()) {
//cout << "key does not match" << endl;
	  break;	/* break switch : key does not match */
	}

	/* check if this is more preferred item */
	ItemMap::iterator i;
	i = itemMap->find(key);
	if (i != itemMap->end()) {	/* key already exists */
	  if ((!i->second.hasLang     && newItem.hasLang    ) ||
	      (!i->second.hasCountry  && newItem.hasCountry ) ||
	      ((!i->second.hasCountry || newItem.hasCountry) &&
	       !i->second.hasModifier && newItem.hasModifier)) {
	    /* replace with new one */
//cout << "replacing with new one" << endl;
	    i->second = newItem;
	  }
	} else {
//cout << "inserting new item : " << key << " = " << newItem.body << endl;
	  itemMap->insert(ItemMap::value_type(key, newItem));
	}
      }
      break;
    }
  }
  fin.close();
//cout << "-" << __FUNCTION__ << endl;
  return itemMapMap;
}
