/**
 * @file libgalago-gtk/galago-gtk-presence-label.c Presence label widget
 *
 * @Copyright (C) 2004-2006 Christian Hammond.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago-gtk/galago-gtk-presence-label.h>
#include <libgalago-gtk/galago-gtk-private.h>
#include <gtk/gtk.h>

struct _GalagoGtkPresenceLabelPriv
{
	GalagoPresence *presence;

	GalagoPerson *person;
	GalagoAccount *account;
	char *username;

	gboolean dim_idle;
	gboolean show_idle_times;

	gulong updated_id;
	gulong account_destroy_id;
	gulong account_display_name_id;
};

static void galago_gtk_presence_label_class_init(GalagoGtkPresenceLabelClass *klass);
static void galago_gtk_presence_label_init(GalagoGtkPresenceLabel *label);
static void galago_gtk_presence_label_finalize(GObject *obj);
static void galago_gtk_presence_label_destroy(GtkObject *obj);
static void update_label(GalagoPresence *presence,
						 GalagoGtkPresenceLabel *label);

static GtkLabelClass *parent_class = NULL;

GType
galago_gtk_presence_label_get_type(void)
{
	static GType type = 0;

	if (!type)
	{
		static const GTypeInfo info =
		{
			sizeof(GalagoGtkPresenceLabelClass),
			NULL,
			NULL,
			(GClassInitFunc)galago_gtk_presence_label_class_init,
			NULL,
			NULL,
			sizeof(GalagoGtkPresenceLabel),
			0,
			(GInstanceInitFunc)galago_gtk_presence_label_init
		};

		type = g_type_register_static(GTK_TYPE_LABEL,
									  "GalagoGtkPresenceLabel", &info, 0);
	}

	return type;
}

static void
galago_gtk_presence_label_class_init(GalagoGtkPresenceLabelClass *klass)
{
	GObjectClass *gobject_class;
	GtkObjectClass *object_class;

	parent_class = g_type_class_peek_parent(klass);

	gobject_class = G_OBJECT_CLASS(klass);
	object_class  = GTK_OBJECT_CLASS(klass);

	gobject_class->finalize = galago_gtk_presence_label_finalize;

	object_class->destroy = galago_gtk_presence_label_destroy;
}

static void
galago_gtk_presence_label_init(GalagoGtkPresenceLabel *label)
{
	label->priv = g_new0(GalagoGtkPresenceLabelPriv, 1);

	label->priv->dim_idle        = TRUE;
	label->priv->show_idle_times = TRUE;
}

static void
galago_gtk_presence_label_finalize(GObject *obj)
{
	GalagoGtkPresenceLabel *label;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_PRESENCE_LABEL(obj));

	label = GALAGO_GTK_PRESENCE_LABEL(obj);

	g_free(label->priv);

	if (G_OBJECT_CLASS(parent_class)->finalize)
		G_OBJECT_CLASS(parent_class)->finalize(obj);
}

static void
galago_gtk_presence_label_destroy(GtkObject *obj)
{
	GalagoGtkPresenceLabel *label;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_PRESENCE_LABEL(obj));

	label = GALAGO_GTK_PRESENCE_LABEL(obj);

	GALAGO_GTK_DISCONNECT_HANDLER(label->priv->presence,
								  label->priv->updated_id);
	GALAGO_GTK_DISCONNECT_HANDLER(label->priv->account,
								  label->priv->account_destroy_id);
	GALAGO_GTK_DISCONNECT_HANDLER(label->priv->account,
								  label->priv->account_display_name_id);

	label->priv->presence = NULL;
	label->priv->account = NULL;

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		GTK_OBJECT_CLASS(parent_class)->destroy(obj);
}

static void
update_label(GalagoPresence *presence, GalagoGtkPresenceLabel *label)
{
	GalagoAccount *account = NULL;
	char *str;

	if (presence == NULL)
		presence = label->priv->presence;

	if (presence != NULL)
		account = galago_presence_get_account(presence);
	else if (label->priv->account != NULL)
		account = label->priv->account;
	else if (label->priv->person != NULL)
	{
		account = galago_person_get_priority_account(label->priv->person);

		if (account == NULL)
		{
			const char *display_name;

			display_name = galago_person_get_display_name(label->priv->person);

			if (galago_gtk_presence_label_get_dim_idle(label))
			{
				str = g_strdup_printf("<span color='dim grey'>%s</span>",
									  display_name);
			}
			else
			{
				str = g_strdup(display_name);
			}

			gtk_label_set_markup(GTK_LABEL(label), str);

			g_free(str);

			return;
		}
	}

	if (presence != NULL && galago_presence_is_idle(presence))
	{
		char *idle_str = NULL;
		time_t idle_time;

		idle_time = galago_presence_get_idle_time(presence);

		if (galago_gtk_presence_label_get_show_idle_times(label) &&
			idle_time > 0)
		{
			int hrs, mins;

			hrs  = idle_time / 3600;
			mins = (idle_time / 60) % 60;

			if (hrs > 0)
				idle_str = g_strdup_printf(" (%dh%02dm)", hrs, mins);
			else
				idle_str = g_strdup_printf(" (%dm)", mins);
		}

		if (galago_gtk_presence_label_get_dim_idle(label))
		{
			str = g_strdup_printf(
				"<span color='dim grey'>%s%s</span>",
				galago_account_get_display_name(account),
				(idle_str == NULL ? "" : idle_str));
		}
		else
		{
			str = g_strdup_printf("%s%s",
								  galago_account_get_display_name(account),
								  (idle_str == NULL ? "" : idle_str));
		}

		if (idle_str != NULL)
			g_free(idle_str);
	}
	else if ((presence == NULL || !galago_presence_is_available(presence)) &&
			 galago_gtk_presence_label_get_dim_idle(label))
	{
		str = g_strdup_printf(
			"<span color='dim grey'>%s</span>",
			galago_account_get_display_name(account));
	}
	else
		str = g_strdup(galago_account_get_display_name(account));

	gtk_label_set_markup(GTK_LABEL(label), str);

	g_free(str);
}

static void
account_destroy_cb(GalagoAccount *account, GalagoGtkPresenceLabel *label)
{
	galago_gtk_presence_label_set_account(label, NULL);
}

static void
account_display_name_changed_cb(GalagoAccount *account,
								GalagoGtkPresenceLabel *label)
{
	gtk_label_set_text(GTK_LABEL(label),
					   galago_account_get_display_name(account));
}

static void
queue_update_label(GalagoGtkPresenceLabel *label)
{
	GALAGO_GTK_DISCONNECT_HANDLER(label->priv->presence,
								  label->priv->updated_id);

	if (label->priv->person == NULL && label->priv->account == NULL)
	{
		char *str = g_strdup_printf("<i>%s</i>", _("No presence"));

		gtk_label_set_markup(GTK_LABEL(label), str);

		g_free(str);
	}
	else
	{
		GalagoPresence *presence = NULL;
		GalagoAccount *account;

		account = (label->priv->person == NULL
				   ? label->priv->account
				   : galago_person_get_priority_account(label->priv->person));

		if (account != NULL)
			presence = galago_account_get_presence(account, TRUE);

		label->priv->presence = presence;

		if (presence != NULL)
		{
			label->priv->updated_id =
				g_signal_connect(G_OBJECT(presence), "changed",
								 G_CALLBACK(update_label), label);
		}

		update_label(presence, label);
	}
}

GtkWidget *
galago_gtk_presence_label_new(void)
{
	GalagoGtkPresenceLabel *label;

	label = g_object_new(GALAGO_GTK_TYPE_PRESENCE_LABEL, NULL);

	queue_update_label(label);

	return GTK_WIDGET(label);
}

void
galago_gtk_presence_label_set_account(GalagoGtkPresenceLabel *label,
									  GalagoAccount *account)
{
	g_return_if_fail(label != NULL);
	g_return_if_fail(GALAGO_GTK_IS_PRESENCE_LABEL(label));

	if (label->priv->account == account)
		return;

	GALAGO_GTK_DISCONNECT_HANDLER(label->priv->account,
								  label->priv->account_destroy_id);
	GALAGO_GTK_DISCONNECT_HANDLER(label->priv->account,
								  label->priv->account_display_name_id);

	label->priv->account = account;

	if (account != NULL)
	{
		label->priv->account_destroy_id =
			g_signal_connect(G_OBJECT(account), "destroy",
							 G_CALLBACK(account_destroy_cb), label);

		label->priv->account_display_name_id =
			g_signal_connect(G_OBJECT(account), "display-name-changed",
							 G_CALLBACK(account_display_name_changed_cb),
							 label);
	}

	queue_update_label(label);
}

void
galago_gtk_presence_label_set_person(GalagoGtkPresenceLabel *label,
									 GalagoPerson *person)
{
	g_return_if_fail(label != NULL);
	g_return_if_fail(GALAGO_GTK_IS_PRESENCE_LABEL(label));

	if (label->priv->person == person)
		return;

	label->priv->person = person;

	queue_update_label(label);
}

void
galago_gtk_presence_label_set_dim_idle(GalagoGtkPresenceLabel *label,
									   gboolean dim_idle)
{
	g_return_if_fail(label != NULL);
	g_return_if_fail(GALAGO_GTK_IS_PRESENCE_LABEL(label));

	if (label->priv->dim_idle == dim_idle)
		return;

	label->priv->dim_idle = dim_idle;

	queue_update_label(label);
}

void
galago_gtk_presence_label_set_show_idle_times(GalagoGtkPresenceLabel *label,
											  gboolean show_idle_times)
{
	g_return_if_fail(label != NULL);
	g_return_if_fail(GALAGO_GTK_IS_PRESENCE_LABEL(label));

	if (label->priv->show_idle_times == show_idle_times)
		return;

	label->priv->show_idle_times = show_idle_times;

	queue_update_label(label);
}

void
galago_gtk_presence_label_update(GalagoGtkPresenceLabel *label)
{
	g_return_if_fail(label != NULL);
	g_return_if_fail(GALAGO_GTK_IS_PRESENCE_LABEL(label));

	queue_update_label(label);
}

GalagoAccount *
galago_gtk_presence_label_get_account(const GalagoGtkPresenceLabel *label)
{
	g_return_val_if_fail(label != NULL, NULL);
	g_return_val_if_fail(GALAGO_GTK_IS_PRESENCE_LABEL(label), NULL);

	return label->priv->account;
}

GalagoPerson *
galago_gtk_presence_label_get_person(const GalagoGtkPresenceLabel *label)
{
	g_return_val_if_fail(label != NULL, NULL);
	g_return_val_if_fail(GALAGO_GTK_IS_PRESENCE_LABEL(label), NULL);

	return label->priv->person;
}

gboolean
galago_gtk_presence_label_get_dim_idle(const GalagoGtkPresenceLabel *label)
{
	g_return_val_if_fail(label != NULL, FALSE);
	g_return_val_if_fail(GALAGO_GTK_IS_PRESENCE_LABEL(label), FALSE);

	return label->priv->dim_idle;
}

gboolean
galago_gtk_presence_label_get_show_idle_times(
	const GalagoGtkPresenceLabel *label)
{
	g_return_val_if_fail(label != NULL, FALSE);
	g_return_val_if_fail(GALAGO_GTK_IS_PRESENCE_LABEL(label), FALSE);

	return label->priv->show_idle_times;
}
