/*
 * Galago Account API
 *
 * Copyright (C) 2004-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-account.h>
#include <libgalago/galago-assert.h>
#include <libgalago/galago-context.h>
#include <libgalago/galago-core.h>
#include <libgalago/galago-private.h>
#include <stdio.h>
#include <string.h>

struct _GalagoAccountPrivate
{
	GalagoService *service;
	GalagoPerson *person;
	GalagoPresence *presence;
	GalagoImage *avatar;
	char *username;
	char *display_name;

	gboolean connected;

	GHashTable *contacts_table;
	GList *contacts;
};

enum
{
	PROP_0,
	PROP_SERVICE,
	PROP_PERSON,
	PROP_PRESENCE,
	PROP_AVATAR,
	PROP_USERNAME,
	PROP_DISPLAYNAME,
	PROP_CONNECTED
};

enum
{
	CONNECTION_STATE_CHANGED,
	DISPLAY_NAME_CHANGED,
	PRESENCE_CREATED,
	PRESENCE_DELETED,
	AVATAR_SET,
	CONTACT_ADDED,
	CONTACT_REMOVED,
	LAST_SIGNAL
};

static void _galago_dbus_account_set_avatar(GalagoAccount *account,
                                            GalagoImage *avatar);
static void _galago_dbus_account_add_contact(GalagoAccount *account,
                                             GalagoAccount *contact);


/**************************************************************************
 * Object initialization
 **************************************************************************/
static void galago_account_destroy(GalagoObject *gobject);
static void galago_account_dbus_message_append(DBusMessageIter *iter,
											   const GalagoObject *object);
static void *galago_account_dbus_message_get(DBusMessageIter *iter);
static void galago_account_dbus_push_full(GalagoObject *object);
static gchar *galago_account_dbus_get_signature(void);
static void galago_account_set_property(GObject *object, guint prop_id,
										const GValue *value, GParamSpec *pspec);
static void galago_account_get_property(GObject *object, guint prop_id,
										GValue *value, GParamSpec *pspec);

static GalagoObjectClass *parent_class = NULL;
static guint signals[LAST_SIGNAL] = {0};

G_DEFINE_TYPE(GalagoAccount, galago_account, GALAGO_TYPE_OBJECT);

static void
galago_account_class_init(GalagoAccountClass *klass)
{
	GalagoObjectClass *object_class  = GALAGO_OBJECT_CLASS(klass);
	GObjectClass      *gobject_class = G_OBJECT_CLASS(klass);

	parent_class = g_type_class_peek_parent(klass);

	object_class->dbus_interface = GALAGO_DBUS_ACCOUNT_INTERFACE;
	object_class->supports_attrs = TRUE;

	object_class->destroy             = galago_account_destroy;
	object_class->dbus_message_append = galago_account_dbus_message_append;
	object_class->dbus_message_get    = galago_account_dbus_message_get;
	object_class->dbus_push_full      = galago_account_dbus_push_full;
	object_class->dbus_get_signature  = galago_account_dbus_get_signature;

	gobject_class->set_property = galago_account_set_property;
	gobject_class->get_property = galago_account_get_property;

	/**
	 * GalagoAccount::connection-state-changed:
	 * @account: The object which received the signal.
	 *
	 * Emitted when the state of the connection changes, when the
	 * account goes online or offline.
	 */
	signals[CONNECTION_STATE_CHANGED] =
		g_signal_new("connection_state_changed",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoAccountClass,
									 connection_state_changed),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__VOID,
					 G_TYPE_NONE, 0);

	/**
	 * GalagoAccount::display-name-changed:
	 * @account: The object which received the signal.
	 *
	 * Emitted whenever the display name of the account changes.
	 */
	signals[DISPLAY_NAME_CHANGED] =
		g_signal_new("display_name_changed",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoAccountClass, display_name_changed),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__VOID,
					 G_TYPE_NONE, 0);

	/**
	 * GalagoAccount::presence-created:
	 * @account: The object which received the signal.
	 * @presence: The new presence object.
	 *
	 * Emitted whenever a new #GalagoPresence object is created for this
	 * account.
	 */
	signals[PRESENCE_CREATED] =
		g_signal_new("presence_created",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoAccountClass, presence_created),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__POINTER,
					 G_TYPE_NONE, 1,
					 G_TYPE_POINTER);

	/**
	 * GalagoAccount::presence-deleted:
	 * @account: The object which received the signal.
	 *
	 * Emitted whenever a #GalagoPresence object associated with this
	 * account is destroyed.
	 */
	signals[PRESENCE_DELETED] =
		g_signal_new("presence_deleted",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoAccountClass, presence_deleted),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__VOID,
					 G_TYPE_NONE, 0);

	/**
	 * GalagoAccount::avatar-set:
	 * @account: The object which received the signal.
	 * @avatar: The avatar set.
	 *
	 * Emitted whenever an avatar is set for this account.
	 */
	signals[AVATAR_SET] =
		g_signal_new("avatar_set",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoAccountClass, avatar_set),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__POINTER,
					 G_TYPE_NONE, 1,
					 G_TYPE_POINTER);

	/**
	 * GalagoAccount::contact-added:
	 * @account: The object which received the signal.
	 * @contact: The contact added to this account.
	 *
	 * Emitted whenever a contact has been added to this account.
	 */
	signals[CONTACT_ADDED] =
		g_signal_new("contact_added",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoAccountClass, contact_added),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__POINTER,
					 G_TYPE_NONE, 1,
					 G_TYPE_POINTER);

	/**
	 * GalagoAccount::contact-removed:
	 * @account: The object which received the signal.
	 * @contact: The contact removed from this account.
	 *
	 * Emitted whenever a contact has been removed from this account.
	 */
	signals[CONTACT_REMOVED] =
		g_signal_new("contact_removed",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoAccountClass, contact_removed),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__POINTER,
					 G_TYPE_NONE, 1,
					 G_TYPE_POINTER);

	g_object_class_install_property(gobject_class, PROP_SERVICE,
		g_param_spec_object("service", "Service",
							"The service the account is on",
							GALAGO_TYPE_SERVICE,
							G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
							G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	g_object_class_install_property(gobject_class, PROP_PERSON,
		g_param_spec_object("person", "Person",
							"The person the account belongs to",
							GALAGO_TYPE_PERSON,
							G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
							G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	g_object_class_install_property(gobject_class, PROP_PRESENCE,
		g_param_spec_object("presence", "Presence",
							"The account's presence",
							GALAGO_TYPE_PRESENCE, G_PARAM_READABLE |
							G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	g_object_class_install_property(gobject_class, PROP_AVATAR,
		g_param_spec_object("avatar", "Avatar",
							"The account's avatar",
							GALAGO_TYPE_IMAGE, G_PARAM_READWRITE |
							G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	g_object_class_install_property(gobject_class, PROP_USERNAME,
		g_param_spec_string("username", "Username",
							"The account's username",
							NULL,
							G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
							G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	g_object_class_install_property(gobject_class, PROP_DISPLAYNAME,
		g_param_spec_string("display-name", "Display Name",
							"The account's username intended for display",
							NULL, G_PARAM_READWRITE |
							G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	g_object_class_install_property(gobject_class, PROP_CONNECTED,
		g_param_spec_boolean("connected", "Connected",
							 "The account's connected state",
							 FALSE, G_PARAM_READWRITE |
							G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
}

static void
galago_account_init(GalagoAccount *account)
{
	account->priv = g_new0(GalagoAccountPrivate, 1);

	account->priv->contacts_table =
		g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);
}

static void
galago_account_destroy(GalagoObject *object)
{
	GalagoAccount *account = GALAGO_ACCOUNT(object);
	GalagoService *service;
	GalagoPerson *person;

	g_return_if_fail(account != NULL);

	if (account->priv != NULL)
	{
		service = galago_account_get_service(account);
		person  = galago_account_get_person(account);

		_galago_service_remove_account(service, account);

		if (person != NULL)
		{
			_galago_person_remove_account(person, account);
			account->priv->person = NULL;
		}

		if (account->priv->presence != NULL)
			galago_object_destroy(GALAGO_OBJECT(account->priv->presence));

		if (account->priv->avatar != NULL)
			galago_object_destroy(GALAGO_OBJECT(account->priv->avatar));

		if (account->priv->contacts_table != NULL)
			g_hash_table_destroy(account->priv->contacts_table);

		if (account->priv->contacts != NULL)
			g_list_free(account->priv->contacts);

		if (account->priv->username != NULL)
			g_free(account->priv->username);

		g_free(account->priv);
		account->priv = NULL;
	}

	if (GALAGO_OBJECT_CLASS(parent_class)->destroy != NULL)
		GALAGO_OBJECT_CLASS(parent_class)->destroy(object);
}

static void
galago_account_dbus_message_append(DBusMessageIter *iter,
                                   const GalagoObject *object)
{
	GalagoAccount *account = (GalagoAccount *)object;
	const char *username;
	gboolean connected;
	const char *obj_path;

	obj_path = galago_object_get_dbus_path(GALAGO_OBJECT(account));
	galago_dbus_message_iter_append_string_or_nil(iter, obj_path);

	/* Service */
	galago_dbus_message_iter_append_object(iter,
		GALAGO_OBJECT(galago_account_get_service(account)));

	/* Person */
	galago_dbus_message_iter_append_object(iter,
		GALAGO_OBJECT(galago_account_get_person(account)));

	/* Username */
	username = galago_account_get_username(account);
	dbus_message_iter_append_basic(iter, DBUS_TYPE_STRING, &username);

	/* Display name */
	galago_dbus_message_iter_append_string_or_nil(iter,
		galago_account_get_display_name(account));

	/* Connected? */
	connected = galago_account_is_connected(account);
	dbus_message_iter_append_basic(iter, DBUS_TYPE_BOOLEAN, &connected);
}

static void *
galago_account_dbus_message_get(DBusMessageIter *iter)
{
	GalagoService *service = NULL;
	GalagoPerson *person = NULL;
	GalagoAccount *account;
	gboolean connected;
	const char *obj_path;
	const char *display_name;
	const char *username = NULL;

	obj_path = galago_dbus_message_iter_get_string_or_nil(iter);
	dbus_message_iter_next(iter);

	service = galago_dbus_message_iter_get_object(iter, GALAGO_TYPE_SERVICE);
	dbus_message_iter_next(iter);

	person = galago_dbus_message_iter_get_object(iter, GALAGO_TYPE_PERSON);
	dbus_message_iter_next(iter);

	dbus_message_iter_get_basic(iter, &username);
	dbus_message_iter_next(iter);

	display_name = galago_dbus_message_iter_get_string_or_nil(iter);
	dbus_message_iter_next(iter);

	dbus_message_iter_get_basic(iter, &connected);

	/* TODO: Set this all at once! */
	account = _galago_account_new(service, person, username, obj_path);
	galago_account_set_display_name(account, display_name);
	galago_account_set_connected(account, connected);

	return account;
}

static void
galago_account_dbus_push_full(GalagoObject *object)
{
	GalagoAccount *account = GALAGO_ACCOUNT(object);
	GalagoPresence *presence = galago_account_get_presence(account, FALSE);
	GList *l;

	if (presence != NULL)
	{
		_galago_dbus_account_create_presence(account, presence);
		galago_dbus_object_push_full(GALAGO_OBJECT(presence));
	}

	_galago_dbus_account_set_avatar(account,
		galago_account_get_avatar(account, FALSE));

	for (l = galago_account_get_contacts(account, FALSE);
		 l != NULL;
		 l = l->next)
	{
		_galago_dbus_account_add_contact(account, (GalagoAccount *)l->data);
	}

	if (GALAGO_OBJECT_CLASS(parent_class)->dbus_push_full != NULL)
		GALAGO_OBJECT_CLASS(parent_class)->dbus_push_full(object);
}

static gchar *
galago_account_dbus_get_signature(void)
{
	return g_strconcat(
		DBUS_TYPE_STRING_AS_STRING,  // object path
		galago_object_type_get_dbus_signature(GALAGO_TYPE_SERVICE), // service
		galago_object_type_get_dbus_signature(GALAGO_TYPE_PERSON),  // person
		DBUS_TYPE_STRING_AS_STRING,  // username
		DBUS_TYPE_STRING_AS_STRING,  // display name
		DBUS_TYPE_BOOLEAN_AS_STRING, // connected
		NULL);
}

static void
galago_account_set_property(GObject *object, guint prop_id,
							const GValue *value, GParamSpec *pspec)
{
	GalagoAccount *account = GALAGO_ACCOUNT(object);

	switch (prop_id)
	{
		case PROP_SERVICE:
			account->priv->service = GALAGO_SERVICE(g_value_get_object(value));
			g_object_notify(G_OBJECT(account), "service");
			break;

		case PROP_PERSON:
			account->priv->person = GALAGO_PERSON(g_value_get_object(value));
			g_object_notify(G_OBJECT(account), "person");
			break;

		case PROP_AVATAR:
			galago_account_set_avatar(account,
				GALAGO_IMAGE(g_value_get_object(value)));
			break;

		case PROP_USERNAME:
			account->priv->username = g_value_dup_string(value);
			g_object_notify(G_OBJECT(account), "username");
			break;

		case PROP_DISPLAYNAME:
			galago_account_set_display_name(account,
											g_value_get_string(value));
			break;

		case PROP_CONNECTED:
			galago_account_set_connected(account, g_value_get_boolean(value));
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void
galago_account_get_property(GObject *object, guint prop_id,
							GValue *value, GParamSpec *pspec)
{
	GalagoAccount *account = GALAGO_ACCOUNT(object);

	switch (prop_id)
	{
		case PROP_SERVICE:
			g_value_set_object(value, galago_account_get_service(account));
			break;

		case PROP_PERSON:
			g_value_set_object(value, galago_account_get_person(account));
			break;

		case PROP_PRESENCE:
			g_value_set_object(value, galago_account_get_presence(account,
																  TRUE));
			break;

		case PROP_AVATAR:
			g_value_set_object(value, galago_account_get_avatar(account,
																TRUE));
			break;

		case PROP_USERNAME:
			g_value_set_string(value, galago_account_get_username(account));
			break;

		case PROP_DISPLAYNAME:
			g_value_set_string(value,
				galago_account_get_display_name(account));
			break;

		case PROP_CONNECTED:
			g_value_set_boolean(value, galago_account_is_connected(account));
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

GalagoAccount *
_galago_account_new(GalagoService *service, GalagoPerson *person,
					const char *username, const char *obj_path)
{
	GalagoAccount *account;
	GalagoOrigin service_origin, person_origin;

	g_return_val_if_fail(service != NULL,                       NULL);
	g_return_val_if_fail(person  != NULL,                       NULL);
	g_return_val_if_fail(GALAGO_IS_SERVICE(service),            NULL);
	g_return_val_if_fail(GALAGO_IS_PERSON(person),              NULL);
	g_return_val_if_fail(username != NULL && *username != '\0', NULL);
	g_return_val_if_fail(obj_path == NULL || *obj_path != '\0', NULL);

	service_origin = galago_object_get_origin(GALAGO_OBJECT(service));
	person_origin  = galago_object_get_origin(GALAGO_OBJECT(person));

	g_return_val_if_fail(service_origin == person_origin, NULL);

	account = galago_service_get_account(service, username, FALSE);

	if (account == NULL)
	{
		galago_context_push(galago_object_get_context(GALAGO_OBJECT(service)));
		account = g_object_new(GALAGO_TYPE_ACCOUNT,
							   "service", service,
							   "person", person,
							   "username", username,
							   "origin", service_origin,
							   NULL);

		/* Set the object path, if specified. */
		if (service_origin == GALAGO_REMOTE)
			galago_object_set_dbus_path(GALAGO_OBJECT(account), obj_path);

		_galago_person_add_account(person, account);
		_galago_service_add_account(service, account);
		galago_context_pop();
	}

	return account;
}


/**************************************************************************
 * GalagoAccount API
 **************************************************************************/

/**
 * galago_account_set_connected
 * @account:   The account.
 * @connected: The connected state.
 *
 * Sets whether or not the account is connected.
 */
void
galago_account_set_connected(GalagoAccount *account, gboolean connected)
{
	GalagoPresence *presence;

	g_return_if_fail(account != NULL);
	g_return_if_fail(GALAGO_IS_ACCOUNT(account));

	if (account->priv->connected == connected &&
		account->priv->presence != NULL)
	{
		return;
	}

	account->priv->connected = connected;

	presence = galago_account_get_presence(account, FALSE);

	if (presence == NULL)
	{
		if (GALAGO_OBJECT_IS_LOCAL(account))
			presence = galago_account_create_presence(account);
	}

	if (presence != NULL)
	{
		galago_context_push(galago_object_get_context(GALAGO_OBJECT(presence)));

		if (connected)
		{
			if (galago_presence_has_status(presence, "offline") ||
				galago_presence_get_active_status(presence) == NULL)
			{
				galago_presence_add_status(presence,
					galago_status_new(GALAGO_STATUS_AVAILABLE, "available",
									  "Available", TRUE));
			}
		}
		else
		{
			galago_presence_set_idle(presence, FALSE, 0);

			if (!galago_presence_has_status(presence, "offline"))
			{
				galago_presence_add_status(presence,
					galago_status_new(GALAGO_STATUS_OFFLINE, "offline",
									  "Offline", TRUE));
			}
		}

		galago_context_pop();
	}

	if (GALAGO_OBJECT_IS_LOCAL(account) && galago_is_connected() &&
		galago_is_feed())
	{
		galago_dbus_send_message(GALAGO_OBJECT(account), "SetConnected",
								 galago_value_new(GALAGO_VALUE_TYPE_BOOLEAN,
												  &connected, NULL),
								 NULL);
	}

	g_object_notify(G_OBJECT(account), "connected");

	/* XXX Is this redundant? */
	g_signal_emit(account, signals[CONNECTION_STATE_CHANGED], 0);
}

/**
 * galago_account_get_service
 * @account: The account.
 *
 * Returns an account's service.
 *
 * Returns: The account's service.
 */
GalagoService *
galago_account_get_service(const GalagoAccount *account)
{
	g_return_val_if_fail(account != NULL,            NULL);
	g_return_val_if_fail(GALAGO_IS_ACCOUNT(account), NULL);

	return account->priv->service;
}

void
_galago_account_set_person(GalagoAccount *account, GalagoPerson *person)
{
	g_return_if_fail(account != NULL);
	g_return_if_fail(person  != NULL);
	g_return_if_fail(GALAGO_IS_ACCOUNT(account));
	g_return_if_fail(GALAGO_IS_PERSON(person));
	g_return_if_fail(galago_object_get_origin(GALAGO_OBJECT(person)) ==
					 galago_object_get_origin(GALAGO_OBJECT(account)));

	if (account->priv->person == person)
		return;

	_galago_person_remove_account(account->priv->person, account);
	account->priv->person = person;
	_galago_person_add_account(person, account);
}

/**
 * galago_account_get_person
 * @account: The account.
 *
 * Returns the person that owns an account.
 *
 * Returns: The person that owns the account.
 */
GalagoPerson *
galago_account_get_person(const GalagoAccount *account)
{
	g_return_val_if_fail(account != NULL,            NULL);
	g_return_val_if_fail(GALAGO_IS_ACCOUNT(account), NULL);

	return account->priv->person;
}

/**
 * galago_account_get_username
 * @account: The account.
 *
 * Returns an account's username.
 *
 * Returns: The account's username.
 */
const char *
galago_account_get_username(const GalagoAccount *account)
{
	g_return_val_if_fail(account != NULL,            NULL);
	g_return_val_if_fail(GALAGO_IS_ACCOUNT(account), NULL);

	return account->priv->username;
}

/**
 * galago_account_is_connected
 * @account: The account.
 *
 * Returns whether or not an account is connected.
 *
 * Returns: The account's connected state.
 */
gboolean
galago_account_is_connected(const GalagoAccount *account)
{
	g_return_val_if_fail(account != NULL,            FALSE);
	g_return_val_if_fail(GALAGO_IS_ACCOUNT(account), FALSE);

	return account->priv->connected;
}

/**
 * galago_account_set_display_name
 * @account:      The account.
 * @display_name: The display name.
 *
 * Sets the account's displayed name.
 *
 * This is the alias that the account may be shown as on another client.
 * It's purely optional.
 */
void
galago_account_set_display_name(GalagoAccount *account,
                                const char *display_name)
{
	g_return_if_fail(account != NULL);
	g_return_if_fail(GALAGO_IS_ACCOUNT(account));

	if (display_name != NULL &&
		(*display_name == '\0' ||
		 !strcmp(display_name, galago_account_get_username(account))))
	{
		display_name = NULL;
	}

	if (display_name == account->priv->display_name ||
		(display_name != NULL && account->priv->display_name != NULL &&
		 !strcmp(account->priv->display_name, display_name)))
	{
		return;
	}

	if (account->priv->display_name != NULL)
		g_free(account->priv->display_name);

	account->priv->display_name = (display_name == NULL
	                               ? NULL : g_strdup(display_name));

	if (GALAGO_OBJECT_IS_LOCAL(account) && galago_is_connected() &&
		galago_is_feed() && !galago_is_daemon())
	{
		galago_dbus_send_message(GALAGO_OBJECT(account), "SetDisplayName",
		                         galago_value_new(GALAGO_VALUE_TYPE_STRING,
		                                          &display_name, NULL),
								 NULL);
	}

	g_object_notify(G_OBJECT(account), "display-name");

	/* XXX Is this redundant? */
	g_signal_emit(account, signals[DISPLAY_NAME_CHANGED], 0);
}

/**
 * galago_account_get_display_name
 * @account: The account.
 *
 * Returns the account's displayed name. If the displayed name is not
 * explicitly set, this will return the screen name.
 *
 * Returns: The displayed name.
 */
const char *
galago_account_get_display_name(const GalagoAccount *account)
{
	g_return_val_if_fail(account != NULL,            NULL);
	g_return_val_if_fail(GALAGO_IS_ACCOUNT(account), NULL);

	if (account->priv->display_name == NULL)
		return galago_account_get_username(account);

	return account->priv->display_name;
}

/**
 * galago_account_is_display_name_set
 * @account: The account.
 *
 * Returns whether or not a custom displayed name is set.
 *
 * Returns: TRUE if a custom displayed name is set, or FALSE.
 */
gboolean
galago_account_is_display_name_set(const GalagoAccount *account)
{
	g_return_val_if_fail(account != NULL,            FALSE);
	g_return_val_if_fail(GALAGO_IS_ACCOUNT(account), FALSE);

	return (account->priv->display_name != NULL);
}

/**
 * galago_account_add_contact
 * @account: The account.
 * @contact: The contact's account to add.
 *
 * Adds a contact to an account.
 */
void
galago_account_add_contact(GalagoAccount *account, GalagoAccount *contact)
{
	const char *username;
	GalagoService *service;

	g_return_if_fail(account != NULL);
	g_return_if_fail(contact != NULL);
	g_return_if_fail(GALAGO_IS_ACCOUNT(account));
	g_return_if_fail(GALAGO_IS_ACCOUNT(contact));
	g_return_if_fail(account != contact);

	username = galago_account_get_username(contact);

	if (galago_account_get_contact(account, username, FALSE) != NULL)
	{
#if 0
		g_warning("A contact with username %s has already been "
				  "added to account %s",
				  username, galago_account_get_username(account));
#endif
		return;
	}

	service = galago_account_get_service(account);

	g_hash_table_insert(account->priv->contacts_table,
						galago_service_normalize(service, username),
						contact);

	account->priv->contacts = g_list_append(account->priv->contacts, contact);

	if (GALAGO_OBJECT_IS_LOCAL(account))
		_galago_dbus_account_add_contact(account, contact);

	g_signal_emit(account, signals[CONTACT_ADDED], 0, contact);
}

/**
 * galago_account_remove_contact
 * @account: The account.
 * @contact: The contact's account to remove.
 *
 * Removes a contact from an account.
 */
void
galago_account_remove_contact(GalagoAccount *account, GalagoAccount *contact)
{
	g_return_if_fail(account != NULL);
	g_return_if_fail(contact != NULL);
	g_return_if_fail(GALAGO_IS_ACCOUNT(account));
	g_return_if_fail(GALAGO_IS_ACCOUNT(contact));
	g_return_if_fail(account != contact);

	g_hash_table_remove(account->priv->contacts_table,
						galago_account_get_username(contact));

	account->priv->contacts = g_list_remove(account->priv->contacts, contact);

	if (GALAGO_OBJECT_IS_LOCAL(account) && galago_is_connected() &&
		galago_is_feed())
	{
		galago_dbus_send_message(GALAGO_OBJECT(account), "RemoveContact",
								 galago_value_new_object(GALAGO_TYPE_ACCOUNT,
														 G_OBJECT(contact)),
								 NULL);
	}

	g_signal_emit(account, signals[CONTACT_REMOVED], 0, contact);
}

/**
 * galago_account_get_contact
 * @account:  The account.
 * @username: The username.
 * @query:    TRUE if a remote query should be done if there is no
 *            local contact found, or FALSE.
 *
 * Returns the contact with the specified username in an account.
 *
 * Returns: The contact's account, if found, or NULL.
 */
GalagoAccount *
galago_account_get_contact(const GalagoAccount *account, const char *username,
                           gboolean query)
{
	GalagoAccount *contact;
	GalagoService *service;
	char *norm_username;

	g_return_val_if_fail(account != NULL,                       NULL);
	g_return_val_if_fail(GALAGO_IS_ACCOUNT(account),            NULL);
	g_return_val_if_fail(username != NULL && *username != '\0', NULL);

	service = galago_account_get_service(account);

	norm_username = galago_service_normalize(service, username);
	contact = g_hash_table_lookup(account->priv->contacts_table,
								  norm_username);
	g_free(norm_username);

	if (contact == NULL && query && GALAGO_OBJECT_IS_REMOTE(account) &&
		!galago_is_daemon() && galago_is_connected())
	{
		galago_context_push(galago_object_get_context(GALAGO_OBJECT(account)));

		contact = galago_dbus_send_message_with_reply(GALAGO_OBJECT(account),
			"GetContact",
			galago_value_new_object(GALAGO_TYPE_ACCOUNT, NULL),
			galago_value_new(GALAGO_VALUE_TYPE_STRING, &username, NULL),
			NULL);

		galago_context_pop();
	}

	return contact;
}

/**
 * galago_account_get_contacts
 * @account: The account.
 * @query:   TRUE if a remote query should be done if there is no
 *           local contact found, or FALSE.
 *
 * Returns a list of accounts of users seen from this account.
 *
 * This may emit a contact-added signal for every object that returns. If
 * your code connects to this signal and calls galago_account_get_contacts()
 * as a result, you will want to add a lock so that you don't end up with
 * unwanted side-effects.
 *
 * Returns: A list of accounts of other users, or NULL.
 */
GList *
galago_account_get_contacts(const GalagoAccount *account, gboolean query)
{
	g_return_val_if_fail(account != NULL,            NULL);
	g_return_val_if_fail(GALAGO_IS_ACCOUNT(account), NULL);

	if (query && GALAGO_OBJECT_IS_REMOTE(account) &&
		!galago_is_daemon() && galago_is_connected())
	{
		GList *temp;

		galago_context_push(galago_object_get_context(GALAGO_OBJECT(account)));

		temp = galago_dbus_send_message_with_reply(GALAGO_OBJECT(account),
			"GetContacts",
			galago_value_new_list(GALAGO_TYPE_OBJECT, NULL,
								  (void *)GALAGO_TYPE_ACCOUNT),
			NULL);
		g_list_free(temp);

		galago_context_pop();
	}

	return account->priv->contacts;
}

/**
 * galago_account_create_presence
 * @account: The account.
 *
 * Creates an account's presence.
 *
 * The account should be a local account. If the account already has a
 * presence, this will return the existing one.
 *
 * Returns: The presence.
 */
GalagoPresence *
galago_account_create_presence(GalagoAccount *account)
{
	g_return_val_if_fail(galago_is_initted(),        NULL);
	g_return_val_if_fail(account != NULL,            NULL);
	g_return_val_if_fail(GALAGO_IS_ACCOUNT(account), NULL);

	return _galago_presence_new(account, NULL);
}

/**
 * galago_account_get_presence
 * @account: The account.
 * @query:   TRUE if a remote query should be done if there is no
 *           local presence found, or FALSE.
 *
 * Returns the account's presence.
 *
 * Returns: The presence, if found, or NULL.
 */
GalagoPresence *
galago_account_get_presence(const GalagoAccount *account, gboolean query)
{
	g_return_val_if_fail(account != NULL,            NULL);
	g_return_val_if_fail(GALAGO_IS_ACCOUNT(account), NULL);

	if (account->priv->presence == NULL && query &&
		GALAGO_OBJECT_IS_REMOTE(account) && !galago_is_daemon() &&
		galago_is_connected())
	{
		account->priv->presence =
			galago_dbus_send_message_with_reply(GALAGO_OBJECT(account),
				"GetPresence",
				galago_value_new_object(GALAGO_TYPE_PRESENCE, NULL),
				NULL);
		g_object_notify(G_OBJECT(account), "presence");
	}

	return account->priv->presence;
}

/**
 * galago_account_set_avatar
 * @account: The account.
 * @avatar:  The avatar to set.
 *
 * Sets the account's avatar. The account should be a local account.
 */
void
galago_account_set_avatar(GalagoAccount *account, GalagoImage *avatar)
{
	g_return_if_fail(account != NULL);
	g_return_if_fail(GALAGO_IS_ACCOUNT(account));

	if (account->priv->avatar == avatar)
		return;

	if (account->priv->avatar != NULL)
	{
		GalagoImage *old_avatar = account->priv->avatar;
		account->priv->avatar = NULL;
		galago_object_destroy(GALAGO_OBJECT(old_avatar));
	}

	account->priv->avatar = avatar;

	if (GALAGO_OBJECT_IS_LOCAL(account))
		_galago_dbus_account_set_avatar(account, avatar);

	g_signal_emit(account, signals[AVATAR_SET], 0, avatar);

	g_object_notify(G_OBJECT(account), "avatar");
}

/**
 * galago_account_get_avatar
 * @account: The account.
 * @query:   TRUE if a remote query should be done if there is no
 *           local avatar found, or FALSE.
 *
 * Returns the account's avatar.
 *
 * Returns: The avatar, if found, or NULL.
 */
GalagoImage *
galago_account_get_avatar(const GalagoAccount *account, gboolean query)
{
	g_return_val_if_fail(account != NULL,            NULL);
	g_return_val_if_fail(GALAGO_IS_ACCOUNT(account), NULL);

	if (account->priv->avatar == NULL && query &&
		GALAGO_OBJECT_IS_REMOTE(account) && !galago_is_daemon() &&
		galago_is_connected())
	{
		account->priv->avatar =
			galago_dbus_send_message_with_reply(GALAGO_OBJECT(account),
				"GetAvatar",
				galago_value_new_object(GALAGO_TYPE_IMAGE, NULL),
				NULL);
		g_object_notify(G_OBJECT(account), "avatar");
	}

	return account->priv->avatar;
}

void
_galago_account_set_presence(GalagoAccount *account, GalagoPresence *presence)
{
	gboolean emit_deleted = FALSE;

	g_return_if_fail(account != NULL);
	g_return_if_fail(GALAGO_IS_ACCOUNT(account));

	if (account->priv->presence != NULL)
	{
		GalagoPresence *old_presence = account->priv->presence;
		account->priv->presence = NULL;
		galago_object_destroy(GALAGO_OBJECT(old_presence));

		emit_deleted = (presence == NULL);
	}

	account->priv->presence = presence;

	g_object_notify(G_OBJECT(account), "presence");

	if (emit_deleted)
		g_signal_emit(account, signals[PRESENCE_DELETED], 0);
}

void
_galago_account_presence_created(GalagoAccount *account,
								 GalagoPresence *presence)
{
	g_return_if_fail(account != NULL);
	g_return_if_fail(GALAGO_IS_ACCOUNT(account));
	g_return_if_fail(presence != NULL);
	g_return_if_fail(GALAGO_IS_PRESENCE(presence));

	/*
	 * We should already have account->priv->presence set, since that's
	 * done in the constructor.
	 */

	g_signal_emit(account, signals[PRESENCE_CREATED], 0, presence);
}

void
_galago_account_presence_deleted(GalagoAccount *account)
{
	g_return_if_fail(account != NULL);
	g_return_if_fail(GALAGO_IS_ACCOUNT(account));

	_galago_account_set_presence(account, NULL);

	g_signal_emit(account, signals[PRESENCE_DELETED], 0);
}


/**************************************************************************
 * D-BUS Functions
 **************************************************************************/
static void
_galago_dbus_account_set_avatar(GalagoAccount *account, GalagoImage *avatar)
{
	if (!galago_is_connected() || !galago_is_feed())
		return;

	if (avatar == NULL)
	{
		galago_dbus_send_message(GALAGO_OBJECT(account), "UnsetAvatar", NULL);
	}
	else
	{
		galago_dbus_send_message(GALAGO_OBJECT(account), "SetAvatar",
								 galago_value_new_object(GALAGO_TYPE_IMAGE,
														 G_OBJECT(avatar)),
								 NULL);
	}
}

static void
_galago_dbus_account_add_contact(GalagoAccount *account,
                                 GalagoAccount *contact)
{
	if (!galago_is_connected() || !galago_is_feed())
		return;

	galago_dbus_send_message(GALAGO_OBJECT(account), "AddContact",
							 galago_value_new_object(GALAGO_TYPE_ACCOUNT,
													 G_OBJECT(contact)),
							 NULL);
}
