/***************************************************************************
 *
 *   KYum - a KDE GUI for yum
 *
 *   Copyright (C) 2005 by Steffen Offermann
 *   steffen_ac@yahoo.com
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <qlayout.h>
#include <qlabel.h>
#include <qcheckbox.h>

#include <klocale.h>

#include "KYumPreferences.h"
#include "DlgInfo.h"


class InfoMsg
  {
    private:
      typedef std::map<DlgInfo::MsgID, InfoMsg *>  MsgMap;

    private:
      static MsgMap   m_msgMap;
      DlgInfo::MsgID  m_msgID;
      QString         m_strText;

    public:
                      InfoMsg   (DlgInfo::MsgID msgID, const char * pText);
      virtual       ~ InfoMsg   ();

      static void     enable    (DlgInfo::MsgID msgID, bool bEnable = true);
      static bool     isEnabled (DlgInfo::MsgID msgID);

      static QString  text      (DlgInfo::MsgID msgID);

      QString         text      () const { return m_strText; }
  };


InfoMsg::MsgMap InfoMsg::m_msgMap;


#define IMPL_INFO(id, text)  static InfoMsg g_msg##id (DlgInfo::c_##id, i18n(text))

#define GLOBAL_STATIC_OK

#ifdef GLOBAL_STATIC_OK
IMPL_INFO(YumConfAndReposSaved, "Repositories and yum.conf have been saved.");
#endif


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

InfoMsg::InfoMsg(DlgInfo::MsgID msgID, const char * pText)

  : m_msgID(msgID),
    m_strText(pText)

{
    m_msgMap[msgID] = this;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

InfoMsg::~InfoMsg()
{
    m_msgMap.erase( m_msgMap.find(m_msgID) );
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void InfoMsg::enable(DlgInfo::MsgID msgID, bool bEnable)
{
    KYumPreferences::enableMsg((int) msgID, bEnable);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

bool InfoMsg::isEnabled(DlgInfo::MsgID msgID)
{
    MsgMap::const_iterator it = m_msgMap.find(msgID);
    return it == m_msgMap.end() ? false :
                                  KYumPreferences::isMsgEnabled((int) msgID);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

QString InfoMsg::text(DlgInfo::MsgID msgID)
{
    MsgMap::const_iterator it = m_msgMap.find(msgID);
    return it == m_msgMap.end() ? "<text not found>" :
                                  it->second->text();
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

DlgInfo::DlgInfo(QWidget * pParent, MsgID msgID)

  : KDialogBase (pParent,
                 "KYum Information",
                 true,
                 i18n("KYum Information"),
                 KDialogBase::Ok/* | KDialogBase::Cancel*/,
                 KDialogBase::Ok,
                 true),
    m_msgID     (msgID)

{
#ifndef GLOBAL_STATIC_OK
    static bool bInitialized = false;

    if ( !bInitialized )
    {
        IMPL_INFO(YumConfAndReposSaved, "Repositories and yum.conf have been saved.");
        bInitialized = true;
    }
#endif


    QWidget     * pTextPanel  = new QWidget(this);
    QVBoxLayout * pTextLayout = new QVBoxLayout(pTextPanel, 5, 20);

    setMainWidget(pTextPanel);
    pTextLayout->setAutoAdd(true);

    QLabel * pLabel = new QLabel(pTextPanel);
    pLabel->setText(InfoMsg::text(msgID));

//    QSpacerItem * pVSpacer = new QSpacerItem(20, 60, QSizePolicy::Maximum, QSizePolicy::Maximum);
//    pTextLayout->addItem(pVSpacer);

    m_pCheckDisable = new QCheckBox(pTextPanel);
    m_pCheckDisable->setText("&Don't show this message again.");

    m_pCheckDisable->setChecked(!isMsgEnabled(msgID));
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

DlgInfo::~DlgInfo()
{
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

bool DlgInfo::isMsgEnabled(MsgID msgID)
{
    return InfoMsg::isEnabled(msgID);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void DlgInfo::enableMsg(MsgID msgID, bool bEnable)
{
    InfoMsg::enable(msgID, bEnable);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void DlgInfo::slotOk()
{
    enableMsg(m_msgID, !m_pCheckDisable->isChecked());
    KDialogBase::slotOk();
}


#include "DlgInfo.moc"
