/*
    KMid Mac OSX Backend
    Copyright (C) 2009-2010 Pedro Lopez-Cabanillas <plcl@users.sf.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#include "macmidioutput.h"
#include "midimapper.h"

#include <KDebug>
#include <QStringList>
#include <QByteArray>
#include <QVarLengthArray>
#include <CoreMIDI/MIDIServices.h>

namespace KMid {

    class MacMIDIOutput::MacMIDIOutputPrivate {
    public:
        MacMIDIOutputPrivate():
            m_client(NULL),
            m_outPort(NULL),
            m_destination(NULL),
            m_mapper(0),
            m_pitchShift(0),
            m_clientFilter(true)
        {
            for (int chan = 0; chan < MIDI_CHANNELS; ++chan) {
                m_lastpgm[chan] = 0;
                m_volumeShift[chan] = 1.0;
                m_volume[chan] = 100;
                m_muted[chan] = false;
                m_locked[chan] = false;
            }
        }

        virtual ~MacMIDIOutputPrivate()
        {
            OSStatus result;
            if (m_client != NULL)
              result = MIDIClientDispose(m_client);
            if (result != noErr)
                kDebug() << "MIDIClientDispose() error:" << result;
        }

        void transformControllerEvent(MIDIPacket *packet)
        {
            int param = packet->data[1];
            if (m_mapper != NULL && m_mapper->isOK()) {
                param = m_mapper->controller(param);
                if (param >= 0 && param < 128)
                    packet->data[1] = param;
            }
            if (param == MIDI_CTL_MSB_MAIN_VOLUME) {
                int chan = packet->data[0] & 0x0f;
                int value = packet->data[2];
                m_volume[chan] = value;
                value = floor(value * m_volumeShift[chan]);
                if (value < 0) value = 0;
                if (value > 127) value = 127;
                packet->data[2] = value;
            }
        }

        void transformNoteEvent(MIDIPacket *packet)
        {
            int note, channel;
            channel = packet->data[0] & 0x0f;
            note = packet->data[1];
            if (channel != MIDI_GM_DRUM_CHANNEL) {
                note += m_pitchShift;
                while (note > 127) note -= 12;
                while (note < 0) note += 12;
                packet->data[1] = note;
            } else if (m_mapper != NULL && m_mapper->isOK()) {
                note = m_mapper->key( channel,
                                      m_lastpgm[channel],
                                      note );
                if (note >= 0 && note < 128)
                    packet->data[1] = note;
            }
        }

        void transformProgramEvent(MIDIPacket *packet)
        {
            if (m_mapper != NULL && m_mapper->isOK()) {
                int channel = packet->data[0] & 0x0f;
                m_lastpgm[channel] = packet->data[1];
                int pgm = m_mapper->patch(channel, m_lastpgm[channel]);
                if (pgm >= 0 && pgm < 128)
                    packet->data[1] = pgm;
            }
        }

        void transformPitchBendEvent(MIDIPacket *packet)
        {
            if (m_mapper != NULL && m_mapper->isOK()) {
                int value = (packet->data[1] + packet->data[2] * 128) - 8192;
                value = m_mapper->pitchBender(value);
                if (value < -8192) value = -8192;
                if (value > 8191) value = 8191;
                quint16 newval = value + 8192;
                packet->data[1] = newval % 0x80;
                packet->data[2] = newval / 0x80;
            }
        }

        void transform(MIDIPacket *packet)
        {
            quint8 status = packet->data[0] & 0xf0;
            quint8 chan = packet->data[0] & 0x0f;
            switch ( status ) {
            case MIDI_STATUS_CONTROLCHANGE:
                transformControllerEvent(packet);
                break;
            case MIDI_STATUS_NOTEOFF:
            case MIDI_STATUS_NOTEON:
                transformNoteEvent(packet);
                break;
            case MIDI_STATUS_PROGRAMCHANGE:
                transformProgramEvent(packet);
                break;
            case MIDI_STATUS_PITCHBEND:
                transformPitchBendEvent(packet);
                break;
            default:
                break;
            }
            if ( status >= MIDI_STATUS_NOTEOFF &&
                 status <  MIDI_STATUS_SYSEX &&
                 m_mapper != NULL && m_mapper->isOK() ) {
                int channel = m_mapper->channel(chan);
                if (channel >= 0 && channel < MIDI_CHANNELS)
                    packet->data[0] = status + channel;
            }
        }

        MIDIClientRef m_client;
        MIDIPortRef m_outPort;
        MIDIEndpointRef m_destination;
        MidiMapper *m_mapper;
        int m_pitchShift;
        bool m_clientFilter;
        QString m_currentOutput;
        QStringList m_outputDevices;

        int m_lastpgm[MIDI_CHANNELS];
        qreal m_volumeShift[MIDI_CHANNELS];
        int m_volume[MIDI_CHANNELS];
        bool m_muted[MIDI_CHANNELS];
        bool m_locked[MIDI_CHANNELS];
        QByteArray m_resetMessage;
    };

    MacMIDIOutput::MacMIDIOutput(QObject *parent) :
        MIDIOutput(parent), d(new MacMIDIOutputPrivate)
    {
        OSStatus result;
        result = MIDIClientCreate(CFSTR("KMid"), NULL, NULL, &d->m_client);
        if ( result != noErr )
            return;
        result = MIDIOutputPortCreate(d->m_client, CFSTR("KMid Output"), &d->m_outPort);
        if ( result != noErr )
            return;
        reloadDeviceList();
    }

    MacMIDIOutput::~MacMIDIOutput()
    {
        delete d;
    }

    QString CFStringToQString(CFStringRef str)
    {
        if (!str)
            return QString();
        CFIndex length = CFStringGetLength(str);
        const UniChar *chars = CFStringGetCharactersPtr(str);
        if (chars)
            return QString(reinterpret_cast<const QChar *>(chars), length);
        QVarLengthArray<UniChar> buffer(length);
        CFStringGetCharacters(str, CFRangeMake(0, length), buffer.data());
        return QString(reinterpret_cast<const QChar *>(buffer.constData()), length);
    }

    qreal MacMIDIOutput::volume(int channel) const
    {
        if (channel >=0 && channel < MIDI_CHANNELS)
            return d->m_volumeShift[channel];
        return -1.0;
    }

    int MacMIDIOutput::outputDevice() const
    {

        return d->m_outputDevices.indexOf(d->m_currentOutput);
    }

    QString MacMIDIOutput::outputDeviceName() const
    {
        return d->m_currentOutput;
    }

    bool MacMIDIOutput::isMuted(int channel) const
    {
        if (channel >= 0 && channel < MIDI_CHANNELS)
            return d->m_muted[channel];
        return false;
    }

    MidiMapper* MacMIDIOutput::midiMap()
    {
        return d->m_mapper;
    }

    int MacMIDIOutput::pitchShift()
    {
        return d->m_pitchShift;
    }

    QStringList MacMIDIOutput::outputDeviceList( bool basicOnly )
    {
        d->m_clientFilter = basicOnly;
        reloadDeviceList();
        return d->m_outputDevices;
    }

    /* SLOTS */

    void MacMIDIOutput::setVolume(int channel, qreal value)
    {
        if (channel >= 0 && channel < MIDI_CHANNELS) {
            d->m_volumeShift[channel] = value;
            sendController(channel, MIDI_CTL_MSB_MAIN_VOLUME, d->m_volume[channel]);
            emit volumeChanged( channel, value );
        } else if ( channel == -1 ) {
            for (int chan = 0; chan < MIDI_CHANNELS; ++chan) {
                d->m_volumeShift[chan] = value;
                sendController(chan, MIDI_CTL_MSB_MAIN_VOLUME, d->m_volume[chan]);
                emit volumeChanged( chan, value );
            }
        }
    }

    static QString getEndpointName(MIDIEndpointRef endpoint)
    {
        QString result;
        CFStringRef str = 0;
        MIDIObjectGetStringProperty (endpoint, kMIDIPropertyName, &str);
        if (str != 0) {
            result = CFStringToQString(str);
            CFRelease(str);
            str = 0;
        }
        MIDIEntityRef entity = 0;
        MIDIEndpointGetEntity(endpoint, &entity);
        if (entity == 0)
            return result;
        if (result.isEmpty()) {
            MIDIObjectGetStringProperty (entity, kMIDIPropertyName, &str);
            if (str != 0) {
                result = CFStringToQString(str);
                CFRelease(str);
                str = 0;
            }
        }
        MIDIDeviceRef device = 0;
        MIDIEntityGetDevice (entity, &device);
        if (device == 0)
            return result;
        MIDIObjectGetStringProperty (device, kMIDIPropertyName, &str);
        if (str != 0) {
            QString s = CFStringToQString(str);
            CFRelease (str);
            str = 0;
            if (!result.startsWith(s, Qt::CaseInsensitive) )
                result = (s + ' ' + result).trimmed();
        }
        return result;
    }

    void MacMIDIOutput::reloadDeviceList()
    {
        int num = MIDIGetNumberOfDestinations();
        d->m_outputDevices.clear();
        for (int i = 0; i < num; ++i) {
            QString result;
            MIDIEndpointRef dest = MIDIGetDestination( i );
            if (dest != 0) {
                QString name = getEndpointName(dest);
                if ( d->m_clientFilter &&
                     name.contains(QLatin1String("IAC"), Qt::CaseSensitive) )
                    continue;
                d->m_outputDevices << name;
            }
        }
        if (!d->m_currentOutput.isEmpty() &&
            !d->m_outputDevices.contains(d->m_currentOutput)) {
            d->m_currentOutput.clear();
            d->m_destination = NULL;
            emit outputDeviceChanged(d->m_currentOutput);
        }
    }

    bool MacMIDIOutput::setOutputDevice(int index)
    {
        if (index >= 0 && index < d->m_outputDevices.count())
            return setOutputDeviceName(d->m_outputDevices[index]);
        return false;
    }

    bool MacMIDIOutput::setOutputDeviceName(const QString &newOutputDevice)
    {
        int index;
        QStringList allOutputDevices;
        int num = MIDIGetNumberOfDestinations();
        for (int i = 0; i < num; ++i) {
            MIDIEndpointRef dest = MIDIGetDestination( i );
            if (dest != 0)
               allOutputDevices << getEndpointName( dest );
        }
        index = allOutputDevices.indexOf(newOutputDevice);
        if (index < 0)
            return false;
        d->m_destination = MIDIGetDestination( index );
        d->m_currentOutput = newOutputDevice;
        return true;
    }

    MIDIClientRef MacMIDIOutput::client() const
    {
        return d->m_client;
    }

    void MacMIDIOutput::setMuted(int channel, bool mute)
    {
        if (channel >= 0 && channel < MIDI_CHANNELS) {
            if (d->m_muted[channel] != mute) {
                if (mute) {
                    sendController(channel, MIDI_CTL_ALL_NOTES_OFF, 0);
                    sendController(channel, MIDI_CTL_ALL_SOUNDS_OFF, 0);
                }
                d->m_muted[channel] = mute;
                emit mutedChanged( channel, mute );
            }
        }
    }

    void MacMIDIOutput::setLocked(int channel, bool lock)
    {
        if (channel >= 0 && channel < MIDI_CHANNELS) {
            if (d->m_locked[channel] != lock) {
                d->m_locked[channel] = lock;
                emit lockedChanged( channel, lock );
            }
        }
    }

    void MacMIDIOutput::setMidiMap(MidiMapper *map)
    {
        d->m_mapper = map;
    }

    void MacMIDIOutput::setPitchShift(int amt)
    {
        if (d->m_pitchShift != amt) {
            allNotesOff();
            d->m_pitchShift = amt;
        }
    }

    void MacMIDIOutput::setResetMessage(const QByteArray& msg)
    {
        d->m_resetMessage = msg;
    }

    /* Realtime MIDI slots */

    void MacMIDIOutput::allNotesOff()
    {
        quint8 data[3];
        quint8 buf[2048];
        MIDIPacketList* pktlist = (MIDIPacketList*) &buf;
        MIDIPacket* packet = MIDIPacketListInit(pktlist);
        for(int chan = 0; chan < MIDI_CHANNELS && packet != NULL; ++chan) {
            data[0] = MIDI_STATUS_CONTROLCHANGE | (chan & 0x0f);
            data[1] = MIDI_CTL_ALL_NOTES_OFF;
            data[2] = 0;
            packet = MIDIPacketListAdd(pktlist, sizeof(buf), packet, 0,
                      sizeof(data), data);
            if (packet != NULL) {
                data[1] = MIDI_CTL_ALL_SOUNDS_OFF;
                packet = MIDIPacketListAdd(pktlist, sizeof(buf), packet, 0,
                           sizeof(data), data);
            }
        }
        if (packet != NULL)
            sendEvents(pktlist);
    }

    void MacMIDIOutput::resetControllers()
    {
        quint8 data[3];
        quint8 buf[2048];
        MIDIPacketList* pktlist = (MIDIPacketList*) &buf;
        MIDIPacket* packet = MIDIPacketListInit(pktlist);
        for(int chan = 0; chan < MIDI_CHANNELS && packet != NULL; ++chan) {
            data[0] = MIDI_STATUS_CONTROLCHANGE | (chan & 0x0f);
            data[1] = MIDI_CTL_RESET_CONTROLLERS;
            data[2] = 0;
            packet = MIDIPacketListAdd(pktlist, sizeof(buf), packet, 0,
                      sizeof(data), data);
            if (packet != NULL) {
                data[1] = MIDI_CTL_MSB_MAIN_VOLUME;
                data[2] = 100;
                packet = MIDIPacketListAdd(pktlist, sizeof(buf), packet, 0,
                           sizeof(data), data);
            }
        }
        if (packet != NULL)
            sendEvents(pktlist);
    }

    void MacMIDIOutput::sendResetMessage()
    {
        if (d->m_resetMessage.size() > 0)
            sendSysexEvent(d->m_resetMessage);
    }

    void MacMIDIOutput::sendNoteOn(int chan, int note, int vel)
    {
        if ( d->m_muted[chan] )
            return;
        quint8 data[3];
        MIDIPacketList pktlist ;
        MIDIPacket* packet = MIDIPacketListInit(&pktlist);
        data[0] = MIDI_STATUS_NOTEON | (chan & 0x0f);
        data[1] = note;
        data[2] = vel;
        packet = MIDIPacketListAdd(&pktlist, sizeof(pktlist), packet, 0,
            sizeof(data), data);
        if (packet != NULL)
            sendEvents(&pktlist);
    }

    void MacMIDIOutput::sendNoteOff(int chan, int note, int vel)
    {
        if ( d->m_muted[chan] )
            return;
        quint8 data[3];
        MIDIPacketList pktlist ;
        MIDIPacket* packet = MIDIPacketListInit(&pktlist);
        data[0] = MIDI_STATUS_NOTEOFF | (chan & 0x0f);
        data[1] = note;
        data[2] = vel;
        packet = MIDIPacketListAdd(&pktlist, sizeof(pktlist), packet, 0,
            sizeof(data), data);
        if (packet != NULL)
            sendEvents(&pktlist);
    }

    void MacMIDIOutput::sendController(int chan, int control, int value)
    {
        if ( d->m_muted[chan] )
            return;
        quint8 data[3];
        MIDIPacketList pktlist ;
        MIDIPacket* packet = MIDIPacketListInit(&pktlist);
        data[0] = MIDI_STATUS_CONTROLCHANGE | (chan & 0x0f);
        data[1] = control;
        data[2] = value;
        packet = MIDIPacketListAdd(&pktlist, sizeof(pktlist), packet, 0,
            sizeof(data), data);
        if (packet != NULL)
            sendEvents(&pktlist);
    }

    void MacMIDIOutput::sendKeyPressure(int chan, int note, int value)
    {
        if ( d->m_muted[chan] )
            return;
        quint8 data[3];
        MIDIPacketList pktlist ;
        MIDIPacket* packet = MIDIPacketListInit(&pktlist);
        data[0] = MIDI_STATUS_KEYPRESURE | (chan & 0x0f);
        data[1] = note;
        data[2] = value;
        packet = MIDIPacketListAdd(&pktlist, sizeof(pktlist), packet, 0,
            sizeof(data), data);
        if (packet != NULL)
            sendEvents(&pktlist);
    }

    void MacMIDIOutput::sendProgram(int chan, int program)
    {
        if ( d->m_muted[chan] )
            return;
        if ( d->m_locked[chan] )
            return;
        quint8 data[2];
        MIDIPacketList pktlist ;
        MIDIPacket* packet = MIDIPacketListInit(&pktlist);
        data[0] = MIDI_STATUS_PROGRAMCHANGE | (chan & 0x0f);
        data[1] = program;
        packet = MIDIPacketListAdd(&pktlist, sizeof(pktlist), packet, 0,
            sizeof(data), data);
        if (packet != NULL)
            sendEvents(&pktlist);
    }

    void MacMIDIOutput::sendChannelPressure(int chan, int value)
    {
        if ( d->m_muted[chan] )
            return;
        quint8 data[2];
        MIDIPacketList pktlist ;
        MIDIPacket* packet = MIDIPacketListInit(&pktlist);
        data[0] = MIDI_STATUS_CHANNELPRESSURE | (chan & 0x0f);
        data[1] = value;
        packet = MIDIPacketListAdd(&pktlist, sizeof(pktlist), packet, 0,
            sizeof(data), data);
        if (packet != NULL)
            sendEvents(&pktlist);
    }

    void MacMIDIOutput::sendPitchBend(int chan, int value)
    {
        if ( d->m_muted[chan] )
            return;
        quint16 val = value + 8192; // value between -8192 and +8191
        quint8 data[3];
        MIDIPacketList pktlist ;
        MIDIPacket* packet = MIDIPacketListInit(&pktlist);
        data[0] = MIDI_STATUS_PITCHBEND | (chan & 0x0f);
        data[1] = val % 0x80; // LSB
        data[2] = val / 0x80; // MSB
        packet = MIDIPacketListAdd(&pktlist, sizeof(pktlist), packet, 0,
            sizeof(data), data);
        if (packet != NULL)
            sendEvents(&pktlist);
    }

    void MacMIDIOutput::sendSysexEvent(const QByteArray& data)
    {
        quint8 buf[4096];
        if (data.size() > 4096)
            return;
        MIDIPacketList* pktlist = (MIDIPacketList*) &buf;
        MIDIPacket* packet = MIDIPacketListInit(pktlist);
        packet = MIDIPacketListAdd(pktlist, sizeof(buf), packet, 0,
            data.size(), (const Byte*)data.data());
        if (packet != NULL)
            sendEvents(pktlist);
    }

    void MacMIDIOutput::sendEvents(const MIDIPacketList* events)
    {
        quint8 buf[4096];
        MIDIPacketList* pktlist = (MIDIPacketList*) &buf;
        MIDIPacket *curpacket = MIDIPacketListInit(pktlist);
        const MIDIPacket *srcpacket = events->packet;
        for ( int i = 0; i < events->numPackets; ++i ) {
            MIDIPacket dstpacket = *srcpacket;
            d->transform(&dstpacket);
            quint8 status = dstpacket.data[0] & 0xf0;
            quint8 chan = dstpacket.data[0] & 0x0f;
            bool ok = ( status >= MIDI_STATUS_NOTEOFF ) &&
                      ( ( status == MIDI_STATUS_SYSEX ) |
                        ( status < MIDI_STATUS_SYSEX &&
                          !d->m_muted[chan] ) );
            if ( ok && status == MIDI_STATUS_PROGRAMCHANGE )
                ok = !d->m_locked[chan];
            if (ok)
               curpacket = MIDIPacketListAdd(pktlist, sizeof(buf), curpacket, 0,
                           dstpacket.length, (const Byte*)&dstpacket.data);
            srcpacket = MIDIPacketNext(srcpacket);
        }
        MIDISend(d->m_outPort, d->m_destination, pktlist);
    }

}

#include "macmidioutput.moc"
