/***************************************************************************
*   Copyright (C) 2004 by Kita Developers                                 *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

/* This class manages the DOM elements  */

#include <kurl.h>
#include <dom/html_element.h>
#include <dom/dom_text.h>

#include "domtree.h"

#include "libkita/kitaconfig.h"
#include "libkita/datmanager.h"
#include "libkita/datinfo.h"
#include "libkita/kita-utf8.h"
#include "libkita/kita_misc.h"


/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/


KitaDomTree::KitaDomTree( const DOM::HTMLDocument& hdoc, const KURL& datURL )
{
    /* config */

    m_templateNum = 20; /* default number of templates */

    /*-----------------------------*/

    m_hdoc = hdoc;
    m_bufLng = 0;
    m_topNum = 65535;
    m_bottomNum = 0;

    /* create the nodes of footer, header, etc. */
    createKokoyon();
    createFooterOrHeader( "footer" );
    createFooterOrHeader( "header" );
    createTugi100();
    createMae100();

    /* get pointer of DatInfo */
    /* Note that m_datURL is already locked in the KitaHTMLPart. */
    m_datInfo = Kita::DatManager::getDatInfoPointer( datURL );
}


KitaDomTree::~KitaDomTree() {}


/*
  This function creates DOM elements of both title and body.
  To show the res, call appendRes().    .                     */ /* public */

bool KitaDomTree::createResElement( int num )
{
    const int mrg = 100;
    const int delta = 500;

    /*---------------------------------*/

    if ( ! m_datInfo ) return FALSE;
    if ( m_hdoc == NULL ) return FALSE;
    if ( num <= 0 ) return FALSE;

    if ( num < m_bufLng ) {
        if ( m_resStatus[ num ] != KITA_HTML_NOTPARSED ) return TRUE; /* already parsed */
    }
    /* resize buffer size */
    else {

        if ( m_bufLng == 0 ) m_bufLng = mrg;
        while ( num >= m_bufLng ) m_bufLng += delta;

        m_titleElm.resize( m_bufLng );
        m_bodyElm.resize( m_bufLng );
        m_resshown.resize( m_bufLng, FALSE );
        m_resStatus.resize( m_bufLng, KITA_HTML_NOTPARSED );
        m_coloredNum.resize( m_bufLng, FALSE );
    }

    /* cleate elements */
    QString titleHTML, bodyHTML;
    m_resStatus[ num ] = m_datInfo->getHTML( num, TRUE, titleHTML, bodyHTML );
    if ( m_resStatus[ num ] == KITA_HTML_NOTPARSED ) return FALSE;

    m_titleElm[ num ] = m_hdoc.createElement( "DIV" );
    m_titleElm[ num ].setAttribute( "class", "res_title" );
    m_titleElm[ num ].setAttribute( "id", QString().setNum( num ) );
    m_titleElm[ num ].setInnerHTML( titleHTML );

    m_bodyElm[ num ] = m_hdoc.createElement( "DIV" );
    m_bodyElm[ num ].setAttribute( "class", "res_body" );
    m_bodyElm[ num ].setAttribute( "id", QString().setNum( num ) );
    m_bodyElm[ num ].setInnerHTML( bodyHTML );

#if KDE_IS_VERSION( 3, 2, 0 )
#else
    if ( KitaConfig::showAA() ) {

        /* clear text in the span */
        DOM::Node tmpnode = m_bodyElm[ num ].firstChild().firstChild();
        while ( !tmpnode.isNull() ) {
            if ( tmpnode.nodeName().string() == "span" ) {
                if ( !tmpnode.firstChild().isNull() ) tmpnode.firstChild().setNodeValue( "" );
            }

            tmpnode = tmpnode.nextSibling();
        }
    }
#endif

    return TRUE;
}



/* append the response */ /* public */
bool KitaDomTree::appendRes( int num )
{
    if ( !createResElement( num ) ) return FALSE;

    m_hdoc.body().appendChild( m_titleElm[ num ] );
    m_hdoc.body().appendChild( m_bodyElm[ num ] );
    m_resshown[ num ] = TRUE;

    if ( num < m_topNum ) m_topNum = num;
    if ( num > m_bottomNum ) m_bottomNum = num;
    if ( m_topNum <= m_templateNum ) m_topNum = 1;

    return TRUE;
}




/* redraw all */ /* public */
void KitaDomTree::redraw( bool force )
{
    if ( ! m_datInfo ) return ;
    if ( m_hdoc == NULL ) return ;
    int readNum = m_datInfo->getReadNum();

    /* don't forget to reset abone here... */
    m_datInfo->resetAbone();

    for ( int i = 1; i <= readNum; i++ ) {

        if ( !m_resshown[ i ] ) continue;

        QString titleHTML, bodyHTML;
        int oldStatus = m_resStatus[ i ];
        m_resStatus[ i ] = m_datInfo->getHTML( i , TRUE, titleHTML, bodyHTML );

        if ( force || oldStatus != m_resStatus[ i ] ) {

            m_titleElm[ i ].setInnerHTML( titleHTML );
            m_bodyElm[ i ].setInnerHTML( bodyHTML );
        }
    }
}


/* change color of number of the res which is responsed. */
/* See also KitaDomTree::changeColorOfNumber() ,
   DatInfo::copyOneLineToResDat(),
   and DatInfo::collectResponsedRes().                   */ /* public */
void KitaDomTree::changeColorOfAllResponsedNumber()
{
    for ( int i = 1; i <= m_bottomNum; ++i ) {

        if ( m_resshown[ i ] && m_datInfo->isResponsed( i ) ) changeColorOfNumber( i );
    }
}



/*-----------------------------------------------------------------*/

/* public information */

/* public */
const bool KitaDomTree::isResShown( int num ) const
{
    return m_resshown[ QMIN( num, m_bufLng - 1 ) ];
}

/* public */
const int KitaDomTree::getTopResNumber() const
{
    return m_topNum;
}

/* public */
const int KitaDomTree::getBottomResNumber() const
{
    return m_bottomNum;
}

/* public */
const int KitaDomTree::getTemplateNumber() const
{
    return m_templateNum;
}

/* public */
const bool KitaDomTree::isMae100Shown() const
{
    if ( m_topNum <= m_templateNum ) return FALSE;

    return TRUE;
}



/*---------------------------------------------------------------------------*/
/* public functions for header Node, footer Node, kokomadeyonda Node, etc... */


/* append templates at the head of thread */ /* public */
void KitaDomTree::appendTemplate( int tmpNum )
{
    if ( tmpNum > m_templateNum ) m_templateNum = tmpNum;

    for ( int i = 1; i <= m_templateNum; i++ ) {

        if ( !createResElement( i ) ) continue;

        m_hdoc.body().appendChild( m_titleElm[ i ] );
        m_hdoc.body().appendChild( m_bodyElm[ i ] );
        m_resshown[ i ] = TRUE;
    }

    if ( m_topNum <= m_templateNum ) m_topNum = 1;
}


/* append footer & header  */ /* public */
void KitaDomTree::appendFooterAndHeader()
{
    if ( ! m_datInfo ) return ;
    if ( m_hdoc == NULL ) return ;

    int readNum = m_datInfo->getReadNum();
    if ( !readNum ) return ;

    if ( readNum > m_headerMaxNum + 100 ) {
        updateFooterOrHeader( m_header, "header" );
        updateFooterOrHeader( m_footer, "footer" );
    }

    m_hdoc.body().insertBefore( m_header, m_hdoc.body().firstChild() );
    m_hdoc.body().appendChild( m_footer );
}



/* append kokomadeyonda & tugi 100 & mae 100*/ /* public */
void KitaDomTree::appendKokoyon()
{
    if ( ! m_datInfo ) return ;
    if ( m_hdoc == NULL ) return ;

    int readNum = m_datInfo->getReadNum();
    if ( !readNum ) return ;

    int viewPos = m_datInfo->getViewPos();
    if ( viewPos == 0
            || ( viewPos > m_templateNum && viewPos < m_topNum )
            || viewPos > m_bottomNum ) return ;

    int i = viewPos + 1;
    while ( !m_resshown[ i ] && i <= readNum ) i++;

    if ( i <= readNum ) m_hdoc.body().insertBefore( m_kokoyon, m_titleElm[ i ] );
    else m_hdoc.body().appendChild( m_kokoyon );
}


/* public */
void KitaDomTree::appendTugi100()
{
    if ( ! m_datInfo ) return ;
    if ( m_hdoc == NULL ) return ;

    int readNum = m_datInfo->getReadNum();
    if ( !readNum ) return ;

    if ( m_bottomNum == readNum ) {
        removeTugi100();
        return ;
    }

    m_hdoc.body().appendChild( m_tugi100 );
}

/* public */
void KitaDomTree::removeTugi100()
{
    if ( ! m_datInfo ) return ;
    if ( m_hdoc == NULL ) return ;

    if ( m_tugi100.parentNode() == m_hdoc.body() ) m_hdoc.body().removeChild( m_tugi100 );
}


/* public */
void KitaDomTree::appendMae100()
{
    if ( ! m_datInfo ) return ;
    if ( m_hdoc == NULL ) return ;

    int readNum = m_datInfo->getReadNum();
    if ( !readNum ) return ;

    if ( m_topNum <= m_templateNum ) {
        removeMae100();
        return ;
    }

    int i = m_topNum;
    if ( i > readNum ) return ;
    while ( !m_resshown[ i ] && i <= readNum ) i++;

    if ( i <= readNum ) {
        m_hdoc.body().insertBefore( m_mae100, m_titleElm[ i ] );
    } else {
        m_hdoc.body().appendChild( m_mae100 );
    }
}

/* public */
void KitaDomTree::removeMae100()
{
    if ( ! m_datInfo ) return ;
    if ( m_hdoc == NULL ) return ;

    if ( !m_mae100.parentNode().isNull() ) m_mae100.parentNode().removeChild( m_mae100 );
}


/* create & append comment node */ /* public */
void KitaDomTree::createCommentNode( const QString& comment, const QString& idstr,
                                     int beforeBR,
                                     int afterBR,
                                     bool binsert )
{
    if ( m_hdoc == NULL ) return ;

    DOM::Element rootnode;

    rootnode = m_hdoc.createElement( "DIV" );
    {
        for ( int i = 0; i < beforeBR; i++ )
            rootnode.appendChild( m_hdoc.createElement( "BR" ) );
        rootnode.setAttribute( "kita_type", "comment" );
        rootnode.setAttribute( "id", idstr );
        rootnode.appendChild( m_hdoc.createTextNode( comment ) );
        for ( int i = 0; i < afterBR; i++ )
            rootnode.appendChild( m_hdoc.createElement( "BR" ) );
    }

    if ( binsert ) {
        if ( m_hdoc.body().firstChild() != NULL ) {
            m_hdoc.body().insertBefore( rootnode, m_hdoc.body().firstChild() );
            return ;
        }
    }

    m_hdoc.body().appendChild( rootnode );
}


/* create belt node */ /* public */
void KitaDomTree::createBeltNode( const QString& idstr )
{
    DOM::Element rootnode;

    QString style = "background-color: #CCCCCC; text-align: center";

    rootnode = m_hdoc.createElement( "DIV" );
    {
        rootnode.setAttribute( "style", style );
        rootnode.setAttribute( "id", idstr );
        rootnode.appendChild( m_hdoc.createElement( "BR" ) );
    }

    m_hdoc.body().appendChild( rootnode );
}


/*--------------------------------------------------------*/

/* private functions */


/* append "A" Node to rootnode */ /* private */
void KitaDomTree::appendANode( DOM::Element rootnode, const QString& linkstr, const QString& comment )
{
    if ( m_hdoc == NULL ) return ;

    DOM::Element tmpelm;
    tmpelm = rootnode.appendChild( m_hdoc.createElement( "A" ) );
    {
        tmpelm.setAttribute( "href", linkstr );
        tmpelm.appendChild( m_hdoc.createTextNode( comment ) );
    }
}


/* update footer or header No  */ /* private */
void KitaDomTree::updateFooterOrHeader(
    DOM::Element& targetelm,    /* = m_header or m_footer */
    QString strType /* = "header" or "footer" */
)
{
    if ( ! m_datInfo ) return ;
    if ( m_hdoc == NULL ) return ;
    if ( strType != "header" && strType != "footer" ) return ;

    DOM::Element bkupElm1, bkupElm2, bkupElm3;
    int readNum = m_datInfo->getReadNum();

    /* remove < a href="#tosaigo"> and <BR> */
    if ( strType == "header" ) {
        bkupElm1 = targetelm.removeChild( targetelm.lastChild() ); /* BR */
        bkupElm2 = targetelm.removeChild( targetelm.lastChild() ); /* BR */
        bkupElm3 = targetelm.removeChild( targetelm.lastChild() ); /* "#tosaigo" */
    } else { /* "footer" */
        bkupElm1 = targetelm.removeChild( targetelm.lastChild() ); /* "#tosaigo" */
    }

    DOM::Node node = targetelm.firstChild(); /* node is now "#kokomade_yonda" */
    node = node.nextSibling(); /* " " */
    node = node.nextSibling(); /* "#zenbu" */
    node = node.nextSibling(); /* " " */
    node = node.nextSibling();

    /* append new "A" nodes */
    for ( int num = 1; num < readNum ; num += 100 ) {

        if ( node == NULL ) {
            QString linkstr = QString( "#%1" ).arg( num );
            appendANode( targetelm, linkstr, QString().setNum( num ) + "-" );
            node = targetelm.appendChild( m_hdoc.createTextNode( " " ) );
            node = node.nextSibling();

            m_headerMaxNum = num;
        } else {
            node = node.nextSibling();
            if ( node != NULL ) node = node.nextSibling();
        }
    }

    /* append < a href="#tosaigo"> and <BR> */
    if ( strType == "header" ) {
        targetelm.appendChild( bkupElm3 ); /* "#tosaigo" */
        targetelm.appendChild( bkupElm2 ); /* BR */
        targetelm.appendChild( bkupElm1 ); /* BR */
    } else { /* "footer" */
        targetelm.appendChild( bkupElm1 ); /* "#tosaigo" */
    }
}


/* create footer or header node   */ /* private */
void KitaDomTree::createFooterOrHeader(
    QString strType /* = "header" or "footer" */
)
{
    if ( m_hdoc == NULL ) return ;
    if ( strType != "header" && strType != "footer" ) return ;

    QString str;
    DOM::Element rootnode, tmpelm;

    rootnode = m_hdoc.createElement( "DIV" );
    {
        rootnode.setAttribute( "kita_type", strType );
        rootnode.setAttribute( "id", strType );

        str = Kita::utf8ToUnicode( KITAUTF8_KOKOYON );
        appendANode( rootnode, "#kokomade_yonda", str );
        rootnode.appendChild( m_hdoc.createTextNode( " " ) );

        str = Kita::utf8ToUnicode( KITAUTF8_ZENBU );
        appendANode( rootnode, "#zenbu", str );
        rootnode.appendChild( m_hdoc.createTextNode( " " ) );

        str = Kita::utf8ToUnicode( KITAUTF8_SAIGO );
        appendANode( rootnode, "#tosaigo", str );

        if ( strType == "header" ) {
            rootnode.appendChild( m_hdoc.createElement( "BR" ) );
            rootnode.appendChild( m_hdoc.createElement( "BR" ) );
        }
    }

    if ( strType == "header" ) m_header = rootnode;
    else m_footer = rootnode;

    m_headerMaxNum = 0;
}


/* create kokomadeyonda node */ /* private */
void KitaDomTree::createKokoyon()
{
    QString str, style;
    DOM::Element rootnode;

    str = Kita::utf8ToUnicode( KITAUTF8_KOKOYON2 );

    rootnode = m_hdoc.createElement( "DIV" );
    {
        rootnode.setAttribute( "class", "kokoyon" );
        rootnode.setAttribute( "kita_type", "kokoyon" );
        rootnode.setAttribute( "id", "kokomade_yonda" );
        rootnode.appendChild( m_hdoc.createTextNode( str ) );
    }

    m_kokoyon = rootnode;
}



/* create tugi 100 node  */ /* private */
void KitaDomTree::createTugi100()
{
    QString str, style;
    DOM::Element rootnode;

    style = "text-align: center";

    rootnode = m_hdoc.createElement( "DIV" );
    {
        rootnode.setAttribute( "kita_type", "tugi100" );
        rootnode.setAttribute( "id", "tugi100" );
        rootnode.setAttribute( "style", style );

        str = Kita::utf8ToUnicode( KITAUTF8_TUGI100 );
        appendANode( rootnode, "#tugi100", str );

        rootnode.appendChild( m_hdoc.createTextNode( "  " ) );

        str = Kita::utf8ToUnicode( KITAUTF8_ZENBUNOKORI );
        appendANode( rootnode, "#nokori", str );

        rootnode.appendChild( m_hdoc.createElement( "BR" ) );
        rootnode.appendChild( m_hdoc.createElement( "BR" ) );
    }

    m_tugi100 = rootnode;
}



/* create mae 100 (template)  node */ /* mae 100 */
void KitaDomTree::createMae100()
{
    QString style;
    DOM::Element rootnode;

    style = "text-align: center";

    rootnode = m_hdoc.createElement( "DIV" );
    {
        rootnode.setAttribute( "kita_type", "mae100" );
        rootnode.setAttribute( "id", "mae100" );
        rootnode.setAttribute( "style", style );

        /* 100 */
        QString tugi100 = Kita::utf8ToUnicode( KITAUTF8_TUGI100 );
        appendANode( rootnode, "#tmp100", tugi100 );

        /* */
        rootnode.appendChild( m_hdoc.createElement( "BR" ) );
        rootnode.appendChild( m_hdoc.createElement( "BR" ) );

        /* */
        style = "background-color: #CCCCCC; text-align: center";
        DOM::Element divElement = rootnode.appendChild( m_hdoc.createElement( "DIV" ) );
        {
            divElement.setAttribute( "style", style );
            QString templateStr = Kita::utf8ToUnicode( KITAUTF8_TEMPLATE );
            divElement.appendChild( m_hdoc.createTextNode( templateStr ) );
        }

        /* */
        rootnode.appendChild( m_hdoc.createElement( "BR" ) );

        /* 100 */
        QString mae100 = Kita::utf8ToUnicode( KITAUTF8_MAE100 );
        appendANode( rootnode, "#mae100", mae100 );

        /* */
        rootnode.appendChild( m_hdoc.createTextNode( "  " ) );

        /*  */
        QString zenbumae = Kita::utf8ToUnicode( KITAUTF8_ZENBUMAE );
        appendANode( rootnode, "#maezenbu", zenbumae );

        /* */
        rootnode.appendChild( m_hdoc.createElement( "BR" ) );
        rootnode.appendChild( m_hdoc.createElement( "BR" ) );
    }

    m_mae100 = rootnode;
}

/* change color of number */
/*
   specify color like this:  "a.coloredLink:link{ color: red; }"   */ /* private */

void KitaDomTree::changeColorOfNumber( int num )
{
    if ( ! m_datInfo ) return ;
    if ( m_hdoc == NULL ) return ;
    if ( m_coloredNum[ num ] ) return ;
    if ( !m_resshown[ num ] ) return ;

    m_coloredNum[ num ] = TRUE;
    DOM::Node node = m_titleElm[ num ];
    node = node.firstChild();
    while ( ! node.isNull() && node.nodeName().string() != "a" ) {
        node = node.firstChild();  /* table tag is used. */
    }
    if ( node.isNull() ) return;

    static_cast< DOM::HTMLElement> ( node ).setAttribute( "class", "coloredLink" );
}
