/***************************************************************************
*   Copyright (C) 2004 by Kita Developers                                 *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

/* DatManager manages all information about thread */

#include <qobject.h>
#include <qmutex.h>
#include <qstringlist.h>
#include <qregexp.h>
#include <qfile.h>

#include "thread.h"
#include "threadinfo.h"
#include "datmanager.h"
#include "datinfo.h"
#include "cache.h"
#include "kita_misc.h"
#include "kita-utf8.h"
#include "kita-utf16.h"
#include "threadindex.h"
#include "boardmanager.h"

using namespace Kita;

#define DMANAGER_MAXQUEUE 16

DatInfoList DatManager::m_datInfoList;
QMutex DatManager::m_mutex;


/*-----------------------------------------------------------------------*/


/* create DatInfo explicitly.                    */
/* Usually, DatInfo is NOT created
   if cache does not exist( i.e. ReadNum == 0 ). */ /* public */
bool DatManager::createDatInfo( const KURL& url )
{
    if ( getDatInfo( url,
                     FALSE /* don't check the existence of cache */
                   ) == NULL ) return FALSE;
    return TRUE;
}


/* get pointer of DatInfo */

/*    !!!  NOTICE  !!!   */
/* It is very dangerous to access to DatInfo directly. */
/* Usually, access to it through DatManager.           */ /* public */
DatInfo * DatManager::getDatInfoPointer( const KURL& url )
{
    return getDatInfo( url );
}


/*------------------------------------------------------------------------*/


/* This function searches instance of DatInfo in m_datInfoList.
 
   If cache exists, create DatInfo and return its pointer.
 
   If checkCached == TRUE and cache does not exist, return NULL.
 
   If checkCached == FALSE and cache does not exist,
   create DatInfo and return its pointer anyway.
 
   see also DatManager::searchDatInfo() and DatManager::createDatInfo() */ /* private */

DatInfo* DatManager::getDatInfo( const KURL& url, bool checkCached )
{
    /* search */
    DatInfo * datInfo = searchDatInfo( url );
    if ( datInfo != NULL ) return datInfo;

    /* create and enroll instance */
    return enrollDatInfo( url, checkCached );
}


/* This function just searches instance of DatInfo specified by datURL
   without creating instance.  */ /* private */
DatInfo* DatManager::searchDatInfo( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    KURL datURL = Kita::getDatURL( url );
    if ( datURL.isEmpty() ) return NULL; /* This url is not enrolled in BoardManager. */
    if ( m_datInfoList.count() == 0 ) return NULL;

    int i = 0;
    DatInfoList::Iterator it;
    DatInfo* datInfo;

    for ( it = m_datInfoList.begin(); it != m_datInfoList.end(); ++it, i++ ) {

        datInfo = ( *it );

        if ( datURL == datInfo->url() ) {

            /* LRU */
            if ( i ) {
                m_datInfoList.remove( it );
                m_datInfoList.prepend( datInfo );
            }

            return datInfo;
        }
    }

    return NULL;
}


/* create and enroll the instance of DatInfo and delete old instances.
   Note that DatInfo::DatInfo() opens cached data and reads it. */  /* private */
DatInfo* DatManager::enrollDatInfo( const KURL& url, bool checkCached )
{
    QMutexLocker locker( &m_mutex );

    KURL datURL = Kita::getDatURL( url );
    if ( datURL.isEmpty() ) return NULL; /* This url is not enrolled in BoardManager. */

    /* create DatInfo & read cached data */
    DatInfo* datInfo = new DatInfo( datURL );

    /* Does cache exist ? */
    /* If cache does not exist, delete DatInfo here. */
    if ( checkCached && datInfo->getReadNum() == 0 ) {
        delete datInfo;
        return NULL;
    }

    m_datInfoList.prepend( datInfo );

    /* delete the all old instances ( LRU algorithm )*/
    if ( m_datInfoList.count() > DMANAGER_MAXQUEUE ) {

        DatInfoList::Iterator it;
        for ( it = m_datInfoList.at( DMANAGER_MAXQUEUE ); it != m_datInfoList.end(); ++it ) {

            if ( ( *it ) == NULL ) continue;
            DatInfo* deleteInfo = ( *it );

            if ( ! deleteInfo->isLocked() ) {
                m_datInfoList.remove( it );
                --it;
                deleteInfo->wait(); /* wait until DatInfo::m_mutex is released. */
                delete deleteInfo;
            }
        }
    }

    return datInfo;
}


/* public */
void DatManager::deleteAllDatInfo()
{
    DatInfoList::Iterator it;
    for ( it = m_datInfoList.begin(); it != m_datInfoList.end(); ++it ) {

        if ( ( *it ) == NULL ) continue;
        ( *it ) ->wait();
        delete ( *it );
    }
}



/*-------------------------------------------------------------*/



/* update cache */   /* public */
bool DatManager::updateCache( const KURL& url , const QObject* parent )
{
    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return FALSE;

    return datInfo->updateCache( parent );
}


/* public */
int DatManager::getResponseCode( const KURL& url )
{
    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return 0;

    return datInfo->getResponseCode();
}

/* public */
int DatManager::getServerTime( const KURL& url )
{
    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return 0;

    return datInfo->getServerTime();
}


/* public */
bool DatManager::deleteCache( const KURL& url )
{
    KURL datURL = Kita::getDatURL( url );
    Kita::Thread* thread = Kita::Thread::getByURLNew( datURL );
    if ( thread == NULL ) return FALSE;
    if ( thread->readNum() == 0 ) return FALSE;

    /* init DatInfo */
    DatInfo * datInfo = searchDatInfo( datURL );
    if ( datInfo ) {
        if ( !datInfo->deleteCache() ) return FALSE;
    }

    /* reset readNum & veiwPos */
    thread->setReadNum( 0 );
    thread->setViewPos( 0 );

    /* delete cache */
    QString cachePath = Kita::Cache::getPath( datURL );
    QString indexPath = Kita::Cache::getIndexPath( datURL );
    QFile::remove( indexPath );
    QFile::remove( cachePath );

    /* delete log from "cache" */
    KitaThreadInfo::removeThreadInfo( datURL.prettyURL() );
    return TRUE;
}


/* public */
bool DatManager::isLoadingNow( const KURL& url )
{
    DatInfo * datInfo = searchDatInfo( url );
    if ( datInfo == NULL ) return FALSE;

    return datInfo->isLoadingNow();
}


/* public */
void DatManager::stopLoading( const KURL& url )
{
    DatInfo * datInfo = searchDatInfo( url );
    if ( datInfo == NULL ) return ;

    return datInfo->stopLoading();
}



/*----------------------*/
/* lock, unlock DatInfo */

/* public */
void DatManager::lock ( const KURL& url )
{
    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return ;

    datInfo->lock ();
}

/* public */
void DatManager::unlock( const KURL& url )
{
    DatInfo * datInfo = searchDatInfo( url );
    if ( datInfo == NULL ) return ;

    datInfo->unlock();
}


/*--------------------------------------*/
/* string data */


/* public */
const QString& DatManager::getDat( const KURL& url, int num )
{
    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return QString::null;

    return datInfo->getDat( num );
}



/* public */
const QString& DatManager::getId( const KURL& url, int num )
{
    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return QString::null;

    return datInfo->getId( num );
}


/* public */
QString DatManager::getPlainName( const KURL& url, int num )
{
    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return QString::null;

    return datInfo->getPlainName( num );
}


/* public */
QString DatManager::getPlainBody( const KURL& url, int num )
{
    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return QString::null;

    return datInfo->getPlainBody( num );
}


/* public */
QString DatManager::getPlainTitle( const KURL& url, int num )
{
    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return QString::null;

    return datInfo->getPlainTitle( num );
}


/* get name (i.e. subject ) of thread from URL of dat file. */ /* public */
const QString DatManager::threadName( const KURL& url )
{
    KURL datURL = Kita::getDatURL( url );
    Kita::Thread* thread = Kita::Thread::getByURLNew( datURL );
    if ( thread != NULL ) return thread->threadName();

    return QString::null;
}


/* public */
const QString DatManager::threadID( const KURL& url )
{
    KURL datURL = Kita::getDatURL( url );
    return datURL.filename().section( ".", 0, 0 );
}


const QString DatManager::getCachePath( const KURL& url )
{
    return Kita::Cache::getPath( url );
}

const QString DatManager::getCacheIndexPath( const KURL& url )
{
    return Kita::Cache::getIndexPath( url );
}

/*---------------------------------------*/
/* HTML data */

/* public */
QString DatManager::getHtml( const KURL& url, int startnum, int endnum, bool checkAbone )
{
    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return QString::null;

    return datInfo->getHTMLString( startnum, endnum, checkAbone );
}



/* public */
QString DatManager::getHtmlByID( const KURL& url, const QString& strid, int &count )
{
    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return QString::null;

    return datInfo->getHtmlByID( strid, count );
}



/* Get HTML document of res tree.*/ /* public */
QString DatManager::getTreeByRes( const KURL& url, const int rootnum, int &count )
{
    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return QString::null;

    return datInfo->getTreeByRes( rootnum, count );
}

/* Get HTML document of reverse res tree.*/ /* public */
QString DatManager::getTreeByResReverse( const KURL& url, const int rootnum, int &count )
{
    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return QString::null;

    return datInfo->getTreeByResReverse( rootnum, count );
}


/* public */
int DatManager::getResNum( const KURL& url )
{
    KURL datURL = Kita::getDatURL( url );
    Kita::Thread* thread = Kita::Thread::getByURLNew( datURL );
    if ( thread != NULL ) return thread->resNum();

    return 0;
}


/* public */
int DatManager::getReadNum( const KURL& url )
{
    KURL datURL = Kita::getDatURL( url );
    Kita::Thread* thread = Kita::Thread::getByURLNew( datURL );
    if ( thread != NULL ) return thread->readNum();

    return 0;
}


/* public */
int DatManager::getViewPos( const KURL& url )
{
    KURL datURL = Kita::getDatURL( url );
    Kita::Thread* thread = Kita::Thread::getByURLNew( datURL );
    if ( thread != NULL ) return thread->viewPos();

    return 0;
}


/* public */
void DatManager::setViewPos( const KURL& url , int num )
{
    KURL datURL = Kita::getDatURL( url );
    Kita::Thread* thread = Kita::Thread::getByURLNew( datURL );
    if ( thread != NULL ) thread->setViewPos( num );

    /* save idx */
    Kita::ThreadIndex::setViewPos( url, num );

    /* save "cache" */
    KitaThreadInfo::setReadNum( datURL.prettyURL(), num );
}


/* public */
int DatManager::getDatSize( const KURL& url )
{
    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return 0;

    return datInfo->getDatSize();
}

/* get number of responses which have same ID. */ /* public */
int DatManager::getNumByID( const KURL& url, const QString& strid )
{
    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return 0;

    return datInfo->getNumByID( strid );
}


/* public */
bool DatManager::isThreadEnrolled( const KURL& url )
{
    if ( Kita::getDatURL( url ).isEmpty() ) return FALSE;

    return TRUE;
}


/* public */
bool DatManager::is2chThread( const KURL& url )
{
    if ( BoardManager::type( url ) != Board_2ch ) return FALSE;
    if ( Kita::getDatURL( url ).isEmpty() ) return FALSE;

    QRegExp url_2ch( ".*\\.2ch\\.net" );
    QRegExp url_bbspink( ".*\\.bbspink\\.com" );

    if ( url_2ch.search( url.host() ) != -1
            || url_bbspink.search( url.host() ) != -1 ) return TRUE;

    return FALSE;
}


/* public */
bool DatManager::isResValid( const KURL& url, int num )
{
    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return FALSE;

    return datInfo->isResValid( num );
}


/* public */
bool DatManager::isBroken( const KURL& url )
{
    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return FALSE;

    return datInfo->isBroken();
}

/* public */
bool DatManager::isResBroken( const KURL& url, int num )
{
    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return FALSE;

    return datInfo->isResBroken( num );
}



/* check if ID == strid  */ /* public */
bool DatManager::checkID( const KURL& url, const QString& strid, int num )
{
    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return FALSE;

    return datInfo->checkID( strid, num );
}


/* check if keywords are included */ /* public */
bool DatManager::checkWord( const KURL& url,
                            QStringList& strlist, int num,
                            bool checkOR /* AND or OR search */
                          )
{
    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return FALSE;

    return datInfo->checkWord( strlist, num, checkOR );
}


/* public */
bool DatManager::isMarked( const KURL& url, int num )
{
    KURL datURL = Kita::getDatURL( url );
    Kita::Thread* thread = Kita::Thread::getByURLNew( datURL );
    if ( thread == NULL ) return FALSE;

    return thread->isMarked( num );
}


/* public */
void DatManager::setMark( const KURL& url, int num, bool mark )
{
    KURL datURL = Kita::getDatURL( url );
    Kita::Thread* thread = Kita::Thread::getByURLNew( datURL );
    if ( thread == NULL ) return ;

    if ( thread->setMark( num, mark ) ) Kita::ThreadIndex::setMarkList( url, thread->markList() );
}


/* public */
bool DatManager::checkAbone( const KURL& url, int num )
{
    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return FALSE;

    return datInfo->checkAbone( num );
}


/* public */
void DatManager::resetAbone( const KURL& url )
{
    DatInfo * datInfo = getDatInfo( url );
    if ( datInfo == NULL ) return ;

    datInfo->resetAbone();
}


/* check if the thread is shown on the main thread tab. */ /* public */
bool DatManager::isMainThreadOpened( const KURL& url )
{
    KURL datURL = Kita::getDatURL( url ).prettyURL();
    Kita::Thread* thread = Kita::Thread::getByURLNew( datURL );
    if ( thread == NULL ) return FALSE;

    return thread->isOpened();
}

void DatManager::setMainThreadOpened( const KURL& url, bool isOpened )
{
    KURL datURL = Kita::getDatURL( url ).prettyURL();
    Kita::Thread* thread = Kita::Thread::getByURLNew( datURL );
    if ( thread != NULL ) thread->setIsOpened( isOpened );
}


/*--------------------------*/
/* obsolete */

/* public */
const QString DatManager::threadURL( const KURL& url )
{
    return Kita::getThreadURL( url );
}

