/***************************************************************************
*   Copyright (C) 2004 by Kita Developers                                 *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/
#include "access.h"
#include "cache.h"
#include "kita_misc.h"
#include "account.h"
#include "boardmanager.h"

#include <config.h>

#include <kurl.h>
#include <kio/slaveconfig.h>
#include <kio/job.h>

#include <qregexp.h>
#include <qfile.h>

using namespace Kita;


Access::Access( const KURL& datURL ) : m_datURL( datURL ), m_currentJob( 0 )
{
    init();
}

/* public */
void Access::init()
{
    m_readNum = 0;
    m_lastLine = QString::null;
    m_bbstype = BoardManager::type( m_datURL );
    m_header = "HTTP/1.1 200 ";  /* dummy header */
    m_dataSize = 0;
    m_threadData = QString::null;
}

/* read data from cache, then emit data to DatInfo. */ /* public */
void Access::getcache()
{
    QString orgData;

    // get cache path.
    QString cachePath = Kita::Cache::getPath( m_datURL );
    if ( cachePath == QString::null ) {
        return;
    }

    // read cache file.
    QFile file( cachePath );
    if ( file.open( IO_ReadOnly ) ) {
        orgData += file.readAll();
        file.close();
    }

    // set data size.
    if ( orgData == QString::null ) return ;
    m_dataSize = orgData.length();

    switch ( m_bbstype ) {

    case Board_2ch:
    case Board_MachiBBS:  /* Machi BBS's data is already parsed as 2ch dat. */

        {
            QString tmpData = Kita::qcpToUnicode( orgData );
            QStringList tmpList = QStringList::split( "\n", tmpData );
            emit receiveData( tmpList );
        }
        break;

    default:
        /* convert data stream into 2ch dat.
        and emit receiveData SIGNAL.        */
        emitDatLineList( orgData );
        break;
    }
}

/* write data to cache */ /* protected */
void Access::writeCacheData()
{
    if ( m_invalidDataReceived ) return ;
    if ( m_threadData.length() == 0 ) return ;

    m_dataSize += m_threadData.length();

    QString cachePath = Kita::Cache::getPath( m_datURL );
    if ( cachePath != QString::null ) {
        FILE * fs = fopen( QFile::encodeName( cachePath ), "a" );
        if ( !fs ) return ;

        fwrite( m_threadData, m_threadData.length(), 1, fs );
        fclose( fs );
    }
    m_threadData = QString::null; /* clear baffer */

    return ;
}

/* update cache file */ /* public */
bool Access::getupdate( int readNum )
{
    /* init */
    m_readNum = readNum;
    m_threadData = QString::null;
    m_firstReceive = FALSE;
    m_invalidDataReceived = FALSE;
    m_lastLine = QString::null;

    /* set URL of data */
    QString getURL;
    switch ( m_bbstype ) {

    case Board_MachiBBS:
        getURL = Kita::getThreadURL( m_datURL );
        if ( m_readNum > 0 ) getURL += "&START=" + QString().setNum( m_readNum + 1 );
        Kita::InitParseMachiBBS();
        break;

    case Board_JBBS:
        getURL = Kita::getThreadURL( m_datURL );
        getURL.replace( "read.cgi", "rawmode.cgi" );  /* adhoc... */
        if ( m_readNum > 0 ) getURL += "/" + QString().setNum( m_readNum + 1 ) + "-";
        break;

    default:
        getURL = m_datURL.prettyURL();
    }

    /* set UserAgent */
    const QString useragent = QString( "Monazilla/1.00 (Kita/%1)" ).arg( VERSION );
    KIO::SlaveConfig::self() ->setConfigData( "http",
            KURL( getURL ).host(),
            "UserAgent", useragent );

    /* create new job */
    KIO::TransferJob* job = KIO::get( getURL, true, false );
    m_currentJob = job;

    connect( job, SIGNAL( data( KIO::Job*, const QByteArray& ) ),
             SLOT( slotReceiveThreadData( KIO::Job*, const QByteArray& ) ) );
    connect( job, SIGNAL( result( KIO::Job* ) ), SLOT( slotThreadResult( KIO::Job* ) ) );

    // use 'HTTP-Headers' metadata.
    job->addMetaData( "PropagateHttpHeader", "true" );

    /* resume */
    if ( m_bbstype != Board_MachiBBS
            && m_bbstype != Board_JBBS
            && m_dataSize > 0 ) {
        m_firstReceive = TRUE; /* remove first char (i.e. \n). see also slotReceiveThreadData() */
        job->addMetaData( "resume", QString::number( m_dataSize - 1 ) );
        job->addMetaData( "AllowCompressedPage", "false" );
    }

    return TRUE;
}

void Access::slotThreadResult( KIO::Job* job )
{
    m_currentJob = 0;
    if ( job->error() ) {
        job->showErrorDialog();
    } else {
        m_header = job->queryMetaData( "HTTP-Headers" );
    }

    writeCacheData();
    emit finishLoad();
}

void Access::slotReceiveThreadData( KIO::Job*, const QByteArray& data )
{
    QByteArray data_tmp = data.copy();
    
    // HACK: crash when data contains '\0'.
    for ( int i=0; i< data_tmp.size(); i++ ) {
        if ( data_tmp[ i ] == '\0' ) data_tmp[ i ] = ' ';
    }

    QString cstr( data_tmp );

    if ( m_bbstype == Board_MachiBBS
            || m_bbstype == Board_JBBS ) {

        emitDatLineList( cstr );
        return ;
    }

    /* check if received data is invalid ( or broken ). */
    if ( ( m_dataSize > 0 && responseCode() != 206 )
            || ( m_firstReceive && cstr[ 0 ] != '\n' )
            || ( m_dataSize == 0 && responseCode() != 200 )
       ) m_invalidDataReceived = TRUE;

    if ( m_invalidDataReceived ) return ;

    /* If this is the first call at resumption, remove LF(\n) at head. */
    if ( m_firstReceive ) {
        cstr = cstr.mid( 1 );
    }
    m_firstReceive = FALSE;

    emitDatLineList( cstr );
}


/* convert data stream into 2ch dat.
   and emit receiveData SIGNAL.        */  /* private */
void Access::emitDatLineList( const QString& dataStream )
{
    QStringList lineList;
    QStringList datLineList;
    if ( dataStream == QString::null ) return ;

    bool endIsLF = FALSE;
    if ( dataStream.at( dataStream.length() - 1 ) == '\n' ) endIsLF = TRUE;

    /* split the stream */
    m_lastLine += dataStream;
    lineList = QStringList::split( "\n", m_lastLine );
    m_lastLine = QString::null;

    /* save the last line */
    if ( !endIsLF ) {

        QStringList::iterator lastit = lineList.end();
        lastit--;
        if ( lastit != lineList.end() ) {

            m_lastLine = ( *lastit );
            lineList.remove( lastit );
        }
    }

    /* filtering */

    /* convert lines into 2ch dat */
    int count = lineList.count();
    for ( int i = 0; i < count ; ++i ) {

        if ( lineList[ i ] != QString::null ) {

            QString line;
            int nextNum = m_readNum + 1;

            /* convert line */
            switch ( m_bbstype ) {

            case Board_MachiBBS:
                line = Kita::ParseMachiBBSOneLine( lineList[ i ] , nextNum );
                break;

            case Board_JBBS:
                line = Kita::ParseJBBSOneLine( lineList[ i ] , nextNum );
                break;

            case Board_FlashCGI:
                line = Kita::ParseFlashCGIOneLine( lineList[ i ] );
                break;

            default:
                line = lineList[ i ];
            }

            if ( line == QString::null ) continue;

            /* add abone lines */
            const QString aboneStr = "abone<><><>abone<>";
            while ( nextNum > m_readNum + 1 ) {
                datLineList += aboneStr;
                m_threadData += aboneStr + "\n";
                ++m_readNum;
            }

            /* save line */
            if ( m_bbstype == Board_MachiBBS ) m_threadData += line + "\n";
            else m_threadData += lineList[ i ] + "\n";
            ++m_readNum;

            /* encode line */
            switch ( m_bbstype ) {

            case Board_JBBS:
                datLineList += Kita::ecuToUnicode( line );
                break;

            default:
                datLineList += Kita::qcpToUnicode( line );
            }
        }
    }

    /* call DatInfo::slotReceiveData() */
    emit receiveData( datLineList );
}



void Access::killJob()
{
    if ( m_currentJob ) m_currentJob->kill();
}

void Access::stopJob()
{
    if ( m_currentJob ) m_currentJob->kill( FALSE ); /* emit result signal */
}

int Access::serverTime()
{
    if ( m_currentJob ) m_header = m_currentJob->queryMetaData( "HTTP-Headers" );
    // parse HTTP headers
    QStringList headerList = QStringList::split( "\n", m_header );
    QRegExp regexp( "Date: (...), (..) (...) (....) (..:..:..) .*" );
    QString dateStr = headerList.grep( regexp ) [ 0 ];
    if ( regexp.search( dateStr ) == -1 ) {
        // invalid date format
        return QDateTime::currentDateTime().toTime_t();
    } else {
        // I hate this format ;p
        QString usLocalDateStr = regexp.cap( 1 ) + " " + regexp.cap( 3 ) + " " +
                                 regexp.cap( 2 ) + " " + regexp.cap( 5 ) + " " +
                                 regexp.cap( 4 );

        // 1970/01/01 00:00:00 GMT
        QDateTime zeroTime( QDate( 1970, 1, 1 ), QTime( 0, 0 ) );
        return zeroTime.secsTo( QDateTime::fromString( usLocalDateStr ) );
    }
}

int Access::responseCode()
{
    if ( m_currentJob ) m_header = m_currentJob->queryMetaData( "HTTP-Headers" );
    // parse HTTP headers
    QStringList headerList = QStringList::split( "\n", m_header );
    QRegExp regexp( "HTTP/1\\.[01] ([0-9]+) .*" );
    QString dateStr = headerList.grep( regexp ) [ 0 ];
    if ( regexp.search( dateStr ) == -1 ) {
        // invalid response
        if ( m_bbstype == Board_JBBS ) return 200; /* adhoc... */
        return 0;
    } else {
        return regexp.cap( 1 ).toInt();
    }
}

/* public */
const int Access::dataSize() const
{
    return m_dataSize;
}

/* public */
const bool Access::invalidDataReceived() const
{
    return m_invalidDataReceived;
}

//
// access offlaw.cgi
//
QString OfflawAccess::get()
{
    QString getURL = Kita::datToOfflaw( m_datURL.url() );
    KURL kgetURL( getURL );
    kgetURL.addQueryItem( "sid", Account::getSessionID() );

    m_threadData = "";
    m_invalidDataReceived = FALSE;

    KIO::SlaveConfig::self() ->setConfigData( "http",
            KURL( getURL ).host(),
            "UserAgent",
            QString( "Monazilla/1.00 (Kita/%1)" ).arg( VERSION ) );

    KIO::TransferJob* job = KIO::get( kgetURL, true, false );
    m_currentJob = job;

    connect( job, SIGNAL( data( KIO::Job*, const QByteArray& ) ),
             SLOT( slotReceiveThreadData( KIO::Job*, const QByteArray& ) ) );
    connect( job, SIGNAL( result( KIO::Job* ) ), SLOT( slotThreadResult( KIO::Job* ) ) );

    // use 'HTTP-Headers' metadata.
    job->addMetaData( "PropagateHttpHeader", "true" );

    return QString::null; /* dummy */
}

void OfflawAccess::slotThreadResult( KIO::Job* job )
{
    m_currentJob = 0;
    if ( job->error() ) {
        job->showErrorDialog();
    } else {
        m_header = job->queryMetaData( "HTTP-Headers" );
    }

    if ( !m_invalidDataReceived && m_threadData.length() ) {
        KURL url = m_datURL;
        writeCacheData();
    }
    emit finishLoad();
}

void OfflawAccess::slotReceiveThreadData( KIO::Job*, const QByteArray& data )
{
    QString cstr( data );

    if ( ( m_dataSize > 0 && responseCode() != 206 )
            || ( m_dataSize == 0 && responseCode() != 200 ) ) {
        m_invalidDataReceived = TRUE;
    }

    if ( m_invalidDataReceived ) return ;

    // "+OK ....../1024K\tLocation:temp/\n"
    if ( m_threadData.isEmpty() && cstr[ 0 ] == '+' ) {
        // skip first line.
        int index = cstr.find( '\n' );
        cstr = cstr.mid( index + 1 );
    }
    emitDatLineList( cstr );
}

#include "access.moc"
