#include "kj.h"

#if RXMMS
extern gint xmms_running, xmms_session;
#endif

extern k_resource res;
extern k_config config;
extern gint cur_track;

static GtkWidget *win_pl = NULL;
static GdkPixmap *bg_pl = NULL;
static GdkGC *gc_pl;
static gint pl_move, pl_button_x, pl_button_y, pl_list_top = 0;
static gint old_track = -1, old_pl_len = -1;

static GtkWidget *tpl_filesel = NULL;
static GList **tpl_filesel_list = NULL;
static gint tpl_load_list;

GList *kj_play_list = NULL;

/* ---------------------------------------------------------------------- */
static void pl_draw_list(GdkWindow *w, k_area *pos);
static void tpl_send_xmms_playlist(GList **pl, gint session);
static void tpl_refresh_playlist(GList *pl);

/* ---------------------------------------------------------------------- */
static void tpl_clear_selection(GList *pl)
{
GList *n;

	for(n = pl; n != NULL; n = g_list_next(n))
		((plist_entry *)n->data)->selected = 0;
}


/* ----------------------------------------------------------------------
 *  Mark nth entry in playlist as selected.
 *  mode: 0 = normal click, 1 = shift click, 2 = ctrl click.
 */
static void tpl_select_nth(GList *pl, int num, int mode)
{
GList *n;
static gint last_sel = -1;
gint strt, len, i;

	if(mode != 2) tpl_clear_selection(pl);
	if(mode == 1 && last_sel >= 0)
		{
		strt = (last_sel > num ? num : last_sel);
		len = abs(last_sel - num);
		for(n = g_list_nth(pl, strt), i = 0; i <= len && n != NULL; i++, n = g_list_next(n))
			((plist_entry *)n->data)->selected = 1;
		}
	else
		{
		last_sel = num;
		n = g_list_nth(pl, num);
		((plist_entry *)n->data)->selected = 1;
		}
}


/* ---------------------------------------------------------------------- */
static gint tpl_delete_selected(GList **pl)
{
GList *n;
plist_entry *ent;
gint change = 0;

	do {
		for(n = *pl; n != NULL; n = g_list_next(n))
			if(((plist_entry *)n->data)->selected)
				{
				ent = (plist_entry *)n->data;
				if(ent->filename) g_free(ent->filename);
				if(ent->title) g_free(ent->title);
				g_free(ent);
				*pl = g_list_remove_link(*pl, n);
				change = 1;
				break;
				}
	} while(n != NULL);

	if(change) tpl_refresh_playlist(*pl);
	return change;
}


/* ---------------------------------------------------------------------- */
static void tpl_playlist_clear(GList **pl)
{
GList *n;
plist_entry *ent;

	if(!*pl) return;

	for(n = *pl; n != NULL; n = g_list_next(n))
		{
		ent = (plist_entry *)n->data;
		if(ent->filename) g_free(ent->filename);
		if(ent->title) g_free(ent->title);
		g_free(ent);
		}

	g_list_free(*pl);
	*pl = NULL;
}


/* ---------------------------------------------------------------------- */
static plist_entry *tpl_add_playlist(GList **pl, gchar *filename, gchar *title, gint length, glong pos)
{
plist_entry *ent;
gchar *p;

	ent = g_malloc0(sizeof(plist_entry));
	if(filename) ent->filename = g_strdup(filename);
	if(title) ent->title = g_strdup(title);
	else
		{
		if((p = strrchr(filename, '/'))) p++;
		else p = filename;
		ent->title = g_strdup(p);
		if((p = strrchr(ent->title, '.'))) *p = '\0';
		}
	ent->length = length;
	ent->selected = FALSE;
	if(pos == -1) *pl = g_list_append(*pl, ent);
	else *pl = g_list_insert(*pl, ent, pos);

	return ent;
}


/* ---------------------------------------------------------------------- */
static void tpl_save_playlist(GList *pl, char *filename)
{
FILE *fp;
GList *n;

	if((fp = fopen(filename, "w")) == NULL) return;
	for(n = pl; n != NULL; n = g_list_next(n))
		fprintf(fp, "%s\n", ((plist_entry *)n->data)->filename);

	fclose(fp);
}


/* ---------------------------------------------------------------------- */
static void tpl_load_playlist(GList **pl, char *filename)
{
FILE *fp;
gchar *line, *p;
int linelen = 1024;

	tpl_playlist_clear(pl);

	if((fp = fopen(filename, "r")) == NULL) return;

	line = g_malloc(linelen);
	while(fgets(line, linelen, fp))
		{
		while(strlen(line) == linelen - 1 && line[strlen(line) - 1] == '\n')
			{
			linelen += 1024;
			line = (gchar *)g_realloc(line, linelen);
			fgets(&line[strlen(line)], 1024, fp);
			}

		for(p = line+strlen(line)-1; (*p == '\r' || *p == '\n') && p > line; p--) *p = '\0';

		tpl_add_playlist(pl, line, NULL, -1, -1);
		}

	fclose(fp);
}


/* ---------------------------------------------------------------------- */
static void tpl_get_xmms_playlist(GList **pl, gint session, gboolean get_time)
{
int i;

	tpl_playlist_clear(pl);

#if RXMMS
	if(xmms_running)
		{
		for(i = 0; i < xmms_remote_get_playlist_length(session); i++)
			{
			if(get_time)
				tpl_add_playlist(pl, xmms_remote_get_playlist_file(session, i),
					xmms_remote_get_playlist_title(xmms_session, i),
					xmms_remote_get_playlist_time(session, i), -1);
			else
				tpl_add_playlist(pl, xmms_remote_get_playlist_file(session, i),
					xmms_remote_get_playlist_title(xmms_session, i), -1, -1);
			}
		}
#endif
}


/* ---------------------------------------------------------------------- */
static void tpl_send_xmms_playlist(GList **pl, gint session)
{
int i, num;
gchar **flist;
GList *n;

#if RXMMS
	if(xmms_running)
		{
		xmms_remote_playlist_clear(session);
		num = g_list_length(*pl);
		flist = g_malloc0(num * sizeof(gchar *));
		for(n = *pl, i = 0; n != NULL; n = g_list_next(n), i++)
			flist[i] = ((plist_entry *)n->data)->filename;
		xmms_remote_playlist(session, flist, num, TRUE);
		g_free(flist);
		}
#endif
}


/* ---------------------------------------------------------------------- */
static void tpl_dump_playlist(GList *pl)
{
plist_entry *ent;
int i;

	for(i = 1; pl != NULL; pl = g_list_next(pl), i++)
		{
		printf("%d. ", i);
		ent = (plist_entry *)pl->data;
		if(ent->filename) printf("%s ", ent->filename);
		if(ent->title) printf("%s ", ent->title);
		printf("%d\n", ent->length);
		}
}


/* ---------------------------------------------------------------------- */
static void tpl_refresh_playlist(GList *pl)
{
	tpl_send_xmms_playlist(&kj_play_list, xmms_session);
	pl_draw_list(bg_pl, res.areas + PLREGION_TEXT);
}


/* ---------------------------------------------------------------------- */
static int tpl_int_compare_func(const void *a, const void *b)
{
	if((gint) a < (gint) b) return -1;
	if((gint) a > (gint) b) return 1;
	else return 0;
}


/* ---------------------------------------------------------------------- */
static void tpl_dirbrowser_add_ok(GtkWidget *w, GtkFileSelection *fs)
{
GList *slist = NULL, *n;
gchar *text, *fname, *path;

	path = gtk_file_selection_get_filename(fs);
	if((text = strrchr(path, '/'))) *(text + 1) = '\0';

		/* create sorted selection list */
	slist = g_list_copy(GTK_CLIST(fs->file_list)->selection);
	slist = g_list_sort(slist, tpl_int_compare_func);

		/* add selected entries */
	for(n = slist; n != NULL; n = g_list_next(n))
		{
		gtk_clist_get_text(GTK_CLIST(fs->file_list), GPOINTER_TO_INT(n->data), 0, &text);
		fname = g_strconcat(path, text, NULL);
		tpl_add_playlist(tpl_filesel_list, fname, text, -1, -1);
		g_free(fname);
		}

	g_list_free(slist);

	tpl_refresh_playlist(*tpl_filesel_list);
	gtk_widget_destroy(tpl_filesel);
}


/* ---------------------------------------------------------------------- */
static void tpl_dirbrowser_add_playlist(GList **pl)
{
	if(tpl_filesel) return;

	tpl_filesel = gtk_file_selection_new("Add file(s)...");

	gtk_clist_set_selection_mode(GTK_CLIST(GTK_FILE_SELECTION(tpl_filesel)->file_list), GTK_SELECTION_EXTENDED);

	gtk_signal_connect(GTK_OBJECT(tpl_filesel), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &tpl_filesel);
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(tpl_filesel)->ok_button), "clicked", GTK_SIGNAL_FUNC(tpl_dirbrowser_add_ok), tpl_filesel);
	gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(tpl_filesel)->cancel_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(tpl_filesel));
	tpl_filesel_list = pl;

	gtk_widget_show(tpl_filesel);
}


/* ---------------------------------------------------------------------- */
static void tpl_dirbrowser_file_ok(GtkWidget *w, GtkFileSelection *fs)
{
gchar *fname, *p;
struct stat buf;

	fname = g_strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs)));
	for(p = fname+strlen(fname)-1; *p == '/' && p > fname; p--) *p = '\0';

	if(stat(fname, &buf) == 0)
		{
		if(S_ISDIR(buf.st_mode))
			{
			p = g_strdup_printf("%s/", fname);
			gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs), p);
			g_free(p);
			g_free(fname);
			return;
			}
		}

	if(fname && *fname)
		{
		if(tpl_load_list)
			{
			tpl_load_playlist(tpl_filesel_list, fname);
			tpl_refresh_playlist(*tpl_filesel_list);
			}
		else tpl_save_playlist(*tpl_filesel_list, fname);
		}

	gtk_widget_destroy(tpl_filesel);
	g_free(fname);
}


/* ---------------------------------------------------------------------- */
static void tpl_dirbrowser_file_playlist(GList **pl, int load)
{
	if(tpl_filesel) return;
	tpl_load_list = load;

	if(load) tpl_filesel = gtk_file_selection_new("Load playlist");
	else tpl_filesel = gtk_file_selection_new("Save playlist");

	gtk_signal_connect(GTK_OBJECT(tpl_filesel), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &tpl_filesel);
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(tpl_filesel)->ok_button), "clicked", GTK_SIGNAL_FUNC(tpl_dirbrowser_file_ok), tpl_filesel);
	gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(tpl_filesel)->cancel_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(tpl_filesel));
	tpl_filesel_list = pl;

	gtk_widget_show(tpl_filesel);
}


/* ---------------------------------------------------------------------- */
static void pl_draw_list(GdkWindow *w, k_area *pos)
{
gint i, len, yp;
gchar *text, buf[20], buf2[30];
GdkFont *font = NULL;
GList *n;

	gdk_draw_pixmap(w, gc_pl, res.pl_background->pix, pos->x1-1, pos->y1-1, pos->x1-1, pos->y1-1, pos->x2 - pos->x1 + 2, pos->y2 - pos->y1 + 2);
	sprintf(buf2, "-adobe-helvetica-medium-r-*-*-%d-*", res.pl_fontsize);

	if((font = gdk_font_load(buf2)))
		{
		yp = pos->y1 + res.pl_fontyspace;

		for(n = g_list_nth(kj_play_list, pl_list_top), i = pl_list_top; n != NULL; n = g_list_next(n), i++)
			{
			if(((plist_entry *)(n->data))->selected)
				{
				gdk_gc_set_foreground(gc_pl, &res.pl_cursel_colour);
				gdk_draw_rectangle(w, gc_pl, 1, pos->x1, yp-res.pl_fontyspace, pos->x2-pos->x1, res.pl_fontyspace);
				}

			if(i == cur_track) gdk_gc_set_foreground(gc_pl, &res.pl_curtrack_colour);
			else gdk_gc_set_foreground(gc_pl, &res.pl_colour);

			text = g_malloc(strlen(((plist_entry *)(n->data))->title) + 20);
			sprintf(text, "%d. %s", i + 1, ((plist_entry *)(n->data))->title);
			len = strlen(text);
			while(gdk_text_width(font, text, len) > (pos->x2 - pos->x1) && len > 4)
				{
				len--;
				text[len - 3] = '.';
				text[len - 2] = '.';
				text[len - 1] = '.';
				text[len] = '\0';
				}

			gdk_draw_text(w, font, gc_pl, pos->x1, yp, text, len);
			g_free(text);

			yp += res.pl_fontyspace;
			if(yp > pos->y2) break;
			}

		gdk_gc_set_foreground(gc_pl, &res.pl_colour);
		sprintf(buf, "%d", g_list_length(kj_play_list));
		gdk_draw_text(w, font, gc_pl, res.areas[PLREGION_TTLTRACK].x1,
			res.areas[PLREGION_TTLTRACK].y1 + res.pl_fontyspace, buf, strlen(buf));

		sprintf(buf, "%d", cur_track);
		gdk_draw_text(w, font, gc_pl, res.areas[PLREGION_SELTRACK].x1,
			res.areas[PLREGION_SELTRACK].y1 + res.pl_fontyspace, buf, strlen(buf));

		gdk_font_unref(font);
		}

	// gdk_window_clear_area(win_pl->window, pos->x1-1, pos->y1-1, pos->x2 - pos->x1 + 2, pos->y2 - pos->y1 + 2);
	gdk_window_clear(win_pl->window);
	gdk_flush();
}


/* ---------------------------------------------------------------------- */
static void pl_select_list(GdkWindow *w, k_area *pos, int type, int x, int y, guint state)
{
int sel_pos, mode = 0;

	sel_pos = pl_list_top + (y - pos->y1)/res.pl_fontyspace;

	if(state & GDK_SHIFT_MASK) mode = 1;
	else if(state & GDK_CONTROL_MASK) mode = 2;
	tpl_select_nth(kj_play_list, sel_pos, mode);

#if RXMMS
	if(type == GDK_2BUTTON_PRESS)
		{
		xmms_remote_set_playlist_pos(xmms_session, sel_pos);
		xmms_remote_play(xmms_session);
		}
#endif

	pl_draw_list(w, pos);
}


/* ---------------------------------------------------------------------- */
static void pl_draw_button(GdkWindow *w, k_area *button)
{
	if(button->type != A_BUTTON && button->type != A_TOGGLE) return;
	if((button->pressed || button->active) && res.pl_pressed)
		gdk_draw_pixmap(w, gc_pl, res.pl_pressed->pix, button->x1, button->y1, 
			button->x1, button->y1, button->x2 - button->x1, button->y2 - button->y1);
	else
		gdk_draw_pixmap(w, gc_pl, res.pl_background->pix, button->x1, button->y1, 
			button->x1, button->y1, button->x2 - button->x1, button->y2 - button->y1);

	gdk_window_clear_area(win_pl->window, button->x1, button->y1, button->x2 - button->x1, button->y2 - button->y1);
}


/* ---------------------------------------------------------------------- */
static int pl_process_button(int type, int x, int y, guint state, int b, int m)
{
int ret = 1, drlist = 0;
Window xwindow;

	if(!m || type == GDK_2BUTTON_PRESS)
		{
		switch(b)
			{
			case PLBUTTON_CLOSE: gtk_widget_destroy(win_pl); break;
			case PLBUTTON_PREFS: kj_configure(); break;
			case PLBUTTON_ABOUT: kj_about(); break;
			case PLBUTTON_MINIMIZE:
				if(win_pl->window)
					{
					xwindow = GDK_WINDOW_XWINDOW(win_pl->window);
					XIconifyWindow(GDK_DISPLAY(), xwindow, DefaultScreen(GDK_DISPLAY()));
					}
				break;
			case PLBUTTON_LINK: config.lock_plist = (config.lock_plist ? 0 : 1); break;
			case PLBUTTON_SELDOWN: pl_list_top++; drlist = 1; break;
			case PLBUTTON_SELUP: if(--pl_list_top < 0) pl_list_top = 0; drlist = 1; break;
			case PLBUTTON_PGDOWN: pl_list_top += 5; drlist = 1; break;
			case PLBUTTON_PGUP: pl_list_top -= 5; if(pl_list_top < 0) pl_list_top = 0; drlist = 1; break;
			case PLBUTTON_HOME: pl_list_top = 0; drlist = 1; break;
			case PLBUTTON_LOAD: tpl_dirbrowser_file_playlist(&kj_play_list, 1); break;
			case PLBUTTON_SAVE: tpl_dirbrowser_file_playlist(&kj_play_list, 0); break;
			case PLBUTTON_ADD: tpl_dirbrowser_add_playlist(&kj_play_list); break;
			case PLBUTTON_DELETE: tpl_delete_selected(&kj_play_list); break;
			case PLBUTTON_SHUFFLE: printf("SHUFFLE\n"); drlist = 1; break;
			case PLBUTTON_RESET: tpl_playlist_clear(&kj_play_list); drlist = 1; break;
			default: ret = 0;
			}

#if RXMMS
		if(!ret && xmms_running)
			{
			ret = 1;
			switch(b)
				{
				case PLBUTTON_END: pl_list_top = xmms_remote_get_playlist_length(xmms_session) - 5; drlist = 1; break;
				case PLBUTTON_PLAY: xmms_remote_play(xmms_session); break;
				case PLBUTTON_STOP: xmms_remote_stop(xmms_session); break;
				case PLBUTTON_PAUSE: xmms_remote_pause(xmms_session); break;
				case PLBUTTON_PREVIOUS: drlist = 1; xmms_remote_playlist_prev(xmms_session); break;
				case PLBUTTON_NEXT: drlist = 1; xmms_remote_playlist_next(xmms_session); break;
				default: ret = 0;
				}
			}
#endif
		}

	if(m && b == PLREGION_TEXT)
		{
		pl_select_list(bg_pl, res.areas + PLREGION_TEXT, type, x, y, state);
		drlist = 1;
		}

	if(drlist) pl_draw_list(bg_pl, res.areas + PLREGION_TEXT);

	return ret;
}


/* ---------------------------------------------------------------------- */
static void pl_button_press_cb(GtkWidget * w, GdkEventButton * event, gpointer data)
{
int b, mflag = 1;

	pl_button_x = event->x;
	pl_button_y = event->y;

	if(event->button == 5) { pl_list_top++; pl_draw_list(bg_pl, res.areas + PLREGION_TEXT); }
	else if(event->button == 4) { if(--pl_list_top < 0) pl_list_top = 0;
		pl_draw_list(bg_pl, res.areas + PLREGION_TEXT); }

	if(event->button != 1) return;

	for(b = NUM_MAIN_AREAS; b < NUM_PL_AREAS; b++)
		{
		if(inside_area(event->x, event->y, res.areas + b) && (res.areas[b].type != A_TEXT))
			{
			if(pl_process_button(event->type, event->x, event->y, event->state, b, 1))
				{
				mflag = 0;
				res.areas[b].focus = 1;
				res.areas[b].pressed = 1;
				pl_draw_button(bg_pl, &res.areas[b]);
				}
			}
		}

	pl_move = mflag;
}


/* ---------------------------------------------------------------------- */
static void pl_button_release_cb(GtkWidget * w, GdkEventButton * event, gpointer data)
{
int b;

	pl_move = 0;

	if(event->button != 1) return;
	for(b = NUM_MAIN_AREAS; b < NUM_PL_AREAS; b++)
		{
		if(res.areas[b].pressed)
			{
			if(res.areas[b].type == A_TOGGLE)
				res.areas[b].active = (res.areas[b].active ? 0 : 1);
			res.areas[b].pressed = 0;
			pl_draw_button(bg_pl, &res.areas[b]);
			pl_process_button(event->type, event->x, event->y, 0, b, 0);
			}
		res.areas[b].focus = 0;
		}
}


/* ---------------------------------------------------------------------- */
static gint pl_key_press_cb(GtkWidget *widget, GdkEventKey *event)
{
	switch(event->keyval)
		{
		case 'p':
		case 'P': gtk_widget_destroy(win_pl);
			break;

		case GDK_Down: pl_process_button(0, 0, 0, 0, PLBUTTON_SELDOWN, 0); break;
		case GDK_Up: pl_process_button(0, 0, 0, 0, PLBUTTON_SELUP, 0); break;
		case GDK_Page_Up: pl_process_button(0, 0, 0, 0, PLBUTTON_PGUP, 0); break;
		case GDK_Page_Down: pl_process_button(0, 0, 0, 0, PLBUTTON_PGDOWN, 0); break;
		case GDK_Home: pl_process_button(0, 0, 0, 0, PLBUTTON_HOME, 0); break;
		case GDK_End: pl_process_button(0, 0, 0, 0, PLBUTTON_END, 0); break;
		default: return FALSE;
		}

	return TRUE;
}


/* ---------------------------------------------------------------------- */
static void pl_motion_notify_cb(GtkWidget *w, GdkEventMotion *event, gpointer data)
{
int b, inside;
gint nx, ny, dx, dy;
GdkModifierType modmask;

	if(pl_move)
		{
		gdk_window_get_pointer(NULL, &nx, &ny, &modmask);
		nx -= pl_button_x;
		ny -= pl_button_y;
		dx = nx - config.plist_x;
		dy = ny - config.plist_y;
		kj_playlist_move(nx, ny);

		if(config.lock_plist)
			kj_main_move(config.win_x + dx, config.win_y + dy);
		}
	else
		{
		for(b = NUM_MAIN_AREAS; b < NUM_PL_AREAS; b++)
			{
			if(res.areas[b].focus)
				{
				inside = inside_area(event->x, event->y, res.areas + b);
				if(inside) pl_process_button(event->type, event->x, event->y, event->state, b, 1);
				if((inside && !res.areas[b].pressed) || (!inside && res.areas[b].pressed))
					{
					res.areas[b].pressed = inside;
					pl_draw_button(bg_pl, &res.areas[b]);
					}
				}
			}
		}
}


/* ---------------------------------------------------------------------- */
void kj_set_pl(void)
{
gint b;

	if(!win_pl) return;

	if(!res.pl_background)
		{
		gtk_widget_destroy(win_pl);
		return;
		}

	if(bg_pl) gdk_pixmap_unref(bg_pl);
	bg_pl = gdk_pixmap_new(win_pl->window, res.pl_background->width, res.pl_background->height, gdk_visual_get_best_depth());

	if(res.pl_background->mask)
		gtk_widget_shape_combine_mask(win_pl, res.pl_background->mask, 0, 0);
	gdk_window_set_hints(win_pl->window, 0, 0, res.pl_background->width,
		res.pl_background->height, res.pl_background->width, res.pl_background->height, GDK_HINT_MIN_SIZE | GDK_HINT_MAX_SIZE);
	gdk_window_resize(win_pl->window, res.pl_background->width, res.pl_background->height);
	gtk_widget_set_usize(win_pl, res.pl_background->width, res.pl_background->height);	/* both... seems to work :) */

	gdk_draw_pixmap(bg_pl, gc_pl, res.pl_background->pix, 0, 0, 0, 0, res.pl_background->width, res.pl_background->height);
	pl_draw_list(bg_pl, res.areas + PLREGION_TEXT);
	for(b = NUM_MAIN_AREAS; b < NUM_PL_AREAS; b++)
		if(res.areas[b].active) pl_draw_button(bg_pl, res.areas + b);

	gdk_window_set_back_pixmap(win_pl->window, bg_pl, 0);
	gdk_window_clear(win_pl->window);
	gdk_flush();
}


/* ---------------------------------------------------------------------- */
void kj_playlist_move(gint x, gint y)
{
	config.plist_x = x;
	config.plist_y = y;
	if(win_pl) gdk_window_move(win_pl->window, x, y);
}


/* ---------------------------------------------------------------------- */
void kj_playlist_cleanup(void)
{
	if(win_pl) gtk_widget_destroy(win_pl);
	if(bg_pl) gdk_pixmap_unref(bg_pl);
	win_pl = NULL;
	bg_pl = NULL;
}


/* ---------------------------------------------------------------------- */
void kj_update_playlist(void)
{
int drlist = 0, cur_plen;

	if(!win_pl) return;

	if(old_track != cur_track)
		{
		drlist = 1;
		old_track = cur_track;
		}

#if RXMMS
	if(xmms_running)
		{
		if((cur_plen = xmms_remote_get_playlist_length(xmms_session)) != old_pl_len)
			{
			tpl_get_xmms_playlist(&kj_play_list, xmms_session, 1);
			drlist = 1;
			old_pl_len = cur_plen;
			}
		}
#endif

	if(drlist) pl_draw_list(bg_pl, res.areas + PLREGION_TEXT);

}


/* ---------------------------------------------------------------------- */
void kj_playlist(void)
{
#if RXMMS
	if((config.playlist_editor_type == 1 || !res.pl_background) && xmms_running)
		{
		/* CVS --- xmms_remote_pl_win_toggle(xmms_session, !xmms_remote_is_pl_win(xmms_session)); */
		xmms_remote_pl_win_toggle(xmms_session, 1);
		return;
		}
#endif

	if(!res.pl_background) return;

	if(win_pl)
		{
		kj_playlist_cleanup();
		return;
		}

	old_pl_len = old_track = -1;
	tpl_get_xmms_playlist(&kj_play_list, xmms_session, 1);

	pl_move = 0;
	res.areas[PLBUTTON_LINK].active = config.lock_plist;

	win_pl = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_widget_set_app_paintable(win_pl, TRUE);
	gtk_window_set_title(GTK_WINDOW(win_pl), "KJ Playlist");
	gtk_window_set_policy(GTK_WINDOW(win_pl), FALSE, FALSE, TRUE);
	gtk_window_set_wmclass(GTK_WINDOW(win_pl), "KJ Playlist", "KJ");

	gtk_widget_set_app_paintable(win_pl, TRUE);
	gtk_widget_set_events(win_pl, GDK_BUTTON_MOTION_MASK | GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK);
	gtk_signal_connect(GTK_OBJECT(win_pl), "button_press_event", GTK_SIGNAL_FUNC(pl_button_press_cb), NULL);
	gtk_signal_connect(GTK_OBJECT(win_pl), "button_release_event", GTK_SIGNAL_FUNC(pl_button_release_cb), NULL);
	gtk_signal_connect(GTK_OBJECT(win_pl), "key_press_event", GTK_SIGNAL_FUNC(pl_key_press_cb), NULL);
	gtk_signal_connect(GTK_OBJECT(win_pl), "motion_notify_event", GTK_SIGNAL_FUNC(pl_motion_notify_cb), NULL);
	gtk_signal_connect(GTK_OBJECT(win_pl), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &win_pl);

	gtk_widget_realize(win_pl);
	gdk_window_set_decorations(win_pl->window, 0);
	gc_pl = gdk_gc_new(win_pl->window);
	kj_set_pl();

	gtk_widget_show(win_pl);

	if(config.save_plist_pos)
		gdk_window_move(win_pl->window, config.plist_x, config.plist_y);
}

