// -*- c++ -*-

/*
 *
 * Copyright (C) 2002 Richard Moore <rich@kde.org>
 * Copyright (C) 2002 George Staikos <staikos@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include <klocale.h>
#include <qdom.h>
#include <qfile.h>
#include <qtextstream.h>

#include <kdebug.h>
#include <ksavefile.h>

#include "channel.h"
#include "channelstore.h"

#include "channeliozapping.h"

//
// Zapping Format Handler
//
ChannelIOFormatZapping::ChannelIOFormatZapping(Kdetv *ktv, QObject *parent, const char *name)
    : KdetvChannelPlugin(ktv, "Zapping Channels", parent, name)
{
    _fmtName  = "zapping";
    _menuName = i18n("Zapping");
    _flags    = FormatRead;
}

//
// Zapping Loading
//
bool ChannelIOFormatZapping::load( ChannelStore *store, ChannelFileMetaInfo * /*info*/,
                                   QIODevice *file, const QString& /*fmt*/ )
{
    kdDebug() << "IOFormatZapping::load(...)" << endl;

    QDomDocument doc;
    if ( !doc.setContent( file ) )
        return false;

    return readDocument( store, doc );
}

bool ChannelIOFormatZapping::readDocument( ChannelStore *store, const QDomDocument& doc)
{
    kdDebug() << "IOFormatZapping::readDocument(...)" << endl;

    QDomElement root = doc.documentElement();
    if ( root.tagName() != "Configuration" )
        return false;

    kdDebug() << "       Found a zapping conf file" << endl;

    QDomNode zapping = FindSubtree(root, "zapping");
    if (zapping.isNull())
        return false;
    QDomNode options = FindSubtree(zapping, "options");
    if (options.isNull())
        return false;
    QDomNode main = FindSubtree(options, "main");
    if (main.isNull())
        return false;
    QDomNode tuned_channels = FindSubtree(main, "tuned_channels");
    if (tuned_channels.isNull())
        return false;

    kdDebug() << "       Found tuned_channels in the conf file" << endl;

    // Read in the channels
    for (QDomNode n = tuned_channels.firstChild(); !n.isNull() && n.isElement();
         n = n.nextSibling()) {

        root = n.toElement();
        kdDebug() << "     Found " << root.attribute("label") << endl;

        if ( root.tagName() != "subtree" ) {
            kdWarning() << "Found a node which is not a subtree!!! - could not parse file" << endl;
            return false;
        }

        Channel* ch = new Channel( store );
        ch->setEnabled(true);
        ch->setNumber(root.attribute("label").toInt());
        for (QDomNode pn = root.firstChild(); !pn.isNull() && pn.isElement();
             pn = pn.nextSibling()) {
            QDomElement z = pn.toElement();
            if (z.attribute("label") == "name")
                ch->setName(z.text());
            else if (z.attribute("label") == "freq") 
                ch->setChannelProperty("frequency", z.text().toULongLong());
        }
        store->addChannel( ch );
    }
    return true;
}

QDomNode ChannelIOFormatZapping::FindSubtree(const QDomNode& node, const QString& name)
{
    for (QDomNode n = node.firstChild(); !n.isNull() && n.isElement();
         n = n.nextSibling()) {
        QDomElement el = n.toElement();
        if (el.tagName() != "subtree") {
            kdDebug() << "Node has the tagname " << el.tagName() << " not 'subtree'" << endl;
            return n;
        }
        if (el.attribute("label") == name)
            return n;
    }
    return QDomNode();
}

bool ChannelIOFormatZapping::handlesFile( const QString& filename, int rflags) const
{
    kdDebug() << "ChannelIOFormatZapping::handlesFile(...) filename = '" << filename
              << "' rflags = " << rflags << endl;
    if (rflags == FormatRead && filename.endsWith("zapping.conf"))
        return true;
    return false;
}

extern "C" KDETV_EXPORT ChannelIOFormatZapping* create_zappingchannels(Kdetv *ktv)
{
    return new ChannelIOFormatZapping(ktv, 0L, "Zapping Channels Plugin");
}
