/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/* gdm-info-provider-battery.c
 *
 * Copyright (C) 2007 David Zeuthen
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <config.h>
#include <glib/gi18n.h>
#include "gdm-info-provider.h"

static gboolean 
get_provider_matches (GdmDevice *device)
{
        return gdm_device_test_capability (device, "battery");
}

static char *
get_icon_name (GdmDevice *device)
{
        /* TODO: select appropriate battery for UPS, etc. */
        return g_strdup ("gpm-primary-100");
}

static char *
get_name (GdmDevice *device, gboolean is_short)
{
        const char *s;
        char *ret;

        ret = NULL;
        s = gdm_device_get_property_string (device, "battery.type");
        if (s != NULL) {
                if (g_ascii_strcasecmp (s, "primary") == 0) {
                        if (is_short)
                                ret = g_strdup (_("Battery"));
                        else
                                ret = g_strdup (_("Laptop Battery Bay"));
                } else if (g_ascii_strcasecmp (s, "ups") == 0) {
                        if (is_short)
                                ret = g_strdup (_("UPS Battery"));
                        else
                                ret = g_strdup (_("Uninterruptable Power Supply"));
                } else if (g_ascii_strcasecmp (s, "mouse") == 0) {
                        if (is_short)
                                ret = g_strdup (_("Mouse Battery"));
                        else
                                ret = g_strdup (_("Wireless Mouse Battery"));
                } else if (g_ascii_strcasecmp (s, "keyboard") == 0) {
                        if (is_short)
                                ret = g_strdup (_("Keyboard Battery"));
                        else
                                ret = g_strdup (_("Wireless Keyboard Battery"));
                } else if (g_ascii_strcasecmp (s, "keyboard_mouse") == 0) {
                        if (is_short)
                                ret = g_strdup (_("Keyboard/Mouse Battery"));
                        else
                                ret = g_strdup (_("Wireless Keyboard/Mouse Battery"));
                }
        }

        return ret;
}

static char *
get_short_name (GdmDevice *device)
{
        return get_name (device, TRUE);
}

static char *
get_long_name (GdmDevice *device)
{
        return get_name (device, FALSE);
}

static char *
get_vendor (GdmDevice *device)
{
        return g_strdup (gdm_device_get_property_string (device, "battery.vendor"));
}

static char *
get_product (GdmDevice *device)
{
        return g_strdup (gdm_device_get_property_string (device, "battery.model"));
}

#define ADD_SUM(p, key, value)                                                       \
        do {                                                                         \
                if ((value) != NULL) {                                               \
                        p = g_slist_append (p, g_strdup (key));                      \
                        p = g_slist_append (p, (value));                             \
                }                                                                    \
        } while (FALSE)

static GSList *
get_summary (GdmDevice *device)
{
        const char *s;
        GSList *p = NULL;

        ADD_SUM (p, _("Battery Present"), 
                        gdm_device_get_property_bool (device, "battery.present") ? 
                        g_strdup (_("Yes")) : g_strdup(_("No")));
        s = gdm_device_get_property_string (device, "battery.vendor");
        if (s != NULL)
                ADD_SUM (p, _("Battery Manufacturer"), g_strdup (s));
        s = gdm_device_get_property_string (device, "battery.model");
        if (s != NULL)
                ADD_SUM (p, _("Battery Model"), g_strdup (s));
        s = gdm_device_get_property_string (device, "battery.serial");
        if (s != NULL)
                ADD_SUM (p, _("Battery Serial"), g_strdup (s));
        s = gdm_device_get_property_string (device, "battery.technology");
        if (s != NULL) {
                if (g_ascii_strcasecmp (s, "lead-acid") == 0) {
                        ADD_SUM (p, _("Battery Technology"), g_strdup (_("Lead Acid")));
                } else if (g_ascii_strcasecmp (s, "lithium-ion") == 0) {
                        ADD_SUM (p, _("Battery Technology"), g_strdup (_("Lithium-ion (Li-ion)")));
                } else if (g_ascii_strcasecmp (s, "lithium-polymer") == 0) {
                        ADD_SUM (p, _("Battery Technology"), g_strdup (_("Lithium ion polymer (LiPO)")));
                } else if (g_ascii_strcasecmp (s, "nickel-metal-hydride") == 0) {
                        ADD_SUM (p, _("Battery Technology"), g_strdup (_("Nickel metal hydride (NiMH)")));
                }
        }
        
        return p;
}

GdmInfoProviderIface gdm_info_provider_battery =
{
        .get_provider_matches = get_provider_matches,
        .get_icon_name        = get_icon_name,
        .get_short_name       = get_short_name,
        .get_long_name        = get_long_name,
        .get_vendor           = get_vendor,
        .get_product          = get_product,
        .get_summary          = get_summary,
};

