#include <utility>
#include <set>
#include <iostream>
#include <fstream>

#include "Block.hpp"
#include "Transform.hpp"

using namespace std;

class CompressMap
{
	struct ltCon
	{
		// Required sorting order is as follows:
		//
		// (offset, level)
		// (  0,      2  )
		// (  0,      1  )
		// (  7,      3  )
		// (  9,      5  )
		// (  9,      4  )
		//
		bool operator()(const Block *b1, const Block *b2) const
		{
			if (b1->offset < b2->offset)
				return true;
			if (b1->offset > b2->offset)
				return false;
			return (b1->level > b2->level);
		}
	};

	typedef multiset<Block *, ltCon> con_t;
	
	con_t m_map;

	/*
	 * Number of the Blocks in the m_map. Also used for
	 * assigning level of the Block in put().
	 */
	unsigned int m_size;

	/*
	 * Maximal length of the Block that has ever been recorded.
	 * Used in get() to optimize search algorithm.
	 */
	size_t m_length;

	Transform *m_transform;

	int store(char *buf, size_t len) const;

	int restore(size_t count, const char *buf, size_t len);

	off_t get_length(off_t			 offset,
	                 con_t::const_iterator	 it) const;

	off_t get_simple(off_t			  offset,
	                 con_t::const_iterator	  oit,
	                 Block			**bl) const;

	con_t::const_iterator get_search_right(off_t                 offset,
	                                       unsigned int          level,
	                                       con_t::const_iterator it) const;

	void print() const;
	
	void prev(con_t::const_iterator &it) const;
	
	void next(con_t::const_iterator &it) const;

public:
	/**
	 * @param transform - Pointer to object of type class Transformation.
	 *                    It will be used to compress index before
	 *                    storing it to the disk or to decompress index
	 *                    after reading it from the disk.
	 */
	CompressMap(Transform *transform);
	~CompressMap();

	/*
	 * Object persistion, store object to file.
	 * 
	 * File format of the index on the disk is as follows:
	 * 	Block  - Block describing compressed buffer with index
	 * 	char[] - Buffer with compressed index
	 */
	off_t store(int fd, off_t to) const;

	/*
	 * Object persistion, restore object from file.
	 */
	off_t restore(int fd, off_t from);

	/*
	 * Get top-layer Block starting on the offset
	 * in uncompressed file.
	 * 
	 * Return:
	 * 	Length of the Block that can be read
	 * 	from that Block. If zero is returned,
	 * 	no Block on that offset was found and
	 * 	pointer to the Block on the nearest bigger
	 * 	offset is returned if it exists or
	 * 	NULL is returned if no such Block exists.
	 */
	off_t get(off_t offset, Block **bl) const;

	/*
	 * Remember new Block.
	 */
	void put(Block *bl);

	/*
	 */
	void truncate(off_t length);

	void set(Transform *transform);
};

