/*
 * Copyright (c) 2008-2009 Internet Initiative Japan Inc. All rights reserved.
 *
 * The terms and conditions of the accompanying program
 * shall be provided separately by Internet Initiative Japan Inc.
 * Any use, reproduction or distribution of the program are permitted
 * provided that you agree to be bound to such terms and conditions.
 *
 * $Id: dkim.h 865 2009-03-31 07:03:38Z takahiko $
 */

#ifndef __DKIM_H__
#define __DKIM_H__

#include <sys/types.h>
#include <stdbool.h>
#include <openssl/evp.h>

#include "dnsresolv.h"
#include "inetmailbox.h"
#include "mailheaders.h"

#ifdef  __cplusplus
extern "C" {
#endif

// Enumerations
typedef enum _dkim_score {
    DKIM_SCORE_NULL = 0,
    DKIM_SCORE_NONE,
    DKIM_SCORE_PASS,
    DKIM_SCORE_FAIL,
    DKIM_SCORE_POLICY,
    DKIM_SCORE_NEUTRAL,
    DKIM_SCORE_TEMPERROR,
    DKIM_SCORE_PERMERROR,
    DKIM_SCORE_MAX, // the number of DKIM_SCORE_* constants
} dkim_score_t;

typedef enum _dkim_adsp_score {
    DKIM_ADSP_SCORE_NULL = 0,
    DKIM_ADSP_SCORE_NONE,
    DKIM_ADSP_SCORE_PASS,
    DKIM_ADSP_SCORE_UNKNOWN,
    DKIM_ADSP_SCORE_FAIL,
    DKIM_ADSP_SCORE_DISCARD,
    DKIM_ADSP_SCORE_NXDOMAIN,
    DKIM_ADSP_SCORE_TEMPERROR,
    DKIM_ADSP_SCORE_PERMERROR,
    DKIM_ADSP_SCORE_MAX,    // the number of DKIM_ADSP_SCORE_* constants
} dkim_adsp_score_t;

#define DSTAT_CATMASK	0xff00
#define DSTATCAT_OK	0x0000
#define DSTATCAT_INFO	0x0100
#define DSTATCAT_VERIFIED	0x0200
#define DSTATCAT_SYSERR	0x0300
#define DSTATCAT_TMPERR	0x0400
#define DSTATCAT_PERMFAIL	0x0500
#define DSTATCAT_CFGERR	0x0600
#define DSTATCAT_WARN	0x0700

#define DSTAT_ISOK(e)	(((e) & DSTAT_CATMASK) == DSTATCAT_OK)
#define DSTAT_ISINFO(e)	(((e) & DSTAT_CATMASK) == DSTATCAT_INFO)
#define DSTAT_ISVERIFIED(e)	(((e) & DSTAT_CATMASK) == DSTATCAT_VERIFIED)
#define DSTAT_ISSYSERR(e)	(((e) & DSTAT_CATMASK) == DSTATCAT_SYSERR)
#define DSTAT_ISTMPERR(e)	(((e) & DSTAT_CATMASK) == DSTATCAT_TMPERR)
#define DSTAT_ISPERMFAIL(e)	(((e) & DSTAT_CATMASK) == DSTATCAT_PERMFAIL)
#define DSTAT_ISCFGERR(e)	(((e) & DSTAT_CATMASK) == DSTATCAT_CFGERR)
#define DSTAT_ISWARN(e)	(((e) & DSTAT_CATMASK) == DSTATCAT_WARN)
#define DSTAT_ISCRITERR(e)	(DSTAT_ISSYSERR(e) || DSTAT_ISCFGERR(e))

// Status Codes
typedef enum _dkim_stat {
    DSTAT_OK = DSTATCAT_OK,
    DSTAT_INFO_DIGEST_MATCH = DSTATCAT_INFO,    // digest value has matched
    DSTAT_INFO_ADSP_NOT_EXIST,  // ADSP record have not found
    DSTAT_INFO_ADSP_NXDOMAIN,   // domain part of "Author" does not exist (NXDOMAIN)
    DSTAT_VERIFIED_NO_SIGNHEADER = DSTATCAT_VERIFIED,   // No DKIM-Signature headers have found
    DSTAT_VERIFIED_AUTHOR,  // Signature is verified as Author Signature
    DSTAT_VERIFIED_THIRDPARTY,  // Signature is verified as Third Party Signature
    // [System Errors]
    DSTAT_SYSERR_DIGEST_UPDATE_FAILURE = DSTATCAT_SYSERR,   // errors on digest update (returned by OpenSSL EVP_DigestUpdate())
    DSTAT_SYSERR_DIGEST_VERIFICATION_FAILURE,   // errors on digital signature verification (returned by OpenSSL EVP_VerifyFinal())
    DSTAT_SYSERR_IMPLERROR, // obvious implementation error
    DSTAT_SYSERR_NORESOURCE,    // memory allocation error
    // [Temporary Errors]
    DSTAT_TMPERR_DNS_LOOKUP_FAILURE = DSTATCAT_TMPERR,  // DNS look up error
    // [DKIM signature verification/generation failures]
    // tag-value object errors
    DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION = DSTATCAT_PERMFAIL,    // tag-value syntax violation
    DSTAT_PERMFAIL_MISSING_REQUIRED_TAG,    // required tag is missing
    DSTAT_PERMFAIL_TAG_DUPLICATED,  // multiple identical tags are found
    DSTAT_PERMFAIL_UNSUPPORTED_PUBKEYALG,   // unsupported public key algorithm
    // verification errors
    DSTAT_PERMFAIL_HEADERDIGEST_MISMATCH,   // digest value of message header is not matched
    DSTAT_PERMFAIL_BODYDIGEST_MISMATCH, // digest value of message body is not matched
    DSTAT_PERMFAIL_AUTHOR_AMBIGUOUS,    // Author header can't be extracted
    DSTAT_PERMFAIL_AUTHOR_NOT_SIGNED,   // Author header is not signed
    // Signature errors
    DSTAT_PERMFAIL_SIGNATURE_INCOMPATIBLE_VERSION,  // unsupported signature version
    DSTAT_PERMFAIL_DOMAIN_MISMATCH, // domains are not matched between sig-i-tag and sig-d-tag
    DSTAT_PERMFAIL_FROM_FIELD_NOT_SIGNED,   // "From:" header header is not signed
    DSTAT_PERMFAIL_SIGNATURE_EXPIRED,   // DKIM-Signature has expired
    DSTAT_PERMFAIL_TIMESTAMP_DISCREPANCY,   // sig-t-tag has later timestamp than sig-x-tag
    DSTAT_PERMFAIL_UNSUPPORTED_CANONALG,    // unsupported canonicalization algorithm
    DSTAT_PERMFAIL_UNSUPPORTED_KEYRETR, // unsupported method to retrieve the public key
    DSTAT_PERMFAIL_UNSUPPORTED_DIGESTALG,   // unsupported digest algorithm
    // Public key errors
    DSTAT_PERMFAIL_PUBKEY_NOT_EXIST,    // Public key record does not exist
    DSTAT_PERMFAIL_PUBKEY_REVOKED,  // Public key record has revoked
    DSTAT_PERMFAIL_PUBKEY_TYPE_MISMATCH,    // key-k-tag and the content of public key (key-p-tag) does not matched
    DSTAT_PERMFAIL_UNSUPPORTED_PUBKEY_VERSION,  // unsupported public key version
    DSTAT_PERMFAIL_UNSUPPORTED_SERVICETYPE, //  unsupported service type (key-s-tag)
    DSTAT_PERMFAIL_PUBKEY_BROKEN,   // Public key is broken (returned by OpenSSL d2i_PUBKEY())
    DSTAT_PERMFAIL_PUBKEY_SRVTYPE_MISMATCH, // service type dose not allow the public key record to be applied to email
    DSTAT_PERMFAIL_PUBKEY_DIGESTALG_MISMATCH,   // digest algorithm of the public key record (key-h-tag) does not match the one of the signature (sig-a-tag-h)
    DSTAT_PERMFAIL_PUBKEY_PUBKEYALG_MISMATCH,   // public key algorithm of the public key record (key-k-tag) does not match the one of the signature (sig-a-tag-k)
    DSTAT_PERMFAIL_PUBKEY_SUBDOMAIN_PROHIBITED, // public key record does not accept subdomain
    DSTAT_PERMFAIL_PUBKEY_GRANULARITY_MISMATCH, // the local-part of "i=" tag of the signature (sig-i-tag) does not match the granularity of the public key record (key-g-tag)
    // ADSP errors
    DSTAT_PERMFAIL_MULTIPLE_ADSP_RECORD,    // multiple ADSP records are found
    // [Misconfigurations]
    DSTAT_CFGERR_SYNTAX_VIOLATION = DSTATCAT_CFGERR,    // syntax error at configuration directives
    DSTAT_CFGERR_EMPTY_VALUE,   // empty value or NULL is specified for configuration
    DSTAT_CFGERR_UNDEFINED_KEYWORD, // undefined keyword is specified for configuration
    // [Warnings]
    DSTAT_WARN_CANONDUMP_OPEN_FAILURE = DSTATCAT_WARN,  // failed to open files to debug
    DSTAT_WARN_CANONDUMP_UPDATE_FAILURE,    // an error on dumping canonicalized text data
} dkim_stat_t;

// type declarations
typedef struct DkimPolicy DkimPolicy;
typedef struct DkimVerifyPolicy DkimVerifyPolicy;
typedef struct DkimVerifySession DkimVerifySession;
typedef struct DkimSignPolicy DkimSignPolicy;
typedef struct DkimSignSession DkimSignSession;

// DkimPolicy
extern void DkimPolicy_setLogger(DkimPolicy *self,
                                 void (*logger) (int priority, const char *message, ...));
extern void DkimPolicy_setSendmail813(DkimPolicy *self, bool flag);
extern dkim_stat_t DkimPolicy_setAuthorPriority(DkimPolicy *self, const char *record,
                                                const char *delim);

// DkimVerifyPolicy
extern DkimVerifyPolicy *DkimVerifyPolicy_new(void);
extern void DkimVerifyPolicy_free(DkimVerifyPolicy *self);
extern void DkimVerifyPolicy_setSignHeaderLimit(DkimVerifyPolicy *self, size_t header_limit);
extern void DkimVerifyPolicy_acceptExpiredSignature(DkimVerifyPolicy *self, bool accept);
#define DkimVerifyPolicy_setLogger(self, logger) DkimPolicy_setLogger((DkimPolicy *)(self), logger)
#define DkimVerifyPolicy_setSendmail813(self, flag) DkimPolicy_setSendmail813((DkimPolicy *)(self), flag)
#define DkimVerifyPolicy_setAuthorPriority(self, record, delim) DkimPolicy_setAuthorPriority((DkimPolicy *)(self), record, delim)

// DkimVerifySession
extern DkimVerifySession *DkimVerifySession_new(const DkimVerifyPolicy *vpolicy,
                                                DnsResolver *resolver);
extern void DkimVerifySession_free(DkimVerifySession *self);
extern dkim_stat_t DkimVerifySession_setHeaders(DkimVerifySession *self,
                                                const MailHeaders *headers);
extern dkim_stat_t DkimVerifySession_startBody(DkimVerifySession *self);
extern dkim_stat_t DkimVerifySession_updateBody(DkimVerifySession *self,
                                                const unsigned char *bodyp, size_t len);
extern dkim_stat_t DkimVerifySession_finishBody(DkimVerifySession *self);
extern dkim_stat_t DkimVerifySession_setCanonDump(DkimVerifySession *self, const char *basedir,
                                                  const char *prefix);
extern const char *DkimVerifySession_getAuthorHeaderName(const DkimVerifySession *self);
extern const InetMailbox *DkimVerifySession_getAuthorMailbox(const DkimVerifySession *self);

extern size_t DkimVerifySession_getVerificationNum(const DkimVerifySession *self);
extern dkim_score_t DkimVerifySession_getVerifyFrameResult(const DkimVerifySession *self,
                                                           size_t signo,
                                                           const InetMailbox **mailbox);
extern dkim_adsp_score_t DkimVerifySession_evalAdsp(DkimVerifySession *self);

// DkimSignPolicy
extern DkimSignPolicy *DkimSignPolicy_new(void);
extern void DkimSignPolicy_free(DkimSignPolicy *self);
extern dkim_stat_t DkimSignPolicy_setCanonAlgorithm(DkimSignPolicy *self,
                                                    const char *headercanon, const char *bodycanon);
extern dkim_stat_t DkimSignPolicy_setDigestAlgorithm(DkimSignPolicy *self, const char *dgstalg);
extern dkim_stat_t DkimSignPolicy_setPublicKeyAlgorithm(DkimSignPolicy *self,
                                                        const char *pubkeyalg);
extern void DkimSignPolicy_setSignatureTTL(DkimSignPolicy *self, long long expiration);
#define DkimSignPolicy_setLogger(self, logger) DkimPolicy_setLogger((DkimPolicy *)(self), logger)
#define DkimSignPolicy_setSendmail813(self, flag) DkimPolicy_setSendmail813((DkimPolicy *)(self), flag)
#define DkimSignPolicy_setAuthorPriority(self, record, delim) DkimPolicy_setAuthorPriority((DkimPolicy *)(self), record, delim)

// DkimSignSession
extern DkimSignSession *DkimSignSession_new(const DkimSignPolicy *spolicy);
extern void DkimSignSession_free(DkimSignSession *self);
extern dkim_stat_t DkimSignSession_setHeaders(DkimSignSession *self, const MailHeaders *headers);
extern dkim_stat_t DkimSignSession_startBody(DkimSignSession *self);
extern dkim_stat_t DkimSignSession_updateBody(DkimSignSession *self, const unsigned char *bodyp,
                                              size_t len);
extern dkim_stat_t DkimSignSession_finishBody(DkimSignSession *self, const char *selector,
                                              EVP_PKEY *pkey, const char **headerf,
                                              const char **headerv);
extern const char *DkimSignSession_getAuthorHeaderName(const DkimSignSession *self);
extern const InetMailbox *DkimSignSession_getAuthorMailbox(const DkimSignSession *self);
extern dkim_stat_t DkimSignSession_setCanonDump(DkimSignSession *self, const char *basedir,
                                                const char *prefix);

extern const char *DKIM_strerror(dkim_stat_t code);
extern const char *DkimEnum_lookupScoreByValue(dkim_score_t val);
extern dkim_score_t DkimEnum_lookupScoreByName(const char *keyword);
extern dkim_score_t DkimEnum_lookupScoreByNameSlice(const char *head, const char *tail);

extern const char *DkimEnum_lookupAdspScoreByValue(dkim_adsp_score_t val);
extern dkim_adsp_score_t DkimEnum_lookupAdspScoreByName(const char *keyword);
extern dkim_adsp_score_t DkimEnum_lookupAdspScoreByNameSlice(const char *head, const char *tail);

#ifdef __cplusplus
}
#endif

#endif /* __DKIM_H__ */
