/*
Copyright (c) 2010 by Drake Justice <djustice.kde@gmail.com>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.
*/

#ifndef DISCBURNER_H
#define DISCBURNER_H

#include <QGraphicsGridLayout>
#include <QGraphicsLinearLayout>
#include <QGraphicsSceneHoverEvent>
#include <QMenu>
#include <QStandardItemModel>
#include <QPropertyAnimation>
#include <QTimer>
#include <QUrl>

#include <Plasma/Applet>
#include <Plasma/Animation>
#include <Plasma/BusyWidget>
#include <Plasma/ComboBox>
#include <Plasma/DataEngine>
#include <Plasma/ExtenderItem>
#include <Plasma/IconWidget>
#include <Plasma/Label>
#include <Plasma/Meter>
#include <Plasma/PopupApplet>
#include <Plasma/PushButton>
#include <Plasma/Separator>
#include <Plasma/Svg>
#include <Plasma/ToolTipContent>
#include <Plasma/TreeView>

#include <Solid/OpticalDisc>

#include <kaction.h>

#include "projectmodel.h"
#include "ui_discburnerConfig.h"

class DiscBurner : public Plasma::PopupApplet
{
    Q_OBJECT

    public:
        DiscBurner(QObject *parent, const QVariantList &args);
        ~DiscBurner();

        void init();
        void paintInterface(QPainter *painter, const QStyleOptionGraphicsItem *option, const QRect &contents);
        void constraintsEvent(Plasma::Constraints constraints);

        enum InteractionState {
            Off = 0,
            Waiting = 1,
            Hovered = 2,
            Rejected = 3,
            DraggedOver = 5
        };

        enum ActionState {
            Unset = 0,
            Idle = 1,
            IdleError = 2,
            IdleSuccess = 4,
            Burning = 8,
            Formatting = 16,
            Reading = 32,
            Converting = 64
        };

    protected slots:
        void hoverEnterEvent(QGraphicsSceneHoverEvent *event);
        void hoverLeaveEvent(QGraphicsSceneHoverEvent *event);
        void configAccepted();
        void readColors();

    protected:
        void dragEnterEvent(QGraphicsSceneDragDropEvent *event);
        void dragLeaveEvent(QGraphicsSceneDragDropEvent *event);
        void dragMoveEvent(QGraphicsSceneDragDropEvent *event);
        void dropEvent(QGraphicsSceneDragDropEvent *event);
        void createConfigurationInterface(KConfigDialog *parent);
        void setEmbedded(const bool embedded);

    private slots:
        void updateApplet();
        void resetActionState();

        void on_createButton_clicked();
        void on_burnButton_clicked();
        void on_backupButton_clicked();
        void on_copyButton_clicked();
        void on_ripButton_clicked();
        void on_formatButton_clicked();
        void on_ejectButton_clicked();
        void on_cancelButton_clicked();

        void on_newAudioCDImage_clicked();
        void on_newMP3CDImage_clicked();
        void on_newVideoCDImage_clicked();
        void on_newVideoDVDImage_clicked();
        void on_newDataCDImage_clicked();
        void on_newDataDVDImage_clicked();
        void on_newMixedModeImage_clicked();

        void setBusy(bool busy);
        void setJobMaximum(int max);
        void setJobProgress(int progress);
        void setJobStatus(QString status);
        void jobCompleted();
        void updateDrives(const QString& udi = "udi"); // *
        void burnCurrentAudioProject();
        void burnCurrentDiscCopy();
        void burnCurrentMP3CDProject();
        void closeOpenProject();
        void ripDisc(const QString& filetype);
        void ripDiscToMp3();
        void ripDiscToOgg();
        void ripDiscToWav();

    signals:
        void cancelProcesses();
        void blankDiscInserted();

    private:
        void initExtenderItem(Plasma::ExtenderItem *item);
        void paintOverlayLabel(QPainter *p, const QRect &contentsRect, const QString& labelText);
        void showDiscBurner(bool show);
        void showOverlayLabel(bool show);
        void setupConnections();
        void burnIso(const QString& path);
        void updateCurrentDisc();
        void addSong(const QString& path);
        void addFile(const QString& path);
        void addFolder(const QString& path);
        void addFolderItem(const QString& path, QStandardItem* item);
        void parseDroppedPath(const QUrl& path, QGraphicsSceneDragDropEvent *event);
        void parseDroppedFolder(const QString& path);
        void setInteractionState(InteractionState state);
        void setActionState(ActionState state);

        QStringList currentProjectFileList();

        bool m_isEmbedded;
        bool m_showMultipleDiscBurners;
        bool m_showOverlayLabel;
        bool m_hasAudioProjectLoaded;
        bool m_hasMP3ProjectLoaded;
        bool m_hasMixedModeProjectLoaded;
        bool m_hasVideoCDProjectLoaded;
        bool m_hasVideoDVDProjectLoaded;
        bool m_hasDataCDProjectLoaded;
        bool m_hasDataDVDProjectLoaded;
        bool m_firstRun;
        bool m_isBusy;
        bool m_driveHasDisc;
        bool m_cancelVisible;
        bool m_ejectVisible;
        bool m_wasCancelled;
        bool m_isHovered;
        bool m_stopDropping;

        int m_jobMaximum;
        int m_jobProgress;
        int m_trackDuplicateCount;
        int m_currentProgress;

        double m_currentProjectSize;

        QGraphicsLinearLayout *m_extenderLayout;
        QGraphicsGridLayout *m_actionsLayout;
        QGraphicsGridLayout *m_discLabelLayout;
        QGraphicsGridLayout *m_appletLayout;

        Plasma::Animation *m_overlayLabelAnimation;
        Plasma::Animation *m_ejectButtonAnimation;

        Plasma::BusyWidget *m_busyWidget;

        Plasma::ComboBox *m_driveCombo;

        Plasma::IconWidget *m_currentDiscLabel;
        Plasma::IconWidget *m_createButton;
        Plasma::IconWidget *m_burnButton;
        Plasma::IconWidget *m_backupButton;
        Plasma::IconWidget *m_copyButton;
        Plasma::IconWidget *m_ripButton;
        Plasma::IconWidget *m_formatButton;
        Plasma::IconWidget *m_ejectButton;
        Plasma::IconWidget *m_cancelButton;
        Plasma::IconWidget *m_closeProjectButton;

        Plasma::Label *m_overlayLabel;

        Plasma::Separator *m_actionsLayoutSeparator;

        Plasma::Svg *m_discIcon;

        Plasma::TreeView *m_tree;

        Plasma::Meter *m_meter;
        Plasma::Meter *m_projectMeter;

        Plasma::ToolTipContent *m_toolTipData;

        DiscBurner *m_extenderApplet;

        Solid::OpticalDisc::ContentTypes m_currentDiscContentType;

        ActionState m_actionState;
        InteractionState m_interactionState;

        QMenu *m_newAudioImageMenu;
        QMenu *m_newVideoImageMenu;
        QMenu *m_newImageMenu;
        QMenu *m_ripToMenu;

        KAction *m_newAudioCDImageAction;
        KAction *m_newMP3CDImageAction;
        KAction *m_newDataCDImageAction;
        KAction *m_newVideoCDImageAction;
        KAction *m_newDataDVDImageAction;
        KAction *m_newVideoDVDImageAction;
        KAction *m_newMixedModeImageAction;
        KAction *m_ripToMp3Action;
        KAction *m_ripToOggAction;
        KAction *m_ripToWavAction;

        QString m_currentDriveUdi;
        QString m_currentDriveDeviceString;
        QString m_currentDiscUdi;
        QString m_currentDiscLabelString;
        QString m_currentDiscUsage;
        QString m_currentDiscCapacity;
        QString m_currentDiscContentTypeString;
        QString m_overlayText;
        QString m_currentJobStatus;

        QHash<QString, QString> m_devicesHash;

        ProjectModel *m_projectModel;

        QUrl m_firstDroppedUrl; // fix me. eradicate via dropEvent later.
	QList<QUrl> m_droppedUrls;

        QColor m_textColor;

        Ui::discburnerConfig ui;
};

K_EXPORT_PLASMA_APPLET(discburner, DiscBurner)

#endif
