/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <QtTest/QtTest>

#include <QDir>

#include <Decibel/DBusNames>

#include "protocolmanager.h"
#include "protocolmanagertest.h"

namespace
{
    const QString manager_extension(".manager");

    const QString application_name_postfix("_TESTSUITE");
}

void ProtocolManagerTest::init()
{
    QCoreApplication::instance()->setOrganizationName(Decibel::organisation_name);
    QCoreApplication::instance()->setApplicationName(Decibel::system_name +
                                                     application_name_postfix);

    m_oldManagerPath = getenv("TELEPATHY_DATA_PATH");
    setenv("TELEPATHY_DATA_PATH", SOURCEDIR "/tests/server", 1);
}

void ProtocolManagerTest::cleanup()
{
    unsetenv("TELEPATHY_DATA_PATH");
    setenv("TELEPATHY_DATA_PATH", m_oldManagerPath.toLocal8Bit().constData(), 1);
}

void ProtocolManagerTest::allConnectionManagersTest()
{
    ProtocolManager protocolMgr;
    QStringList files = getFiles();
    QVERIFY(files.size() > 0);
    QStringList cms = protocolMgr.allConnectionManagers();
    // telepathy always searches for connection managers in the
    // standard places (in addition to the dirs given  in TELEPATHY_DATA_PATH)
    //
    // So allConnectionManagers will return more then just the test CMs
    // we have in our testdir.
    QVERIFY(cms.size() >= files.size());
    QString file;
    foreach(file, files)
    {
        QString name = file.left(file.indexOf('.'));
        QVERIFY(cms.indexOf(name) >= 0);
    }
}

void ProtocolManagerTest::supportedProtocols()
{
    ProtocolManager protocolMgr;

    QStringList protocols_found = getProtocolsFromFiles();
    QVERIFY(protocols_found.size() > 0);

    // get list from protocolmanager:
    QStringList pm_protocols = protocolMgr.supportedProtocols();

    // Compare the two:
    // see comment on getAllConnectionManagers above!
    QVERIFY(pm_protocols.size() >= protocols_found.size());
    QString protocol;
    foreach(protocol, protocols_found)
    { QVERIFY(pm_protocols.indexOf(protocol) >= 0); }
}

void ProtocolManagerTest::connectionManagersForProtocol()
{
    ProtocolManager protocolMgr;
    QStringList known_protocols = getProtocolsFromFiles();
    QVERIFY(known_protocols.size() > 0);

    QString protocol;
    foreach (protocol, known_protocols)
    {
        QStringList file_cms = getManagerFilesForProtocol(protocol);
        QStringList pm_cms = protocolMgr.connectionManagersForProtocol(protocol);

        // see comment on getAllConnectionManagers above!
        QVERIFY(pm_cms.size() >= file_cms.size());
        QString current_cm;
        foreach (current_cm, file_cms)
        {
            QString name = current_cm.left(current_cm.indexOf('.'));
            QVERIFY(pm_cms.indexOf(name) >= 0);
        }
    }
}

void ProtocolManagerTest::defaultConnectionManager()
{
    ProtocolManager protocolMgr;

    QString protocol = "testproto";
    QStringList file_cms = getManagerFilesForProtocol(protocol);
    QString default_cm = protocolMgr.defaultConnectionManagerForProtocol(protocol);
    QVERIFY(file_cms.indexOf(default_cm + manager_extension) >= 0);
}

void ProtocolManagerTest::setDefaultConnectionManager()
{
    ProtocolManager protocolMgr;

    QStringList known_protocols = getProtocolsFromFiles();
    QVERIFY(known_protocols.size() > 0);

    QString protocol = "testproto";
    QStringList file_cms = getManagerFilesForProtocol(protocol);
    QString default_cm = protocolMgr.defaultConnectionManagerForProtocol(protocol);
    int current_idx = file_cms.indexOf(default_cm + manager_extension);
    QVERIFY(current_idx >= 0);

    int other_idx = -1;
    if (file_cms.size() > 1)
    {
        // find other index:
        for (int i = 0; i < file_cms.size(); ++i)
        {
            if (i != current_idx)
            {
                other_idx = i;
                break;
            }
        }
    }
    else
    { other_idx = current_idx; }

    // Set default to other index:
    QString name = file_cms[other_idx];
    name = name.left(name.indexOf('.'));
    protocolMgr.setDefaultConnectionManagerForProtocol(protocol, name);

    // Do we have a new default?
    default_cm = protocolMgr.defaultConnectionManagerForProtocol(protocol);
    QVERIFY(file_cms.indexOf(default_cm + manager_extension) == other_idx);
}

void ProtocolManagerTest::setInvalidDefault()
{
    ProtocolManager protocolMgr;

    QStringList known_protocols = getProtocolsFromFiles();
    QVERIFY(known_protocols.size() > 0);

    QString old_default = protocolMgr.defaultConnectionManagerForProtocol(known_protocols[0]);

    // set invalid default:
    protocolMgr.setDefaultConnectionManagerForProtocol(known_protocols[0], "XXX-invalid-XXX");

    QVERIFY(protocolMgr.defaultConnectionManagerForProtocol(known_protocols[0]) ==
            old_default);
}

void ProtocolManagerTest::simplePersistenceTest()
{
    QString protocol = "testproto";
    QString default_cm;
    {
        ProtocolManager protocolMgr;

        // Precondition: We have more than one CM for our protocol:
        QStringList file_cms = getManagerFilesForProtocol(protocol);
        QVERIFY(file_cms.size() > 1);

        // Get current default:
        default_cm = protocolMgr.defaultConnectionManagerForProtocol(protocol);

        // find other CM:
        int current_idx = file_cms.indexOf(default_cm + manager_extension);
        QVERIFY(current_idx >= 0);

        int other_idx = -1;
        if (file_cms.size() > 1)
        {
            // find other index:
            for (int i = 0; i < file_cms.size(); ++i)
            {
                if (i != current_idx)
                {
                    other_idx = i;
                    break;
                }
            }
        }

        // Change default:
        QString name = file_cms[other_idx];
        name = name.left(name.indexOf('.'));
        protocolMgr.setDefaultConnectionManagerForProtocol(protocol, name);
    }
    {
        ProtocolManager protocolMgr;

        QString new_default = protocolMgr.defaultConnectionManagerForProtocol(protocol);
        QVERIFY(new_default != default_cm);
    }
}

// ---------------------- Helper methods ---------------------- //

QDir ProtocolManagerTest::getDir() const
{
    QString dirname(getenv("TELEPATHY_DATA_PATH"));
    return QDir(dirname + "/managers");
}

QStringList ProtocolManagerTest::getFiles() const
{
    QString dirname(getenv("TELEPATHY_DATA_PATH"));
    return getDir().entryList(QStringList('*' + manager_extension), QDir::Files);
}

QStringList ProtocolManagerTest::getProtocolsFromFiles() const
{
    QStringList protocols_found;

    // find all protocols in test manager files:
    QString file;
    foreach(file, getFiles())
    {
        QSettings settings(getDir().absoluteFilePath(file), QSettings::IniFormat);
        QStringList groups = settings.childGroups();
        QString group;
        foreach(group, groups)
        {
            if (!group.startsWith("Protocol ")) { continue; }
            group = group.mid(9);
            if (protocols_found.indexOf(group) < 0)
            { protocols_found.append(group); }
        }
    }
    return protocols_found;
}

QStringList
ProtocolManagerTest::getManagerFilesForProtocol(const QString proto) const
{
    QStringList cms_found;

    // find all protocols in test manager files:
    QString file;
    foreach(file, getFiles())
    {
        QSettings settings(getDir().absoluteFilePath(file), QSettings::IniFormat);
        QStringList groups = settings.childGroups();
        QString group;
        foreach(group, groups)
        {
            if (!group.startsWith("Protocol ")) { continue; }
            group = group.mid(9);
            if (proto == group)
            { cms_found.append(file); }
        }
    }
    return cms_found;
}

QTEST_MAIN(ProtocolManagerTest)
#include "protocolmanagertest.moc"
