from datetime import datetime
import pkg_resources
pkg_resources.require("SQLObject>=0.8,<=0.10.0")
from turbogears.database import PackageHub
# import some basic SQLObject classes for declaring the data model
# (see http://www.sqlobject.org/SQLObject.html#declaring-the-class)
from sqlobject import SQLObject, SQLObjectNotFound, RelatedJoin, MultipleJoin
# import some datatypes for table columns from SQLObject
# (see http://www.sqlobject.org/SQLObject.html#column-types for more)
from sqlobject.col import StringCol, UnicodeCol, IntCol, DateTimeCol, BoolCol, EnumCol, ForeignKey
from turbogears import identity

__connection__ = hub = PackageHub('luci')


# your data model

# the identity model

class Visit(SQLObject):
    """
    A visit to your site
    """
    class sqlmeta:
        table = 'visit'

    visit_key = StringCol(length=40, alternateID=True,
                          alternateMethodName='by_visit_key')
    created = DateTimeCol(default=datetime.now)
    expiry = DateTimeCol()

    def lookup_visit(cls, visit_key):
        try:
            return cls.by_visit_key(visit_key)
        except SQLObjectNotFound:
            return None
    lookup_visit = classmethod(lookup_visit)


class VisitIdentity(SQLObject):
    """
    A Visit that is link to a User object
    """
    visit_key = StringCol(length=40, alternateID=True,
                          alternateMethodName='by_visit_key')
    user_id = IntCol()


class Group(SQLObject):
    """
    An ultra-simple group definition.
    """
    # names like "Group", "Order" and "User" are reserved words in SQL
    # so we set the name to something safe for SQL
    class sqlmeta:
        table = 'tg_group'

    group_name = UnicodeCol(length=16, alternateID=True,
                            alternateMethodName='by_group_name')
    display_name = UnicodeCol(length=255)
    created = DateTimeCol(default=datetime.now)

    # collection of all users belonging to this group
    users = RelatedJoin('User', intermediateTable='user_group',
                        joinColumn='group_id', otherColumn='user_id')

    # collection of all permissions for this group
    permissions = RelatedJoin('Permission', joinColumn='group_id',
                              intermediateTable='group_permission',
                              otherColumn='permission_id')


class User(SQLObject):
    """
    Reasonably basic User definition.
    Probably would want additional attributes.
    """
    # names like "Group", "Order" and "User" are reserved words in SQL
    # so we set the name to something safe for SQL
    class sqlmeta:
        table = 'tg_user'

    user_name = UnicodeCol(length=16, alternateID=True,
                           alternateMethodName='by_user_name')
    email_address = UnicodeCol(length=255, alternateID=True,
                               alternateMethodName='by_email_address')
    display_name = UnicodeCol(length=255)
    password = UnicodeCol(length=40)
    created = DateTimeCol(default=datetime.now)

    last_login = DateTimeCol(title = "Date and time of last login", alternateID = False, unique = False, notNone = False)
    id = MultipleJoin("user_prefs", joinColumn = 'user_id')
    login_count = IntCol(title = "Number of times the user has logged in", dbName = "login_count", alternateID = False, default = 0, unique = False, notNone = True)

    # groups this user belongs to
    groups = RelatedJoin('Group', intermediateTable='user_group',
                         joinColumn='user_id', otherColumn='group_id')

    def _get_permissions(self):
        perms = set()
        for g in self.groups:
            perms |= set(g.permissions)
        return perms

    def _set_password(self, cleartext_password):
        """Runs cleartext_password through the hash algorithm before saving."""
        password_hash = identity.encrypt_password(cleartext_password)
        self._SO_set_password(password_hash)

    def set_password_raw(self, password):
        """Saves the password as-is to the database."""
        self._SO_set_password(password)


class Permission(SQLObject):
    """
    A relationship that determines what each Group can do
    """
    permission_name = UnicodeCol(length=16, alternateID=True,
                                 alternateMethodName='by_permission_name')
    description = UnicodeCol(length=255)

    groups = RelatedJoin('Group',
                         intermediateTable='group_permission',
                         joinColumn='permission_id',
                         otherColumn='group_id')

class system(SQLObject):
    class sqlmeta:
        table = 'system'
        idName = 'system'
    id = MultipleJoin("task", joinColumn = 'system_id')
    authenticated = BoolCol(title = "authenticated?", dbName = "authenticated", default = "False")
    name = UnicodeCol(title = "System name", dbName = "name", alternateID = True, unique = True, notNone = True)
    addr = UnicodeCol(title = "System's network address", dbName = "addr", alternateID = True, unique = True, notNone = True)
    port = IntCol(title = "Network port ricci listens on", dbName = "port", alternateID = False, default = 11111, unique = False, notNone = True)
    cluster = ForeignKey("cluster", dbName = "cluster")

class cluster(SQLObject):
    class sqlmeta:
        table = 'cluster'
        idName = 'cluster'
    id = MultipleJoin("system", joinColumn = 'cluster_id')
    version = EnumCol(enumValues = ['4', '5'], title = "Cluster version", dbName = "version")
    name = UnicodeCol(title = "Cluster name", dbName = "name", alternateID = True, unique = True, notNone = True)
    alias = UnicodeCol(title = "Cluster alias", dbName = "alias", alternateID = True, unique = False, notNone = True)
    conf = UnicodeCol(title = "Last seen cluster.conf", dbName = "conf", alternateID = False, unique = False, notNone = False)

class task(SQLObject):
    class sqlmeta:
        table = 'task'
        idName = 'task'
    type = IntCol(title = "Task type", dbName = "type", alternateID = True, unique = False, notNone = True)
    batch_id = IntCol(title = "Task's ricci batch id", dbName = "batch_id", alternateID = True, unique = False, notNone = True)
    batch_status = IntCol(title = "Ricci batch status", dbName = "batch_status", alternateID = False, unique = False, notNone = False)
    status = EnumCol(enumValues = ['0', '1', '2'], title = "Task Status (0 = success, 1 = fail, 2 = in progress)", dbName = "status", default = "2")
    start_date = DateTimeCol(title = "Date and time the task started", dbName = "start_date", alternateID = False, unique = False, notNone = True)
    end_date = DateTimeCol(title = "Date and time the task completed", dbName = "end_date", alternateID = False, unique = False, notNone = False)
    description = UnicodeCol(title = "Description of task", dbName = "description", alternateID = False, unique = False, notNone = False)
    status_msg = UnicodeCol(title = "Status message, if any", alternateID = False, unique = False, notNone = False)
    system = ForeignKey("system", dbName = "system")
    initiator = ForeignKey("user", dbName = "User")

class site_prefs(SQLObject):
    class sqlmeta:
        table = 'site_prefs'
        idName = 'site_prefs'
    debug = BoolCol(title = "debug mode?", dbName = "debug", default = "False")

class user_prefs(SQLObject):
    class sqlmeta:
        table = 'user_prefs'
        idName = 'user_prefs'
    user = ForeignKey("user", dbName = "User")
	# user-specific preferences

class cluster_perm(Permission):
    class sqlmeta:
        table = 'cluster_perm'
        idName = 'cluster_perm'
    access = EnumCol(enumValues = ['0', '1', '2'], title = "access type (0 = none, 1 = view only, 2 = full)", dbName = "access", default = "0")
    user = ForeignKey("user", dbName = "User")
    cluster = ForeignKey("cluster", dbName = "cluster")

class system_perm(Permission):
    class sqlmeta:
        table = 'system_perm'
        idName = 'system_perm'
    access = EnumCol(enumValues = ['0', '1', '2'], title = "access type (0 = none, 1 = view only, 2 = full)", dbName = "access", default = "0")
    user = ForeignKey("user", dbName = "User")
    system = ForeignKey("system", dbName = "system")
