/************************************

cardinal - a proper cross platthis twitter client
Copyright (C) 2009  Chris Fuenty <zimmy@zimmy.co.uk>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc.,
51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

************************************/

#include "tweet.h"

tweet::tweet(QWidget *parent, tweet_t twInfo, QString user)
    :QWidget(parent),
    m_ignored(false),
    m_userName(user),
    m_favorite(twInfo.favorite)
{
    if (objectName().isEmpty())
        setObjectName(QString::fromUtf8("StatusView"));

    QSettings settings("Cardinal", "Cardinal");
    if (objectName().isEmpty())
        setObjectName(QString::fromUtf8("this"));
    setSizePolicy(QSizePolicy(QSizePolicy::Preferred, QSizePolicy::Preferred));
    gridLayout = new QGridLayout(this);
    gridLayout->setSpacing(4);
    gridLayout->setContentsMargins(5, 5, 5, 5);
    gridLayout->setObjectName(QString::fromUtf8("gridLayout"));
    userPic = new QLabel(this);
    userPic->setObjectName(QString::fromUtf8("userPic"));
    QSizePolicy sizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
    sizePolicy.setHorizontalStretch(0);
    sizePolicy.setVerticalStretch(0);
    sizePolicy.setHeightForWidth(userPic->sizePolicy().hasHeightForWidth());
    userPic->setSizePolicy(sizePolicy);
    userPic->setMinimumSize(QSize(48, 48));
    userPic->setMaximumSize(QSize(48, 48));
    userPic->setPixmap(QPixmap(QString::fromUtf8(":/Icons/16px/Reload")));
    userPic->setScaledContents(true);

    gridLayout->addWidget(userPic, 0, 1, 1, 1);

    tweetTxt = new QLabel(this);
    tweetTxt->setObjectName(QString::fromUtf8("tweetTxt"));
    QSizePolicy sizePolicy1(QSizePolicy::Preferred, QSizePolicy::Preferred);
    sizePolicy1.setHorizontalStretch(0);
    sizePolicy1.setVerticalStretch(0);
    sizePolicy1.setHeightForWidth(tweetTxt->sizePolicy().hasHeightForWidth());
    tweetTxt->setSizePolicy(sizePolicy1);
    tweetTxt->setScaledContents(false);
    tweetTxt->setWordWrap(true);
    tweetTxt->setOpenExternalLinks(true);
    tweetTxt->setAlignment(Qt::AlignTop|Qt::AlignLeft);

    gridLayout->addWidget(tweetTxt, 0, 2, 1, 2);

    byUser = new QLabel(this);
    byUser->setObjectName(QString::fromUtf8("byUser"));
    QSizePolicy sizePolicy2(QSizePolicy::Preferred, QSizePolicy::Fixed);
    sizePolicy2.setHorizontalStretch(0);
    sizePolicy2.setVerticalStretch(0);
    sizePolicy2.setHeightForWidth(byUser->sizePolicy().hasHeightForWidth());
    byUser->setSizePolicy(sizePolicy2);
    indentSpacer = new QSpacerItem(2, 12, QSizePolicy::Fixed, QSizePolicy::Minimum);

    gridLayout->addWidget(byUser, 3, 2, 1, 1);

    itype = 0;

    menu = new QMenu(this);
    menu->setObjectName(QString::fromUtf8("PopupMenu"));

    reply = new QAction(this);
    reply->setObjectName(QString::fromUtf8("Reply"));
    reply->setText(tr("Reply"));
    reply->setIcon(QIcon(QString(":/Icons/16px/Status-Reply")));
    menu->addAction(reply);
    connect(reply, SIGNAL(triggered()), this, SLOT(replyTriggered()));

    getInfo = new QAction(this);
    getInfo->setObjectName(QString::fromUtf8("GetInfo"));
    getInfo->setText(tr("Get Info On %1").arg(twInfo.screenName));
    getInfo->setIcon(QIcon(QString(":/Icons/16px/Get-Info")));
    connect(getInfo, SIGNAL(triggered()), this, SLOT(getInfoTriggered()));

    action_Delete = new QAction(this);
    action_Retweet = new QAction(this);
    action_MarkAsFavorite = new QAction(this);
    action_SendDM = new QAction(this);

    action_Delete->setVisible(false);

    if(user == twInfo.screenName)
        action_Delete->setVisible(true);

    if(m_favorite == true)
        action_MarkAsFavorite->setText(tr("Remove from Favorites"));
    else
        action_MarkAsFavorite->setText(tr("Mark as Favorite"));

    action_Delete->setText(tr("Delete this Status Update"));
    action_Retweet->setText(tr("Retweet this Update"));
    action_SendDM->setText(tr("Send a Direct Message to %1").arg(twInfo.screenName));
    connect(action_Delete, SIGNAL(triggered()), this, SLOT(deleteUpdateSlot()));
    connect(action_Retweet, SIGNAL(triggered()), this, SLOT(retweet()));
    connect(action_MarkAsFavorite, SIGNAL(triggered()), this, SLOT(markAsFavorite()));
    connect(action_SendDM, SIGNAL(triggered()), this, SLOT(sendDM()));
    menu->addAction(action_MarkAsFavorite);
    menu->addAction(action_Delete);
    menu->addAction(action_Retweet);

    retweeters = new QAction(this);
    retweeters->setText(tr("Show Retweeters..."));
    connect(retweeters, SIGNAL(triggered()), this, SLOT(showRetweeters()));
    menu->addAction(retweeters);
    retweeters->setEnabled(false);

    menu->addSeparator();
    menu->addAction(action_SendDM);
    menu->addAction(getInfo);
    
    QStringList lst = twInfo.tweet.split(" ");
    for(int i = 0; i < lst.size(); i++)
    {
        if(lst.at(i).startsWith("http") || lst.at(i).startsWith("https"))
        {
            lst.replace(i, tr("<a href=\"") + lst.at(i) + tr("\">") + lst.at(i) + tr("</a>"));
        }
    }

    tweetTxt->setText(lst.join(" "));
    connect(tweetTxt, SIGNAL(linkActivated(QString)), this, SLOT(linkActivated(QString)));

    QString dt = QDateTime::fromTime_t(twInfo.time).toString("MMM d h:mm ap");
    byUser->setText(tr("From: ") + twInfo.screenName + tr(" at ") + dt);
    tw = twInfo;

    /* set some colours */
    active = settings.value("activeColor").toString();
    inactive = settings.value("inactiveColor").toString();
    QPalette palette = this->palette();

    if(inactive != NULL)
        palette.setColor(QPalette::Active, QPalette::Text, QColor(inactive));

    setPalette(palette);

    /* hide the user picture if requested */
    if(settings.value("showPic").toBool() == true)
    {
        userPic->setVisible(false);
    }

    /* change the text sizes if desired */
    if(settings.value("textSize").toInt() != 0)
    {
        QFont tFont = this->font();
        tFont.setPointSize(settings.value("textSize").toInt());
        this->setFont(tFont);
    }
}

tweet::~tweet()
{
    gridLayout->removeItem(indentSpacer);
    delete indentSpacer;
}

void tweet::sendDM()
{
    emit requestDM(tw.uid);
}

void tweet::contextMenuEvent(QContextMenuEvent *event)
{
    menu->exec(event->globalPos());
}

void tweet::changeEvent(QEvent *event)
{
    if(event->type() == QEvent::WindowActivate)
        emit activated(tw);

	if(event->type() == QEvent::FocusIn)
		itemActivated();
}

void tweet::itemActivated()
{
        QPalette palette = this->palette();

	if(active != NULL)
                palette.setColor(QPalette::Active, QPalette::Text, QColor(active));

        this->setPalette(palette);
}

void tweet::itemDeactivated()
{
        QPalette palette = this->palette();

	if(inactive != NULL)
               palette.setColor(QPalette::Active, QPalette::Text, QColor(inactive));

        this->setPalette(palette);
}

void tweet::replyTriggered()
{
    emit requestReply(tw);
}

void tweet::prefixDirect()
{
    itype = 1;
}

void tweet::prefixMention()
{
    itype = 3;
}

void tweet::prefixThread()
{
    itype = 4;
}

void tweet::getInfoTriggered()
{
    emit requestUserInfo(this->tw.uid);
}

void tweet::setAltColor()
{
    //tweetTxt->setBackgroundRole(QPalette::AlternateBase);
}

void tweet::prefixFavorite(bool favorite)
{
    QPixmap map(img.copy(QRect(0, 0, 48, 48)));

    if(favorite == true)
    {
        QPixmap reply(":/Icons/16px/Favorite", "png");
        QPainter painter(&map);
        painter.drawPixmap(QRect(28, 28, 20, 20), reply);
        m_favorite = true;
        action_MarkAsFavorite->setText(tr("Remove from Favorites"));
    }

    userPic->setPixmap(map);

    if(favorite == false)
    {
        action_MarkAsFavorite->setText(tr("Mark as Favorite"));
        m_favorite = false;
    }
}

void tweet::setProfileImage(QByteArray data)
{
    if(data.size() == 0)
        return;

    try {
        img.loadFromData(data);
        img = img.scaled(48, 48);

        if(m_favorite == true)
        {
            QPixmap map(img.copy(QRect(0, 0, 48, 48)));
            QPixmap reply(":/Icons/16px/Favorite", "png");
            QPainter painter(&map);
            painter.drawPixmap(QRect(28, 28, 20, 20), reply);
            userPic->setPixmap(map);
            return;
        }

        if(itype == 1)
        {
            QPixmap map(img.copy(QRect(0, 0, 48, 48)));
            QPainter painter(&map);
            painter.setFont(QFont("Arial", 12, 6, false));
            painter.drawText(QRect(28, 28, 20, 20), "D", QTextOption(Qt::AlignLeft));
            userPic->setPixmap(map);
        }
        else if(itype == 3)
        {
            QPixmap map(img.copy(QRect(0, 0, 48, 48)));
            QPixmap reply(":/Icons/16px/New-Status", "png");
            QPainter painter(&map);
            painter.setFont(QFont("Arial", 12, 6, false));
            painter.drawPixmap(QRect(28, 28, 20, 20), reply);
            userPic->setPixmap(map);
            }
        else if(itype == 4)
        {
            QPainter painter(&img);
            painter.fillRect(0, 0, 48, 48, QBrush(QColor(255, 0, 0, 1)));
            userPic->setPixmap(img);
        }
        else
        {
            userPic->setPixmap(img);
        }
    }
    catch (int e)
    {
        Q_UNUSED(e);
    }
}

void tweet::timeout()
{
/*	for(double i = 1.0; i <= 0.0; i - 0.05)
	{
		setWindowOpacity(i);
	}
	emit closing(this);
	this->hide();
*/
}

void tweet::linkActivated(QString url)
{
    QDesktopServices::openUrl(QUrl(url));
}

void tweet::mouseDoubleClickEvent(QMouseEvent *event)
{
    Q_UNUSED(event);
    emit requestReply(tw);
}

tweet_t tweet::returnTweet()
{
    return tw;
}

void tweet::ignore()
{
    m_ignored = true;
}

void tweet::unignore()
{
    m_ignored = false;
}

void tweet::retweet()
{
    emit retweetUpdate(tw.id);
}

void tweet::setRetweetedByMe(bool retweeted)
{
    m_retweeted = retweeted;
    action_Retweet->setEnabled(false);
    QStringList byUserList = byUser->text().split("\n");
    if(m_retweetCount > 0)
        byUser->setText(byUserList[0] + QString("\nRetweeted by You and %1 other%2").arg(m_retweetCount).arg((m_retweetCount == 1 ? "" : "s")));
    else
        byUser->setText(byUserList[0] + QString("\nRetweeted by You"));
}

void tweet::setRetweeters(QList<user_t> users)
{
    m_retweetCount = users.count();
    m_retweeters = users;
    retweeters->setEnabled(true);
    QStringList byUserList = byUser->text().split("\n");

    if(m_retweeted == true)
        byUser->setText(byUserList[0] + QString("\nRetweeted by You and %1 other%2").arg(m_retweetCount).arg((m_retweetCount == 1 ? "" : "s")));
    else
        byUser->setText(byUserList[0] + QString("\nRetweeted by %1 %2").arg(m_retweetCount).arg((m_retweetCount == 1 ? "person" : "people")));
}

void tweet::showRetweeters()
{
    uView = new UserView(m_retweeters, 0);
    connect(uView, SIGNAL(closed()), this , SLOT(uViewClosed()));
    connect(uView, SIGNAL(getUserInfo(unsigned long long int)), 
        this, SIGNAL(requestUserInfo(unsigned long long int)));
    uView->setWindowTitle("Retweeters");
    QString msg(tweetTxt->text());
    msg.truncate(20);
    uView->setInformativeText(QString("Retweeters of a Status Update from %1").arg(tw.screenName));
    uView->show();
}

void tweet::uViewClosed()
{
    delete uView;
}

void tweet::indent()
{
    /* cardinal will only indent 5 levels deep */
    if(m_indentCount < 6)
    {
        m_indentCount++;
        delete indentSpacer;
/* windows has issues with knowing what pixels are, go figure */
//#ifndef _WIN32
        indentSpacer = new QSpacerItem((m_indentCount*8), 12, QSizePolicy::Fixed, QSizePolicy::Minimum);
//#else
//        indentSpacer = new QSpacerItem((m_indentCount*2), 12, QSizePolicy::Fixed, QSizePolicy::Minimum);
//#endif
        gridLayout->addItem(indentSpacer, 0, 0, 1, 1);
    }
}

void tweet::deleteUpdateSlot()
{
    QMessageBox msg(QMessageBox::Question, tr("Delete this Status Update?"),
                    tr("Are you sure you want to delete this status update?"),
                    QMessageBox::Yes|QMessageBox::No, this);
    msg.setInformativeText("This process can not be reversed.");
    int result = msg.exec();
    if(result == QMessageBox::Yes)
        emit deleteUpdate(tw.id);
}

void tweet::markAsFavorite()
{
    if(m_favorite == false)
    {
        emit markFavorite(tw.id, true);
        action_MarkAsFavorite->setText(tr("Remove from Favorites"));
        tw.categories.append(tr("Favorites"));
        prefixFavorite(true);
        m_favorite = true;
    }
    else
    {
        emit markFavorite(tw.id, false);
        action_MarkAsFavorite->setText(tr("Mark as Favorite"));
        tw.categories.removeOne(tr("Favorites"));
        prefixFavorite(false);
        m_favorite = false;
    }
}
