/************************************

cardinal - a proper cross platform twitter client
Copyright (C) 2009  Chris Fuenty <zimmy@zimmy.co.uk>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc.,
51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

************************************/

#include "guimain.h"

Cardinal::Cardinal(QWidget *parent, Qt::WFlags flags)
    : QMainWindow(parent, flags)
{
    hideInTray = false;
    tweetMe = false;
    if (objectName().isEmpty())
        setObjectName(QString::fromUtf8("CardinalClass"));
    resize(361, 481);

    QSettings settings("Cardinal", "Cardinal");

    int result = checkForIdentity();
    if(result == 0)
    {
        m_notify = NULL;
        delete this;
        qApp->exit(0);
        return;
    }

    QString frun = settings.value("firstRun").toString();
    if(frun != "true")
    {
        QMessageBox msgBox;
        msgBox.setIcon(QMessageBox::Information);
        msgBox.setText("Cardinal is PRE RELEASE software.");
        msgBox.setInformativeText("Features are likely to be broken at this point or not implemented period.  If you find any bugs, or have a feature request, please use the ""File a Bug Report"" Help Menu Item.");
        msgBox.exec();
        settings.setValue("firstRun", "true");
    }

    setWindowIcon(QIcon(QString::fromUtf8(":/Cardinal/mainico")));
    setWindowTitle(tr("Cardinal"));

    accounts = new TabWidget(this);
    connect(accounts, SIGNAL(tabChanged(int)), this, SLOT(tabChanged(int)));

    iconLabel = new QLabel(this);
    sbWorkingLabel = new QLabel(this);
    sbWorkingLabel->setFixedSize(20, 16);
    iconLabel->setFixedSize(20, 16);
    iconLabel->setPixmap(QPixmap(":/Cardinal/offline"));

    settings.beginGroup("Accounts");
    QStringList accts = settings.childGroups();
    /* handle no accounts here */
    for (int i = 0; i < accts.count(); ++i)
    {
        settings.beginGroup(accts[i]);
        QString user = settings.value("username").toString();
        QString pass = settings.value("password").toString();
        accounts->addAccount(user, pass);
        settings.endGroup();
    }

    setupActions();
    setCentralWidget(accounts);

    connect(accounts, SIGNAL(setOnline()), this, SLOT(setOnline()));
    connect(accounts, SIGNAL(reply(QString, tweet_t)), this, SLOT(reply(QString, tweet_t)));
    connect(accounts, SIGNAL(requestDM(QString, unsigned long long int)), this, SLOT(directMsg(QString, unsigned long long int)));
    connect(accounts, SIGNAL(hideEditor()), this, SLOT(hideEditor()));
    connect(accounts, SIGNAL(working(bool)), this, SLOT(working(bool)));
    connect(accounts, SIGNAL(updateStatusBar(QString)), this, SLOT(updateStatusBar(QString)));
    connect(accounts, SIGNAL(updateStatusBar(QString)), this, SLOT(updateStatusBar(QString)));
    connect(accounts, SIGNAL(selectedChanged()), this, SLOT(selectedChanged()));

    switchViewMenu();
    
    m_search = new QLineEdit(this);

    m_search->setToolTip(tr("Search"));
    connect(m_search, SIGNAL(returnPressed()), this, SLOT(search()));
    mainToolBar = new QToolBar(tr("Standard"), this);
    mainToolBar->setObjectName(QString::fromUtf8("mainToolBar"));
    addToolBar(Qt::TopToolBarArea, mainToolBar);
    statusBar = new QStatusBar(this);
    statusBar->setObjectName(QString::fromUtf8("statusBar"));
    setStatusBar(statusBar);

    sbInfoLabel = new QLabel(statusBar);
    editor = new Editor(this);
    editor->setVisible(false);

    iconLabel->setParent(statusBar);
    sbWorkingLabel->setParent(statusBar);
    statusBar->addWidget(sbInfoLabel, true);
    statusBar->addWidget(iconLabel, false);
    statusBar->addWidget(sbWorkingLabel);

    mainToolBar->addAction(action_NewTweet);
    mainToolBar->addAction(action_Reply);
    mainToolBar->addAction(action_Reload);
    mainToolBar->addAction(action_GetInfo);
    mainToolBar->addWidget(m_search);
    mainToolBar->setFloatable(false);
    mainToolBar->setMovable(false);

    movie = new QMovie(QString::fromUtf8(":/Cardinal/loading"));

    /* System Tray Icon */
    ticon = new QSystemTrayIcon(this);
    ticon->setIcon(QIcon(":/Cardinal/mainico"));
    ticon->setContextMenu(sysTray);
    connect(ticon, SIGNAL(activated(QSystemTrayIcon::ActivationReason)), this, SLOT(trayActivated(QSystemTrayIcon::ActivationReason)));
    ticon->setToolTip(tr("Cardinal"));
    ticon->show();

    /* Notification Daemon */
    #ifdef USE_SNARL
    m_notify = new NotificationAgent(ticon, winId());
    #else
    m_notify = new NotificationAgent(ticon);
    #endif

    /* Idenitity Initilazation */
    connect(editor, SIGNAL(submitTweet(QString, QString, unsigned long long int)), accounts, SLOT(submitStatus(QString, QString, unsigned long long int)));
    connect(editor, SIGNAL(submitDirectMsg(QString, QString, unsigned long long int)), accounts, SLOT(submitDirectMsg(QString, QString, unsigned long long int)));
    qApp->setQuitOnLastWindowClosed(false);

    /* Windows SuperTweet */
    #ifdef _WIN32
    st = new SuperTweet();
    connect(st, SIGNAL(hotKeyRequested()), this, SLOT(showNewTweet()));
    #endif

    selectedChanged();
}

Cardinal::~Cardinal()
{
    if(m_notify != NULL)
        delete m_notify;
}

void Cardinal::tabChanged(int index)
{
    Q_UNUSED(index);

    selectedChanged();
    updateStatusBar(accounts->statusBarText());
    if(accounts->isOnline())
        setOnline();
    else
        setOffline();

    switchViewMenu();
}

void Cardinal::setupActions()
{
    /* actions and menus are setup here, NOT TOOLBARS */
    action_Quit = new QAction(this);
    action_About = new QAction(this);
    action_NewTweet = new QAction(this);
    action_Preferences = new QAction(this);
    action_Reload = new QAction(this);
    actionHelp = new QAction(this);
    actionBugs = new QAction(this);
    actionWebsite = new QAction(this);
    showWindow = new QAction(this);
    action_GetInfo = new QAction(this);
    action_Reply = new QAction(this);
    sysTray = new QMenu(this);
    menuBar = new QMenuBar(this);
    menu_File = new QMenu(menuBar);
    menu_Help = new QMenu(menuBar);

    connect(showWindow, SIGNAL(triggered()), this, SLOT(show()));
    connect(action_Quit, SIGNAL(triggered()), this, SLOT(quitApp()));
    connect(action_NewTweet, SIGNAL(triggered()), this, SLOT(showNewTweet()));
    connect(action_Preferences, SIGNAL(triggered()), this, SLOT(showPrefs()));
    connect(action_About, SIGNAL(triggered()), this, SLOT(about()));
    connect(action_Reload, SIGNAL(triggered()), accounts, SLOT(reload()));
    connect(actionHelp, SIGNAL(triggered()), this, SLOT(help()));
    connect(actionBugs, SIGNAL(triggered()), this, SLOT(bugs()));
    connect(actionWebsite, SIGNAL(triggered()), this, SLOT(website()));
    connect(action_GetInfo, SIGNAL(triggered()), accounts, SLOT(requestInfo()));
    connect(action_Reply, SIGNAL(triggered()), accounts, SLOT(signalReply()));

    menuBar->setGeometry(QRect(0, 0, 361, 22));

    #ifdef MACOS
    menu_File->setTitle("File");
    #else
    menu_File->setTitle("Cardinal");
    #endif
    menu_Help->setTitle(tr("Help"));
    menu_Help->setObjectName("HelpMenu");
    showWindow->setText(tr("Show Cardinal"));
    action_Quit->setText(tr("Quit"));
    actionHelp->setText(tr("Help"));
    actionWebsite->setText(tr("Cardinal on the Web"));
    actionBugs->setText(tr("File a Bug Report"));
    action_About->setText(tr("About Cardnial"));
    action_NewTweet->setText(tr("Post Status..."));
    action_NewTweet->setIcon(QIcon(QString(":/Icons/16px/New-Status")));
    actionHelp->setShortcut(QApplication::translate("Cardinal", "F1", 0, QApplication::UnicodeUTF8));
    action_NewTweet->setShortcut(QApplication::translate("Cardinal", "Ctrl+T", 0, QApplication::UnicodeUTF8));
    action_Preferences->setText(tr("Preferences..."));

    action_GetInfo->setText(tr("Get Info On"));
    action_GetInfo->setIcon(QIcon(QString(":/Icons/16px/Get-Info")));
    action_Reply->setText(tr("Reply"));
    action_Reply->setShortcut(QKeySequence::fromString("Ctrl+R"));
    action_Reply->setIcon(QIcon(QString(":/Icons/16px/Status-Reply")));
    action_Reload->setText(tr("Reload"));
    action_Reload->setShortcut(QApplication::translate("Cardinal", "F5", 0, QApplication::UnicodeUTF8));
    action_Reload->setIcon(QIcon(QString(":/Icons/16px/Reload")));
    menuBar->addAction(menu_File->menuAction());
    menuBar->addAction(menu_Help->menuAction());
    menu_File->addAction(action_NewTweet);
    menu_File->addAction(action_Reply);
    menu_File->addSeparator();
    menu_File->addAction(action_GetInfo);
    menu_File->addAction(action_Reload);
    menu_File->addAction(action_Preferences);
    menu_File->addSeparator();
    menu_File->addAction(action_Quit);
    menu_Help->addAction(actionHelp);
    menu_Help->addAction(actionBugs);
    menu_Help->addAction(actionWebsite);
    menu_Help->addSeparator();
    menu_Help->addAction(action_About);

    sysTray->addAction(action_NewTweet);
    sysTray->addSeparator();
    sysTray->addAction(showWindow);
    sysTray->addAction(action_Quit);

    setMenuBar(menuBar);
}

void Cardinal::about()
{
    AboutDialog *about = new AboutDialog();
	if(about->exec()) {}
	delete about;
}

int Cardinal::checkForIdentity()
{
    QSettings settings("Cardinal", "Cardinal");
    settings.beginGroup("Accounts");
    QStringList accts = settings.childGroups();
    if(accts.count() == 0)
    {
        PreferencesWindow *prefs = new PreferencesWindow(this);
        if(prefs->exec()) {}
        delete prefs;

        if(settings.childGroups().count() == 0)
        {
            return 0;
        }
    }

    return 1;
}

void Cardinal::settingsChanged()
{
    /* a signal to state settings were changed? */
}

void Cardinal::trayActivated(QSystemTrayIcon::ActivationReason reason)
{
    if (reason == QSystemTrayIcon::DoubleClick)
        show();
}

void Cardinal::showNewTweet()
{
    editor->clear();
    editor->show();
}

void Cardinal::reply(QString account, tweet_t twInfo)
{
    editor->clear();
    editor->setReplyInformation(account, QString("@%1").arg(twInfo.screenName), twInfo.id);
    editor->show();
}

void Cardinal::directMsg(QString account, unsigned long long int tid)
{
    editor->clear();
    editor->setDirectMsgInformation(account, tid);
    editor->show();

}

void Cardinal::updateStatusBar(QString text)
{
    sbInfoLabel->setText(text);
}

void Cardinal::showPrefs()
{
    PreferencesWindow *prefs = new PreferencesWindow(this);
    connect(prefs, SIGNAL(accepted()), this, SLOT(settingsChanged()));
    connect(prefs, SIGNAL(accountAdded(QString)), this, SLOT(addAccount(QString)));
    connect(prefs, SIGNAL(accountRemoved(QString)), this, SLOT(removeAccount(QString)));

    if(prefs->exec()) {}

    delete prefs;
}

void Cardinal::quitApp()
{
    QMessageBox msg(QMessageBox::Question, tr("Quit Cardinal?"), tr("Are you sure you want to quit Cardinal?"), QMessageBox::Yes | QMessageBox::No, this);
    int result = msg.exec();
    if(result == QMessageBox::Yes) {
        //ident->Logout();
        qApp->exit(0);
    }
}

void Cardinal::bugs()
{
    QDesktopServices::openUrl(QUrl("http://devel.zimmy.co.uk/"));
}

void Cardinal::help()
{
    QDesktopServices::openUrl(QUrl("http://cardinal.zimmy.co.uk/help.shtml"));
}

void Cardinal::website()
{
    QDesktopServices::openUrl(QUrl("http://cardinal.zimmy.co.uk/"));
}

void Cardinal::changeEvent(QEvent *evt)
{
    Q_UNUSED(evt);
}

void Cardinal::setOnline()
{
    iconLabel->setPixmap(QPixmap(QString::fromUtf8(":/Cardinal/online")));
    action_NewTweet->setEnabled(true);
    action_Reload->setEnabled(true);
}

void Cardinal::working(bool working)
{
	if(working == true) {
		sbWorkingLabel->setMovie(movie);
		movie->start();
	} else {
		QPixmap pixmap(16, 16);
		pixmap.fill(Qt::transparent);
		sbWorkingLabel->setPixmap(pixmap);
	}
}

void Cardinal::setOffline()
{
    iconLabel->setPixmap(QPixmap(QString::fromUtf8(":/Cardinal/offline")));
    //tabWidget->replyAction()->setEnabled(false);
    action_NewTweet->setEnabled(false);
    action_Reload->setEnabled(false);
}

void Cardinal::showNotification()
{
    m_notify->show();
}

void Cardinal::hideEditor()
{
    editor->hide();
}

void Cardinal::selectedChanged()
{
    if(!accounts->selectedUser().isNull())
    {
        action_GetInfo->setText(tr("Get Info On %1").arg(accounts->selectedUser()));
        action_GetInfo->setDisabled(false);
        action_Reply->setDisabled(false);
    }
    else
    {
        action_GetInfo->setText(tr("Get Info"));
        action_GetInfo->setDisabled(true);
        action_Reply->setDisabled(true);
    }
}

void Cardinal::replyRequested()
{
    return;
}

void Cardinal::switchViewMenu()
{
    foreach(QAction *action, menuBar->actions())
    {
        QString objName = action->objectName();
        if(objName == QString("ViewMenu"))
            menuBar->removeAction(action);
    }

    if(accounts->hasAccounts())
        menuBar->insertMenu(menu_Help->menuAction(), accounts->currentTableView()->viewMenu());
}

void Cardinal::search()
{
    accounts->addSearch(m_search->text());
}

void Cardinal::removeAccount(QString account)
{
    accounts->removeAccount(account);
}

void Cardinal::addAccount(QString account)
{
    accounts->addAccount(account, "");
}
