#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"

#include "ppport.h"

#include <stdio.h>
#include <stdlib.h>

#include <apr_pools.h>

#include <rast/config.h>
#include <rast/rast.h>
#include <rast/db.h>

#include <dlfcn.h>


struct xs_rast_t {
    apr_pool_t *pool;
};
typedef struct xs_rast_t RAST_STATE;

struct xs_rast_db_t {
    apr_pool_t *pool;
    rast_db_t  *db;
    rast_db_t  **dbs;
    int        dbs_num;
};
typedef struct xs_rast_db_t RAST_DB_STATE;

struct xs_rast_result_t {
    apr_pool_t    *pool;
    rast_result_t *result;
    I32           cursor;
    I32           num_properties;
};
typedef struct xs_rast_result_t RAST_RESULT_STATE;


static int
_memory_error(int retcode)
{
    croak("APR MEMORY ERROR\n");
}
void
_not_ref_error()
{
    croak("Not a hash");
}

static void*
get_rast_state_hv(SV *sv)
{
    HV *hv;
    SV **svp;

    sv = SvRV(sv);
    if (!sv || SvTYPE(sv) != SVt_PVHV)
        _not_ref_error();
    hv = (HV *) sv;

    svp = hv_fetch(hv, "_xs_state", 9, 0);
    if (svp) {
        if (SvROK(*svp)) 
            return (void *) INT2PTR(void *, SvIV(SvRV(*svp)));
        else
            croak("_xs_state element is not a ref");
    }
    croak("can't find _xs_state element");
    return;
}


static int
magic_free_rast_state(pTHX_ SV *sv, MAGIC *mg)
{
    return 1;
}

MGVTBL vtbl_free_rast_state = {0, 0, 0, 0, MEMBER_TO_FPTR(magic_free_rast_state)};


MODULE = Rast		PACKAGE = Rast		



int
initialize(self)
        SV *self;
    PREINIT:
        RAST_STATE *state;
        rast_error_t *error;
        SV *sv;
        HV *hv;
        MAGIC *mg;
    CODE:
        sv = SvRV(self);
        if (!sv || SvTYPE(sv) != SVt_PVHV)
            _not_ref_error();
        hv = (HV *) sv;

        apr_initialize();
	//atexit(apr_terminate);
	error = rast_initialize();
        if (error != RAST_OK) {
            croak(error->message);
        }
        //atexit(rast_finalize);

        Newz(9999, state, 1, RAST_STATE);
        state->pool = NULL;
        apr_pool_create_ex(&state->pool, NULL, _memory_error, NULL);
        sv = newSViv(PTR2IV(state));
        sv_magic(sv, 0, '~', 0, 0);
        mg = mg_find(sv, '~');
        assert(mg);
        mg->mg_virtual = &vtbl_free_rast_state;
        SvREADONLY_on(sv);
        hv_store(hv, "_xs_state", 9, newRV_noinc(sv), 0);

        RETVAL = 1;
    OUTPUT:
        RETVAL

void
finalize()
    CODE:
        rast_finalize();
        apr_terminate();


int
_db_create(self, dbpath, options)
    SV *self;
    SV *dbpath;
    SV *options;
    PREINIT:
        rast_db_create_option_t *create_options;
        RAST_STATE *state = (RAST_STATE *) get_rast_state_hv(self);
        rast_error_t *error;
        STRLEN len;
        SV *sv;
        HV *options_hv;
        AV *properties;
        SV **svp;
	I32 i;
        apr_pool_t *pool;
    CODE:
        sv = SvRV(options);
        if (!sv || SvTYPE(sv) != SVt_PVHV)
            _not_ref_error();
        options_hv = (HV *) sv;

        apr_pool_create(&pool, state->pool);
        create_options = rast_db_create_option_create(pool);

        /* preserve_text */
        svp = hv_fetch(options_hv, "preserve_text", 13, 0);
        create_options->preserve_text = 0;
        if (svp) {
            create_options->preserve_text = SvIV(*svp);
        }

        /* byte_order */
        svp = hv_fetch(options_hv, "byte_order", 10, 0);
        create_options->byte_order = 0;
        if (svp) {
            create_options->byte_order = SvIV(*svp);
        }

        /* pos_block_size */
        svp = hv_fetch(options_hv, "pos_block_size", 14, 0);
        create_options->pos_block_size = 0;
        if (svp) {
            create_options->pos_block_size = SvIV(*svp);
        }

        /* encoding */
        svp = hv_fetch(options_hv, "encoding", 8, 0);
        create_options->encoding = "utf8";
        if (svp) {
            create_options->encoding = SvPV(*svp, len);
        }

        /* properties */
        svp = hv_fetch(options_hv, "num_properties", 14, 0);
        create_options->num_properties = 0;
        if (svp) {
            create_options->num_properties = SvIV(*svp);
        }
        if (!create_options->num_properties) {
            apr_pool_destroy(pool);
            croak("no properties");
        }

        svp = hv_fetch(options_hv, "properties", 10, 0);
        if (!svp) {
            apr_pool_destroy(pool);
            croak("no properties");
        }
        properties = (AV *) SvRV(*svp);
	
        create_options->properties = (rast_property_t *) apr_palloc(pool, sizeof(rast_property_t) * create_options->num_properties);

        for (i = 0;i <= av_len(properties);i++) {
            SV **entp;
            AV *ent;

            entp = av_fetch(properties, i, 0);
            ent = (AV *) SvRV(*entp);

            entp = av_fetch(ent, 0, 0);
            create_options->properties[i].name = SvPV(*entp, len);
            entp = av_fetch(ent, 1, 0);
            create_options->properties[i].type = SvIV(*entp);
            entp = av_fetch(ent, 2, 0);
            create_options->properties[i].flags = SvIV(*entp);
        }

        error = rast_db_create(SvPV(dbpath, len), create_options, pool);
        if (error != RAST_OK) {
            apr_pool_destroy(pool);
            croak(error->message);
        }

        apr_pool_destroy(pool);
        RETVAL = 1;
    OUTPUT:
        RETVAL



SV *
_db_open(self, dbpath, flags, options)
    SV *self;
    SV *dbpath;
    SV *flags;
    SV *options;
    PREINIT:
        rast_db_t *db;
        rast_db_open_option_t *open_options;
        RAST_STATE *state = (RAST_STATE *) get_rast_state_hv(self);
        RAST_DB_STATE *db_state;
        rast_error_t *error;
        STRLEN len;
        SV *sv, *rsv;
        HV *options_hv;
        SV **svp;
        MAGIC *mg;
        apr_pool_t *pool;
        apr_pool_t *db_pool;
        rast_db_t **dbs;
        int dbs_num;
    CODE:
        sv = SvRV(options);
        if (!sv || SvTYPE(sv) != SVt_PVHV)
            _not_ref_error();
        options_hv = (HV *) sv;

        apr_pool_create(&pool, state->pool);
        apr_pool_create_ex(&db_pool, NULL, _memory_error, NULL);

        svp = hv_fetch(options_hv, "sync_threshold_chars", 20, 0);
        open_options = NULL;
        if (svp) {
            open_options = (rast_db_open_option_t *) apr_palloc(pool, sizeof(rast_db_open_option_t));
            open_options->sync_threshold_chars = SvIV(*svp);
        }

        sv = SvRV(dbpath);
        if (!sv || SvTYPE(sv) == SVt_PVAV) {
 	    AV *list = (AV *) SvRV(dbpath);
            int i;
            dbs_num = av_len(list) + 1;
            dbs = (rast_db_t **) apr_palloc(db_pool, sizeof(const rast_db_t *) * dbs_num);
            for (i = 0;i < dbs_num;i++) {
  	        SV **entp;
                entp = av_fetch(list, i, 0);
                error = rast_db_open(&dbs[i], SvPV(*entp, len), SvIV(flags), open_options, db_pool);
            }
            if (error == RAST_OK) {
                error = rast_merger_open(&db, dbs, (int) dbs_num, db_pool);
            }
        } else {
            error = rast_db_open(&db, SvPV(dbpath, len), SvIV(flags), open_options, db_pool);
	    dbs_num = 0;
        }
        if (error != RAST_OK) {
            apr_pool_destroy(pool);
            apr_pool_destroy(db_pool);
            croak(error->message);
        }

        apr_pool_destroy(pool);

        /* create Rast::Db */
        ENTER;
        SAVETMPS;
        PUSHMARK(SP);
        if (dbs_num) {
            XPUSHs(sv_2mortal(newSVpv("Rast::Merger", 12)));
            PUTBACK;
            while (1) {
	        int i;
                if (!call_method("Rast::Merger::new", G_SCALAR) <= 0) {
                    rsv = POPs;
                    if (!(!SvROK(rsv) || SvTYPE(SvRV(rsv)) != SVt_PVHV)) {
		        break;
                    }
                }
                rast_db_close(db);
                for (i = 0;i < dbs_num;i++) {
                    rast_db_close(dbs[i]);
                }
                apr_pool_destroy(db_pool);
                croak("Rast::Merger::new did not");
            }
        } else {
            XPUSHs(sv_2mortal(newSVpv("Rast::Db", 8)));
            PUTBACK;
            if (call_method("Rast::Db::new", G_SCALAR) <= 0) {
                rast_db_close(db);
                apr_pool_destroy(db_pool);
                croak("Rast::Db::new did not");
            }
            rsv = POPs;
            if (!SvROK(rsv) || SvTYPE(SvRV(rsv)) != SVt_PVHV) {
                rast_db_close(db);
                apr_pool_destroy(db_pool);
                croak("Rast::Db::new did not");
            }
        }
        rsv = newSVsv(rsv);

        FREETMPS;
        LEAVE;

        Newz(9999, db_state, 1, RAST_DB_STATE);
        db_state->pool = db_pool;
        sv = newSViv(PTR2IV(db_state));
        sv_magic(sv, 0, '~', 0, 0);
        mg = mg_find(sv, '~');
        assert(mg);
        mg->mg_virtual = &vtbl_free_rast_state;
        SvREADONLY_on(sv);
        hv_store((HV *) SvRV(rsv), "_xs_state", 9, newRV_noinc(sv), 0);

        db_state->db = db;
        db_state->dbs = dbs;
        db_state->dbs_num = dbs_num;

        RETVAL = rsv;
    OUTPUT:
        RETVAL


PROTOTYPES: ENABLE

SV *
RAST_LITTLE_ENDIAN()
    CODE:
        RETVAL = newSViv(RAST_LITTLE_ENDIAN);
    OUTPUT:
        RETVAL

SV *
RAST_BIG_ENDIAN()
    CODE:
        RETVAL = newSViv(RAST_BIG_ENDIAN);
    OUTPUT:
        RETVAL

SV *
RAST_NATIVE_ENDIAN()
    CODE:
        RETVAL = newSViv(RAST_NATIVE_ENDIAN);
    OUTPUT:
        RETVAL

SV *
RAST_TYPE_STRING()
    CODE:
        RETVAL = newSViv(RAST_TYPE_STRING);
    OUTPUT:
        RETVAL

SV *
RAST_TYPE_DATE()
    CODE:
        RETVAL = newSViv(RAST_TYPE_DATE);
    OUTPUT:
        RETVAL

SV *
RAST_TYPE_DATETIME()
    CODE:
        RETVAL = newSViv(RAST_TYPE_DATETIME);
    OUTPUT:
        RETVAL

SV *
RAST_TYPE_UINT()
    CODE:
        RETVAL = newSViv(RAST_TYPE_UINT);
    OUTPUT:
        RETVAL

SV *
RAST_PROPERTY_FLAG_SEARCH()
    CODE:
        RETVAL = newSViv(RAST_PROPERTY_FLAG_SEARCH);
    OUTPUT:
        RETVAL

SV *
RAST_PROPERTY_FLAG_TEXT_SEARCH()
    CODE:
        RETVAL = newSViv(RAST_PROPERTY_FLAG_TEXT_SEARCH);
    OUTPUT:
        RETVAL

SV *
RAST_PROPERTY_FLAG_FULL_TEXT_SEARCH()
    CODE:
        RETVAL = newSViv(RAST_PROPERTY_FLAG_FULL_TEXT_SEARCH);
    OUTPUT:
        RETVAL

SV *
RAST_PROPERTY_FLAG_UNIQUE()
    CODE:
        RETVAL = newSViv(RAST_PROPERTY_FLAG_UNIQUE);
    OUTPUT:
        RETVAL

SV *
RAST_PROPERTY_FLAG_OMIT()
    CODE:
        RETVAL = newSViv(RAST_PROPERTY_FLAG_OMIT);
    OUTPUT:
        RETVAL

SV *
RAST_RDWR()
    CODE:
        RETVAL = newSViv(RAST_RDWR);
    OUTPUT:
        RETVAL

SV *
RAST_RDONLY()
    CODE:
        RETVAL = newSViv(RAST_RDONLY);
    OUTPUT:
        RETVAL

SV *
RAST_DB_RDWR()
    CODE:
        RETVAL = newSViv(RAST_DB_RDWR);
    OUTPUT:
        RETVAL

SV *
RAST_DB_RDONLY()
    CODE:
        RETVAL = newSViv(RAST_DB_RDONLY);
    OUTPUT:
        RETVAL

SV *
RAST_RESULT_ALL_ITEMS()
    CODE:
        RETVAL = newSViv(RAST_RESULT_ALL_ITEMS);
    OUTPUT:
        RETVAL

SV *
RAST_SORT_METHOD_SCORE()
    CODE:
        RETVAL = newSViv(RAST_SORT_METHOD_SCORE);
    OUTPUT:
        RETVAL

SV *
RAST_SORT_METHOD_PROPERTY()
    CODE:
        RETVAL = newSViv(RAST_SORT_METHOD_PROPERTY);
    OUTPUT:
        RETVAL

SV *
RAST_SORT_ORDER_DEFAULT()
    CODE:
        RETVAL = newSViv(RAST_SORT_ORDER_DEFAULT);
    OUTPUT:
        RETVAL

SV *
RAST_SORT_ORDER_ASCENDING()
    CODE:
        RETVAL = newSViv(RAST_SORT_ORDER_ASCENDING);
    OUTPUT:
        RETVAL

SV *
RAST_SORT_ORDER_DESCENDING()
    CODE:
        RETVAL = newSViv(RAST_SORT_ORDER_DESCENDING);
    OUTPUT:
        RETVAL

SV *
RAST_SCORE_METHOD_NONE()
    CODE:
        RETVAL = newSViv(RAST_SCORE_METHOD_NONE);
    OUTPUT:
        RETVAL

SV *
RAST_SCORE_METHOD_TFIDF()
    CODE:
        RETVAL = newSViv(RAST_SCORE_METHOD_TFIDF);
    OUTPUT:
        RETVAL



MODULE = Rast		PACKAGE = Rast::Db

PROTOTYPES: ENABLE

void
close(self)
    SV *self;
    PREINIT:
        RAST_DB_STATE *db_state = (RAST_DB_STATE *) get_rast_state_hv(self);
    CODE:
        if (db_state->pool != NULL) {
   	    int i;
            rast_db_close(db_state->db);
            for (i = 0;i < db_state->dbs_num;i++) {
                rast_db_close(db_state->dbs[i]);
            }
            apr_pool_destroy(db_state->pool);
            db_state->pool = NULL;
            db_state->db   = NULL;
        }

void
sync(self)
    SV *self;
    PREINIT:
        RAST_DB_STATE *db_state = (RAST_DB_STATE *) get_rast_state_hv(self);
    CODE:
        if (db_state->pool != NULL) {
            rast_db_sync(db_state->db);
        }

int
get_num_properties(self)
    SV *self;
    PREINIT:
        RAST_DB_STATE *db_state = (RAST_DB_STATE *) get_rast_state_hv(self);
        int num_properties;
        const rast_property_t *rast_properties;
    CODE:
        if (db_state->pool != NULL) {
            rast_properties = rast_db_properties(db_state->db, &num_properties);
            RETVAL = num_properties;
        } else {
            RETVAL = 0;
        }
    OUTPUT:
        RETVAL

int
get_properties_type(self, num)
    SV *self;
    SV *num;
    PREINIT:
        RAST_DB_STATE *db_state = (RAST_DB_STATE *) get_rast_state_hv(self);
        I32 req = SvIV(num);
        int num_properties;
        const rast_property_t *rast_properties;
    CODE:
        RETVAL = 0;
        if (db_state->pool != NULL) {
            rast_properties = rast_db_properties(db_state->db, &num_properties);
            if (num_properties > req) {
                const rast_property_t *property = rast_properties + req;
                RETVAL = property->type;
            }
        }
    OUTPUT:
        RETVAL

int
delete(self, doc_id)
    SV *self;
    SV *doc_id;
    PREINIT:
        RAST_DB_STATE *db_state = (RAST_DB_STATE *) get_rast_state_hv(self);
        rast_error_t *error;
    CODE:
        RETVAL = 0;
        if (db_state->pool != NULL) {
            error = rast_db_delete(db_state->db, SvIV(doc_id));
            if (error == RAST_OK) {
                RETVAL = 1;
            } else {
                rast_error_destroy(error);
            }
        }
    OUTPUT:
        RETVAL


int
_update_register(self, content, length, properties, old_doc)
    SV *self;
    SV *content;
    SV *length;
    SV *properties;
    SV *old_doc;
    PREINIT:
        RAST_DB_STATE *db_state = (RAST_DB_STATE *) get_rast_state_hv(self);
        I32 content_length = SvIV(length);
        rast_value_t *properties_value;
        rast_doc_id_t doc_id;
        rast_doc_id_t old_doc_id;
        rast_error_t *error;
        STRLEN len;
        AV *properties_av;
        apr_pool_t *pool;
        I32 i;
        int num_properties;
        const rast_property_t *rast_properties;
    CODE:
        properties_av = (AV *) SvRV(properties);

        apr_pool_create(&pool, db_state->pool);

        properties_value = (rast_value_t *) apr_palloc(pool, sizeof(rast_value_t) * (av_len(properties_av) + 1));
        rast_properties = rast_db_properties(db_state->db, &num_properties);
        
        for (i = 0;i <=  av_len(properties_av);i++) {
            const rast_property_t *property = rast_properties + i;
            SV **entp;
            entp = av_fetch(properties_av, i, 0);

            switch (property->type) {
                case RAST_TYPE_STRING:
		    rast_value_set_string(&properties_value[i], SvPV(*entp, len));
	            break;
	        case RAST_TYPE_DATE:
		    rast_value_set_date(&properties_value[i], SvPV(*entp, len));
	            break;
		case RAST_TYPE_DATETIME:
 		    rast_value_set_datetime(&properties_value[i], SvPV(*entp, len));
		    break;
		case RAST_TYPE_UINT:
                    rast_value_set_uint(&properties_value[i], SvIV(*entp));
		    break;
            }
        }

        old_doc_id = SvIV(old_doc);
        if (old_doc_id) {
            error = rast_db_update(db_state->db, old_doc_id, SvPV(content, len), content_length, properties_value, &doc_id);
        } else {
            error = rast_db_register(db_state->db, SvPV(content, len), content_length, properties_value, &doc_id);
        }
        apr_pool_destroy(pool);
        if (error != RAST_OK) {
            rast_error_destroy(error);
            RETVAL = 0;
        } else {
            RETVAL = doc_id;
        }
    OUTPUT:
        RETVAL



SV *
_search(self, query, options)
    SV *self;
    SV *query;
    SV *options;
    PREINIT:
        RAST_DB_STATE *db_state = (RAST_DB_STATE *) get_rast_state_hv(self);
        RAST_RESULT_STATE *result_state;
        rast_search_option_t *create_options;
        rast_result_t *result;
        rast_error_t *error;
        STRLEN len;
        SV *sv, **svp, *rsv;
        MAGIC *mg;
        HV *options_hv;
        apr_pool_t *pool;
        apr_pool_t *result_pool;
        I32 i;
        I32 num_properties;
    CODE:
        sv = SvRV(options);
        if (!sv || SvTYPE(sv) != SVt_PVHV)
            _not_ref_error();
        options_hv = (HV *) sv;

        apr_pool_create(&pool, db_state->pool);
        apr_pool_create_ex(&result_pool, NULL, _memory_error, NULL);
        create_options = rast_search_option_create(pool);

        /* start_no */
        svp = hv_fetch(options_hv, "start_no", 8, 0);
        create_options->start_no = 0;
        if (svp) {
            create_options->start_no = SvIV(*svp);
        }

        /* num_items */
        svp = hv_fetch(options_hv, "num_items", 9, 0);
        create_options->num_items = RAST_RESULT_ALL_ITEMS;
        if (svp) {
            create_options->num_items = SvIV(*svp);
        }

        /* need_summary */
        svp = hv_fetch(options_hv, "need_summary", 12, 0);
        create_options->need_summary = 0;
        if (svp) {
            create_options->need_summary = SvIV(*svp);
        }

        /* summary_nchars */
        svp = hv_fetch(options_hv, "summary_nchars", 14, 0);
        create_options->summary_nchars = 100;
        if (svp) {
            create_options->summary_nchars = SvIV(*svp);
        }

        /* sort_method */
        svp = hv_fetch(options_hv, "sort_method", 11, 0);
        create_options->sort_method = RAST_SORT_METHOD_SCORE;
        if (svp) {
            create_options->sort_method = SvIV(*svp);
        }

        /* sort_property */
        svp = hv_fetch(options_hv, "sort_property", 13, 0);
        create_options->sort_property = NULL;
        if (svp) {
            create_options->sort_property = SvPV(*svp, len);
        }

        /* sort_order */
        svp = hv_fetch(options_hv, "sort_order", 10, 0);
        create_options->sort_order = RAST_SORT_ORDER_DEFAULT;
        if (svp) {
            create_options->sort_order = SvIV(*svp);
        }

        /* score_method */
        svp = hv_fetch(options_hv, "score_method", 12, 0);
        create_options->score_method = RAST_SCORE_METHOD_TFIDF;
        if (svp) {
            create_options->score_method = SvIV(*svp);
        }

        svp = hv_fetch(options_hv, "properties", 10, 0);
        if (svp) {
            AV *properties = (AV *) SvRV(*svp);
            create_options->properties = (const char **) apr_palloc(pool, sizeof(const char *) * create_options->num_properties);
            for (i = 0;i <= av_len(properties);i++) {
                SV **entp;
                entp = av_fetch(properties, i, 0);
                create_options->properties[i] = SvPV(*entp, len);
                create_options->num_properties++;
            }
        } else {
            create_options->properties = NULL;
            create_options->num_properties = 0;
        }
        num_properties = create_options->num_properties;

        error = rast_db_search(db_state->db, SvPV(query, len), create_options, &result, result_pool);
        apr_pool_destroy(pool);
        if (error != RAST_OK) {
            rast_error_destroy(error);
            apr_pool_destroy(result_pool);
            RETVAL = newSVuv((uint8_t) 0);
        } else {

            /* create Rast::Result */
            ENTER;
            SAVETMPS;
            PUSHMARK(SP);
            XPUSHs(sv_2mortal(newSVpv("Rast::Result", 12)));
            PUTBACK;
            if (call_method("Rast::Result::new", G_SCALAR) <= 0) {
                apr_pool_destroy(result_pool);
                croak("Rast::Result::new did not");
            }

            rsv = POPs;
            if (!SvROK(rsv) || SvTYPE(SvRV(rsv)) != SVt_PVHV) {
                apr_pool_destroy(result_pool);
                croak("Rast::Db::new did not");
            }
            rsv = newSVsv(rsv);

            FREETMPS;
            LEAVE;

            Newz(9999, result_state, 1, RAST_RESULT_STATE);
            result_state->pool = result_pool;
            sv = newSViv(PTR2IV(result_state));
            sv_magic(sv, 0, '~', 0, 0);
            mg = mg_find(sv, '~');
            assert(mg);
            mg->mg_virtual = &vtbl_free_rast_state;
            SvREADONLY_on(sv);
            hv_store((HV *) SvRV(rsv), "_xs_state", 9, newRV_noinc(sv), 0);

            result_state->result = result;
            result_state->cursor = 0;
            result_state->num_properties = num_properties;

            RETVAL = rsv;
        }
    OUTPUT:
        RETVAL


MODULE = Rast		PACKAGE = Rast::Result

PROTOTYPES: ENABLE

void
destroy(self)
    SV *self;
    PREINIT:
        RAST_RESULT_STATE *result_state = (RAST_RESULT_STATE *) get_rast_state_hv(self);
    CODE:
        if (result_state->pool != NULL) {
            apr_pool_destroy(result_state->pool);
            result_state->result = NULL;
            result_state->pool = NULL;
        }

int
hit_count(self)
    SV *self;
    PREINIT:
        RAST_RESULT_STATE *result_state = (RAST_RESULT_STATE *) get_rast_state_hv(self);
    CODE:
        if (result_state->result != NULL) {
            RETVAL = result_state->result->hit_count;
        } else {
            RETVAL = 0;
        }
    OUTPUT:
        RETVAL

int
num_items(self)
    SV *self;
    PREINIT:
        RAST_RESULT_STATE *result_state = (RAST_RESULT_STATE *) get_rast_state_hv(self);
    CODE:
        if (result_state->result != NULL) {
            RETVAL = result_state->result->num_items;
        } else {
            RETVAL = 0;
        }
    OUTPUT:
        RETVAL

void
rewind(self)
    SV *self;
    PREINIT:
        RAST_RESULT_STATE *result_state = (RAST_RESULT_STATE *) get_rast_state_hv(self);
    CODE:
        if (result_state->result != NULL) {
            result_state->cursor = 0;
        }

void
fetch(self)
    SV *self;
    PREINIT:
        RAST_RESULT_STATE *result_state = (RAST_RESULT_STATE *) get_rast_state_hv(self);
        I32 i, ii;
        HV *rhv;
    PPCODE:
        rhv = NULL;
        if (result_state->result != NULL) {
            if (result_state->result->num_items > result_state->cursor) {
                i = result_state->cursor;
                result_state->cursor++;

                rhv = (HV *) sv_2mortal((SV *)newHV());

                hv_store(rhv, "doc_id", 6, newSViv(result_state->result->items[i]->doc_id), 0);
                hv_store(rhv, "score", 5, newSViv(result_state->result->items[i]->score), 0);
                if (result_state->result->items[i]->summary_nbytes) {
                    hv_store(rhv, "summary", 7, newSVpv(result_state->result->items[i]->summary,
                    result_state->result->items[i]->summary_nbytes), 0);
                }
                if (result_state->num_properties) {
                    rast_value_t *v = result_state->result->items[i]->properties;
                    AV *av = (AV *) sv_2mortal((SV *)newAV());
                    for (ii = 0;ii < result_state->num_properties;ii++) {
  		        char *c;
		        switch (v[ii].type) {
			    case RAST_TYPE_STRING:
                                c = rast_value_string(v + ii);
                                av_push(av, newSVpv(c, strlen(c)));
			        break;
			    case RAST_TYPE_DATE:
                                c = rast_value_date(v + ii);
                                av_push(av, newSVpv(c, strlen(c)));
			        break;
			    case RAST_TYPE_DATETIME:
                                c = rast_value_datetime(v + ii);
                                av_push(av, newSVpv(c, strlen(c)));
			        break;
			    case RAST_TYPE_UINT:
                                av_push(av, newSViv(rast_value_uint(v + ii)));
			        break;
		        }
                    }
                    hv_store(rhv, "properties", 10, (SV *) newRV((SV *) av), 0);
                }
            }
        }

        if (rhv) {
            ST(0) = sv_2mortal(newRV((SV *) rhv));
        } else {
            ST(0) = &PL_sv_undef;
        }
        XSRETURN(1);
