/***************************************************************************
 *   Copyright (C) 2008 by Konstantinos Smanis                             *
 *   kon.smanis@gmail.com                                                  *
 *                                                                         *
 *   This file is part of KGRUBEditor.                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with parent program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor                                       *
 *   Boston, MA 02110-1301, USA.                                          *
 ***************************************************************************/

//Own
#include "root.h"

//Qt
#include <qwidget.h>
#include <qfileinfo.h>

//KDE
#include <kprocess.h>
#include <kdebug.h>

//Ui
#include "../passworddialog.h"

QString Core::Root::Password;

namespace Core
{
	namespace Root
	{
		bool requestPassword( QWidget *parent )
		{
			if ( Password.isEmpty() || KDESu::SuProcess( "root" ).checkInstall( Password.toLocal8Bit() ) != 0 )
			{
				PasswordDialog dlg( parent );
				if ( !dlg.exec() )
					return false;

				Password = dlg.password().toLocal8Bit();
			}
			return true;
		}


		bool makeDirReadableByUserGroup( const QString &dirPath, QWidget *parent )
		{
			if ( dirPath.isEmpty() )
			{
				kError() << "No directory was specified.";
				return false;
			}

			QFileInfo fi( dirPath );
			if ( !fi.exists() )
			{
				kError() << dirPath << "doesn't exist.";
				return false;
			}
			if ( !fi.isDir() )
			{
				kError() << dirPath << "is a file. However, a directory was expected.";
				return false;
			}

			QString userGroup = fi.group();
			if ( userGroup.isEmpty() )
			{
				kError() << "Couldn't determine the group of" << dirPath;
				return false;
			}

			KProcess groups;
			groups.setProgram( "groups" );
			groups.setOutputChannelMode( KProcess::MergedChannels );
			groups.start();
			groups.waitForFinished();
			QString targetUserGroup( groups.readAllStandardOutput().split( ' ' ).first() );
			if ( targetUserGroup.isEmpty() )
			{
				kError() << "Couldn't determine the user's main group.";
				return false;
			}

			if ( userGroup != targetUserGroup )
			{
				kDebug() << "Changing the group of" << dirPath << "from" << userGroup << "to" << targetUserGroup;
				if ( !requestPassword( parent ) )
					return false;

				KDESu::SuProcess chgrp( QByteArray( "root" ), QByteArray( "chgrp " ).append( targetUserGroup ).append( " " ).append( dirPath ) );
				chgrp.exec( Password.toLocal8Bit() );
			}

			if ( !fi.permission( QFile::ReadGroup ) || !fi.permission( QFile::ExeGroup ) )
			{
				kDebug() << "Making" << dirPath << "readable and executable by group" << targetUserGroup;
				if ( requestPassword( parent ) )
					return false;

				KDESu::SuProcess chmod( QByteArray( "root" ), QByteArray( "chmod g+rx " ).append( dirPath ) );
				chmod.exec( Password.toLocal8Bit() );
			}

			return true;
		}
		bool makeDirWritableByUserGroup( const QString &dirPath, QWidget *parent )
		{
			if ( dirPath.isEmpty() )
			{
				kError() << "No directory was specified.";
				return false;
			}

			QFileInfo fi( dirPath );
			if ( fi.exists() && !fi.isDir() )
			{
				kError() << dirPath << "is a file. However, a directory was expected.";
				return false;
			}

			if ( !fi.exists() )
			{
				kDebug() << dirPath << "doesn't exist. Attempting creation.";
				if ( !requestPassword( parent ) )
					return false;

				KDESu::SuProcess mkdir( QByteArray( "root" ), QByteArray( "mkdir " ).append( dirPath ) );
				mkdir.exec( Password.toLocal8Bit() );
				fi.setFile( dirPath );
			}

			QString userGroup = fi.group();
			if ( userGroup.isEmpty() )
			{
				kError() << "Couldn't determine the group of" << dirPath;
				return false;
			}

			KProcess groups;
			groups.setProgram( "groups" );
			groups.setOutputChannelMode( KProcess::MergedChannels );
			groups.start();
			groups.waitForFinished();
			QString targetUserGroup( groups.readAllStandardOutput().split( ' ' ).first() );
			if ( targetUserGroup.isEmpty() )
			{
				kError() << "Couldn't determine the user's main group.";
				return false;
			}

			if ( userGroup != targetUserGroup )
			{
				kDebug() << "Changing the group of" << dirPath << "from" << userGroup << "to" << targetUserGroup;
				if ( !requestPassword( parent ) )
					return false;

				KDESu::SuProcess chgrp( QByteArray( "root" ), QByteArray( "chgrp " ).append( targetUserGroup ).append( " " ).append( dirPath ) );
				chgrp.exec( Password.toLocal8Bit() );
			}

			if ( !fi.permission( QFile::WriteGroup ) )
			{
				kDebug() << "Making" << dirPath << "writable by group" << targetUserGroup;
				if ( !requestPassword( parent ) )
					return false;

				KDESu::SuProcess chmod( QByteArray( "root" ), QByteArray( "chmod g+w " ).append( dirPath ) );
				chmod.exec( Password.toLocal8Bit() );
			}

			return true;
		}
	}
}
