
use AdminUtil;
use DSUtil qw(debug);
# load perldap
use Mozilla::LDAP::Conn;
use Mozilla::LDAP::Utils qw(normalizeDN);
use Mozilla::LDAP::API qw(:constant ldap_url_parse ldap_explode_dn);

sub pre {
    my ($inf, $configdir) = @_;
    my @errs;
    # for each server being updated, we need to convert any information
    # under o=NetscapeRoot for that server from Fedora xxx to 389 xxx
    # e.g. cn=Fedora Directory Server -> cn=389 Directory Server
    # then the rest of the update process will update the version, build
    # number, etc.

    # start with the isie from the adm.conf
    # see if there is already a configds
    my $admConf = AdminUtil::getAdmConf("$configdir/admin-serv");

    if (!$admConf) {
        return ('error_reading_conffile', "$configdir/admin-serv/adm.conf", $!);
    }

    my $basedn = $admConf->{isie};

    if (!$basedn) {
        return ('error_admconf_isie', "$configdir/admin-serv/adm.conf");
    }

    # look for the Fedora branded entries
    $basedn =~ s/cn=389/cn=Fedora/g;

    # start at the host entry - two rdns up from the isie
    my @rdns = ldap_explode_dn($basedn, 0);
    $basedn = join(',', @rdns[-3..-1]);
    my $domaindn = join(',', @rdns[-2..-1]);

    my @ents;
    my $conn = $inf->{configdsconn};
    my $ent = $conn->search($basedn, 'sub', '(objectclass=*)');
    my $rc = $conn->getErrorCode();
    if (($rc != LDAP_SUCCESS) or (!$ent)) {
        if ($rc != LDAP_NO_SUCH_OBJECT) {
            my $errstr = $conn->getErrorString();
            return ('error_finding_isie', $basedn, $conn->{host}, $conn->{port},
                    (($errstr eq "Success") ? 'unknown error' : $errstr));
        } else {
            # nothing to do - just return
            debug(1, "Fedora branded entry $basedn not found - skipping\n");
            return ();
        }
    }
    my @sielist;
    my $count = 0;
    while ($ent) {
        my $olddn = $ent->getDN();
        my $newdn = $olddn;
        $count += ($newdn =~ s/cn=Fedora/cn=389/g);
        $ent->setDN($newdn);
        for my $attr (keys %{$ent}) {
            my @newvals = $ent->getValues($attr);
            for my $val (@newvals) {
                $count += ($val =~ s/cn=Fedora/cn=389/g); # fix DNs
                $count += ($val =~ s/\@fedora-admin/\@389-admin/g); # fix jar names
                $count += ($val =~ s/\@fedora-ds/\@389-ds/g); # fix jar names
                $count += ($val =~ s/Fedora Project/389 Project/g);
                $count += ($val =~ s/Fedora Administration Server/389 Administration Server/g);
                $count += ($val =~ s/Fedora Directory Server/389 Directory Server/g);
            }
            $ent->setValues($attr, @newvals);
        }
        # save the sie
        if ($ent->hasValue('objectclass', 'netscapeServer', 1)) {
            push @sielist, $olddn;
        }
        # save the old DN
        $ent->{_olddn_} = $olddn;
        # add to the list of entries
        push @ents, $ent;
        $ent = $conn->nextEntry();
    }

    if (!$count) {
        # nothing to do - just return
        debug(1, "No Fedora branding found - skipping\n");
        return ();
    }

    # if a prior installation was messed up, there will be both
    # a Fedora branded entry and a 389 branded entry - in this
    # case, the 389 entry is bogus and must be replaced with
    # the converted Fedora->389 entry from above - so we must
    # delete them first - we have to do the deletion in reverse
    # order
    for my $ent (reverse @ents) {
        $conn->delete($ent);
        my $rc = $conn->getErrorCode();
        if ($rc == LDAP_NOT_ALLOWED_ON_NONLEAF) {
            # just update it instead
            if ($conn->update($ent)) {
                $rc = LDAP_SUCCESS;
            } else {
                $rc = $conn->getErrorCode();
            }
        }
        if ($rc != LDAP_SUCCESS) {
            if ($rc == LDAP_NO_SUCH_OBJECT) {
                debug(1, "Entry ", $ent->getDN(), " does not exist - skipping\n");
            } else {
                return ('error_deleteall_entries', $ent->getDN(), $conn->getErrorString());
            }
        }
    }

    # found and fixed all of them, deleted old bogus entries, now try to add
    # if we get Already Exists, just skip
    my @dnstodel = ();
    if ($count) { # have at least one change to make
        for my $ent (@ents) {
            $conn->add($ent);
            my $rc = $conn->getErrorCode();
            if ($rc == LDAP_TYPE_OR_VALUE_EXISTS) {
                # as a result of our corrections above, we have some
                # duplicate values - let's remove them
                # this is a list of attributes that may have DN syntax
                # and are multi valued - we have to normalize them first
                my %mydnattrs = (owner => 'owner', roleoccupant => 'roleoccupant',
                                 member => 'member', seealso => 'seealso',
                                 uniquemember => 'uniquemember',
                                 parentorganization => 'parentorganization',
                                 secretary => 'secretary', manager => 'manager',
                                 aliasedobjectname => 'aliasedobjectname',
                                 associatedname => 'associatedname',
                                 distinguishedname => 'distinguishedname',
                                 documentauthor => 'documentauthor',
                                 nsroledn => 'nsroledn',
                                 nsadminsiedn => 'nsadminsiedn',
                                 nsdirectoryinforef => 'nsdirectoryinforef',
                                 mailenhanceduniquemember => 'mailenhanceduniquemember');
                my %skipattrs = (objectclass => 'objectclass');
                for my $attr (keys %{$ent}) {
                    next if ($skipattrs{lc $attr});
                    my @newvals = $ent->getValues($attr);
                    my %uniq = ();
                    # the keys of the uniq hash will be the normalized values
                    # the hash table will just throw away dups, so the
                    # resultant table will have as the keys the unique
                    # normalized values, and will have as the values the
                    # original unique un-normalized values
                    if ($mydnattrs{lc $attr}) {
                        %uniq = map { normalizeDN($_) => $_ } @newvals;
                    } else {
                        %uniq = map { lc $_ => $_ } @newvals;
                    }
                    $ent->setValues($attr, values %uniq);
                }
                if ($conn->update($ent)) {
                    $rc = LDAP_SUCCESS;
                } else {
                    $rc = $conn->getErrorCode();
                }
            } elsif ($rc == LDAP_SUCCESS) {
                push @dnstodel, $ent->{_olddn_};
            }
            if ($rc != LDAP_SUCCESS) {
                if ($rc != LDAP_ALREADY_EXISTS) {
                    # just bail - it's unlikely that we would get this error from
                    # far down in the tree, if we didn't already get this at
                    # the top level
                    return ('error_adding_entry', $ent->getDN(), $conn->getErrorString());
                } else {
                    debug(1, "Entry ", $ent->getDN(), " already exists - skipping\n");
                }
            }
        }
    }

    # now fix adm.conf and local.conf
    $count = 0; # reset
    while (my ($key, $val) = each %{$admConf}) {
        $count += ($val =~ s/cn=Fedora/cn=389/g); # fix DNs
        $count += ($val =~ s/\@fedora-admin/\@389-admin/g); # fix jar names
        $count += ($val =~ s/\@fedora-ds/\@389-ds/g); # fix jar names
        $admConf->{$key} = $val;
    }
    if (($count > 0) and !AdminUtil::updateAdmConf($admConf, "$configdir/admin-serv")) {
        return ('error_updating_conffile', "$configdir/admin-serv/adm.conf", $!);
    }

    my $localconf = "$configdir/admin-serv/local.conf";
    if (!open(LOCALCONF, "$localconf")) {
        return ('error_reading_conffile', $localconf, $!);
    }
    my @lines = <LOCALCONF>;
    close LOCALCONF;
    $count = 0; # reset
    for my $line (@lines) {
        $count += ($line =~ s/cn=Fedora/cn=389/g); # fix DNs
        $count += ($line =~ s/\@fedora-admin/\@389-admin/g); # fix jar names
        $count += ($line =~ s/\@fedora-ds/\@389-ds/g); # fix jar names
    }
    
    if ($count > 0) {
        if (!open(LOCALCONF, ">$localconf")) {
            return ('error_updating_conffile', $localconf, $!);
        }
        for my $line (@lines) {
            print LOCALCONF $line;
        }
        close LOCALCONF;
    }

    # config files were successfully updated - delete the old entries
    while (@dnstodel) {
        # use pop because we have to delete in reverse order
        my $dn = pop @dnstodel;
        if ($dn) {
            $conn->delete($dn);
            my $rc = $conn->getErrorCode();
            if ($rc != LDAP_SUCCESS) {
                return ('error_deleteall_entries', $dn, $conn->getErrorString());
            }
        }
    }

    # fix the names in this entry
    # dn: cn=Client, ou=Admin, ou=Global Preferences, ou=%domain%, o=NetscapeRoot
    my $dn = "cn=Client, ou=Admin, ou=Global Preferences, $domaindn";
    $ent = $conn->search($dn, 'base', "(objectclass=*)");
    if ($ent) {
        $count = 0;
        @newvals = $ent->getValues('nsNickName');
        map { s/Fedora/389/g } @newvals;
        $ent->setValues('nsNickName', @newvals);
        $conn->update($ent);
    }    

    # remove the sie entries under ou=User Preferences, ou=%domain%, o=NetscapeRoot
    # they will be recreated
    for my $siedn (@sielist) {
        $ent = $conn->search($domaindn, 'sub', "(|(ou=$siedn)(ou=\"$siedn\"))");
        if ($ent) {
            $conn->delete($ent);
        }
    }

    return ();
}
