/*
 * Copyright © 2007 Peter Hutterer
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Soft-
 * ware"), to deal in the Software without restriction, including without
 * limitation the rights to use, copy, modify, merge, publish, distribute,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, provided that the above copyright
 * notice(s) and this permission notice appear in all copies of the Soft-
 * ware and that both the above copyright notice(s) and this permission
 * notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABIL-
 * ITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT OF THIRD PARTY
 * RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR HOLDERS INCLUDED IN
 * THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL INDIRECT OR CONSE-
 * QUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFOR-
 * MANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder shall
 * not be used in advertising or otherwise to promote the sale, use or
 * other dealings in this Software without prior written authorization of
 * the copyright holder.
 *
 * Authors:
 *   Peter Hutterer <peter@cs.unisa.edu.au>
 */


/**
 * X input driver for the nintendo wiimote.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <X11/extensions/XIproto.h>
#include <xf86.h>
#include <xf86_OSlib.h>
#include <xf86Xinput.h>

#include "wiimote.h"

/* internals */
static int _wii_init_buttons(DeviceIntPtr device);
static int _wii_init_axes(DeviceIntPtr device);
static int _wii_process(InputInfoPtr pInfo);
static int _wii_process_buttons(InputInfoPtr pInfo);

/* module */
static void     WiiUnplug(pointer p);
static pointer  WiiPlug(pointer        module,
                        pointer        options,
                        int            *errmaj,
                        int            *errmin);

/* driver */
static InputInfoPtr WiiPreInit(InputDriverPtr  drv,
                               IDevPtr         dev,
                               int             flags);

static void         WiiUnInit(InputDriverPtr   drv,
                              InputInfoPtr     pInfo,
                              int              flags);
static void         WiiReadInput(InputInfoPtr  pInfo);
static int          WiiControl(DeviceIntPtr    device,
                               int             what);
static void         WiiPtrCtrl(DeviceIntPtr    devices,
                               PtrCtrl         *ctrl);


/**
 * Driver Rec, fields are used when device is initialised/removed.
 */
_X_EXPORT InputDriverRec WIIMOTE = {
    1,
    "wiimote",
    NULL,
    WiiPreInit,
    WiiUnInit,
    NULL,
    0
};


/**
 * Module versioning information.
 */
static XF86ModuleVersionInfo WiiVersionRec =
{
    "wiimote",
    MODULEVENDORSTRING,
    MODINFOSTRING1,
    MODINFOSTRING2,
    XORG_VERSION_CURRENT,
    PACKAGE_VERSION_MAJOR, PACKAGE_VERSION_MINOR, PACKAGE_VERSION_PATCHLEVEL,
    ABI_CLASS_XINPUT,
    ABI_XINPUT_VERSION,
    MOD_CLASS_XINPUT,
    {0, 0, 0, 0}
};

/**
 * Module control. Fields are used when module is loaded/unloaded.
 */
_X_EXPORT XF86ModuleData wiimoteModuleData =
{
    &WiiVersionRec,
    WiiPlug,
    WiiUnplug
};


/***************************************************************************
 *                            Module procs                                 *
 ***************************************************************************/


/**
 * Called when module is unloaded.
 */
static void
WiiUnplug(pointer p)
{
}

/**
 * Called when module is loaded.
 */
static pointer
WiiPlug(pointer        module,
        pointer        options,
        int            *errmaj,
        int            *errmin)
{
    xf86AddInputDriver(&WIIMOTE, module, 0);
    return module;
}

/***************************************************************************
 *                            Driver procs                                 *
 ***************************************************************************/

/**
 * Called each time a new device is to be initialized.
 */
static InputInfoPtr WiiPreInit(InputDriverPtr  drv,
                               IDevPtr         dev,
                               int             flags)
{
    InputInfoPtr        pInfo;
    WiimotePtr          wii;
    char*               devname;
    char*               buttonmap;
    char*               axismap;
    int                 i;

    if (!(pInfo = xf86AllocateInput(drv, 0)))
        return NULL;

    pInfo->name         = xstrdup(dev->identifier);
    pInfo->flags        = 0;
    pInfo->type_name    = "WIIMOTE";
    pInfo->conf_idev    = dev;
    pInfo->read_input   = WiiReadInput; /* new data avl */
    pInfo->switch_mode  = NULL;         /* toggle absolute/relative mode */
    pInfo->device_control = WiiControl; /* enable/disable dev */
    pInfo->flags       |= XI86_OPEN_ON_INIT;
    pInfo->flags       |= XI86_CONFIGURED;
    pInfo->fd           = -1;

    xf86CollectInputOptions(pInfo, NULL, NULL);
    xf86ProcessCommonOptions(pInfo, pInfo->options);

    devname = xf86CheckStrOption(dev->commonOptions, "Device", 0);
    if (!devname)
        goto cleanup;
    xf86Msg(X_INFO, "%s: Using device %s.\n", pInfo->name, devname);


    wii = xcalloc(1, sizeof(Wiimote));
    if (!wii)
        goto cleanup;

    wii->btaddr = strdup(devname);

    wii->acc_on = xf86CheckBoolOption(dev->commonOptions, "Accelerometer", 0);
    wii->ir_on = xf86CheckBoolOption(dev->commonOptions, "IR", 1);
    wii->tilt_on = xf86CheckBoolOption(dev->commonOptions, "Tilt", 1);
    wii->nunchuk_on = xf86CheckBoolOption(dev->commonOptions, "Nunchuk", 0);

    wii->leds.led1 = xf86CheckBoolOption(dev->commonOptions, "LED1", 0);
    wii->leds.led2 = xf86CheckBoolOption(dev->commonOptions, "LED2", 0);
    wii->leds.led3 = xf86CheckBoolOption(dev->commonOptions, "LED3", 0);
    wii->leds.led4 = xf86CheckBoolOption(dev->commonOptions, "LED4", 0);

    if ((buttonmap = xf86CheckStrOption(dev->commonOptions, "ButtonMapping",
                    NULL)))
    {
        int b, n = 0;
        char *s = buttonmap;

        /* buttonmap order is in the order of the defines in wiimote.h */
        while(s && n < WII_NUM_BUTTONS && ((b = strtol(s, &s, 10)) != 0))
        {
            if (b < 0 || b > WII_NUM_BUTTONS)
            {
                xf86Msg(X_WARNING, "%s: Invalid button mapping: %d\n",
                        pInfo->name, b);
                break;
            }
            wii->bmap[n++] = b;
        }
    } else {
        for (i = 0; i < WII_NUM_BUTTONS; i++)
            wii->bmap[i] = i;
    }

    /* set axismap to default, then overwrite with the mappings */
    for (i = 0; i < WII_NUM_AXES; i++)
        wii->axismap[i] = i - ((wii->ir_on) ? 0 : 2);

    if ((axismap = xf86CheckStrOption(dev->commonOptions, "AxisMapping",
                    NULL)))
    {
        int a, n = 2; /* first two in axismap array are IR axes */
        char* s = axismap;

        /* axismap order is the order of defines in wiimote.h */
        while(s && n < WII_NUM_AXES && ((a = strtol(s, &s, 10)) != 0))
        {
            if (a < 0 || a > WII_NUM_AXES)
            {
                xf86Msg(X_WARNING, "%s: Invalid axis mapping: %d\n",
                        pInfo->name, a);
                break;
            }
            /* screwed up axis mapping due to IR axes */
            wii->axismap[n++] = (a - 1) + ((wii->ir_on) ? 2 : 0);
        }
    }

    pInfo->private = wii;
    return pInfo;
cleanup:
    if (wii->btaddr)
        xfree(wii->btaddr);
    if (wii)
        xfree(wii);
    if (pInfo)
        xf86DeleteInput(pInfo, 0);
    return NULL;
}

/**
 * Called each time a device is to be removed.
 * Clean up your mess here.
 */
static void WiiUnInit(InputDriverPtr drv,
                      InputInfoPtr   pInfo,
                      int            flags)
{
    xf86DeleteInput(pInfo, 0);
}

/**
 * Called when data is available on the socket.
 * We always read in a full packet, containing all the info for all users.
 */
static void WiiReadInput(InputInfoPtr pInfo)
{
    WiimotePtr wii = (WiimotePtr)pInfo->private;
    while(xf86WaitForInput(pInfo->fd, 0) > 0)
    {
        if (wiimote_update(wii->wii) == WIIMOTE_ERROR)
        {
            xf86Msg(X_ERROR, "%s: wiimote update failed. Disabling "
                    "device.\n", pInfo->name);
            xf86DisableDevice(pInfo->dev, TRUE);
            return;
        }
        _wii_process(pInfo);
    }
}


/**
 * Called when the device is to be enabled/disabled, etc.
 * @return Success or X error code.
 */
static int WiiControl(DeviceIntPtr    device,
                      int             what)
{
    InputInfoPtr        pInfo = device->public.devicePrivate;
    WiimotePtr          wii   = (WiimotePtr)pInfo->private;

    switch(what)
    {
        case DEVICE_INIT:
            _wii_init_buttons(device);
            _wii_init_axes(device);
            break;

        /* Switch device on. Establish socket, start event delivery.  */
        case DEVICE_ON:
            xf86Msg(X_INFO, "%s: On.\n", pInfo->name);
            if (device->public.on)
                break;

            xf86Msg(X_INFO, "%s: trying to connect to %s.\n",
                    pInfo->name, wii->btaddr);
            wii->wii = wiimote_open(wii->btaddr);
            if (!wii->wii)
            {
                xf86Msg(X_INFO, "%s: Failed to connect to %s.\n",
                        pInfo->name, wii->btaddr);
                break;
            }

            /* wiimote doesn't send data if acc is off, so force it on */
            wii->wii->mode.acc = 1;
            wii->wii->mode.ir = wii->ir_on;
            wii->wii->mode.ext = wii->nunchuk_on;
            wii->wii->led.bits = 0;
            wii->wii->led.one = wii->leds.led1;
            wii->wii->led.two = wii->leds.led2;
            wii->wii->led.three = wii->leds.led3;
            wii->wii->led.four = wii->leds.led4;
            wiimote_update(wii->wii);

            pInfo->fd = wii->wii->link.s_intr;
            AddEnabledDevice(pInfo->fd);
            device->public.on = TRUE;
            break;
        /**
         * Shut down device.
         */
        case DEVICE_OFF:
            xf86Msg(X_INFO, "%s: Off.\n", pInfo->name);
            if (!device->public.on)
                break;

            wiimote_close(wii->wii);
            wii->wii = NULL;
            RemoveEnabledDevice(pInfo->fd);
            pInfo->fd = -1;

            device->public.on = FALSE;
            break;
        case DEVICE_CLOSE:
            break;
    }
    return Success;
}

static void
WiiPtrCtrl(DeviceIntPtr device,  PtrCtrl *ctrl)
{
    /* dix takes care of everything */
}

/***************************************************************************
 *                            internal procs                               *
 ***************************************************************************/

static int
_wii_init_buttons(DeviceIntPtr device)
{
    InputInfoPtr        pInfo = device->public.devicePrivate;
    WiimotePtr          wii = (WiimotePtr)pInfo->private;
    int                 ret = Success;

    if (!InitButtonClassDeviceStruct(device, WII_NUM_BUTTONS, wii->bmap)) {
        xf86Msg(X_ERROR, "%s: Failed to register buttons.\n", pInfo->name);
        ret = BadAlloc;
    }

    return ret;
}

static int
_wii_init_axes(DeviceIntPtr device)
{
    InputInfoPtr        pInfo = device->public.devicePrivate;
    int                 i;
    WiimotePtr          wii = (WiimotePtr)(pInfo->private);


    if (!InitValuatorClassDeviceStruct(device,
                                       WII_NUM_AXES,
                                       GetMotionHistory,
                                       GetMotionHistorySize(),
                                       0))
        return BadAlloc;

    /* The X server doesn't allow mixing of absolute and relative valuators.
     * Depending on whether IR is enabled, we claim to report absolute or
     * relative. Note that all axes except IR are always relative.
     */
    pInfo->dev->valuator->mode = (wii->ir_on) ? Absolute: Relative;

    for (i = 0; i < WII_NUM_AXES; i++)
    {
        /* XXX: clip IR axes here? */
        xf86InitValuatorAxisStruct(device, i, -1, -1, 1, 1, 1);
        xf86InitValuatorDefaults(device, i);
    }

    if (!InitPtrFeedbackClassDeviceStruct(device, WiiPtrCtrl))
        return BadAlloc;

    return Success;
}

static int
_wii_process(InputInfoPtr pInfo)
{
    WiimotePtr  wii                     = (WiimotePtr)pInfo->private;
    int         motion[WII_NUM_AXES]    = {0};
    BOOL        report                  = FALSE;

#define AMAP(field) wii->axismap[WII_##field]

    /* IR data is absolute. XInput specs don't allow mixing absolute and
     * relative values. So the limitation we impose is that IR is enabled,
     * they ALWAYS report as axes 0 and 1. This way we can abuse the specs and
     * send an "incomplete" event with only two valuators (the specs tell us
     * that we always need to report all valuators but I guess noboy will kill
     * us for that).
     *
     * To the user this means that if IR is enabled, it will always control
     * the system cursor. Axis mapping will only kick in from axis 2 onwards.
     * Tough luck.
     */
    if (wii->ir_on)
    {
        /* Testing shows value range is 0 - 0x6FF.
         * The wiimote does not report values between 256 and 512,
         * likewise with 768 and 1024. So we just merge them together.
         */
        if (((wii->ir.ir1_x != wii->wii->ir1.x) ||
            (wii->ir.ir1_y != wii->wii->ir1.y)) &&
                (wii->wii->ir1.x != 0x6FF && wii->wii->ir1.y != 0x6FF))
        {
            wii->ir.ir1_x = wii->wii->ir1.x;
            wii->ir.ir1_y = wii->wii->ir1.y;
            /* x right to left, so swap it back */
            motion[0] = WII_MAX_X - (((wii->ir.ir1_x >> 1) & ~0xFF) |
                (wii->ir.ir1_x & 0xFF));
            motion[1] = (((wii->ir.ir1_y >> 1) & ~0xFF) |
                (wii->ir.ir1_y & 0xFF));

            xf86PostMotionEventP(pInfo->dev, TRUE, 0, 2, motion);

            /* reset so IR axis can be intermixed with other valuators*/
            motion[0] = 0;
            motion[1] = 0;
        }
    }

    if (wii->acc_on)
    {
        if (abs(wii->acc.x - wii->wii->axis.x) > WII_ACCEL_THRESH)
        {
            motion[AMAP(ACC_X)] = (wii->wii->axis.x - wii->acc.x);
            wii->acc.x = wii->wii->axis.x;
            report = TRUE;
        }

        if (abs(wii->acc.y - wii->wii->axis.y) > WII_ACCEL_THRESH)
        {
            motion[AMAP(ACC_Y)] = (wii->acc.y - wii->wii->axis.y);
            wii->acc.y = wii->wii->axis.y;
            report = TRUE;
        }

        if (abs(wii->acc.z - wii->wii->axis.z) > WII_ACCEL_THRESH)
        {
            motion[AMAP(ACC_Z)] = (wii->wii->axis.z - wii->acc.z);
            wii->acc.z = wii->wii->axis.z;
            report = TRUE;
        }
    }

    if (wii->tilt_on)
    {
        if (abs(wii->wii->tilt.x) > WII_TILT_THRESH)
        {
            motion[AMAP(TILT_X)] = wii->wii->tilt.x * WII_TILT_SCALE;
            wii->tilt.x = wii->wii->tilt.x;
            report = TRUE;
        }

        if (abs(wii->wii->tilt.y) > WII_TILT_THRESH)
        {
            motion[AMAP(TILT_Y)] = wii->wii->tilt.y * WII_TILT_SCALE;
            wii->tilt.y = wii->wii->tilt.y;
            report = TRUE;
        }

        if (abs(wii->wii->tilt.z) > WII_TILT_THRESH)
        {
            motion[AMAP(TILT_Z)] = wii->wii->tilt.z * WII_TILT_SCALE;
            wii->tilt.z = wii->wii->tilt.z;
            report = TRUE;
        }
    }

    if (wii->nunchuk_on)
    {
        /* neutral position of the nunchuk joystick is 125/125 */
        if (wii->wii->ext.nunchuk.joyx != wii->nunchuk.joyx)
        {
            motion[AMAP(NUN_JOYX)] = wii->wii->ext.nunchuk.joyx -
                WII_NUNCHUK_NEUTRAL;
            wii->nunchuk.joyx = wii->wii->ext.nunchuk.joyx;
            report = TRUE;
        }

        if (wii->wii->ext.nunchuk.joyy != wii->nunchuk.joyy)
        {
            motion[AMAP(NUN_JOYY)] = wii->wii->ext.nunchuk.joyy -
                WII_NUNCHUK_NEUTRAL;
            wii->nunchuk.joyy = wii->wii->ext.nunchuk.joyy;
            report = TRUE;
        }

        if (abs(wii->wii->ext.nunchuk.axis.x - wii->nunchuk.x)
                > WII_ACCEL_THRESH)
        {
            motion[AMAP(NUN_X)] = (wii->wii->ext.nunchuk.axis.x - wii->nunchuk.x);
            wii->nunchuk.x = wii->wii->ext.nunchuk.axis.x;
            report = TRUE;
        }

        if (abs(wii->wii->ext.nunchuk.axis.y - wii->nunchuk.y)
                > WII_ACCEL_THRESH)
        {
            motion[AMAP(NUN_Y)] = (wii->wii->ext.nunchuk.axis.y - wii->nunchuk.y);
            wii->nunchuk.y = wii->wii->ext.nunchuk.axis.y;
            report = TRUE;
        }

        if (abs(wii->wii->ext.nunchuk.axis.z - wii->nunchuk.z)
                > WII_ACCEL_THRESH)
        {
            motion[AMAP(NUN_Z)] = (wii->wii->ext.nunchuk.axis.z - wii->nunchuk.z);
            wii->nunchuk.z = wii->wii->ext.nunchuk.axis.z;
            report = TRUE;
        }
    }

    if (report)
    {
        xf86PostMotionEventP(pInfo->dev, FALSE, 0, WII_NUM_AXES, motion);
    }

    _wii_process_buttons(pInfo);
    return report;
}


static int
_wii_process_buttons(InputInfoPtr pInfo)
{
    WiimotePtr  wii             = (WiimotePtr)pInfo->private;
    wiimote_keys_t* keys        = &wii->wii->keys;

#define CHECK_SET_POST(field) \
    if ((keys->field != 0) != ((wii->bstate & WII_BT_##field) != 0)) { \
        if (keys->field) wii->bstate |= WII_BT_##field; \
        else wii->bstate &= ~WII_BT_##field; \
        xf86PostButtonEvent(pInfo->dev, FALSE, wii->bmap[WII_##field], \
                keys->field, 0, 0); \
    }

    CHECK_SET_POST(left);
    CHECK_SET_POST(right);
    CHECK_SET_POST(up);
    CHECK_SET_POST(down);
    CHECK_SET_POST(one);
    CHECK_SET_POST(two);
    CHECK_SET_POST(a);
    CHECK_SET_POST(b);
    CHECK_SET_POST(home);
    CHECK_SET_POST(minus);
    CHECK_SET_POST(plus);
    if (wii->nunchuk_on)
    {
        if ((wii->wii->ext.nunchuk.keys.z != 0) !=
                ((wii->bstate & WII_BT_z) != 0))
        {
            if (wii->wii->ext.nunchuk.keys.z) wii->bstate |= WII_BT_z;
            else wii->bstate &= ~WII_BT_z;
            xf86PostButtonEvent(pInfo->dev, FALSE, wii->bmap[WII_z],
                    wii->wii->ext.nunchuk.keys.z, 0, 0);

        }

        if ((wii->wii->ext.nunchuk.keys.c != 0) !=
                ((wii->bstate & WII_BT_c) != 0))
        {
            if (wii->wii->ext.nunchuk.keys.c) wii->bstate |= WII_BT_c;
            else wii->bstate &= ~WII_BT_c;
            xf86PostButtonEvent(pInfo->dev, FALSE, wii->bmap[WII_c],
                    wii->wii->ext.nunchuk.keys.c, 0, 0);

        }
    }

    return 0;
}
