/*
 * Copyright © 2007 Peter Hutterer
 * Copyright © 2008 University of South Australia
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Soft-
 * ware"), to deal in the Software without restriction, including without
 * limitation the rights to use, copy, modify, merge, publish, distribute,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, provided that the above copyright
 * notice(s) and this permission notice appear in all copies of the Soft-
 * ware and that both the above copyright notice(s) and this permission
 * notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABIL-
 * ITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT OF THIRD PARTY
 * RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR HOLDERS INCLUDED IN
 * THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL INDIRECT OR CONSE-
 * QUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFOR-
 * MANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder shall
 * not be used in advertising or otherwise to promote the sale, use or
 * other dealings in this Software without prior written authorization of
 * the copyright holder.
 *
 * Authors:
 *   Peter Hutterer <peter@cs.unisa.edu.au>
 */

/**
 * X input driver for the MERL DiamondTouch.
 * Touch is interpreted as button press, moves as drag motions. If no value
 * above the given threshold is found, a release event is sent.
 *
 *   - supports hotplugging.
 *   - allows configurable hotspot search functions.
 *   - signal threshold configurable per user.
 *   - blob delivery
 *
 * TODO:
 *   - only works for single screen
 *   - device-to-screen coordinate mapping is crude (_dt_scale_to_screen)
 *   - the signal isn't reported as valuator yet.
 *
 * Naming conventions:
 * DtFooBar ... Driver-specific stuff, used by the X interface.
 * _dt_foo_bar ... Driver-internal stuff
 *
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <X11/extensions/XIproto.h>
#include <xf86.h>
#include <xf86_OSlib.h>
#include <xf86Xinput.h>

#include <diamondtouch/ee_defs.h>

#include "diamondtouch.h"

/* module */
static void     DtUnplug(pointer p);
static pointer  DtPlug(pointer        module,
                           pointer        options,
                           int            *errmaj,
                           int            *errmin);

/* driver */
static InputInfoPtr DtPreInit(InputDriverPtr  drv,
                              IDevPtr         dev,
                              int             flags);

static void         DtUnInit(InputDriverPtr   drv,
                             InputInfoPtr     pInfo,
                             int              flags);
static void         DtReadInput(InputInfoPtr  pInfo);
static int          DtControl(DeviceIntPtr    device,
                              int             what);
static void         DtPtrCtrl(DeviceIntPtr    devices,
                              PtrCtrl         *ctrl);

/* internal */
static BOOL _dt_setup_device(InputInfoPtr pInfo, char* path, BOOL probeonly);
static BOOL _dt_setup_read_buffer(InputInfoPtr pInfo);
static int _dt_init_buttons(DeviceIntPtr device);
static int _dt_init_axes(DeviceIntPtr device);
static int _dt_parse_data(InputInfoPtr pInfo);
static int _dt_bounding_box(DtDevicePtr dt, int uid,
                            int *x1, int *y1, int *x2, int *y2);
static void _dt_scale_to_screen(DtDevicePtr dt, int *x, int *y);
static void _dt_parse_supported_options(DtDevicePtr dt, IDevPtr dev, int num);

/* hotspot searching algorithms */
static int _dt_signal_peak(DtDevicePtr dt, int uid, int *x, int *y);
static int _dt_bb_center(DtDevicePtr dt, int uid, int *x, int *y);

#ifdef BLOB_SUPPORT
static int  _dt_init_blob(DeviceIntPtr device, DtDevicePtr dt);
static void _dt_send_blob_events(DtDevicePtr dt, int uid, BlobPtr blobs);

/* blob searching procs */
static BlobPtr _dt_create_blobs(DtDevicePtr dt, int uid);
static int _dt_find_continuing_blobs(DtDevicePtr dt,
                                     int uid,
                                     BlobPtr *known_blobs,
                                     BlobPtr blobs);

static void _dt_free_blob_list(BlobPtr blob, BOOL freeData);
static void _dt_remove_stale_blobs(BlobPtr* blobs);
static BOOL _dt_blobs_are_equal(BlobPtr first, BlobPtr second, int noise);
static void _dt_mark_blob(BlobPtr list, BlobPtr comparer);
#endif

/* Holds all devices. Note that this is only > 2 if you have multiple PHYSICAL
 * DiamondTouch devices attached. Multiple users on one physical will share
 * one DtDeviceRec.
 */
static DtDevicePtr alldevices = NULL;

/* To keep track of already known blobs. */
static unsigned long    next_blobid     = 0;
static unsigned long    iteration       = 0;

static struct _search_funcs {
    char* name;
    int (*searchproc) (DtDevicePtr, int, int*, int*);
} search_funcs [2] = {
    {"SignalPeak", _dt_signal_peak},
    {"BBCenter", _dt_bb_center }
};

/**
 * Driver Rec, fields are used when device is initialised/removed.
 */
_X_EXPORT InputDriverRec DIAMONDTOUCH = {
    1,
    "diamondtouch",
    NULL,
    DtPreInit,
    DtUnInit,
    NULL,
    0
};

/**
 * Module versioning information.
 */
static XF86ModuleVersionInfo DtVersionRec =
{
    "diamondtouch",
    MODULEVENDORSTRING,
    MODINFOSTRING1,
    MODINFOSTRING2,
    XORG_VERSION_CURRENT,
    PACKAGE_VERSION_MAJOR, PACKAGE_VERSION_MINOR, PACKAGE_VERSION_PATCHLEVEL,
    ABI_CLASS_XINPUT,
    ABI_XINPUT_VERSION,
    MOD_CLASS_XINPUT,
    {0, 0, 0, 0}
};

/**
 * Module control. Fields are used when module is loaded/unloaded.
 */
_X_EXPORT XF86ModuleData diamondtouchModuleData =
{
    &DtVersionRec,
    DtPlug,
    DtUnplug
};


/***************************************************************************
 *                            Module procs                                 *
 ***************************************************************************/


/**
 * Called when module is unloaded.
 */
static void
DtUnplug(pointer p)
{
}

/**
 * Called when module is loaded.
 */
static pointer
DtPlug(pointer        module,
           pointer        options,
           int            *errmaj,
           int            *errmin)
{
    xf86AddInputDriver(&DIAMONDTOUCH, module, 0);
    return module;
}

/***************************************************************************
 *                            Driver procs                                 *
 ***************************************************************************/

/**
 * Called each time a new device is to be initialized.
 * This driver allows attaching devices: when the same Device option is given
 * for more than one device section, they will all share the same DtDeviceRec
 * struct.
 */
static InputInfoPtr DtPreInit(InputDriverPtr  drv,
                              IDevPtr         dev,
                              int             flags)
{
    InputInfoPtr        pInfo;
    DtDevicePtr         dt = NULL;
    DtDevicePtr         it;
    char                *devname;
    int                 i, usernum;
    int                 recurse = 0;

    if (!(pInfo = xf86AllocateInput(drv, 0)))
        return NULL;

    pInfo->name         = xstrdup(dev->identifier);
    pInfo->flags        = 0;
    pInfo->type_name    = XI_TOUCHSCREEN;
    pInfo->conf_idev    = dev;
    pInfo->read_input   = DtReadInput; /* new data avl */
    pInfo->switch_mode  = NULL;        /* toggle absolute/relative mode */
    pInfo->device_control = DtControl; /* enable/disable dev */
    pInfo->flags       |= XI86_OPEN_ON_INIT;
    pInfo->flags       |= XI86_CONFIGURED;
    pInfo->fd           = -1;

    xf86CollectInputOptions(pInfo, NULL, NULL);
    xf86ProcessCommonOptions(pInfo, pInfo->options);

    devname = xf86CheckStrOption(dev->commonOptions, "Device", DFLT_DEVICE);

    usernum = xf86CheckIntOption(dev->commonOptions, "UserNumber", -1);

    /* If the same device file is re-used, attach to other device */
    for (it = alldevices; it; it = it->next)
    {
        if (strcmp(it->device, devname) == 0) /* same name */
        {
            if (it->nusers == it->refcnt)
            {
                xf86Msg(X_ERROR, "%s: too many users. Ignoring this user.\n",
                        pInfo->name);
                goto cleanup;
            }

            /* if usernum wasn't specified, find first unused spot. */
            if (usernum == -1)
            {
                for (i = 0; i < it->nusers; i++)
                {
                    if (!it->attached[i] || !it->attached[i]->dev)
                    {
                        usernum = i;
                        break;
                    }
                }
            } else if (usernum >= it->nusers)
            {
                xf86Msg(X_ERROR,
                        "%s: invalid user. Maximum number of users: %d\n",
                        pInfo->name, it->nusers);
                goto cleanup;

            } else if (it->attached[usernum])
            {
                xf86Msg(X_ERROR, "%s: user already attached. Ignoring.\n",
                        pInfo->name);
                goto cleanup;
            }

            xf86Msg(X_INFO,
                    "%s: attaching to existing device as user %d.\n",
                    pInfo->name, usernum);

            it->refcnt++;
            it->attached[usernum] = pInfo;
            pInfo->private = it;

            _dt_parse_supported_options(it, dev, usernum);

            return pInfo;
        }
    }

    if (usernum == -1)
        usernum = 0;

    /* no attachment, create new dtdevice */
    dt = xcalloc(1, sizeof(DtDeviceRec));
    if (!dt)
        goto cleanup;

    pInfo->private      = dt;
    dt->refcnt          = 1;
    dt->device          = devname;

    if (!_dt_setup_device(pInfo, dt->device, FALSE))
        goto cleanup;

    dt->attached        = xcalloc(dt->nusers, sizeof(InputInfoPtr));
    dt->attached[usernum] = pInfo;

    _dt_parse_supported_options(dt, dev, usernum);

    xf86Msg(X_INFO, "%s: Using device %s.\n", pInfo->name, dt->device);
    xf86Msg(X_INFO, "%s: Thresholds: %d (rows), %d (cols)\n",
            pInfo->name,
            dt->users[usernum].row_thresh,
            dt->users[usernum].col_thresh);


    /* all happy, close fd and return */
    close (pInfo->fd);
    pInfo->fd   = -1;
    dt->fd      = -1;
    dt->next    = alldevices;
    alldevices  = dt;

    /* Do we need to create additional devices? */
    /* the hal code in the server doesn't do int options for now, so we need
     * to atoi it manually */
    recurse = atoi(xf86CheckStrOption(dev->commonOptions, "NumDevices", "1"));
    while (--recurse)
    {
        DeviceIntPtr d;
        InputOption options[4];
        char name[512] = {0};
        int ret;

        snprintf(name, sizeof(name) - 1, "%s %d", pInfo->name, recurse);

        /* NIDR uses different options structs than we have in PreInit. Yay */
        options[0].key   = strdup("Device");
        options[0].value = strdup(devname);
        options[0].next  = &options[1];

        options[1].key   = strdup("driver");
        options[1].value = strdup("diamondtouch");
        options[1].next  = &options[2];

        options[2].key   = strdup("name");
        options[2].value = strdup(name);
        options[2].next  = NULL;

        ret = NewInputDeviceRequest(options, &d);
        if (ret != Success)
            xf86Msg(X_ERROR, "%s: Failed to create device %d (error %d)\n",
                    pInfo->name, recurse, ret);
    }


    return pInfo;

cleanup:
    if (dt)
        alldevices = dt->next;

    if (pInfo->fd)
    {
        close(pInfo->fd);
        pInfo->fd = -1;
    }
    if (dt && dt->device)
        xfree(dt->device);
    if (dt)
        xfree(dt);
    if (pInfo)
        xf86DeleteInput(pInfo, 0);
    return NULL;
}

/**
 * Called each time a device is to be removed.
 * Clean up your mess here.
 */
static void DtUnInit(InputDriverPtr drv,
                     InputInfoPtr   pInfo,
                     int            flags)
{
    DtDevicePtr       dt = (DtDevicePtr)pInfo->private;

    if (dt->refcnt == 0)
    {
        if (dt->device)
        {
            xfree(dt->device);
            dt->device = NULL;
        }

        if (dt == alldevices)
            alldevices = dt->next;
        else
        {
            DtDevicePtr it = alldevices;
            while (it->next != dt)
                it = it->next;
            it->next = dt->next;
        }
        xfree(dt);
    }

    if (pInfo->fd > -1)
    {
        close(pInfo->fd);
        pInfo->fd = -1;
    }

    xf86DeleteInput(pInfo, 0);
}

/**
 * Called when data is available on the socket.
 * We always read in a full packet, containing all the info for all users.
 */
static void DtReadInput(InputInfoPtr pInfo)
{
    DtDevicePtr         dt = (DtDevicePtr)pInfo->private;
    int                 err;

    while(xf86WaitForInput(pInfo->fd, 0) > 0)
    {
        err = read(pInfo->fd, dt->buffer, dt->packet_size);
        if (err < 0 && errno == EAGAIN)
            break;
        else if (err < 0)
        {
            int i;
            xf86Msg(X_ERROR, "%s: error reading data (%s).\n", pInfo->name,
                    strerror(errno));
            RemoveEnabledDevice(pInfo->fd);
            close(pInfo->fd);
            for (i = 0; i < dt->nusers; i++)
            {
                if (dt->attached[i])
                {
                    dt->attached[i]->fd = -1;
                    xf86DisableDevice(dt->attached[i]->dev, TRUE);
                }
            }
            return;
        } else if (err < dt->packet_size)
        {
            xf86Msg(X_ERROR, "%s: packet too small. and now?\n", pInfo->name);
            break;
        }

        _dt_parse_data(pInfo);
    }
}

/**
 * Called when the device is to be enabled/disabled, etc.
 * @return Success or X error code.
 */
static int DtControl(DeviceIntPtr    device,
                     int             what)
{
    InputInfoPtr        pInfo = device->public.devicePrivate;
    DtDevicePtr         dt = pInfo->private;
    int                 i;

    switch(what)
    {
        case DEVICE_INIT:
            _dt_init_buttons(device);
            _dt_init_axes(device);
#ifdef BLOB_SUPPORT
            _dt_init_blob(device, dt);
#endif
            break;
        /* Switch device on. Establish socket, start event delivery.  */
        case DEVICE_ON:
            xf86Msg(X_INFO, "%s: On.\n", pInfo->name);
            if (device->public.on)
                break;

            if (dt->fd == -1)
            {
                SYSCALL(pInfo->fd = open(dt->device, O_RDONLY | O_NONBLOCK));
                if (pInfo->fd < 0)
                {
                    xf86Msg(X_ERROR, "%s: cannot open device.\n", pInfo->name);
                    return BadRequest;
                }
                _dt_setup_device(pInfo, dt->device, TRUE);

                AddEnabledDevice(pInfo->fd);
                dt->fd = pInfo->fd;
            }
            pInfo->fd = dt->fd;
            device->public.on = TRUE;
            break;
        /**
         * Shut down device.
         */
        case DEVICE_OFF:
            {
                xf86Msg(X_INFO, "%s: Off.\n", pInfo->name);
                if (!device->public.on)
                    break;

                pInfo->fd = -1;
                /* if no other device uses the FD anymore, close it */
                for (i = 0; i < dt->nusers; i++)
                {
                    if (dt->attached[i] && dt->attached[i]->fd != -1)
                        break;
                }
                if (i == dt->nusers)
                {
                    RemoveEnabledDevice(dt->fd);
                    close(dt->fd);
                }

                device->public.on = FALSE;
            }
            break;
        case DEVICE_CLOSE:
            for (i = 0; i < dt->nusers; i++)
            {
                if (dt->attached[i] &&
                        dt->attached[i]->dev == device)
                {
                    dt->attached[i] = NULL;
                    break;
                }
            }
            /* free what we have to free */
            if (--dt->refcnt == 0)
                xfree(dt->buffer);
            break;
    }
    return Success;
}

static void
DtPtrCtrl(DeviceIntPtr device,  PtrCtrl *ctrl)
{
    /* dix takes care of everything */
}


/***************************************************************************
 *                            internal procs                               *
 ***************************************************************************/

/**
 * Parse device options as given to us by the kernel.
 *
 * This function will print out the error message to the log.
 *
 * @param probeonly Only test ioctl call, don't read all parameters.
 * @return TRUE on success or False otherwise.
 */
static BOOL
_dt_setup_device(InputInfoPtr pInfo, char* path, BOOL probeonly)
{
    DtDevicePtr         dt = (DtDevicePtr)pInfo->private;
    unsigned char       ee_data[64];
    unsigned char*      orientation;

    SYSCALL(pInfo->fd = open(dt->device, O_RDONLY | O_NONBLOCK));
    if (pInfo->fd == -1)
    {
        xf86Msg(X_ERROR, "%s: failed to open %s.\n",
                pInfo->name, dt->device);
        return FALSE;
    }


    if ((ioctl(pInfo->fd, DT_IOCGPARAMS, ee_data)) < 0)
    {
        xf86Msg(X_ERROR, "%s: ioctl failed (%s).\n", pInfo->name,
                strerror(errno));
        return FALSE;
    }

    if (probeonly)
        return TRUE;

    dt->pic_version = ee_data[DT_EE_PIC_FIRMWARE_VERSION];

    dt->nusers      = ee_data[DT_EE_NUM_SUPPORTED_USERS];
    dt->ncols       = ee_data[DT_EE_NUM_COLUMNS_MSB] << 8 |
                      ee_data[DT_EE_NUM_COLUMNS_LSB];
    dt->nrows       = ee_data[DT_EE_NUM_ROWS_MSB] << 8 |
                      ee_data[DT_EE_NUM_ROWS_LSB];
    dt->row_w       = (ee_data[DT_EE_ROW_WIDTH_MSB] << 8 |
                       ee_data[DT_EE_ROW_WIDTH_LSB]) / 1.0e5;
    dt->col_w       = (ee_data[DT_EE_COLUMN_WIDTH_MSB] << 8 |
                       ee_data[DT_EE_COLUMN_WIDTH_LSB]) / 1.0e5;

    orientation     = &ee_data[DT_EE_ROW_COLUMN_ORIENTATION];

    dt->rows_first  = ((*orientation & DT_EE_ROWS_FIRST) == DT_EE_ROWS_FIRST);
    dt->left2right  = ((*orientation & DT_EE_COLUMNS_LEFT_TO_RIGHT) == DT_EE_COLUMNS_LEFT_TO_RIGHT);
    dt->top2bottom  = ((*orientation & DT_EE_ROWS_BOTTOM_TO_TOP) != DT_EE_ROWS_BOTTOM_TO_TOP);

    dt->tbu    = (ee_data[DT_EE_UPDATE_PERIOD_MSB] << 8 |
                  ee_data[DT_EE_UPDATE_PERIOD_LSB]) / 1.0e5;

    dt->packet_size = 4 + dt->nusers * (dt->nrows + dt->ncols);

    /* shield drive only supported from PIC 2. Adds some extra bytes to the
     * packet (one byte per user). */
    if (dt->pic_version >= 2)
    {
        if (ee_data[DT_EE_SHIELD_DRV_FLAGS] & DT_EE_SHIELD_DRV_ON)
        {
            dt->shield_drv_on = TRUE;
            dt->packet_size  += dt->nusers;
        }
    }

    return _dt_setup_read_buffer(pInfo);
}

/**
 * set up buffer and the respective pointers into the buffer
 *
 * The packet format coming from the DT is
 * 4 bytes header
 * nrows bytes row data for user 1
 * ncols bytes column data for user 1
 * [one byte shield drive data for user 1, if applicable]
 * nrows bytes row data for user 2
 *      etc.
 *
 * There's one global read buffer (dt->buffer), and each dtuser points into
 * the respective areas into this read buffer.
 */
static BOOL
_dt_setup_read_buffer(InputInfoPtr pInfo)
{
    DtDevicePtr dt      = (DtDevicePtr)pInfo->private;
    DtUserPtr   dtuser;
    int         i;

    dt->buffer = xcalloc(dt->packet_size, sizeof(char));
    if (!dt->buffer)
        goto cleanup;

    dt->users  = xcalloc(dt->nusers, sizeof(DtUserRec));
    if (!dt->users)
        goto cleanup;

    for (dtuser = dt->users, i = 0; i < dt->nusers; i++, dtuser++)
    {
        dtuser->uid = i;
        if (dt->rows_first)
        {
            dtuser->rows   = dt->buffer + 4 + i * (dt->nrows + dt->ncols);
            dtuser->cols   = dtuser->rows + dt->nrows;
            dtuser->shield = dtuser->cols + dt->ncols;
        } else
        {
            dtuser->cols   = dt->buffer + 4 + i * (dt->nrows + dt->ncols);
            dtuser->rows   = dtuser->cols + dt->ncols;
            dtuser->shield = dtuser->rows + dt->nrows;
        }
    }

    return TRUE;

cleanup:
    if (dt->buffer)
        xfree(dt->buffer);
    if (dt->users)
        xfree(dt->users);

    xf86Msg(X_ERROR, "%s: Failed to set up read buffers.\n", pInfo->name);
    return FALSE;
}


/**
 * Init the button map for the dt device.
 * @return Success or X error code on failure.
 */
static int
_dt_init_buttons(DeviceIntPtr device)
{
    InputInfoPtr        pInfo = device->public.devicePrivate;
    DtDevicePtr         dt = (DtDevicePtr)pInfo->private;
    CARD8               *map;
    int                 i;
    int                 ret = Success;

    map = xcalloc(dt->buttons, sizeof(CARD8));

    for (i = 0; i < dt->buttons; i++)
        map[i] = i;

    if (!InitButtonClassDeviceStruct(device, dt->buttons, map)) {
        xf86Msg(X_ERROR, "%s: Failed to register buttons.\n", pInfo->name);
        ret = BadAlloc;
    }

    xfree(map);
    return ret;
}


/**
 * Init the valuators for the dt device.
 * Only absolute mode is supported.
 * @return Success or X error code on failure.
 */
static int
_dt_init_axes(DeviceIntPtr device)
{
    InputInfoPtr        pInfo = device->public.devicePrivate;
    int                 i;
    DtDevicePtr         dt = (DtDevicePtr)pInfo->private;

    if (!InitValuatorClassDeviceStruct(device,
                                       dt->axes,
#if GET_ABI_MAJOR(ABI_XINPUT_VERSION) < 3
                                       GetMotionHistory,
#endif
                                       GetMotionHistorySize(),
                                       0))
        return BadAlloc;

    pInfo->dev->valuator->mode = Relative;
    if (!InitAbsoluteClassDeviceStruct(device))
        return BadAlloc;

    for (i = 0; i < dt->axes; i++)
    {
        xf86InitValuatorAxisStruct(device, i, -1, -1, 1, 1, 1);
        xf86InitValuatorDefaults(device, i);
    }

    if (!InitPtrFeedbackClassDeviceStruct(device, DtPtrCtrl))
        return BadAlloc;

    return Success;
}

#ifdef BLOB_SUPPORT
/**
 * Init the blob struct for the dt_device.
 * @return Success or X error code on failure.
 */
static int
_dt_init_blob(DeviceIntPtr device, DtDevicePtr dt)
{
    AxisInfo axes[2];

    axes[0].resolution = 100; /* XXX: no idea what the dpi rate is */
    axes[0].min_value = 0;
    axes[0].max_value = dt->ncols;

    axes[1].resolution = 100; /*XXX: */
    axes[1].min_value = 0;
    axes[1].max_value = dt->nrows;

    if (!InitBlobClassDeviceStruct(device,
                                   0, /* num buttons */
                                   BC_Multiblob,
                                   axes,
                                   0, /* num_identifiers */
                                   NULL))
        return BadAlloc;

    return Success;
}
#endif

/**
 * Parse the data packet coming from the kernel.
 *
 * Searches for the bounding box for each user and then posts a button event
 * with the hotspot in the center of the bounding box.
 */
static int
_dt_parse_data(InputInfoPtr pInfo)
{
    DtDevicePtr         dt = (DtDevicePtr)pInfo->private;
    int                 uid = 0;
    int                 hotspot_x;
    int                 hotspot_y;

    iteration++;
    for (uid = 0; uid < dt->nusers; uid++)
    {
        if (!dt->attached[uid] ||
                !dt->attached[uid]->dev ||
                !dt->attached[uid]->dev->public.on)
            continue;

#ifdef BLOB_SUPPORT
        /* blob event delivery */
        if (dt->users[uid].deliver_blobs)
        {
            BlobPtr blobs = _dt_create_blobs(dt, uid);
            _dt_find_continuing_blobs(dt,
                                      uid,
                                      &dt->users[uid].known_blobs,
                                      blobs);
            _dt_send_blob_events(dt, uid, dt->users[uid].known_blobs);
            _dt_remove_stale_blobs(&dt->users[uid].known_blobs);
            _dt_free_blob_list(blobs, TRUE);
        } else /* pure mouse emulation */
#endif
        {
            if (!dt->users[uid].search_proc)
                continue;

            if (dt->users[uid].search_proc(dt, uid, &hotspot_x, &hotspot_y))
            {
                _dt_scale_to_screen(dt, &hotspot_x, &hotspot_y);

                if (dt->users[uid].last_x != hotspot_x ||
                        dt->users[uid].last_y != hotspot_y)
                {
                    xf86PostMotionEvent(dt->attached[uid]->dev,
                            TRUE, /*is absolute */
                            0,
                            dt->axes,
                            hotspot_x,
                            hotspot_y);
                    dt->users[uid].last_x = hotspot_x;
                    dt->users[uid].last_y = hotspot_y;
                }

                if (!dt->users[uid].is_down)
                {
                    xf86PostButtonEvent(dt->attached[uid]->dev,
                            FALSE,
                            1, /* button 1 */
                            TRUE,
                            0,
                            0);
                    dt->users[uid].is_down = TRUE;
                    dt->users[uid].last_x = hotspot_x;
                    dt->users[uid].last_y = hotspot_y;
                }
            } else if (dt->users[uid].is_down)
            {
                xf86PostButtonEvent(dt->attached[uid]->dev,
                        FALSE,
                        1, /* button 1 */
                        FALSE,
                        0,
                        0);
                dt->users[uid].is_down = FALSE;
            }
        }
    }
    return 0;
}

/**
 * Find the biggest bounding box that encloses all points above threshold.
 * If there are two spots separated by a number of pixels, the bounding box
 * will go around both of them.
 * Parameters are changed in place.
 */
static int
_dt_bounding_box(DtDevicePtr dt, int uid, int *x1, int *y1, int *x2, int *y2)
{
    DtUserPtr   dtuser = &dt->users[uid];
    int         i;

    *x1 = *y1 = -1;

    for (i = 0; i < dt->nrows; i++)
    {
        if (dtuser->rows[i] >= dt->users[uid].row_thresh)
        {
            if (*y1 == -1)
                *y1 = i;
            *y2 = i;
        }
    }

    for (i = 0; i < dt->ncols; i++)
    {
        if (dtuser->cols[i] >= dt->users[uid].col_thresh)
        {
            if (*x1 == -1)
                *x1 = i;
            *x2 = i;
        }
    }
    return 0;
}

/**
 * Scale from DT coordinates to screen coordinates.
 * Parameters modified in places.
 * Only works for the first screen, very crude mapping.
 */
static void
_dt_scale_to_screen(DtDevicePtr dt, int *x, int *y)
{
    *x = (*x) * (1.0 * screenInfo.screens[0]->width/dt->ncols);
    *y = (*y) * (1.0 * screenInfo.screens[0]->height/dt->nrows);

    if (!dt->left2right)
        *x = screenInfo.screens[0]->width - *x;
    if (!dt->top2bottom)
        *y = screenInfo.screens[0]->height - *y;
}

/**
 * Parse DT-specific options. Note that the parameters are user-specific, and
 * do not affect the device settings.
 * @param dt The device to configure.
 * @param dev The device an configuration as given by the DDX.
 * @param usernum The user number (essentially the DT pad). 0 to dt->nusers - 1.
 */
static void
_dt_parse_supported_options(DtDevicePtr dt, IDevPtr dev, int usernum)
{
    int i;
    char* search_algo = xf86CheckStrOption(dev->commonOptions,
                                           "SearchFunction",
                                           DFLT_SEARCHFUNC);

    for (i = 0; i < sizeof(search_funcs)/sizeof(struct _search_funcs); i++)
    {
        if (!strcasecmp(search_algo, search_funcs[i].name))
        {
            dt->users[usernum].search_proc = search_funcs[i].searchproc;
            break;
        }
    }
    xfree(search_algo);

    dt->users[usernum].row_thresh =
        xf86CheckIntOption(dev->commonOptions, "RowThreshold",
                                         DFLT_ROW_THRESH);
    dt->users[usernum].col_thresh =
        xf86CheckIntOption(dev->commonOptions, "ColThreshold",
                                         DFLT_COL_THRESH);
#ifdef BLOB_SUPPORT
    dt->users[usernum].move_thresh =
        xf86CheckIntOption(dev->commonOptions, "MoveThreshold",
                                         DFLT_MOVE_THRESH);
    dt->users[usernum].noiselimit =
        xf86CheckIntOption(dev->commonOptions, "NoiseLimit",
                                         DFLT_NOISELIMIT);

    dt->users[usernum].deliver_blobs =
        xf86CheckBoolOption(dev->commonOptions, "DeliverBlobs",
                                         DFLT_DELIVER_BLOB);
#endif

    /* global device options, are hardcoded anyway */

    /* if ReportValues is set, the third axis is the data.
       TODO: well, eventually anyway. */
    dt->axes = 2;
    if (xf86CheckBoolOption(dev->commonOptions, "ReportValues", FALSE))
                dt->axes = 3;
    dt->buttons = 2;
}


/*************************************************************************
 *                     Hotspot search functions                          *
 *************************************************************************/

/**
 * Find bounding box center.
 * x/y are modified in place.
 * @param x x coordinate of hotspot in device coordinates (cols)
 * @param y x coordinate of hotspot in device coordinates (rows)
 * @return 0 if no hotspot was found, 1 otherwise.
 */
static int
_dt_bb_center(DtDevicePtr dt, int uid, int *x, int *y)
{
    int x1, x2, y1, y2;
    _dt_bounding_box(dt, uid, &x1, &y1, &x2, &y2);

    if (x1 == -1 || y1 == -1)
        return 0;

    *x = x1 + (x2 - x1)/2;
    *y = y1 + (y2 - y1)/2;
    return 1;
}

/**
 * Find the strongest signal.
 * x/y are modified in place.
 * @param x x coordinate of hotspot in device coordinates (cols)
 * @param y x coordinate of hotspot in device coordinates (rows)
 * @return 0 if no hotspot was found, 1 otherwise.
 */
static int
_dt_signal_peak(DtDevicePtr dt, int uid, int *x, int *y)
{
    int max = -1;
    int i;

    for (i = 0; i < dt->ncols; i++)
    {
        if (dt->users[uid].cols[i] > dt->users[uid].col_thresh)
        {
            max = dt->users[uid].cols[i];
            *x = i;
        }
    }

    if (max == -1)
        return 0;
    max = -1;

    for (i = 0; i < dt->nrows; i++)
    {
        if (dt->users[uid].rows[i] > dt->users[uid].row_thresh)
        {
            max = dt->users[uid].rows[i];
            *y = i;
        }
    }

    return (max != -1);
}

#ifdef BLOB_SUPPORT
/****************************************************************************
 *                             Blob support                                 *
 ****************************************************************************/


/**
 * Create a blob for a touch area.
 * The way how the diamondtouch is designed, we can't actually tell where
 * exactly a touchpoint happened if we have more than one touchpoint by one
 * user.
 * What we do here is simply creating a new blob for each area where there
 * could be a possible touchpoint.
 *
 * Caller is responsible for freeing memory.
 *
 * @return The list of blobs found
 */
static BlobPtr
_dt_create_blobs(DtDevicePtr dt, int uid)
{
    unsigned char* data;
    int         x       = 0;
    int         y       = 0;
    BlobPtr     blobs   = NULL,
                current = NULL;
    int         nblobs  = 0;

    Range       *xranges = NULL;
    int         nxranges = 0;
    BOOL        inXrange = FALSE;

    Range       *yranges = NULL;
    int         nyranges = 0;
    BOOL        inYrange = FALSE;

    data = dt->users[uid].cols;
    for (x = 0; x < dt->ncols; x++)
    {
        if (data[x] > dt->users[uid].col_thresh)
        {
            if (!inXrange)
            {
                xranges = xrealloc(xranges, (++nxranges) * sizeof(Range));
                xranges[nxranges - 1].min = x;
                inXrange = TRUE;
            }
            xranges[nxranges - 1].max = x;
        } else
        {
            if (inXrange)
                inXrange = FALSE;
        }
    }

    data = dt->users[uid].rows;
    for (y = 0; y < dt->nrows; y++)
    {
        if (data[y] > dt->users[uid].row_thresh)
        {
            if (!inYrange)
            {
                yranges = xrealloc(yranges, (++nyranges) * sizeof(Range));
                yranges[nyranges - 1].min = y;
                inYrange = TRUE;
            }
            yranges[nyranges - 1].max = y;
        } else
        {
            if (inYrange)
                inYrange = FALSE;
        }
    }

    /* create a blob for each area */
    for (x = 0; x < nxranges; x++)
    {
        for (y = 0; y < nyranges; y++)
        {
            current = xcalloc(1, sizeof(BlobRec));
            current->device     = dt->attached[uid]->dev->id;
            current->subdevice  = nblobs;
            current->blobid     = 0;
            current->next       = blobs;
            blobs               = current;

            /* bounding box */
            current->x1         = xranges[x].min;
            current->x2         = xranges[x].max;
            current->y1         = yranges[y].min;
            current->y2         = yranges[y].max;
            _dt_scale_to_screen(dt, &current->x1, &current->y1);
            _dt_scale_to_screen(dt, &current->x2, &current->y2);

            if (!dt->left2right)
            {
                int swap = current->x1;
                current->x1 = current->x2;
                current->x2 = swap;
            }
            if (!dt->top2bottom)
            {
                int swap = current->y1;
                current->y1 = current->y2;
                current->y2 = swap;
            }

            /* hotspot */
            current->hsx        = current->x1 + (current->x2 - current->x1)/2;
            current->hsy        = current->y1 + (current->y2 - current->y1)/2;

            /* data bitmap */
            current->datalen    = ((current->x2 - current->x1)
                                    * (current->y2 - current->y1) + 7)/8;
            if (current->datalen == 0)
                current->datalen = 1;
            current->data       = (char*)xcalloc(current->datalen,
                                                 sizeof(char));

            /*
             * The DT hardware doesn't really allow true blob detection. We
             * get x/y values, but those don't really allow us to get a real
             * blob.
             * So we might as well skip all the work and just memset a square
             * bitmap.
             */
            memset(current->data, 0xff, current->datalen);
        }
    }

    return blobs;
}


/**
 * Compare the list of blobs with a stored list of blobs.
 *
 * If a blob in known_blobs is found that is within the threshold area of a
 * given blob, it's data is updated and we assume it's a continuation.
 *
 * If not, the new blob is prepended to the list.
 *
 * @param known_blobs will be modified to reflect an updated list.
 */
static int
_dt_find_continuing_blobs(DtDevicePtr dt,
                          int uid,
                          BlobPtr *known_blobs,
                          BlobPtr blobs)
{
    int         mvthr   = dt->users[uid].move_thresh;
    int         noise   = dt->users[uid].noiselimit;
    BlobPtr     current = NULL,
                known   = NULL,
                tmp     = NULL;

    /* if there's only one known blob and only one new blob, we assume it's a
       continuation */

    if (*known_blobs && blobs &&
        (*known_blobs)->next == NULL &&
         blobs->next == NULL)
    {
        BlobPtr known = *known_blobs;
        known->iteration = iteration;
        known->blobid |= BlobContinue;

        if (!_dt_blobs_are_equal(known, blobs, noise))
        {
            known->x1 = blobs->x1;
            known->y1 = blobs->y1;
            known->x2 = blobs->x2;
            known->y2 = blobs->y2;
            known->hsx = blobs->hsx;
            known->hsy = blobs->hsy;
            known->datalen = blobs->datalen;
            xfree(known->data);
            known->data = xalloc(known->datalen);
            memcpy(known->data, blobs->data, known->datalen);
            known->dirty = TRUE;
        }
        return 0;
    }

    current = blobs;
    while (current)
    {
        /* search for previous instance of blob */
        known = *known_blobs;
        while(known)
        {
            if (    current->marked <= 0 &&
                    known->x1 - mvthr < current->x1 &&
                    known->x1 + mvthr > current->x1 &&
                    known->y1 - mvthr < current->y1 &&
                    known->y2 + mvthr > current->y2)
            {
                /* seems to be the same, update known blob. */
                known->iteration = iteration;
                known->blobid |= BlobContinue;

                /* check if we actually need to update the blob */
                if (_dt_blobs_are_equal(current, known, noise))
                    known->dirty = FALSE;
                else
                {
                    known->x1 = current->x1;
                    known->x2 = current->x2;
                    known->y1 = current->y1;
                    known->y2 = current->y2;
                    known->hsx = current->hsx;
                    known->hsy = current->hsy;
                    known->datalen = current->datalen;
                    xfree(known->data);
                    known->data = xalloc(known->datalen);
                    memcpy(known->data, current->data, known->datalen);
                    known->dirty = TRUE;
                }

                /* mark all other blobs with the same x/y range as potential
                 * duplicates */
                _dt_mark_blob(blobs, current);
                current->marked = -1;

                break;
            }
            known = known->next;
        }
        current = current->next;
    }

    /* in the first loop, we marked all the ones that are potential duplicates
       of our known blobs. So we may have blobs left that are unmarked.
       Those ones are ones we haven't known yet. Each blob is added, others in
       the same x/y range are marked. Only unmarked ones are added.
     */

    current = blobs;
    while(current)
    {
        /* unknown blob, prepend to list */
        if (current->marked == 0)
        {
            tmp                 = xcalloc(1, sizeof(BlobRec));
            *tmp                = *current;
            tmp->next           = *known_blobs;
            tmp->iteration      = iteration;
            tmp->blobid         = (++next_blobid % BLOBID_MASK);
            tmp->dirty          = TRUE;
            tmp->data           = xalloc(current->datalen);
            memcpy(tmp->data, current->data, current->datalen);
            *known_blobs        = tmp;
            _dt_mark_blob(blobs, tmp);
        }
        current = current->next;
    }

    return 0;
}

static void
_dt_free_blob_list(BlobPtr blob, BOOL freeData)
{
    BlobPtr next;

    while(blob)
    {
        next = blob->next;

        if (freeData)
            xfree(blob->data);
        xfree(blob);

        blob = next;
    }
}

/**
 * Remove stale blobs, i.e. blobs that weren't updated in the last iteration.
 * @param blobs the list to be modified in place.
 */
static void
_dt_remove_stale_blobs(BlobPtr* blobs)
{
    BlobPtr current, prev;

    current = *blobs;
    prev = NULL;

    while(current)
    {
        if (current->iteration != iteration)
        {
            if (!prev)
                *blobs = current->next;
            else
                prev->next = current->next;

            xfree(current->data);
            xfree(current);
            if (!prev)
                current =  *blobs;
            else
                current = prev->next;

        } else
        {
            prev = current;
            current = current->next;
        }
    }
}

/**
 * Send an event for each blob in the list.
 *
 * X applications don't like repeating press/release pointer events from the
 * same device and button. We can only let one blob at a time emulate the
 * mouse, otherwise the apps will be very unhappy.
 * The first blob is automatically picked for pointer emulation. Once it
 * disappears, the next blob to newly appear will emulate the pointer. Rinse,
 * wash, repeat.
 * Blobs that start to appear while another blob already emulates the mouse
 * will not cause mouse emulation.
 */
static void
_dt_send_blob_events(DtDevicePtr dt, int uid, BlobPtr blobs)
{
    BlobPtr     current         = blobs;
    int         dlen;
    char*       data;
    BOOL        emulatePointer;
        /* primary_id is the blob that sends mouse events */
    static long primary_id      = ~BLOBID_MASK;
    static blobEvent event;

    while(current)
    {
        emulatePointer = FALSE;
        if (current->iteration != iteration)
        {
            current->blobid &= BLOBID_MASK;
            current->blobid |= BlobStop;

            dlen = 0;
            data = NULL;
            current->dirty = TRUE;
        } else
        {
            if ((primary_id == ~BLOBID_MASK) &&
                 !(current->blobid & BlobContinue))
            {
                primary_id = current->blobid;
            }

            dlen = current->datalen;
            data = current->data;
        }

        if (current->dirty)
        {
            if ((current->blobid & BLOBID_MASK) == primary_id)
                emulatePointer = TRUE;

            event.blobid = current->blobid;
            event.hot_x_root = current->hsx;
            event.hot_y_root = current->hsy;
            event.bb_x1_root = current->x1;
            event.bb_y1_root = current->y1;
            event.bb_x2_root = current->x2;
            event.bb_y2_root = current->y2;
            event.elevation  = 0;
            event.rotation   = 0;
            event.intensity  = 0;
            event.format     = BlobFormatMonochrome;
            event.bits_per_pixel = 1;

            xf86PostBlobEvent(dt->attached[uid]->dev,
                    emulatePointer,
                    1, /* button */
                    &event,
                    data,
                    NULL);
            current->dirty = FALSE;

            if ((current->blobid & BlobStop) &&
                ((current->blobid & BLOBID_MASK) == primary_id))
            {
                primary_id = ~BLOBID_MASK;
            }
        }


        current = current->next;
    }
}

static BOOL
_dt_blobs_are_equal(BlobPtr first, BlobPtr second, int noise)
{
    return (first->x1 > second->x1 - noise &&
            first->x1 < second->x1 + noise &&
            first->y1 > second->y1 - noise &&
            first->y1 < second->y1 + noise &&
            first->x2 > second->x2 - noise &&
            first->x2 < second->x2 + noise &&
            first->y2 > second->y2 - noise &&
            first->y2 < second->y2 + noise &&
            first->hsx > second->hsx - noise &&
            first->hsx < second->hsx + noise &&
            first->hsy > second->hsy - noise &&
            first->hsy < second->hsy + noise &&
            second->datalen == first->datalen &&
            memcmp(second->data, first->data, second->datalen) == 0);
}

/**
 * Mark all those blobs in list with a x or y range equal to the given blob.
 */
static void
_dt_mark_blob(BlobPtr list, BlobPtr comparer)
{
    BlobPtr marker = list;
    while(marker)
    {
        if ((marker->x1 == comparer->x1
                    && marker->x2 == comparer->x2) ||
                ( marker->y1 == comparer->y1
                  && marker->y2 == comparer->y2))
        {
            if (marker->marked == -1)
                ErrorF("This should not happen.\n");
            else
                marker->marked++;
        }
        marker = marker->next;
    }
}
#endif
