/* Sylph-Searcher - full-text search program for Sylpheed
 * Copyright (C) 2007 Sylpheed Development Team
 */

#include <sylph/sylmain.h>
#include <sylph/prefs_common.h>
#include <sylph/account.h>
#include <sylph/folder.h>
#include <sylph/procmsg.h>
#include <sylph/procheader.h>
#include <sylph/codeconv.h>
#include <sylph/utils.h>

#include <string.h>
#include <time.h>
#include <libpq-fe.h>
#include <mecab.h>

#include "common.h"

#define PG_CONN(dbinfo) ((PGconn *)dbinfo->dbdata)

#ifdef G_OS_WIN32
static CharSet mecab_encoding = C_SHIFT_JIS;
#else
static CharSet mecab_encoding = C_EUC_JP;
#endif

gchar *sql_escape_str(DBInfo *conn, const gchar *str)
{
	gchar *esc_str;
	size_t len;

	if (!str)
		return g_strdup("");

	len = strlen(str);
	esc_str = g_malloc(len * 2 + 1);
	esc_str[0] = '\0';
	PQescapeStringConn(PG_CONN(conn), esc_str, str, len, NULL);

	return esc_str;
}

#define MAXWAKACHILEN 128

gchar *get_wakachi_text(const gchar *str)
{
	gchar buf[MAXWAKACHILEN + 1];
	mecab_t *mecab;
	const mecab_node_t *node;
	gchar *mecabenc_str;
	gchar *utf8_str;
	GString *string;
	gint len;

	mecab = mecab_new2("");
	if (!mecab) {
		g_warning("mecab_new2 returned error");
		exit(1);
	}

	if (mecab_encoding == C_EUC_JP)
		mecabenc_str = conv_codeset_strdup(str, CS_UTF_8, CS_EUC_JP);
	else if (mecab_encoding == C_SHIFT_JIS || mecab_encoding == C_CP932)
		mecabenc_str = conv_codeset_strdup(str, CS_UTF_8, CS_CP932);
	else
		mecabenc_str = g_strdup(str);

	//node = mecab_sparse_tonode(mecab, mecabenc_str);
	node = mecab_sparse_tonode2(mecab, mecabenc_str,
				    strlen(mecabenc_str) + 1);

	string = g_string_new("");

	for (; node; node = node->next) {
		if (node->stat != MECAB_BOS_NODE &&
		    node->stat != MECAB_EOS_NODE) {
			len = MIN(MAXWAKACHILEN, node->length);
			if (node->length > MAXWAKACHILEN) {
				g_warning("wakachi: node->length(%d) too long. max: %d", node->length, MAXWAKACHILEN);
				continue;
			}

			strncpy(buf, node->surface, len);
			*(buf + len) = '\0';

			if (*buf != '\0') {
				if (string->len > 0)
					g_string_append_c(string, ' ');
				g_string_append(string, buf);
			}
		}
	}

	g_free(mecabenc_str);
	mecab_destroy(mecab);

	if (mecab_encoding == C_EUC_JP)
		utf8_str = conv_codeset_strdup(string->str, CS_EUC_JP, CS_UTF_8);
	else if (mecab_encoding == C_SHIFT_JIS || mecab_encoding == C_CP932)
		utf8_str = conv_codeset_strdup(string->str, CS_CP932, CS_UTF_8);
	else
		utf8_str = g_strdup(string->str);

	g_string_free(string, TRUE);
	return utf8_str;
}

static void usage(const gchar *cmdname)
{
	g_print("Usage: %s [-d dbname] [-h hostname] [-p port] [-U username] [-P password] [--mecab-encoding encoding] [-nrv] ...\n", cmdname);
	exit(1);
}

gint parse_cmdline(gint argc, gchar *argv[],
		   gchar **dbname, gchar **hostname, gushort *port,
		   gchar **user, gchar **pass)
{
	gint i;
	gchar *p;

	if (dbname)
		*dbname = NULL;
	if (hostname)
		*hostname = NULL;
	if (port)
		*port = 0;
	if (user)
		*user = NULL;
	if (pass)
		*pass = NULL;

	for (i = 1; i < argc; i++) {
		p = argv[i];
		if (*p == '-') {
			if (i + 1 == argc)
				usage(argv[0]);

			if (*(p + 1) == 'd')
				*dbname = argv[i + 1];
			else if (*(p + 1) == 'h')
				*hostname = argv[i + 1];
			else if (*(p + 1) == 'p')
				*port = atoi(argv[i + 1]);
			else if (*(p + 1) == 'U')
				*user = argv[i + 1];
			else if (*(p + 1) == 'P')
				*pass = argv[i + 1];
			else if (*(p + 1) == 'n' ||
				 *(p + 1) == 'r' ||
				 *(p + 1) == 'v')
				continue;
			else if (*(p + 1) == '-') {
				if (!strncmp(p + 2, "mecab-encoding", 14)) {
					CharSet charset;
					charset = conv_get_charset_from_str(argv[i + 1]);
					if (charset != C_AUTO) {
						g_print("mecab_encoding = %s\n", argv[i + 1]);
						mecab_encoding = charset;
					}
				}
			} else
				usage(argv[0]);

			i++;
		} else
			break;
	}

	return i;
}

gboolean cmdline_has_option(gint argc, gchar *argv[], const gchar *opt)
{
	gint i;

	for (i = 1; i < argc; i++) {
		if (!strcmp(argv[i], opt))
			return TRUE;
	}

	return FALSE;
}

static AppConfig tmp_config;

static PrefParam param[] = {
	{"dbname", NULL, &tmp_config.dbname, P_STRING},
	{"hostname", NULL, &tmp_config.hostname, P_STRING},
	{"port", "0", &tmp_config.port, P_USHORT},
	{"user", NULL, &tmp_config.user, P_STRING},
	{"pass", NULL, &tmp_config.pass, P_STRING},
	{"res_limit", "0", &tmp_config.res_limit, P_INT},
	{NULL, NULL, NULL, P_OTHER}
};

gint read_config(AppConfig *config)
{
	gchar *path;

	g_return_val_if_fail(config != NULL, -1);

	memset(&tmp_config, 0, sizeof(tmp_config));

	path = g_strconcat(get_rc_dir(), G_DIR_SEPARATOR_S, "searcherrc", NULL);
	if (!is_file_exist(path)) {
		g_free(path);
		return -1;
	}
	prefs_read_config(param, "Searcher", path, NULL);
	g_free(path);

	*config = tmp_config;
	memset(&tmp_config, 0, sizeof(tmp_config));

	return 0;
}

gint write_config(AppConfig *config)
{
	g_return_val_if_fail(config != NULL, -1);

	tmp_config = *config;
	prefs_write_config(param, "Searcher", "searcherrc");
	memset(&tmp_config, 0, sizeof(tmp_config));

	return 0;
}

DBInfo *db_connect(const gchar *dbname, const gchar *hostname, gushort port,
		   const gchar *user, const gchar *pass)
{
	DBInfo *dbinfo;
	PGconn *conn;
	GString *conninfo;

	conninfo = g_string_new("");
	if (dbname && *dbname)
		g_string_append_printf(conninfo, "dbname = '%s' ", dbname);
	if (hostname && *hostname)
		g_string_append_printf(conninfo, "host = '%s' ", hostname);
	if (port > 0)
		g_string_append_printf(conninfo, "port = %d ", port);
	if (user && *user)
		g_string_append_printf(conninfo, "user = '%s' ", user);
	if (pass && *pass)
		g_string_append_printf(conninfo, "password = '%s' ", pass);

	conn = PQconnectdb(conninfo->str);
	dbinfo = g_new(DBInfo, 1);
	dbinfo->dbdata = conn;
	g_string_free(conninfo, TRUE);

	if (PQstatus(conn) != CONNECTION_OK) {
		db_error_message(dbinfo, "connection to database failed");
		db_disconnect(dbinfo);
		return NULL;
	}

	return dbinfo;
}

gint db_disconnect(DBInfo *conn)
{
	if (conn) {
		PQfinish(PG_CONN(conn));
		g_free(conn);
	}

	return 0;
}

void db_error_message(DBInfo *conn, const gchar *msg)
{
	gchar *pqmsg;

	pqmsg = g_locale_to_utf8(PQerrorMessage(PG_CONN(conn)),
				 -1, NULL, NULL, NULL);
	g_warning("%s: %s", msg ? msg : "",
		  pqmsg ? pqmsg : PQerrorMessage(PG_CONN(conn)));
	g_free(pqmsg);
}

gint db_start_transaction(DBInfo *conn)
{
	PGresult *res;

	res = PQexec(PG_CONN(conn), "BEGIN");
	if (PQresultStatus(res) != PGRES_COMMAND_OK) {
		db_error_message(conn, "BEGIN failed");
		PQclear(res);
		return -1;
	}

	PQclear(res);
	return 0;
}

gint db_end_transaction(DBInfo *conn)
{
	PGresult *res;

	res = PQexec(PG_CONN(conn), "COMMIT");
	if (PQresultStatus(res) != PGRES_COMMAND_OK) {
		db_error_message(conn, "COMMIT failed");
		PQclear(res);
		return -1;
	}

	PQclear(res);
	return 0;
}

gint db_get_sid_from_msgid(DBInfo *conn, const gchar *msgid, gulong *sid)
{
	gchar *query;
	PGresult *res;

	query = g_strdup_printf("SELECT msg_sid FROM msginfo WHERE hdr_msgid = E'%s'", msgid);
	res = PQexec(PG_CONN(conn), query);
	g_free(query);

	if (PQresultStatus(res) != PGRES_TUPLES_OK) {
		db_error_message(conn, "SELECT from msginfo failed");
		PQclear(res);
		return -1;
	}

	if (PQntuples(res) > 0) {
		gchar *val;
		val = PQgetvalue(res, 0, 0);
		*sid = atol(val);
	} else
		*sid = 0;

	PQclear(res);

	return 0;
}

gint db_get_sid_from_folderinfo(DBInfo *conn, const gchar *folder_id,
                                guint msgnum, gulong *sid)
{
	gchar *query;
	PGresult *res;

	query = g_strdup_printf("SELECT msg_sid FROM msg_folderinfo WHERE folder_id = E'%s' AND msgnum = %u", folder_id, msgnum);
	res = PQexec(PG_CONN(conn), query);
	g_free(query);

	if (PQresultStatus(res) != PGRES_TUPLES_OK) {
		db_error_message(conn, "SELECT from msg_folderinfo failed");
		PQclear(res);
		return -1;
	}

	if (PQntuples(res) > 0) {
		gchar *val;
		val = PQgetvalue(res, 0, 0);
		*sid = atol(val);
	} else
		*sid = 0;

	PQclear(res);

	return 0;
}

gint db_delete_msg_from_folderinfo(DBInfo *conn, const gchar *folder_id,
				   guint msgnum)
{
	gchar *query;
	PGresult *res;

	query = g_strdup_printf("DELETE FROM msg_folderinfo WHERE folder_id = E'%s' AND msgnum = %u", folder_id, msgnum);
	res = PQexec(PG_CONN(conn), query);
	g_free(query);

	if (PQresultStatus(res) != PGRES_COMMAND_OK) {
		db_error_message(conn, "DELETE FROM msg_folderinfo failed");
		PQclear(res);
		return -1;
	}

	PQclear(res);
	return 0;
}

gint db_is_sid_exist_in_folderinfo(DBInfo *conn, gulong sid)
{
	gchar *query;
	PGresult *res;

	query = g_strdup_printf("SELECT msg_sid FROM msg_folderinfo WHERE msg_sid = %u LIMIT 1", sid);
	res = PQexec(PG_CONN(conn), query);
	g_free(query);

	if (PQresultStatus(res) != PGRES_TUPLES_OK) {
		db_error_message(conn, "SELECT from msg_folderinfo failed");
		PQclear(res);
		return -1;
	}

	if (PQntuples(res) > 0) {
		PQclear(res);
		return 1;
	}

	PQclear(res);
	return 0;
}

gint db_delete_msg(DBInfo *conn, gulong sid)
{
	gchar *query;
	PGresult *res;

	query = g_strdup_printf("DELETE FROM msginfo WHERE msg_sid = %u", sid);
	res = PQexec(PG_CONN(conn), query);
	g_free(query);

	if (PQresultStatus(res) != PGRES_COMMAND_OK) {
		db_error_message(conn, "DELETE FROM msginfo failed");
		PQclear(res);
		return -1;
	}

	PQclear(res);
	return 0;
}
