# Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
#
# This file is part of Rast.
# See the file COPYING for redistribution information.
#

require "test/unit"

require "rast/xmlrpc-server-testable"
require "rast/merged-doc-id-check"

require "rast_test"

module Rast
  class XMLRPCServerTest < Test::Unit::TestCase
    include XMLRPCServerTestable
    include MergedDocIDCheck

    def test_command_arguments
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      log_filename = File.join(db_name, "abyss.log")
      kick_xmlrpc_server("--log-file", log_filename, db_name) do
        assert_equal(true, File.exist?(log_filename))
      end
    end

    def test_register
      register_test_simple
      register_test_only_property
    end

    def register_test_simple
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      kick_xmlrpc_server("--read-write", db_name) do |server|
        result = server.call("rast.register", {"text" => "本日は晴天なり"})
        assert_equal(1, result["doc_id"])
      end
      LocalDB.open(db_name) do |db|
        result = db.search("晴天")
        assert_equal(1, result.num_docs)
        assert_equal(1, result.hit_count)
        assert_equal("晴天", result.terms[0].term)
        assert_equal(1, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(1, result.items[0].doc_id)
        assert_equal(0, result.items[0].db_index)
        assert_equal(nil, result.items[0].properties)
        assert_equal(1, result.items.length)
      end

      kick_xmlrpc_server(db_name) do |server|
        result = server.call("rast.register", {"text" => "本日は晴天なり"})
        assert_equal("rast", result["error"]["type"])
        assert_kind_of(Integer, result["error"]["code"])
        assert_kind_of(String, result["error"]["message"])
      end

      kick_xmlrpc_server("--read-only", db_name) do |server|
        result = server.call("rast.register", {"text" => "本日は晴天なり"})
        assert_equal("rast", result["error"]["type"])
        assert_kind_of(Integer, result["error"]["code"])
        assert_kind_of(String, result["error"]["message"])
      end

      kick_xmlrpc_server("--read-only", "--read-write", db_name) do |server|
        result = server.call("rast.register", {"text" => "本日は晴天なり"})
        assert_equal(2, result["doc_id"])
      end

      kick_xmlrpc_server("--read-write", "--read-only", db_name) do |server|
        result = server.call("rast.register", {"text" => "本日は晴天なり"})
        assert_equal("rast", result["error"]["type"])
        assert_kind_of(Integer, result["error"]["code"])
        assert_kind_of(String, result["error"]["message"])
      end
    end

    def register_test_only_property
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "title",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "modified_date",
            "type" => Rast::PROPERTY_TYPE_DATE,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "modified",
            "type" => Rast::PROPERTY_TYPE_DATETIME,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "serial",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      time = DateTime.new(2005, 2, 17, 19, 4, 38)
      kick_xmlrpc_server("--read-write", db_name) do |server|
        result = server.call("rast.register",
                             {
                               "text" => "",
                               "properties" => [
                                 "test", time.strftime("%F"),
                                 time.strftime("%FT%T"), 10,
                               ]
                             })
        assert_equal(1, result["doc_id"])
      end
      LocalDB.open(db_name) do |db|
        result = db.search("serial = 10",
                           {
                             "properties" => [
                               "title",
                               "modified_date",
                               "modified",
                               "serial",
                             ]
                           })
        assert_equal(1, result.items[0].doc_id)
        assert_equal("test", result.items[0].properties[0])
        assert_equal(time.strftime("%F"), result.items[0].properties[1])
        assert_equal(time.strftime("%FT%T"), result.items[0].properties[2])
        assert_equal(10, result.items[0].properties[3])
        assert_equal(4, result.items[0].properties.length)
      end
    end

    def test_search
      search_test_simple
      search_test_summary
      search_test_text_and_property
      search_test_property_and_summary
      search_test_property_types
      search_test_sort_with_score
      search_test_calc_score
      search_test_calc_received_all_num_docs_score
      search_test_sort_with_property
      search_test_set_range
      search_test_merger
      search_test_rast_error
      # search_test_apr_error
      # search_test_bdb_error
    end

    def search_test_simple
      options = {
        "encoding" => "utf8",
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("昨日は雨天なり", {})
        db.register("本日は晴天なり", {})
        db.register("明日は雪っすり", {})
      end
      kick_xmlrpc_server(db_name) do |server|
        result = server.call("rast.search",
                             {
                               "query" => "晴天",
                               "options" => {}
                             })
        assert_equal(1, result["num_indices"])
        assert_equal(3, result["num_docs"])
        assert_equal(1, result["hit_count"])
        assert_equal("晴天", result["terms"][0]["term"])
        assert_equal(1, result["terms"][0]["doc_count"])
        assert_equal(1, result["terms"].length)
        assert_equal(2, result["items"][0]["doc_id"])
        assert_equal(0, result["items"][0]["db_index"])
        assert_equal(0, result["items"][0]["properties"].length)
        assert_equal(1, result["items"].length)

        result = server.call("rast.search",
                             {
                               "query" => "なり",
                               "options" => {}
                             })
        assert_equal(2, result["hit_count"])
        assert_equal("なり", result["terms"][0]["term"])
        assert_equal(2, result["terms"][0]["doc_count"])
        assert_equal(1, result["terms"].length)
        assert_equal(1, result["items"][0]["doc_id"])
        assert_equal(0, result["items"][0]["properties"].length)
        assert_equal(2, result["items"][1]["doc_id"])
        assert_equal(0, result["items"][1]["properties"].length)
        assert_equal(2, result["items"].length)
      end
    end

    def search_test_summary
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "title",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => true,
            "unique" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("0123456789", {"title" => "full text search property"})
        db.register("零一二三四五六七八九",
                    {"title" => "full text search property"})
        db.register("零1二3四5六7八9",
                    {"title" => "full text search property"})
      end
      kick_xmlrpc_server(db_name) do |server|
        search_options = {"need_summary" => true, "summary_nchars" => 6}
        result = server.call("rast.search",
                             {
                               "query" => "5",
                               "options" => search_options,
                             })
        assert_equal(1, result["items"][0]["doc_id"])
        assert_equal("345678", result["items"][0]["summary"])
        assert_equal(3, result["items"][1]["doc_id"])
        assert_equal("3四5六7八", result["items"][1]["summary"])
        assert_equal(2, result["items"].length)
      end
    end

    def search_test_text_and_property
      options = {
        "encoding" => "utf8",
        "preserve_text" => false,
        "properties" => [
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => true,
          },
        ],
      }

      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("bcxy abc", {"filename" => "xyz"})
        db.register("bcxy", {"filename" => "123"})
      end
      kick_xmlrpc_server(db_name) do |server|
        search_options = {"need_summary" => true, "summary_nchars" => 6}
        result = server.call("rast.search",
                              {
                                "query" => "abc",
                                "options" => {},
                              })
        assert_equal(1, result["items"][0]["doc_id"])
        assert_equal(1, result["hit_count"])
        result = server.call("rast.search",
                              {
                                "query" => "xyz",
                                "options" => {},
                              })
        assert_equal(1, result["items"][0]["doc_id"])
        assert_equal(1, result["hit_count"])
        result = server.call("rast.search",
                              {
                                "query" => "abcxyz",
                                "options" => {},
                              })
        assert_equal(0, result["hit_count"])
        result = server.call("rast.search",
                              {
                                "query" => '"abc xyz"',
                                "options" => {},
                              })
        assert_equal(0, result["hit_count"])
      end
    end

    def search_test_property_and_summary
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
        ],
      }

      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("昨日は雨天なり", {"filename" => "ame.txt"})
        db.register("a" * 100 + "晴天" + "b" * 100,
                    {"filename" => "hare.txt"})
        db.register("明日は雪っす", {"filename" => "yuki.txt"})
        db.register("", {"filename" => "yuki.txt"})
      end
      kick_xmlrpc_server(db_name) do |server|
        result = server.call("rast.search",
                             {
                               "query" => "晴天",
                               "options" => {
                                 "properties" => ["filename"],
                                 "need_summary" => true,
                               }
                             })
        assert_equal(1, result["hit_count"])
        assert_equal("晴天", result["terms"][0]["term"])
        assert_equal(1, result["terms"][0]["doc_count"])
        assert_equal(1, result["terms"].length)
        assert_equal(2, result["items"][0]["doc_id"])
        assert_equal("hare.txt", result["items"][0]["properties"][0]["value"])
        assert_equal(1, result["items"][0]["properties"].length)
        expected_summary = "a" * 49 + "晴天" + "b" * 49
        assert_equal(expected_summary, result["items"][0]["summary"])
        assert_equal(1, result["items"].length)
      end

      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("1" + "abc" * 100, {"filename" => "abc.txt"})
      end
      kick_xmlrpc_server(db_name) do |server|
        result = server.call("rast.search",
                             {
                               "query" => "abc",
                               "options" => {
                                 "summary_nchars" => 301,
                                 "need_summary" => true,
                               }
                             })
        assert_equal(1, result["hit_count"])
        assert_equal("abc", result["terms"][0]["term"])
        assert_equal(1, result["terms"][0]["doc_count"])
        assert_equal(1, result["terms"].length)
        assert_equal(1, result["items"][0]["doc_id"])
        assert_equal("1" + "abc" * 100, result["items"][0]["summary"])
        assert_equal(1, result["items"].length)
      end
    end

    def search_test_property_types
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "serial",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "modified_date",
            "type" => Rast::PROPERTY_TYPE_DATE,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "modified",
            "type" => Rast::PROPERTY_TYPE_DATETIME,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
        ],
      }

      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("晴天",
                    {
                      "filename" => "hare.txt",
                      "serial" => 2,
                      "modified_date" => "2005-04-16",
                      "modified" => "2005-04-16T14:25:33+0900",
                    })
      end
      kick_xmlrpc_server(db_name) do |server|
        result = server.call("rast.search",
                             {
                               "query" => "晴天",
                               "options" => {
                                 "properties" => [
                                   "filename", "serial",
                                   "modified_date", "modified"
                                 ],
                                 "need_summary" => true,
                               }
                             })
        assert_equal(1, result["hit_count"])
        assert_equal("晴天", result["terms"][0]["term"])
        assert_equal(1, result["terms"][0]["doc_count"])
        assert_equal(1, result["terms"].length)
        assert_equal(1, result["items"][0]["doc_id"])
        assert_equal("string",
                     result["items"][0]["properties"][0]["type"])
        assert_equal("hare.txt", result["items"][0]["properties"][0]["value"])
        assert_equal("uint",
                     result["items"][0]["properties"][1]["type"])
        assert_equal(2, result["items"][0]["properties"][1]["value"])
        assert_equal("date",
                     result["items"][0]["properties"][2]["type"])
        assert_equal("2005-04-16",
                     result["items"][0]["properties"][2]["value"])
        assert_equal("datetime",
                     result["items"][0]["properties"][3]["type"])
        assert_equal("2005-04-16T14:25:33",
                     result["items"][0]["properties"][3]["value"])
        assert_equal(4, result["items"][0]["properties"].length)
        assert_equal(1, result["items"].length)
      end
    end

    def search_test_sort_with_score
      options = {
        "encoding" => "utf8",
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("あい" * 99 + "晴天", {}) 
        db.register("晴天" * 100, {})
        db.register("あい" * 49 + "晴天", {}) 
      end
      kick_xmlrpc_server(db_name) do |server|
        result = server.call("rast.search",
                             {
                               "query" => "晴天",
                               "options" => {}
                             })
        # descend
        assert_equal(3, result["hit_count"])
        assert_equal("晴天", result["terms"][0]["term"])
        assert_equal(3, result["terms"][0]["doc_count"])
        assert_equal(1, result["terms"].length)
        assert_equal(2, result["items"][0]["doc_id"])
        assert_equal(3, result["items"][1]["doc_id"])
        assert_equal(1, result["items"][2]["doc_id"])
        assert_equal(3, result["items"].length)

        # ascend
        search_options = {
          "sort_order" => "ascending"
        }
        result = server.call("rast.search",
                             {
                               "query" => "晴天",
                               "options" => search_options,
                             })
        assert_equal(3, result["hit_count"])
        assert_equal("晴天", result["terms"][0]["term"])
        assert_equal(3, result["terms"][0]["doc_count"])
        assert_equal(1, result["terms"].length)
        assert_equal(1, result["items"][0]["doc_id"])
        assert_equal(3, result["items"][1]["doc_id"])
        assert_equal(2, result["items"][2]["doc_id"])
        assert_equal(3, result["items"].length)
      end
    end

    def search_test_calc_score
      options = {
        "encoding" => "utf8",
        "properties" => [
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("aaa" * 100, {"filename" => "abc.txt"})
      end
      kick_xmlrpc_server(db_name) do |server|
        result = server.call("rast.search",
                             {
                               "query" => "aaa",
                               "options" => {
                                 "score_method" => "none",
                               },
                             })
        assert_equal(0, result["items"][0]["score"])

        result = server.call("rast.search",
                             {
                               "query" => "aaa",
                               "options" => {
                                 "score_method" => "tfidf",
                               },
                             })
        assert_operator(0, :<, result["items"][0]["score"])

        result = server.call("rast.search",
                             {
                               "query" => "aaa",
                               "options" => {},
                             })
        assert_operator(0, :<, result["items"][0]["score"])

        result = server.call("rast.search",
                             {
                               "query" => "aaa",
                               "options" => {
                                 "sort_method" => "property",
                                 "sort_property" => "filename",
                               },
                             })
        assert_operator(0, :<, result["items"][0]["score"])

        result = server.call("rast.search",
                             {
                               "query" => "aaa",
                               "options" => {
                                 "score_method" => "tfidf",
                                 "sort_method" => "property",
                                 "sort_property" => "filename",
                               },
                             })
        assert_operator(0, :<, result["items"][0]["score"])
      end
    end

    def search_test_calc_received_all_num_docs_score
      options = {
        "encoding" => "utf8",
        "properties" => [
        ],
      }
      db_name1 = generate_db_name
      LocalDB.create(db_name1, options)
      LocalDB.open(db_name1) do |db|
        db.register("qwerty abcdef", {})
        db.register("qwerty", {})
      end
      result1 = LocalDB.open(db_name1) do |db|
        db.search("qwe abc", {})
      end
      db_name2 = generate_db_name
      LocalDB.create(db_name2, options)
      LocalDB.open(db_name2) do |db|
        db.register("qwerty abcdef", {})
      end
      kick_xmlrpc_server(db_name2) do |server|
        terms = result1.terms.collect do |term|
          term.doc_count
        end
        result2 = server.call("rast.search",
                              {
                                "query" => "qwe abc",
                                "options" => {
                                  "all_num_docs" => 2,
                                  "terms" => terms
                                }
                              })
        assert_equal(result1["items"][0]["score"],
                     result2["items"][0]["score"])
      end
    end

    def search_test_sort_with_property
      options = {
        "encoding" => "utf8",
        "properties" => [
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "serial",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("昨日は雨天なり",
                    {"filename" => "2.txt", "serial" => 10})
        db.register("本日は晴天なり",
                    {"filename" => "3.txt", "serial" => 5})
        db.register("明日は雪っすり",
                    {"filename" => "1.txt", "serial" => 15})
      end

      kick_xmlrpc_server(db_name) do |server|
        result = server.call("rast.search",
                             {
                               "query" => "日は",
                               "options" => {
                                 "sort_method" => "property",
                                 "sort_property" => "filename",
                                 "properties" => ["filename"],
                               }
                             })
        assert_equal(3, result["hit_count"])
        assert_equal(3, result["terms"][0]["doc_count"])
        assert_equal(1, result["terms"].length)
        assert_equal(3, result["items"][0]["doc_id"])
        assert_equal(1, result["items"][1]["doc_id"])
        assert_equal(2, result["items"][2]["doc_id"])
        assert_equal(3, result["items"].length)

        result = server.call("rast.search",
                             {
                               "query" => "日は",
                               "options" => {
                                 "sort_method" => "property",
                                 "sort_property" => "filename",
                                 "sort_order" => "ascending",
                                 "properties" => ["filename"],
                               }
                             })
        assert_equal(3, result["hit_count"])
        assert_equal(3, result["terms"][0]["doc_count"])
        assert_equal(1, result["terms"].length)
        assert_equal(3, result["items"][0]["doc_id"])
        assert_equal(1, result["items"][1]["doc_id"])
        assert_equal(2, result["items"][2]["doc_id"])
        assert_equal(3, result["items"].length)

        result = server.call("rast.search",
                             {
                               "query" => "日は",
                               "options" => {
                                 "sort_method" => "property",
                                 "sort_property" => "filename",
                                 "sort_order" => "descending",
                                 "properties" => ["filename"],
                               }
                             })
        assert_equal(3, result["hit_count"])
        assert_equal(3, result["terms"][0]["doc_count"])
        assert_equal(1, result["terms"].length)
        assert_equal(2, result["items"][0]["doc_id"])
        assert_equal(1, result["items"][1]["doc_id"])
        assert_equal(3, result["items"][2]["doc_id"])
        assert_equal(3, result["items"].length)

        result = server.call("rast.search",
                             {
                               "query" => "日は",
                               "options" => {
                                 "sort_method" => "property",
                                 "sort_property" => "serial",
                                 "properties" => ["serial"],
                               }
                             })
        assert_equal(3, result["hit_count"])
        assert_equal(3, result["terms"][0]["doc_count"])
        assert_equal(1, result["terms"].length)
        assert_equal(2, result["items"][0]["doc_id"])
        assert_equal(1, result["items"][1]["doc_id"])
        assert_equal(3, result["items"][2]["doc_id"])
        assert_equal(3, result["items"].length)

        result = server.call("rast.search",
                             {
                               "query" => "日は",
                               "options" => {
                                 "sort_method" => "property",
                                 "sort_property" => "serial",
                                 "sort_order" => "ascending",
                                 "properties" => ["serial"],
                               }
                             })
        assert_equal(3, result["hit_count"])
        assert_equal(3, result["terms"][0]["doc_count"])
        assert_equal(1, result["terms"].length)
        assert_equal(2, result["items"][0]["doc_id"])
        assert_equal(1, result["items"][1]["doc_id"])
        assert_equal(3, result["items"][2]["doc_id"])
        assert_equal(3, result["items"].length)

        result = server.call("rast.search",
                             {
                               "query" => "日は",
                               "options" => {
                                 "sort_method" => "property",
                                 "sort_property" => "serial",
                                 "sort_order" => "descending",
                                 "properties" => ["serial"],
                               }
                             })
        assert_equal(3, result["hit_count"])
        assert_equal(3, result["terms"][0]["doc_count"])
        assert_equal(1, result["terms"].length)
        assert_equal(3, result["items"][0]["doc_id"])
        assert_equal(1, result["items"][1]["doc_id"])
        assert_equal(2, result["items"][2]["doc_id"])
        assert_equal(3, result["items"].length)
      end
    end

    def search_test_set_range
      options = {
        "encoding" => "utf8",
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("1", {}) 
        db.register("1", {}) 
        db.register("1", {}) 
        db.register("1", {}) 
        db.register("1", {}) 
      end

      kick_xmlrpc_server(db_name) do |server|
        result = server.call("rast.search",
                             {
                               "query" => "1",
                               "options" => {
                                 "start_no" => 0,
                                 "num_items" => 3,
                               },
                             })
        assert_equal(5, result["hit_count"])
        assert_equal(3, result["items"].length)

        result = server.call("rast.search",
                             {
                               "query" => "1",
                               "options" => {
                                 "start_no" => 1,
                                 "num_items" => 3,
                               },
                             })
        assert_equal(5, result["hit_count"])
        assert_equal(3, result["items"].length)

        result = server.call("rast.search",
                             {
                               "query" => "1",
                               "options" => {
                                 "start_no" => 4,
                                 "num_items" => 10,
                               },
                             })
        assert_equal(5, result["hit_count"])
        assert_equal(1, result["items"].length)

        result = server.call("rast.search",
                             {
                               "query" => "1",
                               "options" => {
                                 "start_no" => 20,
                                 "num_items" => 10,
                               },
                             })
        assert_equal(5, result["hit_count"])
        assert_equal(0, result["items"].length)
      end
    end

    def search_test_merger
      options = {
        "encoding" => "utf8",
        "properties" => [],
      }
      db_names = []
      db_names.push(generate_db_name)
      LocalDB.create(db_names[0], options)
      LocalDB.open(db_names[0]) do |db|
        db.register("昨日は雨天なり", {})
        db.register("本日は晴天なり", {})
        db.register("明日は雪っすり", {})
      end

      db_names.push(generate_db_name)
      LocalDB.create(db_names[1], options)
      LocalDB.open(db_names[1]) do |db|
        db.register("昨日は雨天なり", {})
        db.register("本日は晴天なり", {})
        db.register("明日は雪っすり", {})
      end

      kick_xmlrpc_server(db_names[0], db_names[1]) do |server|
        result = server.call("rast.search",
                             {
                               "query" => "晴天",
                               "options" => {}
                             })
        assert_equal(2, result["num_indices"])
        assert_equal(6, result["num_docs"])
        assert_equal(2, result["hit_count"])
        assert_equal("晴天", result["terms"][0]["term"])
        assert_equal(2, result["terms"][0]["doc_count"])
        assert_equal(1, result["terms"].length)
        assert_equal(DocIDInfo.new(2, 0),
                     split_doc_id(result["items"][0]["doc_id"]))
        assert_equal(0, result["items"][0]["db_index"])
        assert_equal(0, result["items"][0]["properties"].length)
        assert_equal(DocIDInfo.new(2, 1),
                     split_doc_id(result["items"][1]["doc_id"]))
        assert_equal(1, result["items"][1]["db_index"])
        assert_equal(0, result["items"][1]["properties"].length)
        assert_equal(2, result["items"].length)
      end
    end

    def search_test_rast_error
      options = {
        "encoding" => "utf8",
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("昨日は雨天なり", {})
      end
      kick_xmlrpc_server(db_name) do |server|
        result = server.call("rast.search",
                             {
                               "query" => "晴天",
                               "options" => {
                                 "properties" => ["invalid_property"],
                               },
                             })
        assert_equal("rast", result["error"]["type"])
        assert_kind_of(Integer, result["error"]["code"])
        assert_kind_of(String, result["error"]["message"])
      end
    end

    def test_delete
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("本日は晴天なり", {})
      end
      LocalDB.open(db_name) do |db|
        result = db.search("晴天")
        assert_equal(1, result.hit_count)
      end
      kick_xmlrpc_server("--read-write", db_name) do |server|
        server.call("rast.delete", {"doc_id" => 1})
      end
      LocalDB.open(db_name) do |db|
        result = db.search("晴天")
        assert_equal(0, result.hit_count)
      end
    end

    def test_update
      update_test_simple
      update_test_property
    end

    def update_test_simple
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("本日は晴天なり", {})
      end
      LocalDB.open(db_name) do |db|
        result = db.search("晴天")
        assert_equal(1, result.hit_count)
      end
      kick_xmlrpc_server("--read-write", db_name) do |server|
        result = server.call("rast.update",
                             {
                               "doc_id" => 1,
                               "text" => "明日は雪っす",
                               "properties" => [],
                             })
        assert_equal(2, result["doc_id"])
      end
      LocalDB.open(db_name) do |db|
        result = db.search("晴天")
        assert_equal(0, result.hit_count)
        result = db.search("雪っ")
        assert_equal(1, result.hit_count)
      end
    end

    def update_test_property
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => true,
          },
          {
            "name" => "count",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("本日は晴天なり", {"filename" => "a.txt", "count" => 10})
      end
      LocalDB.open(db_name) do |db|
        result = db.search("晴天")
        assert_equal(1, result.hit_count)
      end
      kick_xmlrpc_server("--read-write", db_name) do |server|
        result = server.call("rast.update",
                             {
                               "doc_id" => 1,
                               "text" => "明日は雪っす",
                               "properties" => ["b.txt", 20],
                             })
        assert_equal(2, result["doc_id"])
      end
      LocalDB.open(db_name) do |db|
        result = db.search("晴天")
        assert_equal(0, result.hit_count)
        result = db.search("雪っ")
        assert_equal(1, result.hit_count)
        result = db.search("count = 10")
        assert_equal(0, result.hit_count)
        result = db.search("count = 20")
        assert_equal(1, result.hit_count)
      end
    end

    def test_get_text
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      text = "本日は晴天なり"
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register(text, {})
      end
      kick_xmlrpc_server(db_name) do |server|
        result = server.call("rast.get_text", {"doc_id" => 1})
        assert_equal(text, result["text"])
      end

      options = {
        "encoding" => "utf8",
        "preserve_text" => false,
        "properties" => [],
      }
      db_name = generate_db_name
      text = "本日は晴天なり"
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register(text, {})
      end
      kick_xmlrpc_server(db_name) do |server|
        result = server.call("rast.get_text", {"doc_id" => 1})
        assert_equal("", result["text"])
      end
    end

    def test_db_encoding
      options = {
        "encoding" => "mecab_euc_jp",
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      kick_xmlrpc_server(db_name) do |server|
        result = server.call("rast.encoding")

        assert_equal({"encoding" => "EUC-JP"}, result)
      end
    end

    def test_db_properties
      options = {
        "encoding" => "utf8",
        "properties" => [
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => true,
            "full_text_search" => false,
          },
          {
            "name" => "modified_date",
            "type" => Rast::PROPERTY_TYPE_DATE,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "modified",
            "type" => Rast::PROPERTY_TYPE_DATETIME,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "serial",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      kick_xmlrpc_server(db_name) do |server|
        result = server.call("rast.properties")

        assert_equal("filename", result[0]["name"])
        assert_equal("string", result[0]["type"])
        assert_equal(0x0002, result[0]["flags"] )
        assert_equal("modified_date", result[1]["name"])
        assert_equal("date", result[1]["type"])
        assert_equal(0x0000, result[1]["flags"] )
        assert_equal("modified", result[2]["name"])
        assert_equal("datetime", result[2]["type"])
        assert_equal(0x0000, result[2]["flags"] )
        assert_equal("serial", result[3]["name"])
        assert_equal("uint", result[3]["type"])
        assert_equal(0x0001, result[3]["flags"] )
        assert_equal(4, result.length)
      end
    end

    def kick_xmlrpc_server(*args)
      kick_rast_xmlrpc_server_process(*args) do |port|
        yield(XMLRPC::Client.new("localhost", "/RPC2", port))
      end
    end
  end
end
