/****************************************************************************
** Class Utils implementation ...
**
**   Created : Wed Jun 17 07:53:05 2008
**        by : Varol Okan
** Copyright : (c) Varol Okan
**   License : GPL v 2.0
**
** Here we have some misc functions which are needed by a few classes 
** but can not really be assigned to any of those classes.
**
****************************************************************************/

#include <QColor>
#include <QPainter>
#include <QFileInfo>

#include "utils.h"
#include "render.h"
#include "slideshow.h"

namespace Render
{

Slideshow::Slideshow ( Render::Render *pParent )
{
  m_pParent = pParent;
  m_size = QSize ( 720, 480 ); // NTSC
}

Slideshow::Slideshow ( )
{
  m_pParent = NULL;
  m_size    = QSize ( 720, 480 ); // NTSC
}

Slideshow::~Slideshow ( )
{
}

void Slideshow::initMe ( CXmlSlideshow *pSlideshow, QString &qsTempPath )
{
  m_pSlideshow = pSlideshow;
  m_qsTempPath = qsTempPath;
  m_qsFileName = m_qsTempPath + pSlideshow->slideshow_name + ".vob";
}

void Slideshow::setFileName ( QString &qsFileName )
{
  if ( qsFileName.isEmpty ( ) )
    m_qsFileName = m_qsTempPath + m_pSlideshow->slideshow_name + ".vob";
  else
    m_qsFileName = qsFileName;
}

void Slideshow::displayProgress ( float fProgress )
{
  if ( m_pParent )
    m_pParent->sendProgress ( fProgress );
  else {
    printf ( "." );
    if ( fProgress >= 100.0f )
      printf ( "\n" );
    fflush ( stdout );
  }
}

QString Slideshow::getHashName ( QString qsInput )
{
  if ( m_pParent )
    return m_pParent->getHashName ( qsInput );
  return qsInput;
}

bool Slideshow::exec ( )
{
  if ( ! m_pSlideshow )
    return false;

  // The slideshow has to start with something
  int iX = m_pSlideshow->xres;
  int iY = m_pSlideshow->yres;
  if ( iX < 10 )
       iX = 480;
  if ( iY < 0 )
       iY = 720;
  m_size = QSize ( iX, iY );

  double fFPS = 29.97;
  if ( ( iX == 720) && ( iY == 576 ) )
    fFPS = 25.0;
  else if ( ( iX == 704 ) && ( iY == 576 ) )
    fFPS = 25.0;
  else if ( ( iX == 352 ) && ( iY == 576 ) )
    fFPS = 25.0;
  else if ( ( iX == 352 ) && ( iY == 288 ) )
    fFPS = 25.0;

  CXmlSlideshow::img_struct *pXmlStart = createStartEnd (  true );
  CXmlSlideshow::img_struct *pXmlEnd   = createStartEnd ( false );
  CXmlSlideshow::img_struct *pXmlCurrent, *pXmlPrevious;

  createBackground ( );

  Encoder::FFmpeg theEncoder;
  theEncoder.initStream ( m_qsFileName, Encoder::Encoder::vfNTSC, Encoder::Encoder::afAC3, 1000 );

  pXmlPrevious     = pXmlStart;
  uint t, iCount   = m_pSlideshow->countImg ( );
  uint iAudioCount = m_pSlideshow->audio_list.count ( );
  float fProgress  = 0.0f;
  float fCount2    = 2.0f * ( iCount + iAudioCount ) + 2.0f;
  int iProgress    = 0;
  int iStart       = 0;

  if ( ! m_pSlideshow->intro_page )  {
    // The user chose to skip the intro page with the slideshow title
    iStart        = 1;
    pXmlPrevious  = m_pSlideshow->getImg ( 0 );
    fCount2      -= 2.0f;
  }

  for ( t=iStart; t<iCount; t++ )  {
    pXmlCurrent   = m_pSlideshow->getImg ( t );
//printf ( "Encoding               <%s>\n", (const char *)pXmlPrevious->src.toUtf8 ( ) );
    // The frame is to be displayed for a bit ...
    createFromImage  ( pXmlPrevious, &theEncoder, fFPS );
    fProgress = 100.0f * ( (float)iProgress++ / fCount2 );
    displayProgress ( fProgress );
//printf ( "Encoding Transition to <%s>\n", (const char *)pXmlCurrent->src.toUtf8 ( ) );
    createTransition ( pXmlPrevious, pXmlCurrent, &theEncoder, fFPS );

    fProgress = 100.0f * ( (float)iProgress++ / fCount2 );
    displayProgress ( fProgress );

    pXmlPrevious = pXmlCurrent;
  }
  // The frame is to be displayed for a bit ...
//printf ( "Encoding               <%s>\n", (const char *)pXmlPrevious->src.toUtf8 ( ) );
  createFromImage  ( pXmlPrevious, &theEncoder, fFPS );
  fProgress = 100.0f * ( (float)iProgress++ / fCount2 );
  displayProgress ( fProgress );
//printf ( "Encoding Transition to <%s>\n", (const char *)pXmlEnd->src.toUtf8 ( ) );
  createTransition ( pXmlPrevious, pXmlEnd, &theEncoder, fFPS );

  // the last step is to add the audio stream if present
  // Not working correctly: Postponed to the next release ...
  if ( 0 )  { // m_pSlideshow->audio_list.count ( ) > 0 )  {
    QString qsCommand;
    // For now I will simply use the command line until I figure out how to do this
    // through libffmpeg
    // ffmpeg -y -i "Slide 1.vob" -i "Audio.ogg" -vcodec copy -acodec ac3 -ar 48000 -ab 192k -shortest -map 0:0 -map 1:0 -target ntsc-dvd  /tmp/aa.vob
    qsCommand  = QString ( "mv \"%1\" \"%2\".1; " ).arg ( m_qsFileName ).arg ( m_qsFileName );
    qsCommand += QString ( "ffmpeg -y -i \"%1\".1 " ).arg ( m_qsFileName );
    for ( t=0; t<(uint)m_pSlideshow->audio_list.count ( ); t++ )  {
      if ( m_pSlideshow->audio_list[t].length ( ) > 1 )
        qsCommand += QString ( " -i \"%1\" " ).arg ( m_pSlideshow->audio_list[t] );
    }
    qsCommand += QString ( "-vcodec copy -acodec ac3 -ar 48000 -ab 192k -shortest -map 0:0 -map 1.0 " );
    qsCommand += QString ( "%1" ).arg ( m_qsFileName );
    system ( (const char *)qsCommand.toUtf8 ( ) );
    printf ( "CMD <%s>\n", (const char *)qsCommand.toUtf8 ( ) );

    qsCommand  = QString ( "mv \"%1\".1 \"%2\"" ).arg ( m_qsFileName ).arg ( m_qsFileName );
    QFileInfo   fileInfo ( m_qsFileName + ".1" );
    if  ( ! fileInfo.exists ( ) || fileInfo.size ( ) < 10 )
      system ( (const char *)qsCommand.toUtf8 ( ) );
  }

  displayProgress ( 100.0f );
  displayProgress ( 100.0f );

  theEncoder.endStream  ( );
  delete pXmlStart;
  delete pXmlEnd;
  return true;
}

void Slideshow::createBackground ( )
{
  m_background = QImage ( m_size, QImage::Format_ARGB32_Premultiplied );
  QString qsBackground = m_pSlideshow->background;
  if ( ! qsBackground.isEmpty ( ) )  {
    QColor theColor ( qsBackground );
    if ( theColor.isValid ( ) )
      m_background.fill ( theColor.rgba ( ) );
    else  {
      QFileInfo fileInfo ( qsBackground );
      if ( fileInfo.exists ( ) )  {
        QPainter thePainter;
        QImage img ( qsBackground );
        img  = img.scaled ( m_size, Qt::IgnoreAspectRatio, Qt::SmoothTransformation );
        thePainter.begin ( &m_background );
        thePainter.drawImage ( 0, 0, img, 0, 0 );
        thePainter.end ( );
      }
      else
        m_background.fill ( 0xFF000000 );
    }
  }
  else
    m_background.fill ( 0xFF000000 );
}

CXmlSlideshow::img_struct *Slideshow::createStartEnd ( bool bStart )
{
  // This frunction creates the first frame and the last frame.
  int iX = m_size.width  ( );
  int iY = m_size.height ( );

  QFont theFont       ( "Helvetica [Cronyx]", 24, QFont::Bold );
  QImage canvas       (    m_size, QImage::Format_ARGB32_Premultiplied );
  QPainter thePainter (   &canvas );
  thePainter.setPen   ( QColor ( "#AFAF15" ) ); // Yellow
  thePainter.setBrush ( QColor ( "#000000" ) );   // Qt::blue );
  thePainter.drawRect ( 0,0,iX,iY );
  thePainter.setFont  (   theFont );

  QString qsFileName;
  if ( bStart )  {
    thePainter.drawText ( iX / 3.0, iY / 2.0 - 10, m_pSlideshow->slideshow_name );
    qsFileName = m_qsTempPath + "start.jpg";
  }
  else
    qsFileName = m_qsTempPath + "end.jpg";

  // Note this is the same algol as in Render::fileFromSocket
  QString qsNewFileName = getHashName ( qsFileName );
  // Store the hash file name so we can access
  canvas.save ( qsNewFileName, "JPEG", 100 );

  CXmlSlideshow::img_struct *pXml = new CXmlSlideshow::img_struct;
  pXml->src    = qsFileName;
  pXml->width  = iX;
  pXml->height = iY;
  return pXml;
}

void Slideshow::createFromImage ( CXmlSlideshow::img_struct *pXmlImage, Encoder::FFmpeg *pEncoder, double fFPS )
{
  if ( ! pXmlImage )
    return;

  QPainter thePainter;
  QString  qsFileName = getHashName ( pXmlImage->src );
  QImage   canvas   ( m_size, QImage::Format_ARGB32_Premultiplied );
  QImage   imgStart ( qsFileName );

  if ( pXmlImage->pMatrix )
       imgStart = imgStart.transformed ( *pXmlImage->pMatrix, Qt::SmoothTransformation );
  imgStart = imgStart.scaled ( m_size, Qt::KeepAspectRatio );

  int iX, iY;
  iX = (int)( ( m_size.width  ( ) - imgStart.width  ( ) ) / 2.0 );
  iY = (int)( ( m_size.height ( ) - imgStart.height ( ) ) / 2.0 );

  canvas = m_background;
  thePainter.begin ( &canvas );
  thePainter.drawImage ( iX, iY, imgStart, 0, 0 );
  thePainter.end ( );

  float fDuration = pXmlImage->fDuration;
  if (  fDuration < 0.0f )
        fDuration = (float)m_pSlideshow->delay;
  if (  fDuration < 0.0f )
        fDuration = 5.0f;
  fDuration *= 1000.0f; // in mSeconds please.

  // And at the end we will add this to the VOB compliant video TS
  int iNumberOfImages = (int)( fFPS * fDuration / 1000.0f );
  pEncoder->addImage ( &canvas, iNumberOfImages );
}

void Slideshow::createTransition ( CXmlSlideshow::img_struct *pStart, CXmlSlideshow::img_struct *pStop, Encoder::FFmpeg *pEncoder, double fFPS )
{
  if ( ! pStart || ! pStop )
    return;

  QPainter thePainter;
  QString  qsFileName = getHashName ( pStart->src );
  QImage   canvas   ( m_size, QImage::Format_ARGB32_Premultiplied );
  QImage   imgStart ( qsFileName );

  if ( pStart->pMatrix )
       imgStart = imgStart.transformed ( *pStart->pMatrix, Qt::SmoothTransformation );
  imgStart = imgStart.scaled ( m_size, Qt::KeepAspectRatio );

  int iX, iY;
  iX = (int)( ( m_size.width  ( ) - imgStart.width  ( ) ) / 2.0 );
  iY = (int)( ( m_size.height ( ) - imgStart.height ( ) ) / 2.0 );

  canvas = m_background;
  thePainter.begin ( &canvas );
  thePainter.drawImage ( iX, iY, imgStart, 0, 0 );
  thePainter.end ( );
  imgStart = canvas;

  qsFileName = getHashName ( pStop->src );
  QImage  imgStop  ( qsFileName );

  if ( pStop->pMatrix )
       imgStop = imgStop.transformed ( *pStop->pMatrix, Qt::SmoothTransformation );
  imgStop = imgStop.scaled ( m_size, Qt::KeepAspectRatio );

  iX = (int)( ( m_size.width  ( ) - imgStop.width  ( ) ) / 2.0 );
  iY = (int)( ( m_size.height ( ) - imgStop.height ( ) ) / 2.0 );

  canvas = m_background;
  thePainter.begin ( &canvas );
  thePainter.drawImage ( iX, iY, imgStop, 0, 0 );
  thePainter.end ( );
  imgStop = canvas;

  float fDuration = (float)m_pSlideshow->filter_delay;
  if  ( fDuration < 0.0f )
        fDuration = pStop->fDuration;
  if  ( fDuration < 0.0f )
        fDuration = 3.0f;
  fDuration *= 1000.0f; // in mSeconds please.

  int t,  iNumberOfImages = (int)( fFPS * fDuration / 1000.0f );
  QImage  result, temp, alpha ( m_size, QImage::Format_ARGB32_Premultiplied );

  qreal fAlpha;
  for ( t=0; t<iNumberOfImages; t++ )  {
    result = imgStart;
    fAlpha = (qreal)t / ( iNumberOfImages - 1 );

    QPainter thePainter   ( &result );
    thePainter.setOpacity (  fAlpha ); // 0 = transparent 1=opaque
    thePainter.drawImage  ( 0, 0, imgStop, 0, 0 );

    // And at the end we will add this to the VOB compliant video TS
    pEncoder->addImage   ( &result, 1 );
  }
}

}; // end of namespace Render
