/******************************************************************
 **
 ** This file is shared between 
 ** QRender    ( Qt 4.x ) and
 ** QSlideshow ( Qt 3.x )
 **
 ** It would not be wise to re-write the whole file or 
 ** to have two copies of essentially the same code in
 ** a project. This way changes in one art of the app
 ** will automagically be available in the other part.
 **
 *****************************************************************/
#include "xml_slideshow.h"

//#if (QT_VERSION > 0x0301FF)
#ifdef QDVD_RENDER
  #include <QDomDocument>
  #include <QFile>
  #include <QColor>
  #include <QMatrix>
  #include <QListView>
  #include <QFileDialog>
  #include <QMessageBox>
#else
  #include <qdom.h>
  #include <qfile.h>
  #include <qcolor.h>
  #include <qwmatrix.h>
  #include <qlistview.h>
  #include <qfiledialog.h>
  #include <qmessagebox.h>

  #include "xml_dvd.h"
  #include "qslideshow/modifiers2.h"

  #ifndef toAscii
    #define toAscii ascii
  #endif
#endif

CXmlSlideshow::CXmlSlideshow ( )
{
	node_name     = QString ( SLIDESHOW_TAG );
	delay         = -1.0;
	filter_delay  = -1.0;
	xres          =  720;
	yres          =  480; // NTSC for now
	intro_page    = true;
	ppArrayImg    = NULL;
	ppArrayFilter = NULL;
	id            = NULL;
}

CXmlSlideshow::~CXmlSlideshow ()
{
//printf ( "CXmlSlideshow::~CXmlSlideshow <%s>\n", slideshow_name.ascii ( ) );
	clear ( );
}

void CXmlSlideshow::clear ( )
{
	clearObjects ( );
	delay        = -1.0f;
	filter_delay = -1.0f;
	xres         = 720;
	yres         = 480;	// NTSC for now
	intro_page   = true;
	id           = NULL;
	background   = QString ( "" );
	audio_list.clear ( );
}

void CXmlSlideshow::clearObjects ( )
{
	// Lets clean up a little bit ...
	int t=0;
	// The timObjectArray holds Img, and Filter objects, thus no need to delete these again.
	time_object *pTimeObject;
	for ( t=0; t<(int)m_listTimeObjects.count(); t++ )	{
		pTimeObject = m_listTimeObjects[t];
		delete pTimeObject;
	}

	if (ppArrayImg)
		delete []ppArrayImg;
	if (ppArrayFilter)
		delete []ppArrayFilter;
	ppArrayImg    = NULL;
	ppArrayFilter = NULL;
	m_listTimeObjects.clear ( );
}

CXmlSlideshow &CXmlSlideshow::operator = (CXmlSlideshow &theOther)
{
	uint t;
	time_object *pTimeObject, *pNewObject;

	clear ();
	delay          = theOther.delay;
	filter_delay   = theOther.filter_delay;
	xres           = theOther.xres;
	yres           = theOther.yres;
	intro_page     = theOther.intro_page;
	id             = theOther.id;
	background     = theOther.background;
	slideshow_name = theOther.slideshow_name;
	audio_list     = theOther.audio_list;
	// Here we copy the objects over ..
	for (t=0;t<theOther.count();t++)	{
		pTimeObject = theOther.getTimeObject(t);
		if (pTimeObject->node_name == "img")
			pNewObject = addImg    ( );  //new img_struct;
		else
			pNewObject = addFilter ( );  //new filter_struct;
		*pNewObject = *pTimeObject;
	}
	return *this;
}

bool CXmlSlideshow::operator == ( CXmlSlideshow &theOther )
{
	uint t;
	time_object *p1, *p2;

	bool bReturn = (
		( delay          == theOther.delay          ) &&
		( filter_delay   == theOther.filter_delay   ) &&
		( xres           == theOther.xres           ) &&
		( yres           == theOther.yres           ) &&
		( intro_page     == theOther.intro_page     ) &&
		( background     == theOther.background     ) &&
		( slideshow_name == theOther.slideshow_name ) &&
		( audio_list     == theOther.audio_list     )  );

	if ( ! bReturn )
		return false;

	if ( count ( )  != theOther.count ( ) )
		return false;

	// Here we copy the objects over ..
	for ( t=0; t<count ( ); t++ )  {
		p1 = getTimeObject ( t );
		p2 = theOther.getTimeObject ( t );
		if ( ! ( *p1 == *p2 ) )
			return false;
	}
	// Went through all this hassle. Deserves a true 'true' :)
	return true;
}

bool CXmlSlideshow::readXml ()
{
	// Here we read in a xml - file and create the neccesary underlying structure.
	//
	// For now we are going to ask for the file name here and handle the QDom...
	// Later on this is done a level further up and only QDomNode * is sent.
	//
	//////////////////////////////////////////////////////////////////////////////////
//#if (QT_VERSION > 0x0301FF)
#ifdef QDVD_RENDER
	QString fileName = QFileDialog::getOpenFileName ( NULL, QString("./"), QObject::tr ("Slideshow files ( *.slide)"));
#else
	QString fileName = QFileDialog::getOpenFileName ( QString("./"), QObject::tr ("Slideshow files ( *.slide)"));
#endif
	return readXml (fileName);
}

bool CXmlSlideshow::readXml ( QString &fileName, bool bMessageBox )
{
	// Here we read in a xml - file and create the neccesary underlying structure.
	//
	//////////////////////////////////////////////////////////////////////////////////
	// Assign the file
	QFile projectFile ( fileName );
	QString qsErrorMsg;
	int iErrorLine;
	int iErrorCol;

#ifdef QDVD_RENDER
	if ( ! projectFile.open ( QIODevice::ReadWrite ) )
		return false;
#else
	if ( ! projectFile.open ( IO_ReadWrite ) )
		return false;
#endif
	QDomDocument xmlDoc ( SLIDESHOW_DOCTYPE );
	if ( ! xmlDoc.setContent ( &projectFile, true, &qsErrorMsg, &iErrorLine, &iErrorCol ) )  {
		// Error handling ...
		projectFile.close ( );
		QString qsHeader = QObject::tr ( "Slideshow file seems to be defective." );
		QString qsBody   = QObject::tr ( "There is a problem at line %1 at column %2\nwith your slideshow file %3\nError string = %4\n\nDo you want to try to load another slideshow file ?" ) 
			.arg ( iErrorLine ).arg ( iErrorCol). arg ( fileName ).arg ( qsErrorMsg);
		if ( bMessageBox )  {
			int iReturn = QMessageBox::warning ( NULL, qsHeader, qsBody, QMessageBox::Yes, QMessageBox::No );
			if (iReturn == QMessageBox::Yes)
				return readXml (); // Okay, user wants to specify another project file.
		}
		else  {
			printf ( "Slideshow file <%s> seems to be defective\n", (const char *)fileName.toAscii ( ) );
			printf ( "Error row<%d> col<%d> . ErrorString<%s>\n", iErrorLine, iErrorCol, (const char *)qsErrorMsg.toAscii ( ) );
			return false;
		}
	}

	QDomElement docElem = xmlDoc.documentElement ( );
	bool bReturn = readXml ( &docElem );
	projectFile.close ( );
	return bReturn;
}

bool CXmlSlideshow::readXml ( QDomElement *pElement )
{
	// And now read in all remaining nodes and handle them accordingly.
	img_struct    tempImg;     // temp to get the node_name.
	filter_struct tempFilter;  // temp to get the node_name.
	time_object   *pTimeObject = NULL;
	bool bReturn = false;
	intro_page   = true;

	QDomNode xmlNode = pElement->firstChild ( );
	QDomAttr a = pElement->attributeNode ( SLIDESHOW_DELAY );
	delay = a.value ( ).toFloat ( );
	a = pElement->attributeNode ( SLIDESHOW_INTRO );
	if ( !  a.value ( ).isEmpty ( ) )
		intro_page = a.value( ).toInt ( );
	a = pElement->attributeNode ( SLIDESHOW_FILTER_DELAY );
	filter_delay    = a.value ( ).toFloat ( );
	a = pElement->attributeNode ( SLIDESHOW_BACKGROUND );
	background      = a.value ( );
	a = pElement->attributeNode ( SLIDESHOW_NAME );
	slideshow_name  = a.value ( );
	a = pElement->attributeNode ( SLIDESHOW_XRES );
	xres  = a.value ( ).toInt ( );
	a = pElement->attributeNode ( SLIDESHOW_YRES );
	yres  = a.value ( ).toInt ( );
	a = pElement->attributeNode ( SLIDESHOW_AUDIO_LIST );
#ifdef QDVD_RENDER
	audio_list = a.value ( ).split ( "+-+");
#else
	audio_list = QStringList::split ( "+-+", a.value ( ) );
#endif
	if (xres < 1)
		xres = 720;
	if (yres < 1)
		yres = 480;

	if ( delay <= 0.0f )
		delay = 5.0f;
	if (filter_delay <= 0.0f )
		filter_delay = 3.0f;
	float fCurrentTime = 0.0f;
	while( ! xmlNode.isNull ( ) ) {
		QDomElement searchTree = xmlNode.toElement ( );
		pTimeObject = NULL;
		if (searchTree.tagName ( ) == tempImg.node_name )
			pTimeObject = (time_object *)addImg ( );
		else if ( searchTree.tagName ( ) == tempFilter.node_name )
			pTimeObject = (time_object *)addFilter ( );
		if ( pTimeObject )
			bReturn = pTimeObject->readXml ( &searchTree );
		else
			return false;
		// If there has been a problem then return false.
		if ( ! bReturn )
			return false;

		// And here we calculate some extra values ...
		if (pTimeObject->fDuration <= 0.0f) {
			if (searchTree.tagName() == tempImg.node_name)
				pTimeObject->fDuration = delay;
			else
				pTimeObject->fDuration = filter_delay;
		}
		pTimeObject->fStartTime = fCurrentTime;
		fCurrentTime += pTimeObject->fDuration;
		pTimeObject->fEndTime = fCurrentTime;
		// Otherwise go to the next node ...
		xmlNode = xmlNode.nextSibling ( );
	}
	return true;
}

bool CXmlSlideshow::writeXml ()
{
	// Here we write to a xml - file and create the neccesary underlying structure.
	//
	// For now we are going to ask for the file name here and handle the QDom...
	// Later on this is done a level further up and only QDomNode * is sent.
	//
	//////////////////////////////////////////////////////////////////////////////////
#ifdef QDVD_RENDER
	QString fileName = QFileDialog::getSaveFileName ( NULL, QString("./"), QObject::tr ("Slideshow files ( *.slide)"));
#else
	QString fileName = QFileDialog::getSaveFileName ( QString("./"), QObject::tr ("Slideshow files ( *.slide)"));
#endif
	if (fileName.isNull())
		return false;
	return writeXml(fileName);
}

bool CXmlSlideshow::writeXml (QString &fileName)
{
//	debug_out ("CXmlSlideshow::writeXml <%s><%s>\n",(const char *)pNodeElement->tagName(), (const char *)node_name);
	// Assign the file
	QFile projectFile ( fileName );
#ifdef QDVD_RENDER
	if ( ! projectFile.open ( QIODevice::WriteOnly ) )
		return false;
#else
	if ( ! projectFile.open ( IO_WriteOnly ) )
		return false;
#endif
	QDomDocument xmlDoc ( SLIDESHOW_DOCTYPE );	// <""> for now.
	QDomElement  slideshowNode = xmlDoc.createElement ( node_name );  // <slideshow>
	bool bReturn = writeXml ( &slideshowNode );
	QString xml  = xmlDoc.toString  ( );
//	printf ("%s\n", (const char *)xml);
#ifdef QDVD_RENDER
	projectFile.write ( xml.toUtf8 ( ), (qint64)xml.length ( ) );
#else
	projectFile.writeBlock ( xml.utf8 ( ), (Q_INT64)xml.length ( ) );
#endif
	projectFile.close ( );
	return bReturn;
}

bool CXmlSlideshow::writeXml ( QDomElement *pElement )
{
	int t;
	QDomDocument xmlDoc = pElement->ownerDocument();

	if ( delay > 0.0f )
		pElement->setAttribute( SLIDESHOW_DELAY,  QString ( "%1" ).arg ( delay ) );
	if ( filter_delay > 0.0f )
		pElement->setAttribute( SLIDESHOW_FILTER_DELAY,  QString ( "%1" ).arg ( filter_delay ) );
	if ( ! background.isNull ( ) )
		pElement->setAttribute( SLIDESHOW_BACKGROUND, background );
	if ( ! slideshow_name.isNull  ( ) )
		pElement->setAttribute( SLIDESHOW_NAME, slideshow_name );
	if ( xres > 0 )
		pElement->setAttribute( SLIDESHOW_XRES,  QString ("%1").arg(xres));
	if ( yres > 0 )
		pElement->setAttribute( SLIDESHOW_YRES,  QString ("%1").arg(yres));
	if ( ! intro_page )
		pElement->setAttribute( SLIDESHOW_INTRO,  QString ("%1").arg ( (int)intro_page ) );
	if ( audio_list.count ( ) > 0 )
		pElement->setAttribute( SLIDESHOW_AUDIO_LIST, audio_list.join (QString ("+-+")) );
	// And now proceed to writing the rest of the file.
	xmlDoc.appendChild ( *pElement );

	for ( t=0; t<(int)m_listTimeObjects.count ( ); t++ )
		m_listTimeObjects[t]->writeXml ( &xmlDoc, pElement );
	return true;
}

uint CXmlSlideshow::count ()
{
	return m_listTimeObjects.count();
}

int  CXmlSlideshow::findImg (CXmlSlideshow::img_struct *pImg)
{
	uint t;
	for (t=0;t<count();t++)	{
		if (m_listTimeObjects[t] == (time_object *)pImg)
			return t;
	}
	return -1;
}

uint CXmlSlideshow::countImg ()
{
	if (!ppArrayImg)
		return 0;
	int iCount = 0;
	img_struct *pImg = ppArrayImg[iCount];
	while (pImg)	{
		iCount ++;
		pImg = ppArrayImg[iCount];
	}
	return iCount;
}

uint CXmlSlideshow::countFilter ()
{
	if (!ppArrayFilter)
		return 0;
	int iCount = 0;
	filter_struct *pFilter = ppArrayFilter[iCount];
	while (pFilter)	{
		iCount ++;
		pFilter = ppArrayFilter[iCount];
	}
	return iCount;
}

CXmlSlideshow::time_object *CXmlSlideshow::getTimeObject(uint iObjectNo)
{
	if (iObjectNo < (uint)m_listTimeObjects.count())
		return m_listTimeObjects[iObjectNo];
	else
		return NULL;
}

CXmlSlideshow::img_struct *CXmlSlideshow::getImg(uint iImgNo)
{
	if (iImgNo < countImg())
		return ppArrayImg[iImgNo];
	else
		return NULL;
}

CXmlSlideshow::filter_struct *CXmlSlideshow::getFilter(uint iFilterNo)
{
	if (iFilterNo < countFilter())
		return ppArrayFilter[iFilterNo];
	else
		return NULL;
}

CXmlSlideshow::img_struct *CXmlSlideshow::addImg()
{
	// This function simply enlarges the array of available titlesets.
	// That'll keep the structure dynamic.
	if (!ppArrayImg)	{
		// This is the first ...
		ppArrayImg=new img_struct *[2];
		ppArrayImg[0]=new img_struct;
		ppArrayImg[1]=NULL;
		m_listTimeObjects.append (ppArrayImg[0]);
		return ppArrayImg[0];
	}

	int i,t=0;
	img_struct *pImg = ppArrayImg[t];
	img_struct **ppNewArray=NULL, **ppOldArray=NULL;
	while (pImg)	{
		t ++;
		pImg = ppArrayImg[t];
	}
	// Now we have the count of actual titlesets.
	ppNewArray=new img_struct *[t+2];
	for (i=0;i<t;i++)	{
		ppNewArray[i]=ppArrayImg[i];
	}
	ppNewArray[i] = new img_struct();
	ppNewArray[i+1] = NULL;
	// Now we can delete th old array (but not the contents.
	ppOldArray = ppArrayImg;
	ppArrayImg = ppNewArray;
	delete []ppOldArray;
	// And finally return the latest addition ...
	m_listTimeObjects.append (ppArrayImg[i]);
	return ppArrayImg[i];
}

void CXmlSlideshow::delImg ( CXmlSlideshow::img_struct *pImg )
{
  if ( ! ppArrayImg )
    return;

#ifdef QDVD_RENDER
  m_listTimeObjects.removeAll ( pImg );
#else
  m_listTimeObjects.remove ( pImg );
#endif

  int t=0;
  CXmlSlideshow::img_struct *pArray = ppArrayImg[0];
  // First we find where the Img is ...
  while ( pArray && ( pArray != pImg ) )
    pArray = ppArrayImg[++t];

  // Make sure we did not run through the whole array
  if ( pImg == pArray ) {
    // and shift all remaining entries one to the left
    while ( pArray )  {
      pArray = ppArrayImg[++t];
      ppArrayImg[t-1] = pArray;
    }

    delete pImg;
  }
}

CXmlSlideshow::filter_struct *CXmlSlideshow::addFilter()
{
	// This function simply enlarges the array of available titlesets.
	// That'll keep the structure dynamic.
	if (!ppArrayFilter)	{
		// This is the first ...
		ppArrayFilter = new filter_struct *[2];
		ppArrayFilter[0]=new filter_struct;
		ppArrayFilter[1]=NULL;
		m_listTimeObjects.append (ppArrayFilter[0]);
		return ppArrayFilter[0];
	}

	int i,t=0;
	filter_struct *pFilter = ppArrayFilter[t];
	filter_struct **ppNewArray=NULL, **ppOldArray=NULL;
	while (pFilter)	{
		t ++;
		pFilter = ppArrayFilter[t];
	}
	// Now we have the count of actual titlesets.
	ppNewArray=new filter_struct *[t+2];
	for (i=0;i<t;i++)	{
		ppNewArray[i]=ppArrayFilter[i];
	}
	ppNewArray[i] = new filter_struct();
	ppNewArray[i+1] = NULL;
	// Now we can delete th old array (but not the contents.
	ppOldArray = ppArrayFilter;
	ppArrayFilter = ppNewArray;
	delete []ppOldArray;
	// And finally return the latest addition ...
	m_listTimeObjects.append (ppArrayFilter[i]);
	return ppArrayFilter[i];
}

CXmlSlideshow::time_object::time_object ()
{
	node_name	= QString ("");
	fDuration	= -1.0f;	// Define the duration in seconds
	fStartTime	= -1.0f;
	fEndTime	= -1.0f;
	pModifier 	= NULL;
}

CXmlSlideshow::img_struct::img_struct ()
	: time_object ()
{
	node_name = QString ( "img");
	src       = QString (   "" );
	text      = QString (   "" );
	width     =    0;
	height    =    0;
	pMatrix   = NULL;
	rotate    = 0.0f;
	effect[0].node_name = effect[0].node_name + "_start";
	effect[1].node_name = effect[1].node_name + "_end";
}

CXmlSlideshow::img_struct::~img_struct ()
{
  if ( pMatrix )
    delete pMatrix;
  pMatrix = NULL;
#ifndef QDVD_RENDER
	if (pModifier)	{
		ImageManipulator *pManipulator = (ImageManipulator *)pModifier;
		delete pManipulator;
	}
#endif
}

CXmlSlideshow::filter_struct::filter_struct ()
	: time_object ()
{
	node_name = QString ("filter");
	name = QString ("");	// the type of filter, e.g. crossfade/fadein/fadeout
	subtitle = QString ("");
}

CXmlSlideshow::effect_struct::effect_struct ()
{
	node_name = QString ("effect");
	QString name;		// either crop/kenburns/scroll
	x0  = 0;
	y0  = 0;
	x1  = 0;
	y1  = 0;
	xe0 = 0;
	ye0 = 0;
	xe1 = 0;
	ye1 = 0;
	scroll = QString ("");	// either left or right
}

bool CXmlSlideshow::img_struct::readXml (QDomElement *pDocElem)
{
//	bool bReturn = true;
	if (pDocElem->tagName() != node_name)
		return false;
	// So lets get first the attributes for this node.
	QDomAttr a = pDocElem->attributeNode ( IMG_SRC );
	src = a.value();
	a = pDocElem->attributeNode ( IMG_TEXT );
	text = a.value();
	a = pDocElem->attributeNode ( IMG_WIDTH );
	width = a.value().toInt();
	a = pDocElem->attributeNode ( IMG_HEIGHT );
	height = a.value().toInt();
	a = pDocElem->attributeNode ( IMG_ROTATE );
	rotate = a.value().toFloat();
	a = pDocElem->attributeNode ( IMG_DURATION );
	fDuration = a.value().toFloat();
	a = pDocElem->attributeNode ( IMG_MATRIX );
	if ( pMatrix )
		delete pMatrix;
	pMatrix = NULL;
#ifdef QDVD_RENDER
	if ( ! a.value ( ).isEmpty ( ) )  {
		QStringList list = a.value ( ).split ( "," );
		qreal fValues[10]; //m11, m12, m21, m22, dx, dy;
		for ( int t=0; t<list.count ( ); t++ )
			fValues[t] = list[t].toDouble ( );
		pMatrix = new QMatrix ( fValues[0], fValues[1], fValues[2], fValues[3], fValues[4], fValues[5] );
//		pMatrix = new QMatrix ( qreal m11, qreal m12, qreal m21, qreal m22, qreal dx, qreal dy );
	}
#else
	if ( ! a.value ( ).isEmpty ( ) )  {
		QStringList list = QStringList::split ( ",", a.value ( ) );
		double fValues[10]; //m11, m12, m21, m22, dx, dy;
		for ( int t=0; t<(int)list.count ( ); t++ )
			fValues[t] = list[t].toDouble ( );
		pMatrix = new QMatrix ( fValues[0], fValues[1], fValues[2], fValues[3], fValues[4], fValues[5] );
	}
	bool bReturn = false;
	QDomNode xmlNode = pDocElem->firstChild ( );
	ImageManipulator tmpImageManipulator;
	while( !xmlNode.isNull() ) {
		QDomElement searchTree = xmlNode.toElement();
		ImageManipulator *pImageManipulator = new ImageManipulator;
		if (searchTree.tagName() == effect[0].node_name)
			bReturn = effect[0].readXml(&searchTree);
		if (searchTree.tagName() == effect[1].node_name)
			bReturn = effect[1].readXml(&searchTree);
		if (searchTree.tagName() == tmpImageManipulator.node_name)	{
			pModifier = (void *)pImageManipulator;
			bReturn = pImageManipulator->readProjectFile( searchTree );
		}
		// If there has been a problem then return false.
		if (!bReturn)
			return false;
		// Otherwise go to the next node ...
		xmlNode = xmlNode.nextSibling();
	}
#endif
	return true;
}

bool CXmlSlideshow::effect_struct::readXml (QDomElement *pDocElem)
{
	if (pDocElem->tagName() != node_name)
		return false;
	// So lets get first the attributes for this node.
	QDomAttr a = pDocElem->attributeNode ( EFFECT_NAME );
	name = a.value();
	a = pDocElem->attributeNode ( EFFECT_X0 );
	x0 = a.value().toInt();
	a = pDocElem->attributeNode ( EFFECT_Y0 );
	y0 = a.value().toInt();
	a = pDocElem->attributeNode ( EFFECT_X1 );
	x1 = a.value().toInt();
	a = pDocElem->attributeNode ( EFFECT_Y1 );
	y1 = a.value().toInt();

	a = pDocElem->attributeNode ( EFFECT_XE0 );
	xe0 = a.value().toInt();
	a = pDocElem->attributeNode ( EFFECT_YE0 );
	ye0 = a.value().toInt();
	a = pDocElem->attributeNode ( EFFECT_XE1 );
	xe1 = a.value().toInt();
	a = pDocElem->attributeNode ( EFFECT_YE1 );
	ye1 = a.value().toInt();
	a = pDocElem->attributeNode ( EFFECT_SCROLL );
	scroll = a.value();
	return true;
}

bool CXmlSlideshow::filter_struct::readXml (QDomElement *pDocElem)
{
	if (pDocElem->tagName() != node_name)
		return false;
	// So lets get first the attributes for this node.
	QDomAttr a = pDocElem->attributeNode ( FILTER_NAME );
	name = a.value();
	a = pDocElem->attributeNode ( FILTER_DURATION );
	fDuration = a.value().toFloat();
	a = pDocElem->attributeNode ( FILTER_SUBTITLE );
	subtitle = a.value();
	return true;
}

bool CXmlSlideshow::img_struct::writeXml(QDomDocument *pDocument, QDomElement *pNodeElement)
{
//	debug_out ("CXmlSlideshow::img_struct::writeXml <%s><%s>\n",(const char *)pNodeElement->tagName(), (const char *)node_name);
	QDomElement imgNode = pDocument->createElement( node_name );	// <img>

	if (!src.isEmpty())
		imgNode.setAttribute( IMG_SRC, src );
	if (!text.isEmpty())
		imgNode.setAttribute( IMG_TEXT, text );
	if (width > 1)
		imgNode.setAttribute( IMG_WIDTH, QString ("%1").arg(width) );
	if (height > 1)
		imgNode.setAttribute( IMG_HEIGHT, QString ("%1").arg(height) );
	if (rotate != 0.0)
		imgNode.setAttribute( IMG_ROTATE, QString ("%1").arg(rotate) );
	if (fDuration > -1.0f)
		imgNode.setAttribute( IMG_DURATION, QString ("%1").arg(fDuration) );
//	if (pModifier)
//		((ImageManipulator *)pModifier)->writeProjectFile( imgNode );
	if ( pMatrix )  {
		QString qsMatrix = QString ( "%1,%2,%3,%4,%5,%6" )
		.arg ( pMatrix->m11 ( ) ).arg ( pMatrix->m12 ( ) )
		.arg ( pMatrix->m21 ( ) ).arg ( pMatrix->m22 ( ) )
		.arg ( pMatrix->dx  ( ) ).arg ( pMatrix->dy  ( ) );
		imgNode.setAttribute ( IMG_MATRIX, qsMatrix );
	}

	pNodeElement->appendChild( imgNode );
	effect[0].writeXml(pDocument, pNodeElement);
	return effect[1].writeXml(pDocument, pNodeElement);
}

bool CXmlSlideshow::effect_struct::writeXml(QDomDocument *pDocument, QDomElement *pNodeElement)
{
//	debug_out ("CXmlSlideshow::effect_struct::writeXml <%s><%s>\n",(const char *)pNodeElement->tagName(), (const char *)node_name);
	QDomElement effectNode = pDocument->createElement( node_name );	// <img>

	if (!name.isNull())
		effectNode.setAttribute( EFFECT_NAME, name );
	else
		return true;
	if (x0 > 1)
		effectNode.setAttribute( EFFECT_X0, QString ("%1").arg(x0) );
	if (y0 > 1)
		effectNode.setAttribute( EFFECT_Y0, QString ("%1").arg(y0) );
	if (x1 > 1)
		effectNode.setAttribute( EFFECT_X1, QString ("%1").arg(x1) );
	if (y1 > 1)
		effectNode.setAttribute( EFFECT_Y1, QString ("%1").arg(y1) );
	if (xe0 > 1)
		effectNode.setAttribute( EFFECT_XE0, QString ("%1").arg(xe0) );
	if (ye0 > 1)
		effectNode.setAttribute( EFFECT_YE0, QString ("%1").arg(ye0) );
	if (xe1 > 1)
		effectNode.setAttribute( EFFECT_XE1, QString ("%1").arg(xe1) );
	if (ye1 > 1)
		effectNode.setAttribute( EFFECT_YE1, QString ("%1").arg(ye1) );
	if (!scroll.isEmpty())
		effectNode.setAttribute( EFFECT_SCROLL, scroll );
	pNodeElement->appendChild( effectNode );
	return true;
}

bool CXmlSlideshow::filter_struct::writeXml(QDomDocument *pDocument, QDomElement *pNodeElement)
{
//	debug_out ("CXmlSlideshow::effect_struct::writeXml <%s><%s>\n",(const char *)pNodeElement->tagName(), (const char *)node_name);
	QDomElement filterNode = pDocument->createElement( node_name );	// <img>

	if (!name.isNull())
		filterNode.setAttribute( FILTER_NAME, name );
	if (fDuration > 0.0f)
		filterNode.setAttribute( FILTER_DURATION, QString ("%1").arg(fDuration) );
	if (!subtitle.isEmpty())
		filterNode.setAttribute( FILTER_SUBTITLE, subtitle );
	pNodeElement->appendChild( filterNode );
	return true;
}

CXmlSlideshow::effect_struct& CXmlSlideshow::effect_struct::operator =(CXmlSlideshow::effect_struct &theOther)
{
	name = theOther.name;
	x0 = theOther.x0;
	y0 = theOther.y0;
	x1 = theOther.x1;
	y1 = theOther.y1;
	xe0 = theOther.xe0;
	ye0 = theOther.ye0;
	xe1 = theOther.xe1;
	ye1 = theOther.ye1;
	scroll = theOther.scroll;
	return *this;
}
bool CXmlSlideshow::effect_struct::operator == ( CXmlSlideshow::effect_struct &theOther )
{
	return ( ( name   == theOther.name   ) &&
		 ( x0     == theOther.x0     ) &&
		 ( y0     == theOther.y0     ) &&
		 ( x1     == theOther.x1     ) &&
		 ( y1     == theOther.y1     ) &&
		 ( xe0    == theOther.xe0    ) &&
		 ( ye0    == theOther.ye0    ) &&
		 ( xe1    == theOther.xe1    ) &&
		 ( ye1    == theOther.ye1    ) &&
		 ( scroll == theOther.scroll ) );
}

CXmlSlideshow::time_object& CXmlSlideshow::img_struct::operator =(CXmlSlideshow::time_object &theOther)
{
	CXmlSlideshow::img_struct *pTheOther = (CXmlSlideshow::img_struct *)&theOther;
#ifndef QDVD_RENDER
	ImageManipulator *pMan, *pOtherMan;
	pMan = (ImageManipulator *)pModifier;
	// check if we already have a modifier ...
	if ( pMan )
		delete pMan;
	pMan = NULL;
	if (pTheOther->pModifier)	{
		pOtherMan = (ImageManipulator *)pTheOther->pModifier;
		//create a modifier, and copy over ...
		pMan = new ImageManipulator;
		*pMan = *pOtherMan;
		pModifier = pMan;
	}
#endif
	if ( pMatrix )
		delete pMatrix;
	pMatrix = NULL;
	if ( pTheOther->pMatrix )
		pMatrix = new QMatrix ( *pTheOther->pMatrix );

	src = pTheOther->src;
	text = pTheOther->text;
	width = pTheOther->width;
	height = pTheOther->height;
	rotate  = pTheOther->rotate;
	effect[0]= pTheOther->effect[0];
	effect[1] = pTheOther->effect[1];
	fEndTime   = pTheOther->fEndTime;
	fDuration   = pTheOther->fDuration;
	fStartTime   = pTheOther->fStartTime;
	return *this;
}
bool CXmlSlideshow::img_struct::operator == ( CXmlSlideshow::time_object &theOther )
{
	CXmlSlideshow::img_struct *pTheOther = (CXmlSlideshow::img_struct *)&theOther;
	bool bReturn = ( 
		 ( src == pTheOther->src                 ) &&
		 ( text == pTheOther->text               ) &&
		 ( width == pTheOther->width             ) &&
		 ( height == pTheOther->height           ) &&
		 ( rotate  == pTheOther->rotate          ) &&
		 ( effect[0]== pTheOther->effect[0]      ) &&
		 ( effect[1] == pTheOther->effect[1]     ) &&
		 ( fEndTime   == pTheOther->fEndTime     ) &&
		 ( fDuration   == pTheOther->fDuration   ) &&
		 ( fStartTime   == pTheOther->fStartTime ) );

	if ( ! bReturn )
		return false;

	if ( ( pMatrix == NULL ) && ( pTheOther->pMatrix == NULL ) )
		return true;

	if ( pMatrix && pTheOther->pMatrix )
		return ( *pMatrix == *pTheOther->pMatrix );

	return false;
}

CXmlSlideshow::time_object& CXmlSlideshow::filter_struct::operator =(CXmlSlideshow::time_object &theOther)
{
	CXmlSlideshow::filter_struct *pTheOther = (CXmlSlideshow::filter_struct *)&theOther;
	name   = pTheOther->name;
	subtitle= pTheOther->subtitle;
	fEndTime = pTheOther->fEndTime;
	fDuration = pTheOther->fDuration;
	fStartTime = pTheOther->fStartTime;
	return *this;
}
bool CXmlSlideshow::filter_struct::operator == ( CXmlSlideshow::time_object &theOther )
{
	CXmlSlideshow::filter_struct *pTheOther = (CXmlSlideshow::filter_struct *)&theOther;
	return ( ( name   == pTheOther->name           ) &&
		 ( subtitle== pTheOther->subtitle      ) &&
		 ( fEndTime == pTheOther->fEndTime     ) &&
		 ( fDuration == pTheOther->fDuration   ) &&
		 ( fStartTime == pTheOther->fStartTime ) );
}

