/*
 * Copyright (c) 2004-2005 The Trustees of Indiana University and Indiana
 *                         University Research and Technology
 *                         Corporation.  All rights reserved.
 * Copyright (c) 2004-2005 The University of Tennessee and The University
 *                         of Tennessee Research Foundation.  All rights
 *                         reserved.
 * Copyright (c) 2004-2005 High Performance Computing Center Stuttgart,
 *                         University of Stuttgart.  All rights reserved.
 * Copyright (c) 2004-2005 The Regents of the University of California.
 *                         All rights reserved.
 * $COPYRIGHT$
 *
 * Additional copyrights may follow
 *
 * $HEADER$
 */
/** @file:
 *
 * The Open MPI general purpose registry - support functions.
 *
 */

/*
 * includes
 */

#include "orte_config.h"

#include "orte/mca/errmgr/errmgr.h"
#include "orte/mca/ns/ns.h"
#include "opal/util/output.h"
#include "opal/util/trace.h"

#include "orte/mca/gpr/replica/api_layer/gpr_replica_api.h"
#include "orte/mca/gpr/replica/transition_layer/gpr_replica_tl.h"
#include "orte/mca/gpr/replica/functional_layer/gpr_replica_fn.h"


/*
 * GENERAL REGISTRY TRIGGER FUNCTIONS
 */
int
orte_gpr_replica_register_subscription(orte_gpr_replica_subscription_t **subptr,
                                    orte_process_name_t *requestor,
                                    orte_gpr_subscription_t *subscription)
{
    int rc;
    orte_std_cntr_t i, j, k, num_tokens, num_keys;
    orte_gpr_replica_subscription_t *sub, **subs;
    orte_gpr_replica_requestor_t *req, **reqs;
    orte_gpr_replica_addr_mode_t tok_mode, key_mode;
    orte_gpr_replica_itag_t itag, *tokentags=NULL;
    orte_gpr_replica_ivalue_t *ival;

    OPAL_TRACE(3);

    /* if this is a named subscription, see if that name has
     * already been entered on the replica. If it has, then we
     * simply attach this recipient to that subscription -
     * this indicates that this recipient would also like a
     * copy of the data generated by that subscription
     */
    if (NULL != subscription->name) {
        /* look for this name on current list */
        subs = (orte_gpr_replica_subscription_t**)(orte_gpr_replica.subscriptions)->addr;
        for (k=0, j=0; j < orte_gpr_replica.num_subs &&
                       k < (orte_gpr_replica.subscriptions)->size; k++) {
            if (NULL != subs[k]) {
                j++;
                if (NULL != subs[k]->name && NULL != subscription->name &&
                0 == strcmp(subs[k]->name, subscription->name)) {
                    /* found name on list - add another recipient to that
                     * subscription
                     */
                    sub = subs[k];
                    goto ADDREQ;
                }
            }
        }
    }

    /* Either this is NOT a named subscription, or it is named
     * but that name is NOT on the current list of subscriptions.
     * Either way, we add this subscription to the replica's list.
     *
     * NOTE that you CANNOT add yourself as a recipient to a non-named
     * subscription - even if all the subscription specifications are
     * identical. This is done in the interest of speed as checking
     * all the specifications would take some time. Subscriptions are
     * "named" because they are intended to be used by multiple processes.
     * Un-named subscriptions are, therefore, assumed to be specialty
     * subscriptions that do not merit such consideration.
     */

     /* see if another subscription is available on the system */
    if (ORTE_GPR_SUBSCRIPTION_ID_MAX-1 < orte_gpr_replica.num_subs) { /* none left! */
        ORTE_ERROR_LOG(ORTE_ERR_OUT_OF_RESOURCE);
        return ORTE_ERR_OUT_OF_RESOURCE;
    }

    sub = OBJ_NEW(orte_gpr_replica_subscription_t);
    if (NULL == sub) {
        ORTE_ERROR_LOG(ORTE_ERR_OUT_OF_RESOURCE);
        return ORTE_ERR_OUT_OF_RESOURCE;
    }
    sub->idtag = orte_gpr_replica.num_subs;

    if (NULL != subscription->name) {
        sub->name = strdup(subscription->name);
    }
    sub->action = subscription->action;
    if (ORTE_GPR_NOTIFY_STARTS_AFTER_TRIG & sub->action) {
        sub->active = false;
    } else {
        sub->active = true;
    }

    /* store all the data specifications for this subscription */
    for (i=0; i < subscription->cnt; i++) {
        ival = OBJ_NEW(orte_gpr_replica_ivalue_t);
        if (NULL == ival) {
            ORTE_ERROR_LOG(ORTE_ERR_OUT_OF_RESOURCE);
            OBJ_RELEASE(sub);
            return ORTE_ERR_OUT_OF_RESOURCE;
        }

        /* find and store the segment */
        if (ORTE_SUCCESS != (rc = orte_gpr_replica_find_seg(&(ival->seg), true,
                                           subscription->values[i]->segment))) {
            ORTE_ERROR_LOG(rc);
            OBJ_RELEASE(sub);
            OBJ_RELEASE(ival);
            return rc;
        }
        tok_mode = ORTE_GPR_REPLICA_TOKMODE((subscription->values[i])->addr_mode);
        if (0x00 == tok_mode) {  /* default token address mode to AND */
            subscription->values[i]->addr_mode = subscription->values[i]->addr_mode | ORTE_GPR_TOKENS_AND;
        }
        key_mode = ORTE_GPR_REPLICA_KEYMODE((subscription->values[i])->addr_mode);
        if (0x00 == key_mode) {  /* default key address mode to OR */
            key_mode = subscription->values[i]->addr_mode = subscription->values[i]->addr_mode | ORTE_GPR_KEYS_OR;
        }
        ival->addr_mode = ORTE_GPR_REPLICA_REMOVE_OVERWRITE(subscription->values[i]->addr_mode);

        if (NULL != subscription->values[i]->tokens &&
            0 < subscription->values[i]->num_tokens) {
            num_tokens = subscription->values[i]->num_tokens; /* indicates non-NULL terminated list */
            if (ORTE_SUCCESS != (rc = orte_gpr_replica_get_itag_list(&tokentags, ival->seg,
                                    subscription->values[i]->tokens, &num_tokens))) {
                ORTE_ERROR_LOG(rc);
                OBJ_RELEASE(sub);
                OBJ_RELEASE(ival);
                return rc;

            }
            if (ORTE_SUCCESS != (rc = orte_value_array_set_size(&(ival->tokentags), (orte_std_cntr_t)num_tokens))) {
                ORTE_ERROR_LOG(rc);
                OBJ_RELEASE(sub);
                OBJ_RELEASE(ival);
                return rc;
            }
            for (j=0; j < num_tokens; j++) {
                ORTE_VALUE_ARRAY_SET_ITEM(&(ival->tokentags), orte_gpr_replica_itag_t,
                                                j, tokentags[j]);
            }
            free(tokentags);
            tokentags = NULL;
        }

        if (NULL != subscription->values[i]->keyvals &&
            0 < subscription->values[i]->cnt) {
            num_keys = subscription->values[i]->cnt;
            if (ORTE_SUCCESS != (rc = orte_value_array_set_size(&(ival->keytags), num_keys))) {
                ORTE_ERROR_LOG(rc);
                OBJ_RELEASE(sub);
                OBJ_RELEASE(ival);
                return rc;
            }
            for (j=0; j < num_keys; j++) {
                if (ORTE_SUCCESS != (rc = orte_gpr_replica_create_itag(&itag,
                                     ival->seg,
                                     subscription->values[i]->keyvals[j]->key))) {
                    ORTE_ERROR_LOG(rc);
                    OBJ_RELEASE(sub);
                    OBJ_RELEASE(ival);
                    return rc;
                }
                ORTE_VALUE_ARRAY_SET_ITEM(&(ival->keytags), orte_gpr_replica_itag_t,
                                                j, itag);
            }
        }
            /* add the object to the subscription's value pointer array */
        if (0 > (rc = orte_pointer_array_add(&(ival->index), sub->values, ival))) {
            ORTE_ERROR_LOG(ORTE_ERR_OUT_OF_RESOURCE);
            OBJ_RELEASE(sub);
            OBJ_RELEASE(ival);
            return ORTE_ERR_OUT_OF_RESOURCE;
        }
        (sub->num_values)++;
    }
    /* add the object to the replica's subscriptions pointer array */
    if (0 > (rc = orte_pointer_array_add(&(sub->index), orte_gpr_replica.subscriptions, sub))) {
        ORTE_ERROR_LOG(ORTE_ERR_OUT_OF_RESOURCE);
        OBJ_RELEASE(sub);
        return ORTE_ERR_OUT_OF_RESOURCE;
    }
    (orte_gpr_replica.num_subs)++;

ADDREQ:
    /* see if this requestor and subscription id is already attached to
     * this subscription - if so, ignore it to avoid duplicates
     */
    reqs = (orte_gpr_replica_requestor_t**)(sub->requestors)->addr;
    for (i=0, j=0; j < sub->num_requestors &&
                   i < (sub->requestors)->size; i++) {
        if (NULL != reqs[i]) {
            j++;
            if ((NULL == reqs[i]->requestor && NULL != requestor) ||
                (NULL != reqs[i]->requestor && NULL == requestor)) {
                continue;
            }
            if (reqs[i]->idtag == subscription->id &&
                ((NULL == reqs[i]->requestor && NULL == requestor) ||
                 (ORTE_EQUAL == orte_dss.compare(reqs[i]->requestor, requestor, ORTE_NAME)))) {
                /* found this requestor - do not add it again */
                goto DONESUB;
            }
        }
    }

    /* get here if requestor is not already on this subscription
     * add this requestor to the subscription
     */
    req = OBJ_NEW(orte_gpr_replica_requestor_t);
    if (NULL == req) {
        ORTE_ERROR_LOG(ORTE_ERR_OUT_OF_RESOURCE);
        return ORTE_ERR_OUT_OF_RESOURCE;
    }

    if (NULL != requestor) {
        if (ORTE_SUCCESS != (rc = orte_dss.copy((void**)&(req->requestor), requestor, ORTE_NAME))) {
              ORTE_ERROR_LOG(rc);
              return rc;
        }
    } else {
         req->requestor = NULL;
    }

    if (0 > (rc = orte_pointer_array_add(&(req->index), sub->requestors, req))) {
        ORTE_ERROR_LOG(ORTE_ERR_OUT_OF_RESOURCE);
        return ORTE_ERR_OUT_OF_RESOURCE;
    }
    (sub->num_requestors)++;

    /* store the requestor's subscription id so they can ask
     * us to cancel their subscription at a later time,
     * if they choose to do so, and so that we can tell
     * them which callback function to use when we send
     * them a datagram
     */
    req->idtag = subscription->id;

    /*
     * New subscription, send initial values?
     */
    if(sub->active && subscription->action & ORTE_GPR_NOTIFY_PRE_EXISTING) {

        if(ORTE_SUCCESS != (rc = orte_gpr_replica_register_callback(sub, NULL))) {
            ORTE_ERROR_LOG(rc);
            return rc;
        }
    }
DONESUB:
    /* record where the subscription went */
    *subptr = sub;

    return ORTE_SUCCESS;
}

int
orte_gpr_replica_register_trigger(orte_gpr_replica_trigger_t **trigptr,
                                  orte_process_name_t *requestor,
                                  orte_gpr_trigger_t *trigger)
{
    orte_gpr_replica_trigger_t *trig, **trigs;
    int rc;
    orte_std_cntr_t i, j, k, m, num_tokens, index;
    orte_gpr_replica_addr_mode_t tok_mode, key_mode;
    orte_gpr_replica_segment_t *seg;
    orte_gpr_replica_container_t **cptr, *cptr2;
    orte_gpr_replica_itag_t itag, *tokentags=NULL;
    orte_gpr_replica_itagval_t *iptr;
    orte_gpr_replica_counter_t *cntr;
    orte_gpr_replica_trigger_requestor_t *req, **reqs;
    bool found;

    OPAL_TRACE(3);

    /* set a default response value */
    *trigptr = NULL;

    /* if this is a named trigger, see if that name has
     * already been entered on the replica. If it has, then we
     * can simply return the pointer to the existing trigger.
     */
    if (NULL != trigger->name) {
        /* look for this name on current list */
        trigs = (orte_gpr_replica_trigger_t**)(orte_gpr_replica.triggers)->addr;
        for (k=0, j=0; j < orte_gpr_replica.num_trigs &&
                       k < (orte_gpr_replica.triggers)->size; k++) {
            if (NULL != trigs[k]) {
                j++;
                if (trigs[k]->name && NULL != trigger->name &&
                0 == strcmp(trigs[k]->name, trigger->name)) {
                    /* found name on list - add recipient's
                     * trigger info to that trigger
                     */
                    trig = trigs[k];
                    goto ADDREQ;
                }
            }
        }
    }

    /* Either this is NOT a named trigger, or it is named
     * but that name is NOT on the current list of triggers.
     * Either way, we add this trigger to the replica's list.
     *
     * NOTE that you CANNOT add a subscription to a pre-entered non-named
     * trigger - even if all the trigger specifications are
     * identical. This is done in the interest of speed as checking
     * all the specifications would take some time. Triggers are
     * "named" because they are intended to be used by multiple processes.
     * Un-named triggers are, therefore, assumed to be specialty
     * triggers that do not merit such consideration.
     */

    /* see if another trigger is available */
    if (ORTE_GPR_TRIGGER_ID_MAX-1 < orte_gpr_replica.num_trigs) { /* none left! */
        ORTE_ERROR_LOG(ORTE_ERR_OUT_OF_RESOURCE);
        return ORTE_ERR_OUT_OF_RESOURCE;
    }

    trig = OBJ_NEW(orte_gpr_replica_trigger_t);
    if (NULL == trig) {
        ORTE_ERROR_LOG(ORTE_ERR_OUT_OF_RESOURCE);
        return ORTE_ERR_OUT_OF_RESOURCE;
    }
    trig->idtag = orte_gpr_replica.num_trigs;

    /* if a name for this trigger has been provided, copy it over */
    if (NULL != trigger->name) {
        trig->name = strdup(trigger->name);
    }
    /* copy the action field */
    trig->action = trigger->action;

    /* put this trigger on the replica's list */
    if (0 > (rc = orte_pointer_array_add(&(trig->index), orte_gpr_replica.triggers, trig))) {
        ORTE_ERROR_LOG(ORTE_ERR_OUT_OF_RESOURCE);
        return ORTE_ERR_OUT_OF_RESOURCE;
    }
    (orte_gpr_replica.num_trigs)++;

    /* locate and setup the trigger's counters */
    for (i=0; i < trigger->cnt; i++) {
        /* get this counter's addressing modes */
        tok_mode = ORTE_GPR_REPLICA_TOKMODE((trigger->values[i])->addr_mode);
        if (0x00 == tok_mode) {  /* default token address mode to AND */
            tok_mode = ORTE_GPR_REPLICA_AND;
        }
        key_mode = ORTE_GPR_REPLICA_KEYMODE((trigger->values[i])->addr_mode);
        if (0x00 == key_mode) {  /* default key address mode to OR */
            key_mode = ORTE_GPR_REPLICA_OR;
        }

        /* locate this counter's segment - this is where the counter will be */
        if (ORTE_SUCCESS != (rc = orte_gpr_replica_find_seg(&seg, true,
                                        trigger->values[i]->segment))) {
            ORTE_ERROR_LOG(rc);
            OPAL_THREAD_UNLOCK(&orte_gpr_replica_globals.mutex);
            return rc;
        }

        /* convert the counter's tokens to an itaglist */
        if (NULL != (trigger->values[i])->tokens &&
               0 < (trigger->values[i])->num_tokens) {
            num_tokens = (trigger->values[i])->num_tokens; /* indicates non-NULL terminated list */
            if (ORTE_SUCCESS != (rc = orte_gpr_replica_get_itag_list(&tokentags, seg,
                                (trigger->values[i])->tokens, &num_tokens))) {
                ORTE_ERROR_LOG(rc);
                goto CLEANUP;
            }
        }

        /* find the specified container(s) */
        if (ORTE_SUCCESS != (rc = orte_gpr_replica_find_containers(seg, tok_mode,
                                        tokentags, num_tokens))) {
            ORTE_ERROR_LOG(rc);
            goto CLEANUP;
        }

        if (0 == orte_gpr_replica_globals.num_srch_cptr) {
            /* no existing container found - create one using all the tokens */
            if (ORTE_SUCCESS != (rc = orte_gpr_replica_create_container(&cptr2, seg,
                                                num_tokens, tokentags))) {
                ORTE_ERROR_LOG(rc);
                goto CLEANUP;
            }

            /* ok, store all of this counter's values in the new container, adding a pointer to each
             * one in the trigger's counter array
             */
            for (j=0; j < (trigger->values[i])->cnt; j++) {
                if (ORTE_SUCCESS != (rc =
                        orte_gpr_replica_add_keyval(&iptr, seg, cptr2,
                                    (trigger->values[i])->keyvals[j]))) {
                    ORTE_ERROR_LOG(rc);
                    goto CLEANUP;
                }
                cntr = OBJ_NEW(orte_gpr_replica_counter_t);
                if (NULL == cntr) {
                    ORTE_ERROR_LOG(ORTE_ERR_OUT_OF_RESOURCE);
                    return ORTE_ERR_OUT_OF_RESOURCE;
                }
                cntr->seg = seg;
                cntr->cptr = cptr2;
                cntr->iptr = iptr;
                /* if the trigger is at a level, then the requestor MUST specify the
                 * level in the provided keyval. Otherwise, we only need to store
                 * the iptr since we will be comparing levels between multiple
                 * counters
                 */
                if (trigger->action & ORTE_GPR_TRIG_AT_LEVEL) {
                    if (NULL == trigger->values[i]->keyvals) {
                        ORTE_ERROR_LOG(ORTE_ERR_BAD_PARAM);
                        rc = ORTE_ERR_BAD_PARAM;
                        goto CLEANUP;
                    }
                    cntr->trigger_level.value = OBJ_NEW(orte_data_value_t);
                    if (NULL == cntr->trigger_level.value) {
                        ORTE_ERROR_LOG(ORTE_ERR_OUT_OF_RESOURCE);
                        rc = ORTE_ERR_OUT_OF_RESOURCE;
                        goto CLEANUP;
                    }
                    cntr->trigger_level.value->type = ((trigger->values[i])->keyvals[j])->value->type;
                    if (ORTE_SUCCESS != (rc = orte_dss.copy(&((cntr->trigger_level.value)->data),
                                                            ((trigger->values[i])->keyvals[j])->value->data,
                                                            ((trigger->values[i])->keyvals[j])->value->type))) {
                        ORTE_ERROR_LOG(rc);
                        goto CLEANUP;
                    }
                }
                if (0 > orte_pointer_array_add(&index, trig->counters, cntr)) {
                    ORTE_ERROR_LOG(ORTE_ERR_OUT_OF_RESOURCE);
                    rc = ORTE_ERR_OUT_OF_RESOURCE;
                    goto CLEANUP;
                }
            }
            trig->num_counters += (trigger->values[i])->cnt;
        } else {  /* For each counter, go through the list of containers and
                     see if it already exists in container. Only allow each
                     counter to be identified once - error if either a counter is never
                     found or already existing in more than one place. */
            cptr = (orte_gpr_replica_container_t**)(orte_gpr_replica_globals.srch_cptr)->addr;
            for (j=0; j < (trigger->values[i])->cnt; j++) {
                found = false;
                if (ORTE_SUCCESS != orte_gpr_replica_dict_lookup(&itag, seg,
                            ((trigger->values[i])->keyvals[j])->key)) {
                    ORTE_ERROR_LOG(ORTE_ERR_NOT_FOUND);
                    return ORTE_ERR_NOT_FOUND;
                }
                for (k=0, m=0; m < orte_gpr_replica_globals.num_srch_cptr &&
                               k < (orte_gpr_replica_globals.srch_cptr)->size; k++) {
                    if (NULL != cptr[k]) {
                        m++;
                        if (ORTE_SUCCESS == orte_gpr_replica_search_container(
                                                ORTE_GPR_REPLICA_OR,
                                                &itag, 1, cptr[k]) &&
                            0 < orte_gpr_replica_globals.num_srch_ival) {
                            /* this key already exists - make sure it's unique
                             */
                            if (1 < orte_gpr_replica_globals.num_srch_ival || found) {
                                /* not unique - error out */
                                ORTE_ERROR_LOG(ORTE_ERR_BAD_PARAM);
                                rc = ORTE_ERR_BAD_PARAM;
                                goto CLEANUP;
                            }
                            /* okay, add to trigger's counter array */
                            found = true;
                            iptr = (orte_gpr_replica_itagval_t*)((orte_gpr_replica_globals.srch_ival)->addr[0]);
                            cntr = OBJ_NEW(orte_gpr_replica_counter_t);
                            if (NULL == cntr) {
                                ORTE_ERROR_LOG(ORTE_ERR_OUT_OF_RESOURCE);
                                return ORTE_ERR_OUT_OF_RESOURCE;
                            }
                            cntr->seg = seg;
                            cntr->cptr = cptr[k];
                            cntr->iptr = iptr;
                            /* if the trigger is at a level, then the requestor MUST specify the
                             * level in the provided keyval. Otherwise, we only need to store
                             * the iptr since we will be comparing levels between multiple
                             * counters
                             */
                            if (trigger->action & ORTE_GPR_TRIG_AT_LEVEL) {
                                if (NULL == trigger->values[i]->keyvals) {
                                    ORTE_ERROR_LOG(ORTE_ERR_BAD_PARAM);
                                    rc = ORTE_ERR_BAD_PARAM;
                                    goto CLEANUP;
                                }
                                cntr->trigger_level.value = OBJ_NEW(orte_data_value_t);
                                if (NULL == cntr->trigger_level.value) {
                                    ORTE_ERROR_LOG(ORTE_ERR_OUT_OF_RESOURCE);
                                    rc = ORTE_ERR_OUT_OF_RESOURCE;
                                    goto CLEANUP;
                                }
                                cntr->trigger_level.value->type = ((trigger->values[i])->keyvals[j])->value->type;
                                if (ORTE_SUCCESS != (rc = orte_dss.copy(&((cntr->trigger_level.value)->data),
                                                                        ((trigger->values[i])->keyvals[j])->value->data,
                                                                        ((trigger->values[i])->keyvals[j])->value->type))) {
                                    ORTE_ERROR_LOG(rc);
                                    goto CLEANUP;
                                }
                            }
                            if (0 > orte_pointer_array_add(&index, trig->counters, cntr)) {
                                ORTE_ERROR_LOG(ORTE_ERR_OUT_OF_RESOURCE);
                                rc = ORTE_ERR_OUT_OF_RESOURCE;
                                goto CLEANUP;
                            }
                            (trig->num_counters)++;
                        }  /* end if found */
                    }  /* end if cptr NULL */
                }  /* end for k */
                if (!found) {  /* specified counter never found - error */
                    ORTE_ERROR_LOG(ORTE_ERR_BAD_PARAM);
                    rc = ORTE_ERR_BAD_PARAM;
                    goto CLEANUP;
                } /* end if found */
            }  /* end for j */
        }  /* end if/else container found */
    } /* end for i */

ADDREQ:
    /* see if this requestor and trigger id is already attached to
     * this trigger - if so, ignore it to avoid duplicates
     */
    reqs = (orte_gpr_replica_trigger_requestor_t**)(trig->attached)->addr;
    for (i=0, j=0; j < trig->num_attached &&
                   i < (trig->attached)->size; i++) {
        if (NULL != reqs[i]) {
            j++;
            /* if one is NULL and the other isn't, then they can't possibly match */
            if ((NULL == reqs[i]->requestor && NULL != requestor) ||
                (NULL != reqs[i]->requestor && NULL == requestor)) {
                continue;
            }
            if (reqs[i]->idtag == trigger->id &&
                ((NULL == reqs[i]->requestor && NULL == requestor) ||
                 (ORTE_EQUAL == orte_dss.compare(reqs[i]->requestor, requestor, ORTE_NAME)))) {
                /* found this requestor - do not add it again */
                goto DONETRIG;
            }
        }
    }

    /* add this requestor to the trigger's list of "attached" callers */
    req = OBJ_NEW(orte_gpr_replica_trigger_requestor_t);
    if (NULL == req) {
        ORTE_ERROR_LOG(ORTE_ERR_OUT_OF_RESOURCE);
        return ORTE_ERR_OUT_OF_RESOURCE;
    }

    if (NULL != requestor) {
        if (ORTE_SUCCESS != (rc = orte_dss.copy((void**)&(req->requestor), requestor, ORTE_NAME))) {
              ORTE_ERROR_LOG(rc);
              return rc;
        }
    } else {
        req->requestor = NULL;
    }

    if (0 > (rc = orte_pointer_array_add(&(req->index), trig->attached, req))) {
        ORTE_ERROR_LOG(ORTE_ERR_OUT_OF_RESOURCE);
        return ORTE_ERR_OUT_OF_RESOURCE;
    }
    (trig->num_attached)++;

    /* store the requestor's trigger id so they can ask
     * us to cancel their subscription at a later time,
     * if they choose to do so.
     */
    req->idtag = trigger->id;

    /* see if the ROUTE_DATA_TO_ME flag is set. This indicates
     * that the requestor wants all data sent to them and
     * is assuming all responsibility for properly routing
     * the data
     */
    if (ORTE_GPR_TRIG_ROUTE_DATA_THRU_ME & trig->action) {
        if (NULL == trig->master) {
            /* someone already requested this responsibility.
             * if I'm a singleton, this is NOT an error - the
             * initial "launch" has recorded the stage gate
             * triggers using the [-1,-1,-1] name, so we need to
             * overwrite that with my name so I get the notifications.
             */
#if 0
            if (orte_process_info.singleton || orte_process_info.seed) {
opal_output(0, "Trigger master being redefined");
                trig->master = req;
            } else {
                /* if i'm not a singleton, then this is an error - report it */
                ORTE_ERROR_LOG(ORTE_ERR_NOT_AVAILABLE);
            }
        } else {
#endif
            trig->master = req;
        }
    }

DONETRIG:
    /* report the location of this trigger */
    *trigptr = trig;

    /* record that we had success */
    rc = ORTE_SUCCESS;

CLEANUP:
    if (NULL != tokentags) {
        free(tokentags);
    }

    return rc;
}

/*
 * Remove a subscription from the system. Note that the requestor only
 * knows their local subscription id, so that is what has been provided.
 * We need to find the specified combination of requestor and
 * subscription id, and then delete it
 */
int
orte_gpr_replica_remove_subscription(orte_process_name_t *requestor,
                                     orte_gpr_subscription_id_t id)
{
    orte_gpr_replica_subscription_t **subs, *sub;
    orte_gpr_replica_requestor_t **reqs, *req;
    orte_gpr_replica_trigger_t **trigs;
    orte_std_cntr_t i, j, k, m;
    bool found;

    OPAL_TRACE(3);

    /* find this subscription on the list */
    subs = (orte_gpr_replica_subscription_t**)(orte_gpr_replica.subscriptions)->addr;
    for (i=0, j=0; j < orte_gpr_replica.num_subs &&
                   i < (orte_gpr_replica.subscriptions)->size; i++) {
        if (NULL != subs[i]) {
            j++;
            reqs = (orte_gpr_replica_requestor_t**)(subs[i]->requestors)->addr;
            for (k=0, m=0; m < subs[i]->num_requestors &&
                           k < (subs[i]->requestors)->size; k++) {
                if (NULL != reqs[k]) {
                    m++;
                    if (id == reqs[k]->idtag &&
                        ((NULL == requestor && NULL == reqs[k]->requestor) ||
                        (NULL != requestor && NULL != reqs[k]->requestor &&
                        ORTE_EQUAL == orte_dss.compare(reqs[k]->requestor, requestor, ORTE_NAME)))) {
                        /* this is the subscription */
                        sub = subs[i];
                        req = reqs[k];
                        goto PROCESS;
                    }
                }
            }
        }
    }
    /* if we arrive here, then we were
     * unable to find a matching subscription. report that fact
     * and exit
     */
    ORTE_ERROR_LOG(ORTE_ERR_NOT_FOUND);
    return ORTE_ERR_NOT_FOUND;

PROCESS:
    /* remove the specified requestor. if this was the last
     * requestor on this subscription, remove the subscription
     * as well
     */
    /* must release the requestor object PRIOR to setting
     * the indexed location to NULL or we lose the pointer
     */
    i = req->index;
    OBJ_RELEASE(req);
    orte_pointer_array_set_item(sub->requestors, i, NULL);
    (sub->num_requestors)--;
    if (0 == sub->num_requestors) { /* nobody left */
        /* NOTE: cannot release sub here as we still need the
         * object so we can check for it in the list of triggers
         */
        orte_pointer_array_set_item(orte_gpr_replica.subscriptions, sub->index, NULL);
        (orte_gpr_replica.num_subs)--;
    }

    /* check for this subscription throughout the list of triggers
     * and remove it wherever found
     */
    trigs = (orte_gpr_replica_trigger_t**)(orte_gpr_replica.triggers)->addr;
    for (i=0, j=0; j < orte_gpr_replica.num_trigs &&
                   i < (orte_gpr_replica.triggers)->size; i++) {
        if (NULL != trigs[i]) {
            j++;
            found = false;
            subs = (orte_gpr_replica_subscription_t**)(trigs[i]->subscriptions)->addr;
            for (k=0, m=0; !found && m < trigs[i]->num_subscriptions &&
                           k < (trigs[i]->subscriptions)->size; k++) {
                if (NULL != subs[k]) {
                    m++;
                    if (sub == subs[k]) { /* match found */
                        orte_pointer_array_set_item(trigs[i]->subscriptions, k, NULL);
                        (trigs[i]->num_subscriptions)--;
                        /* if that was the last subscription on this trigger, then
                         * remove the trigger - not needed any more
                         */
                        if (0 == trigs[i]->num_subscriptions) {
                            OBJ_RELEASE(trigs[i]);
                            orte_pointer_array_set_item(orte_gpr_replica.triggers, i, NULL);
                        }
                        found = true;
                    }
                }
            }
        }
    }
    /* done with sub, so now can release it if we need to do so */
    if (0 == sub->num_requestors) OBJ_RELEASE(sub);

    /* ALL DONE! */
    return ORTE_SUCCESS;
}

/*
 * Remove a trigger from the system. Note that the requestor only
 * knows their local trigger id, so that is what has been provided.
 * We need to find the specified combination of requestor and
 * trigger id, and then delete it
 */
int
orte_gpr_replica_remove_trigger(orte_process_name_t *requestor,
                                orte_gpr_trigger_id_t id)
{
    orte_gpr_replica_subscription_t **subs;
    orte_gpr_replica_trigger_requestor_t **reqs, *req;
    orte_gpr_replica_trigger_t **trigs, *trig;
    orte_std_cntr_t i, j, k, m;

    OPAL_TRACE(3);

    /* find this trigger on the list */
    trigs = (orte_gpr_replica_trigger_t**)(orte_gpr_replica.triggers)->addr;
    for (i=0, j=0; j < orte_gpr_replica.num_trigs &&
                   i < (orte_gpr_replica.triggers)->size; i++) {
        if (NULL != trigs[i]) {
            j++;
            reqs = (orte_gpr_replica_trigger_requestor_t**)(trigs[i]->attached)->addr;
            for (k=0, m=0; m < trigs[i]->num_attached &&
                           k < (trigs[i]->attached)->size; k++) {
                if (NULL != reqs[k]) {
                    m++;
                    if (id == reqs[k]->idtag &&
                        ((NULL == requestor && NULL == reqs[k]->requestor) ||
                        (NULL != requestor && NULL != reqs[k]->requestor &&
                        ORTE_EQUAL == orte_dss.compare(reqs[k]->requestor, requestor, ORTE_NAME)))) {
                        /* this is the trigger */
                        trig = trigs[i];
                        req = reqs[k];
                        goto PROCESS;
                    }
                }
            }
        }
    }
    /* if we arrive here, then we had a remote requestor but were
     * unable to find a matching trigger. report that fact
     * and exit
     */
    ORTE_ERROR_LOG(ORTE_ERR_NOT_FOUND);
    return ORTE_ERR_NOT_FOUND;

PROCESS:
    /* remove the specified requestor. if this was the last
     * requestor on this trigger, remove the trigger
     * as well
     */
    /* must release the requestor object PRIOR to setting
     * the indexed location to NULL or we lose the pointer
     */
    i = req->index;
    OBJ_RELEASE(req);
    orte_pointer_array_set_item(trig->attached, i, NULL);
    (trig->num_attached)--;
    if (0 == trig->num_attached) { /* nobody left */
        /* NOTE: cannot release trig here as we still need the
         * object so we can clear any attached subscriptions
         */
        orte_pointer_array_set_item(orte_gpr_replica.triggers, trig->index, NULL);
        (orte_gpr_replica.num_trigs)--;
    }

    /* now need to check any attached subscriptions. if the subscription
     * was flagged to be deleted after the trigger fired, or was flagged
     * to only start once the trigger had fired, then we need
     * to delete it here. otherwise, we leave the subscription alone.
     */
    subs = (orte_gpr_replica_subscription_t**)(trig->subscriptions)->addr;
    for (i=0, j=0; j < trig->num_subscriptions &&
                   i < (trig->subscriptions)->size; i++) {
        if (NULL != subs[i]) {
            j++;
            if (ORTE_GPR_NOTIFY_STARTS_AFTER_TRIG & subs[i]->action ||
                ORTE_GPR_NOTIFY_DELETE_AFTER_TRIG & subs[i]->action) {
                OBJ_RELEASE(subs[i]);
            }
        }
    }

    /* done processing trigger - can release it now, if we need to do so */
    if (0 == trig->num_attached) OBJ_RELEASE(trig);

    /* ALL DONE! */
    return ORTE_SUCCESS;
}


int orte_gpr_replica_record_action(orte_gpr_replica_segment_t *seg,
                                   orte_gpr_replica_container_t *cptr,
                                   orte_gpr_replica_itagval_t *iptr,
                                   orte_gpr_replica_action_t action)
{
    orte_gpr_replica_action_taken_t *new_action;
    orte_std_cntr_t index;
    int rc;

    OPAL_TRACE(3);

    new_action = OBJ_NEW(orte_gpr_replica_action_taken_t);
    if (NULL == new_action) {
        ORTE_ERROR_LOG(ORTE_ERR_OUT_OF_RESOURCE);
        return ORTE_ERR_OUT_OF_RESOURCE;
    }
    new_action->action = action;

    /* store pointers to the affected itagval */
    new_action->seg = seg;
    new_action->cptr = cptr;
    new_action->iptr = iptr;

    /* "retain" ALL of the respective objects so they can't disappear until
     * after we process the actions
     */
    OBJ_RETAIN(seg);
    OBJ_RETAIN(cptr);
    OBJ_RETAIN(iptr);

    /* add the new action record to the array */
    if (0 > (rc = orte_pointer_array_add(&index, orte_gpr_replica_globals.acted_upon, new_action))) {
        ORTE_ERROR_LOG(ORTE_ERR_OUT_OF_RESOURCE);
        return ORTE_ERR_OUT_OF_RESOURCE;
    }

    /* increment the number acted upon */
    (orte_gpr_replica_globals.num_acted_upon)++;

    return ORTE_SUCCESS;
}


int orte_gpr_replica_update_storage_locations(orte_gpr_replica_itagval_t *new_iptr)
{
    orte_gpr_replica_trigger_t **trig;
    orte_gpr_replica_counter_t **cntrs;
    orte_gpr_replica_itagval_t **old_iptrs;
    orte_std_cntr_t i, j, k, m, n, p;
    bool replaced;

    OPAL_TRACE(3);

    trig = (orte_gpr_replica_trigger_t**)((orte_gpr_replica.triggers)->addr);
    for (i=0, m=0; m < orte_gpr_replica.num_trigs &&
                   i < (orte_gpr_replica.triggers)->size; i++) {
        if (NULL != trig[i]) {
            m++;
            cntrs = (orte_gpr_replica_counter_t**)((trig[i]->counters)->addr);
            for (j=0, n=0; n < trig[i]->num_counters &&
                           j < (trig[i]->counters)->size; j++) {
                if (NULL != cntrs[j]) {
                    n++;
                    old_iptrs = (orte_gpr_replica_itagval_t**)((orte_gpr_replica_globals.srch_ival)->addr);
                    for (k=0, p=0; p < orte_gpr_replica_globals.num_srch_ival &&
                                   k < (orte_gpr_replica_globals.srch_ival)->size; k++) {
                        replaced = false;
                        if (NULL != old_iptrs[k]) {
                            p++;
                            if (old_iptrs[k] == cntrs[j]->iptr) {
                                if (NULL == new_iptr || replaced) {
                                    orte_pointer_array_set_item(trig[i]->counters, j, NULL);
                                    (trig[i]->num_counters)--;
                                } else if (!replaced) {
                                    cntrs[j]->iptr = new_iptr;
                                    replaced = true;
                                }
                            }
                        }
                    }
                }
            }
        }
    }
    return ORTE_SUCCESS;
}


int orte_gpr_replica_check_events(void)
{
    orte_gpr_replica_trigger_t **trigs;
    orte_gpr_replica_subscription_t **subs;
    orte_gpr_replica_action_taken_t **ptr;
    orte_std_cntr_t i, j;
    int rc;

    OPAL_TRACE(3);

    /* we first check all the subscriptions to see if any are "active".
     * this needs to be done BEFORE we check triggers to ensure that
     * triggers that turn "on" a subscription don't cause duplicate
     * messages to their requestor
     */
    subs = (orte_gpr_replica_subscription_t**)((orte_gpr_replica.subscriptions)->addr);
    for (i=0, j=0; j < orte_gpr_replica.num_subs &&
              i < (orte_gpr_replica.subscriptions)->size; i++) {
        if (NULL != subs[i]) {
            j++;
            if (subs[i]->active) {
                /* this is an active subscription - check to see if
                 * any of the recorded actions match its specified
                 * conditions and process it if so
                 */
                if (ORTE_SUCCESS != (rc = orte_gpr_replica_check_subscription(subs[i]))) {
                    ORTE_ERROR_LOG(rc);
                    return rc;
                }
            } /* if notify */
        }
    }

    /* check for triggers that might have fired.
     * NOTE: MUST DO THIS *AFTER* THE NOTIFY CHECK. If the trigger was
     * set to start notifies after firing, then checking notifies
     * AFTER the triggers were processed causes the notification to
     * be sent twice.
     */
    trigs = (orte_gpr_replica_trigger_t**)((orte_gpr_replica.triggers)->addr);
    for (i=0, j=0; j < orte_gpr_replica.num_trigs &&
              i < (orte_gpr_replica.triggers)->size; i++) {
        if (NULL != trigs[i] && !trigs[i]->processing) {
            j++;
            /* check the trigger */
            if (ORTE_SUCCESS != (rc = orte_gpr_replica_check_trig(trigs[i]))) {
                ORTE_ERROR_LOG(rc);
                return rc;
            }
        }  /* if trig not NULL */
    }

    /* clean up the action record. The recorded actions from a given
     * call into the registry are only needed through the "check_events"
     * function call.
     */
    ptr = (orte_gpr_replica_action_taken_t**)((orte_gpr_replica_globals.acted_upon)->addr);
    for (i=0, j=0; j < orte_gpr_replica_globals.num_acted_upon &&
                   i < (orte_gpr_replica_globals.acted_upon)->size; i++) {
        if (NULL != ptr[i]) {
           j++;
           OBJ_RELEASE(ptr[i]);
        }
    }
    orte_gpr_replica_globals.num_acted_upon = 0;

    return ORTE_SUCCESS;
}

/*
 * Check a trigger to see if it has fired based on the current
 * state of its counters
 */
int orte_gpr_replica_check_trig(orte_gpr_replica_trigger_t *trig)
{
    orte_gpr_replica_subscription_t **subs;
    orte_gpr_replica_counter_t **cntr;
    orte_gpr_replica_itagval_t *base_value=NULL;
    orte_data_type_t base_type = ORTE_UNDEF;
    bool first, fire;
    orte_std_cntr_t i, j;
    int rc;

    OPAL_TRACE(3);

    if (ORTE_GPR_TRIG_CMP_LEVELS & trig->action) { /* compare the levels of the counters */
        cntr = (orte_gpr_replica_counter_t**)((trig->counters)->addr);
        first = true;
        fire = true;
        for (i=0, j=0; j < trig->num_counters &&
                       i < (trig->counters)->size && fire; i++) {
            if (NULL != cntr[i]) {
                j++;
                if (first) {
                    base_value = cntr[i]->iptr;
                    base_type = cntr[i]->iptr->value->type;
                    first = false;
                } else {
                   if (base_type != cntr[i]->iptr->value->type) {
                       ORTE_ERROR_LOG(ORTE_ERR_COMPARE_FAILURE);
                       return ORTE_ERR_COMPARE_FAILURE;
                   }
                   if (ORTE_EQUAL != orte_dss.compare(base_value->value->data, cntr[i]->iptr->value->data, base_type)) {
                        fire = false;
                   }
                }
            }
        }
        if (fire) { /* all levels were equal */
            goto FIRED;
        }
        return ORTE_SUCCESS;

    } else if (ORTE_GPR_TRIG_AT_LEVEL & trig->action) { /* see if counters are at a level */
        cntr = (orte_gpr_replica_counter_t**)((trig->counters)->addr);
        fire = true;
        for (i=0, j=0; j < trig->num_counters &&
                       i < (trig->counters)->size && fire; i++) {
            if (NULL != cntr[i]) {
                j++;
                if (cntr[i]->iptr->value->type != cntr[i]->trigger_level.value->type) {
                       ORTE_ERROR_LOG(ORTE_ERR_COMPARE_FAILURE);
                       return ORTE_ERR_COMPARE_FAILURE;
                }
                if (ORTE_EQUAL != orte_dss.compare(cntr[i]->iptr->value->data,
                                                   cntr[i]->trigger_level.value->data,
                                                   cntr[i]->iptr->value->type)) {
                    fire = false;
               }
            }
        }
        if (fire) { /* all counters at specified trigger level */
            goto FIRED;
        }
        return ORTE_SUCCESS;
    }

    return ORTE_SUCCESS;  /* neither cmp nor at level set */

FIRED:
    /* if this trigger wants everything routed through a "master", then we register
     * this as a trigger_callback.
     */
    if (NULL != trig->master) {
        if (ORTE_SUCCESS != (rc = orte_gpr_replica_register_trigger_callback(trig))) {
            ORTE_ERROR_LOG(rc);
            return rc;
        }
        /* for each subscription assocated with this trigger, check to see if
        * the subscription needs any special treatment
        */
        subs = (orte_gpr_replica_subscription_t**)(trig->subscriptions)->addr;
        for (i=0, j=0; j < trig->num_subscriptions &&
                        i < (trig->subscriptions)->size; i++) {
            if (NULL != subs[i]) {
                j++;
                /* if ORTE_GPR_NOTIFY_STARTS_AFTER_TRIG set, set the subscription
                    * "active" to indicate that trigger fired
                    */
                if (ORTE_GPR_NOTIFY_STARTS_AFTER_TRIG & subs[i]->action) {
                    subs[i]->active = true;
                }
                /* if ORTE_GPR_NOTIFY_DELETE_AFTER_TRIG set, then set the flag
                    * so it can be cleaned up later
                    */
                if (ORTE_GPR_NOTIFY_DELETE_AFTER_TRIG & subs[i]->action) {
                    subs[i]->cleanup = true;
                }
            }
        }
    } else {
        /* for each subscription associated with this trigger, we need to
         * register a callback to the requestor that returns the specified
         * data
         */
        subs = (orte_gpr_replica_subscription_t**)(trig->subscriptions)->addr;
        for (i=0, j=0; j < trig->num_subscriptions &&
                       i < (trig->subscriptions)->size; i++) {
            if (NULL != subs[i]) {
                j++;
                if (ORTE_SUCCESS != (rc = orte_gpr_replica_register_callback(subs[i], NULL))) {
                    ORTE_ERROR_LOG(rc);
                    return rc;
                }
                /* if ORTE_GPR_NOTIFY_STARTS_AFTER_TRIG set, set the subscription
                    * "active" to indicate that trigger fired
                    */
                if (ORTE_GPR_NOTIFY_STARTS_AFTER_TRIG & subs[i]->action) {
                    subs[i]->active = true;
                }
                /* if ORTE_GPR_NOTIFY_DELETE_AFTER_TRIG set, then set the flag
                    * so it can be cleaned up later
                    */
                if (ORTE_GPR_NOTIFY_DELETE_AFTER_TRIG & subs[i]->action) {
                    subs[i]->cleanup = true;
                }
            }
        }
    }


    /* set the processing flag so we don't go into infinite loop if
     * any callback functions modify the registry
     */
    trig->processing = true;

    /* if this trigger was a one-shot, set flag to indicate it has fired
     * so it can be cleaned up later
     */
    if (ORTE_GPR_TRIG_ONE_SHOT & trig->action) {
        trig->one_shot_fired = true;
    }

    return ORTE_SUCCESS;
}

/*
 * Check subscriptions to see if any were fired by any of the
 * recorded actions that have occurred on the registry.
 */
int orte_gpr_replica_check_subscription(orte_gpr_replica_subscription_t *sub)
{
    orte_gpr_replica_action_taken_t **ptr;
    orte_std_cntr_t i, j, k;
    orte_gpr_value_t *value;
    orte_gpr_addr_mode_t addr_mode;
    int rc=ORTE_SUCCESS;

    OPAL_TRACE(3);

    /* When entering this function, we know that the specified
     * subscription is active since that was tested above. What we now need
     * to determine is whether or not any of the data
     * objects pointed to by the subscription were involved in a change. The
     * subscription could describe a container - e.g., the subscriber might want to know
     * if anything gets added to a container - or could be a container plus one or
     * more keys when the subscriber wants to know when a specific value gets changed.
     */
    ptr = (orte_gpr_replica_action_taken_t**)((orte_gpr_replica_globals.acted_upon)->addr);
    for (i=0, k=0; k < orte_gpr_replica_globals.num_acted_upon &&
                   i < (orte_gpr_replica_globals.acted_upon)->size; i++) {
        if (NULL != ptr[i]) {
            k++;
            if (
                (((sub->action & ORTE_GPR_NOTIFY_ADD_ENTRY) &&
                (ptr[i]->action & ORTE_GPR_REPLICA_ENTRY_ADDED)) ||

                ((sub->action & ORTE_GPR_NOTIFY_DEL_ENTRY) &&
                (ptr[i]->action & ORTE_GPR_REPLICA_ENTRY_DELETED)) ||

                ((sub->action & ORTE_GPR_NOTIFY_VALUE_CHG) &&
                (ptr[i]->action & ORTE_GPR_REPLICA_ENTRY_CHG_TO)) ||

                ((sub->action & ORTE_GPR_NOTIFY_VALUE_CHG) &&
                (ptr[i]->action & ORTE_GPR_REPLICA_ENTRY_CHG_FRM)) ||

                ((sub->action & ORTE_GPR_NOTIFY_VALUE_CHG) &&
                (ptr[i]->action & ORTE_GPR_REPLICA_ENTRY_CHANGED)))

                && orte_gpr_replica_check_notify_matches(&addr_mode, sub, ptr[i])) {

                /* if the notify matched one of the subscription values,
                 * then the address mode will have
                 * been stored for us. we now need to send back
                 * the segment name and tokens from the container that is
                 * being addressed!
                 */
                /* Construct the base structure for returned data so it can be
                 * sent to the user, if required
                 */

                if (ORTE_GPR_REPLICA_STRIPPED(addr_mode)) {
                    if (ORTE_SUCCESS != (rc = orte_gpr_base_create_value(&value, addr_mode,
                                                                         NULL, 1, 0))) {
                        ORTE_ERROR_LOG(rc);
                        return rc;
                    }
                } else {
                    if (ORTE_SUCCESS != (rc = orte_gpr_base_create_value(&value, addr_mode,
                                                                         ptr[i]->seg->name,
                                                                         1, ptr[i]->cptr->num_itags))) {
                        ORTE_ERROR_LOG(rc);
                        return rc;
                    }
                    for (j=0; j < value->num_tokens; j++) {
                        if (ORTE_SUCCESS != (rc = orte_gpr_replica_dict_reverse_lookup(
                                                                                       &(value->tokens[j]),
                                                                                       ptr[i]->seg,
                                                                                       ptr[i]->cptr->itags[j]))) {
                            ORTE_ERROR_LOG(rc);
                            goto CLEANUP;
                        }
                    }
                }

                /* send back the recorded data */
                value->keyvals[0] = OBJ_NEW(orte_gpr_keyval_t);
                if (ORTE_SUCCESS != (rc = orte_gpr_replica_dict_reverse_lookup(
                                        &((value->keyvals[0])->key), ptr[i]->seg,
                                        ptr[i]->iptr->itag))) {
                    ORTE_ERROR_LOG(rc);
                    goto CLEANUP;
                }
                (value->keyvals[0])->value = OBJ_NEW(orte_data_value_t);
                if (NULL == value->keyvals[0]->value) {
                    ORTE_ERROR_LOG(ORTE_ERR_OUT_OF_RESOURCE);
                    OBJ_RELEASE(value);
                    return ORTE_ERR_OUT_OF_RESOURCE;
                }
                value->keyvals[0]->value->type = ptr[i]->iptr->value->type;
                if (ORTE_SUCCESS != (rc = orte_dss.copy(&((value->keyvals[0]->value)->data), ptr[i]->iptr->value->data, ptr[i]->iptr->value->type))) {
                    ORTE_ERROR_LOG(rc);
                    goto CLEANUP;
                }
                if (ORTE_SUCCESS != (rc = orte_gpr_replica_register_callback(sub, value))) {
                    ORTE_ERROR_LOG(rc);
                    goto CLEANUP;
                }
                /* register that this subscription is being processed
                 * to avoid potential infinite loops
                 */
                sub->processing = true;
            }
        }
    }

CLEANUP:
    return rc;
}


bool orte_gpr_replica_check_notify_matches(orte_gpr_addr_mode_t *addr_mode,
                                           orte_gpr_replica_subscription_t *sub,
                                           orte_gpr_replica_action_taken_t *ptr)
{
    orte_gpr_replica_addr_mode_t tokmod;
    orte_std_cntr_t i, j;
    orte_gpr_replica_ivalue_t **ivals;

    OPAL_TRACE(3);

    /* we need to run through all of this subscription's defined
     * values to see if any of them match the acted upon one.
     */
    ivals = (orte_gpr_replica_ivalue_t**)(sub->values)->addr;
    for (i=0, j=0; j < sub->num_values &&
                   i < (sub->values)->size; i++) {
        if (NULL != ivals[i]) {
            j++;
            /* first, check to see if the segments match */
            if (ivals[i]->seg != ptr->seg) { /* don't match - return false */
                continue;
            }

            /* next, check to see if the containers match */
            tokmod = ORTE_GPR_REPLICA_TOKMODE(ivals[i]->addr_mode);
            if (!orte_gpr_replica_check_itag_list(tokmod,
                        orte_value_array_get_size(&(ivals[i]->tokentags)),
                        ORTE_VALUE_ARRAY_GET_BASE(&(ivals[i]->tokentags), orte_gpr_replica_itag_t),
                        (ptr->cptr)->num_itags,
                        (ptr->cptr)->itags)) {
                /* not this container */
                continue;
            }
            /* next, check to see if this keyval was on the list */
            if (orte_gpr_replica_check_itag_list(ORTE_GPR_REPLICA_OR,
                        orte_value_array_get_size(&(ivals[i]->keytags)),
                        ORTE_VALUE_ARRAY_GET_BASE(&(ivals[i]->keytags), orte_gpr_replica_itag_t),
                        1,
                        &(ptr->iptr->itag))) {
                /* keyval is on list - return the address mode */
                *addr_mode = ivals[i]->addr_mode;
                return true;
            }
        }
    }

    /* if we get here, then the acted upon value was
     * nowhere on the subscription's defined values */
    return false;
}


int orte_gpr_replica_purge_subscriptions(orte_process_name_t *proc)
{
#if 0
    orte_gpr_replica_trigger_t **trig;
    orte_std_cntr_t i;
    int rc;

    OPAL_TRACE(3);

    /* locate any notification events that have proc as the requestor
     * and remove them
     */
    trig = (orte_gpr_replica_triggers_t**)((orte_gpr_replica.triggers)->addr);
    for (i=0; i < (orte_gpr_replica.triggers)->size; i++) {
        if (NULL != trig[i]) {
            if (NULL == proc && NULL == trig[i]->requestor) {
                if (ORTE_SUCCESS != (rc = orte_pointer_array_set_item(orte_gpr_replica.triggers,
                                                trig[i]->index, NULL))) {
                    ORTE_ERROR_LOG(rc);
                    return rc;
                }
                OBJ_RELEASE(trig);
            } else if (NULL != proc && NULL != trig[i]->requestor &&
                       ORTE_EQUAL == orte_dss.compare(Oproc, trig[i]->requestor, ORTE_NAME)) {
                if (ORTE_SUCCESS != (rc = orte_pointer_array_set_item(orte_gpr_replica.triggers,
                                                trig[i]->index, NULL))) {
                    ORTE_ERROR_LOG(rc);
                    return rc;
                }
                OBJ_RELEASE(trig);
            }
        }
    }
#endif

    return ORTE_SUCCESS;
}
