/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: worker.c,v 1.5.2.1 2004/06/26 18:02:59 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"

#include "monitor.h"
#include "worker.h"

#include <glib.h>
#include <ghttp.h>

#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>


static Monitor *normal_job_monitor = NULL;
static volatile GSList *job_list = NULL;
static volatile GSList *worker_threads = NULL;
static int maximum_number_of_worker_threads;
static volatile int number_of_idle_threads;
static volatile int number_of_threads;

static WorkerThread *worker_thread_new(void);
static void *worker_thread_main(void *args);



static Monitor *modest_job_monitor = NULL;
static volatile GSList *modest_job_list = NULL;
static volatile GSList *modest_worker_threads = NULL;
static int maximum_number_of_modest_threads;
static volatile int number_of_idle_modest_threads;
static volatile int number_of_modest_threads;

static WorkerThread *modest_worker_thread_new(void);
static void *modest_worker_thread_main(void *args);



static Monitor *loader_job_monitor = NULL;
static volatile GSList *loader_job_list = NULL;
static volatile GSList *loader_worker_threads = NULL;
static int maximum_number_of_loader_threads;
static volatile int number_of_idle_loader_threads;
static volatile int number_of_loader_threads;

static WorkerThread *loader_worker_thread_new(void);
static void *loader_worker_thread_main(void *args);




static pthread_attr_t worker_thread_attribute;



#define NEED_LARGER_STACK	0


void
initialize_worker(int initial_num_threads, int maximum_num_threads,
		  int num_modest_threads, int num_loader_threads)
{
  int i;
#if NEED_LARGER_STACK
  size_t stack_size;
#endif
  int initial_num_modest_threads;
  int initial_num_loader_threads;

  job_list = NULL;
  worker_threads = NULL;
  if (maximum_num_threads > 0)
    maximum_number_of_worker_threads = maximum_num_threads;
  else
    maximum_number_of_worker_threads = 0x7fffffff;	/* ޤǡ */
  number_of_threads = initial_num_threads;
  number_of_idle_threads = 0;

  if (maximum_num_threads < initial_num_threads)
    {
      fprintf(stderr,
	      "initialize_worker: Invalid args(maximum(%d) < initial(%d).\n",
	      maximum_num_threads, initial_num_threads);
      abort();
    }

  normal_job_monitor = ochusha_monitor_new(NULL);
  modest_job_monitor = ochusha_monitor_new(NULL);
  loader_job_monitor = ochusha_monitor_new(NULL);

  if (pthread_attr_init(&worker_thread_attribute) != 0)
    {
      fprintf(stderr, "Couldn't init a worker threads' attribute.\n");
      abort();
    }

#if NEED_LARGER_STACK
  /* å­ʤΤȻפä㤦ХǤ뤳ȤȽΤǼΤ */
  if (pthread_attr_getstacksize(&worker_thread_attribute, &stack_size) != 0)
    {
      fprintf(stderr, "Couldn't get default stack_size.\n");
    }
  else
    fprintf(stderr, "default stack_size = 0x%x\n", stack_size);

  if (pthread_attr_setstacksize(&worker_thread_attribute, stack_size * 2) != 0)
    {
      fprintf(stderr, "Couldn't set worker threads' stack_size as 0x%x.\n",
	      stack_size * 2);
      abort();
    }
#endif

  for (i = 0; i < initial_num_threads; i++)
    worker_thread_new();

  if (num_modest_threads > 0)
    initial_num_modest_threads = MIN(num_modest_threads, 4);
  else
    initial_num_modest_threads = 0;

  maximum_number_of_modest_threads = num_modest_threads;

  for (i = 0; i < initial_num_modest_threads; i++)
    modest_worker_thread_new();

  if (num_loader_threads > 0)
    initial_num_loader_threads = MIN(num_loader_threads, 4);
  else
    initial_num_loader_threads = 0;

  maximum_number_of_loader_threads = num_loader_threads;

  for (i = 0; i < initial_num_loader_threads; i++)
    loader_worker_thread_new();
}


static void
set_die_now_command_for_all_threads(gpointer data, gpointer unused)
{
  WorkerThread *thread = (WorkerThread *)data;
  thread->command = DIE_NOW;
}


void
terminate_workers(void)
{
  g_slist_foreach((GSList *)worker_threads,
		  set_die_now_command_for_all_threads, NULL);

  g_slist_foreach((GSList *)modest_worker_threads,
		  set_die_now_command_for_all_threads, NULL);

  g_slist_foreach((GSList *)loader_worker_threads,
		  set_die_now_command_for_all_threads, NULL);
}


void
join_workers(void)
{
  GSList *list_entry;

  while ((list_entry = (GSList *)worker_threads) != NULL)
    {
      WorkerThread *thread;
      worker_threads = g_slist_remove_link((GSList *)worker_threads,
					   list_entry);
      thread = (WorkerThread *)list_entry->data;
      g_slist_free_1(list_entry);
      ochusha_monitor_notify_all(normal_job_monitor);

      if (pthread_join(thread->id, NULL) != 0)
	{
	  fprintf(stderr, "Couldn't join a thread\n");
	  abort();
	}

      G_FREE(thread);
    }
  ochusha_monitor_free(normal_job_monitor);
  normal_job_monitor = NULL;

  while ((list_entry = (GSList *)modest_worker_threads) != NULL)
    {
      WorkerThread *thread;
      modest_worker_threads
	= g_slist_remove_link((GSList *)modest_worker_threads, list_entry);
      thread = (WorkerThread *)list_entry->data;
      g_slist_free_1(list_entry);
      ochusha_monitor_notify_all(modest_job_monitor);

      if (pthread_join(thread->id, NULL) != 0)
	{
	  fprintf(stderr, "Couldn't join a thread\n");
	  abort();
	}

      G_FREE(thread);
    }
  ochusha_monitor_free(modest_job_monitor);
  modest_job_monitor = NULL;

  while ((list_entry = (GSList *)loader_worker_threads) != NULL)
    {
      WorkerThread *thread;
      loader_worker_threads
	= g_slist_remove_link((GSList *)loader_worker_threads, list_entry);
      thread = (WorkerThread *)list_entry->data;
      g_slist_free_1(list_entry);
      ochusha_monitor_notify_all(loader_job_monitor);

      if (pthread_join(thread->id, NULL) != 0)
	{
	  fprintf(stderr, "Couldn't join a thread\n");
	  abort();
	}

      G_FREE(thread);
    }
  ochusha_monitor_free(loader_job_monitor);
  loader_job_monitor = NULL;
}


void
cancel_pending_jobs(void)
{
  GSList *tmp_list;

  ochusha_monitor_enter(normal_job_monitor);
  tmp_list = (GSList *)job_list;
  job_list = NULL;
  ochusha_monitor_exit(normal_job_monitor);

  g_slist_foreach(tmp_list, (GFunc)g_free, NULL);
  g_slist_free(tmp_list);


  ochusha_monitor_enter(modest_job_monitor);
  tmp_list = (GSList *)modest_job_list;
  modest_job_list = NULL;
  ochusha_monitor_exit(modest_job_monitor);

  g_slist_foreach(tmp_list, (GFunc)g_free, NULL);
  g_slist_free(tmp_list);


  ochusha_monitor_enter(loader_job_monitor);
  tmp_list = (GSList *)loader_job_list;
  loader_job_list = NULL;
  ochusha_monitor_exit(loader_job_monitor);

  g_slist_foreach(tmp_list, (GFunc)g_free, NULL);
  g_slist_free(tmp_list);
}


static WorkerThread *
worker_thread_new(void)
{
  WorkerThread *thread = G_NEW0(WorkerThread, 1);
  thread->command = GO_AHEAD;

  if (pthread_create(&thread->id, &worker_thread_attribute,
		     worker_thread_main, thread) != 0)
    {
      fprintf(stderr, "Couldn't create a worker thread.\n");
      abort();
    }

  worker_threads = g_slist_append((GSList *)worker_threads, thread);
  number_of_threads++;

  return thread;
}


void
commit_job(WorkerJob *job)
{
  ochusha_monitor_enter(normal_job_monitor);
  {

#if DEBUG_THREAD_MOST
    fprintf(stderr, "commit_job: job committed.\n");
    fflush(stderr);
#endif

    job_list = g_slist_append((GSList *)job_list, job);

    if (number_of_idle_threads > 0)
      {
	number_of_idle_threads--;

#if DEBUG_THREAD_MOST
	fprintf(stderr, "signaling to awake idle thread.\n");
	fflush(stderr);
#endif

	ochusha_monitor_notify(normal_job_monitor);
	ochusha_monitor_exit(normal_job_monitor);
#if DEBUG_THREAD_MOST
	fprintf(stderr, "Job must be started!\n");
	fflush(stderr);
#endif
	return;
      }

#if DEBUG_THREAD_MOST
    fprintf(stderr, "There's no idle thread.\n");
    fflush(stderr);
#endif

    if (number_of_threads < maximum_number_of_worker_threads)
      worker_thread_new();
  }
  ochusha_monitor_exit(normal_job_monitor);
}


static void *
worker_thread_main(void *args)
{
  WorkerThread *self = (WorkerThread *)args;
  ochusha_monitor_enter(normal_job_monitor);
  {
#if DEBUG_THREAD_MOST
    fprintf(stderr, "WorkerThread is created.\n");
    fflush(stderr);
#endif

    while (1)
      {
	GSList *list_entry;
	WorkerJob *job;
	if (self->command == DIE_NOW)
	  {
	    ochusha_monitor_exit(normal_job_monitor);
#if DEBUG_THREAD_MOST
	    fprintf(stderr, "WorkerThread is killed.\n");
	    fflush(stderr);
#endif
	    pthread_exit(NULL);
	  }

	list_entry = (GSList *)job_list;
	if (list_entry == NULL)
	  {
#if DEBUG_THREAD_MOST
	    fprintf(stderr, "There's no job to be done.\n");
	    fflush(stderr);
#endif
	    number_of_idle_threads++;
	    ochusha_monitor_wait(normal_job_monitor);
	    continue;
	  }

	job_list = g_slist_remove_link((GSList *)job_list, list_entry);
	job = (WorkerJob *)list_entry->data;
	g_slist_free_1(list_entry);

	ochusha_monitor_exit(normal_job_monitor);

	(*job->job)(self, job->args);
	G_FREE(job);

	ochusha_monitor_enter(normal_job_monitor);
      }
  }
}


static WorkerThread *
modest_worker_thread_new(void)
{
  WorkerThread *thread = G_NEW0(WorkerThread, 1);
  thread->command = GO_AHEAD;

  if (pthread_create(&thread->id, &worker_thread_attribute,
		     modest_worker_thread_main, thread) != 0)
    {
      fprintf(stderr, "Couldn't create a worker thread.\n");
      abort();
    }

  modest_worker_threads
    = g_slist_append((GSList *)modest_worker_threads, thread);
  number_of_modest_threads++;

  return thread;
}


void
commit_modest_job(WorkerJob *job)
{
  if (maximum_number_of_modest_threads <= 0)
    {
      commit_job(job);
      return;
    }

  ochusha_monitor_enter(modest_job_monitor);
  {

#if DEBUG_THREAD_MOST
    fprintf(stderr, "commit_job: job committed.\n");
    fflush(stderr);
#endif

    modest_job_list = g_slist_append((GSList *)modest_job_list, job);

    if (number_of_idle_modest_threads > 0)
      {
	number_of_idle_modest_threads--;

#if DEBUG_THREAD_MOST
	fprintf(stderr, "signaling to awake idle thread.\n");
	fflush(stderr);
#endif
	ochusha_monitor_notify(modest_job_monitor);
	ochusha_monitor_exit(modest_job_monitor);
#if DEBUG_THREAD_MOST
	fprintf(stderr, "Job must be started!\n");
	fflush(stderr);
#endif
	return;
      }

#if DEBUG_THREAD_MOST
    fprintf(stderr, "There's no idle thread.\n");
    fflush(stderr);
#endif

    if (number_of_modest_threads < maximum_number_of_modest_threads)
      modest_worker_thread_new();
  }
  ochusha_monitor_exit(modest_job_monitor);
}


static void *
modest_worker_thread_main(void *args)
{
  WorkerThread *self = (WorkerThread *)args;
  ochusha_monitor_enter(modest_job_monitor);
  {
#if DEBUG_THREAD_MOST
    fprintf(stderr, "WorkerThread is created.\n");
    fflush(stderr);
#endif

    while (1)
      {
	GSList *list_entry;
	WorkerJob *job;
	if (self->command == DIE_NOW)
	  {
	    ochusha_monitor_exit(modest_job_monitor);
#if DEBUG_THREAD_MOST
	    fprintf(stderr, "WorkerThread is killed.\n");
	    fflush(stderr);
#endif
	    pthread_exit(NULL);
	  }

	list_entry = (GSList *)modest_job_list;
	if (list_entry == NULL)
	  {
#if DEBUG_THREAD_MOST
	    fprintf(stderr, "There's no job to be done.\n");
	    fflush(stderr);
#endif
	    number_of_idle_modest_threads++;
	    ochusha_monitor_wait(modest_job_monitor);
	    continue;
	  }

	modest_job_list
	  = g_slist_remove_link((GSList *)modest_job_list, list_entry);
	job = (WorkerJob *)list_entry->data;
	g_slist_free_1(list_entry);

	ochusha_monitor_exit(modest_job_monitor);

	(*job->job)(self, job->args);
	G_FREE(job);

	ochusha_monitor_enter(modest_job_monitor);
      }
  }
}


static WorkerThread *
loader_worker_thread_new(void)
{
  WorkerThread *thread = G_NEW0(WorkerThread, 1);
  thread->command = GO_AHEAD;

  if (pthread_create(&thread->id, &worker_thread_attribute,
		     loader_worker_thread_main, thread) != 0)
    {
      fprintf(stderr, "Couldn't create a worker thread.\n");
      abort();
    }

  loader_worker_threads
    = g_slist_append((GSList *)loader_worker_threads, thread);
  number_of_loader_threads++;

  return thread;
}


void
commit_loader_job(WorkerJob *job)
{
  if (maximum_number_of_loader_threads <= 0)
    {
      commit_job(job);
      return;
    }

  ochusha_monitor_enter(loader_job_monitor);
  {

#if DEBUG_THREAD_MOST
    fprintf(stderr, "commit_job: job committed.\n");
    fflush(stderr);
#endif

    loader_job_list = g_slist_append((GSList *)loader_job_list, job);

    if (number_of_idle_loader_threads > 0)
      {
	number_of_idle_loader_threads--;

#if DEBUG_THREAD_MOST
	fprintf(stderr, "signaling to awake idle thread.\n");
	fflush(stderr);
#endif
	ochusha_monitor_notify(loader_job_monitor);
	ochusha_monitor_exit(loader_job_monitor);
#if DEBUG_THREAD_MOST
	fprintf(stderr, "Job must be started!\n");
	fflush(stderr);
#endif
	return;
      }

#if DEBUG_THREAD_MOST
    fprintf(stderr, "There's no idle thread.\n");
    fflush(stderr);
#endif

    if (number_of_loader_threads < maximum_number_of_loader_threads)
      loader_worker_thread_new();
  }
  ochusha_monitor_exit(loader_job_monitor);
}


static void *
loader_worker_thread_main(void *args)
{
  WorkerThread *self = (WorkerThread *)args;
  ochusha_monitor_enter(loader_job_monitor);
  {
#if DEBUG_THREAD_MOST
    fprintf(stderr, "WorkerThread is created.\n");
    fflush(stderr);
#endif

    while (1)
      {
	GSList *list_entry;
	WorkerJob *job;
	if (self->command == DIE_NOW)
	  {
	    ochusha_monitor_exit(loader_job_monitor);
#if DEBUG_THREAD_MOST
	    fprintf(stderr, "WorkerThread is killed.\n");
	    fflush(stderr);
#endif
	    pthread_exit(NULL);
	  }

	list_entry = (GSList *)loader_job_list;
	if (list_entry == NULL)
	  {
#if DEBUG_THREAD_MOST
	    fprintf(stderr, "There's no job to be done.\n");
	    fflush(stderr);
#endif
	    number_of_idle_loader_threads++;
	    ochusha_monitor_wait(loader_job_monitor);
	    continue;
	  }

	loader_job_list
	  = g_slist_remove_link((GSList *)loader_job_list, list_entry);
	job = (WorkerJob *)list_entry->data;
	g_slist_free_1(list_entry);

	ochusha_monitor_exit(loader_job_monitor);

	(*job->job)(self, job->args);
	G_FREE(job);

	ochusha_monitor_enter(loader_job_monitor);
      }
  }
}
