/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_board_jbbs.c,v 1.17.2.6 2005/08/15 17:15:07 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_board_jbbs.h"
#include "ochusha_bbs_thread.h"
#include "ochusha_thread_jbbs.h"

#include "htmlutils.h"
#include "utils.h"

#include <glib-object.h>
#include <glib.h>

#if HAVE_ONIG_ONIGPOSIX_H
# include <onig/onigposix.h>
#else
# include <onigposix.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>


#define ANALYSIS_INTERVAL	200


static void ochusha_board_jbbs_class_init(OchushaBoardJBBSClass *klass);
static void ochusha_board_jbbs_init(OchushaBoardJBBS *board);

static const char *ochusha_board_jbbs_get_subject_txt_encoding(
						OchushaBulletinBoard *board);
static const char *ochusha_board_jbbs_get_subback_html_encoding(
						OchushaBulletinBoard *board);

static char *ochusha_board_jbbs_generate_base_path(OchushaBulletinBoard *board,
						   const char *url);
static char *ochusha_board_jbbs_generate_board_id(OchushaBulletinBoard *board,
						  const char *url);
static OchushaBBSThread *ochusha_board_jbbs_thread_new(
						OchushaBulletinBoard *board,
						const char *id,
						const gchar *title);

static OchushaAsyncBuffer *ochusha_board_jbbs_get_threadlist_source(
					OchushaBulletinBoard *board,
					OchushaNetworkBroker *broker,
					OchushaAsyncBuffer *buffer,
					OchushaNetworkBrokerCacheMode mode);
static gboolean ochusha_board_jbbs_refresh_threadlist(
						OchushaBulletinBoard *board,
						OchushaAsyncBuffer *buffer,
						EachThreadCallback *cb,
						StartParsingCallback *start_parsing_cb,
						BeforeWaitCallback *before_wait_cb,
						AfterWaitCallback *after_wait_cb,
						EndParsingCallback *end_parsing_cb,
						gpointer callback_data);

static const char *ochusha_board_jbbs_get_response_character_encoding(
						OchushaBulletinBoard *board);
static iconv_helper *ochusha_board_jbbs_get_response_iconv_helper(
						OchushaBulletinBoard *board);
static char *ochusha_board_jbbs_get_read_cgi_url(OchushaBoard2ch *board_2ch);

static gboolean ochusha_board_jbbs_is_new_thread_supported(
						OchushaBulletinBoard *board);
static gboolean ochusha_board_jbbs_create_new_thread(
					OchushaBulletinBoard *board,
					OchushaNetworkBroker *broker,
					const gchar *title,
					const OchushaBBSResponse *response);


GType
ochusha_board_jbbs_get_type(void)
{
  static GType board_jbbs_type = 0;

  if (board_jbbs_type == 0)
    {
      static const GTypeInfo board_jbbs_info =
	{
	  sizeof(OchushaBoardJBBSClass),
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)ochusha_board_jbbs_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof(OchushaBoardJBBS),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)ochusha_board_jbbs_init,
	};

      board_jbbs_type = g_type_register_static(OCHUSHA_TYPE_BOARD_2CH,
					      "OchushaBoardJBBS",
					      &board_jbbs_info, 0);
    }

  return board_jbbs_type;
}


static OchushaBoard2chClass *parent_class = NULL;
static GQuark idle_checker_id;


static void
ochusha_board_jbbs_class_init(OchushaBoardJBBSClass *klass)
{
  OchushaBulletinBoardClass *b_class = OCHUSHA_BULLETIN_BOARD_CLASS(klass);
  OchushaBoard2chClass *class_2ch = OCHUSHA_BOARD_2CH_CLASS(klass);

  parent_class = g_type_class_peek_parent(klass);

  b_class->generate_base_path = ochusha_board_jbbs_generate_base_path;
  b_class->generate_board_id = ochusha_board_jbbs_generate_board_id;
  b_class->thread_new = ochusha_board_jbbs_thread_new;
  b_class->get_threadlist_source = ochusha_board_jbbs_get_threadlist_source;
  b_class->refresh_threadlist = ochusha_board_jbbs_refresh_threadlist;
  b_class->get_response_character_encoding
    = ochusha_board_jbbs_get_response_character_encoding;
  b_class->get_response_iconv_helper
    = ochusha_board_jbbs_get_response_iconv_helper;

  b_class->new_thread_supported
    = ochusha_board_jbbs_is_new_thread_supported;
  b_class->create_new_thread
    = ochusha_board_jbbs_create_new_thread;

  class_2ch->get_read_cgi_url = ochusha_board_jbbs_get_read_cgi_url;

  idle_checker_id
    = g_quark_from_static_string("OchushaBoardJBBS::IdleChecker");
}


static void
ochusha_board_jbbs_init(OchushaBoardJBBS *board)
{
  OCHUSHA_BULLETIN_BOARD(board)->bbs_type = OCHUSHA_BBS_TYPE_JBBS;
}


static const char *
ochusha_board_jbbs_get_subject_txt_encoding(OchushaBulletinBoard *board)
{
  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_MACHIBBS:
    case OCHUSHA_BBS_TYPE_JBBS:
    case OCHUSHA_BBS_TYPE_MITINOKU:
      return "CP932";

    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
      return "EUC-JP";

    default:
      return NULL;
    }
}


static const char *
ochusha_board_jbbs_get_subback_html_encoding(OchushaBulletinBoard *board)
{
  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_MITINOKU:
      return "EUC-JP";

#if 0	/* BBSsubject.txtȤ */
    case OCHUSHA_BBS_TYPE_JBBS:
    case OCHUSHA_BBS_TYPE_MACHIBBS:
    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
#endif
    default:
      return NULL;
    }
}


static char *
ochusha_board_jbbs_generate_base_path(OchushaBulletinBoard *board,
				      const char *url)
{
  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
      {
	char *path;
	char *abs_path = ochusha_utils_url_extract_http_absolute_path(url);
	char *tmp_pos;
	g_return_val_if_fail(abs_path != NULL, NULL);

	if (abs_path[0] != '/')
	  {
	    G_FREE(abs_path);
	    return NULL;
	  }
	tmp_pos = strchr(abs_path + 1, '/');

	if (tmp_pos != NULL)
	  {
	    tmp_pos[1] = '\0';
	    path = G_STRDUP(abs_path);
	  }
	else
	  {
	    path = G_STRDUP("/");
	  }
	G_FREE(abs_path);
	return path;
      }

    default:
      return G_STRDUP("/");
    }
}


static char *
ochusha_board_jbbs_generate_board_id(OchushaBulletinBoard *board,
				    const char *url)
{
  char *id;
  char *abs_path = ochusha_utils_url_extract_http_absolute_path(url);
  if (abs_path != NULL && abs_path[0] != '/')
    {
      G_FREE(abs_path);
      abs_path = NULL;
    }
  g_return_val_if_fail(abs_path != NULL, NULL);

  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_MACHIBBS:
    case OCHUSHA_BBS_TYPE_MITINOKU:
      {
	char *tmp_pos = strchr(abs_path + 1, '/');
	if (tmp_pos != NULL)
	  {
	    *tmp_pos = '\0';
	  }
	id = G_STRDUP(abs_path + 1);
	break;
      }

    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
      {
	char *tmp_pos = strchr(abs_path + 1, '/');
	if (tmp_pos == NULL)
	  id = G_STRDUP(abs_path + 1);
	else
	  {
	    char *id_tail = strchr(tmp_pos + 1, '/');
	    if (id_tail != NULL)
	      *id_tail = '\0';
	    id = G_STRDUP(tmp_pos + 1);
	  }
	break;
      }

    default:
      id = NULL;	/* ̤ݡ */
      break;
    }

  G_FREE(abs_path);

  return id;
}


static OchushaBBSThread *
ochusha_board_jbbs_thread_new(OchushaBulletinBoard *board, const char *id,
			     const gchar *title)
{
  OchushaBBSThread *thread;
  g_return_val_if_fail(OCHUSHA_IS_BOARD_JBBS(board) && id != NULL, NULL);

  thread = ochusha_thread_jbbs_new(OCHUSHA_BOARD_JBBS(board), id, title);
  g_return_val_if_fail(thread != NULL, NULL);

  return thread;
}


static const char *
ochusha_board_jbbs_get_response_character_encoding(OchushaBulletinBoard *board)
{
  g_return_val_if_fail(OCHUSHA_IS_BOARD_JBBS(board), NULL);

  switch (((OchushaBulletinBoard *)board)->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_MACHIBBS:
    case OCHUSHA_BBS_TYPE_JBBS:
      return "CP932";

    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
    case OCHUSHA_BBS_TYPE_MITINOKU:
      return "EUC-JP";

    default:
      return NULL;
    }
}


static iconv_helper *
ochusha_board_jbbs_get_response_iconv_helper(OchushaBulletinBoard *board)
{
  g_return_val_if_fail(OCHUSHA_IS_BOARD_JBBS(board), NULL);

  switch (((OchushaBulletinBoard *)board)->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_MACHIBBS:
    case OCHUSHA_BBS_TYPE_JBBS:
      return cp932_to_utf8_helper;

    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
    case OCHUSHA_BBS_TYPE_MITINOKU:
      return eucjp_to_utf8_helper;

    default:
      return NULL;
    }
}


static char *
ochusha_board_jbbs_get_read_cgi_url(OchushaBoard2ch *board_2ch)
{
  OchushaBulletinBoard *board = OCHUSHA_BULLETIN_BOARD(board_2ch);
  char url[PATH_MAX];
  g_return_val_if_fail(OCHUSHA_IS_BOARD_JBBS(board_2ch), NULL);

  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
      if (snprintf(url, PATH_MAX, "http://%s%sbbs/read.cgi",
		   ochusha_bulletin_board_get_server(board),
		   ochusha_bulletin_board_get_base_path(board)) >= PATH_MAX)
	return NULL;
      break;

    case OCHUSHA_BBS_TYPE_JBBS:
      if (snprintf(url, PATH_MAX, "http://%s/bbs/read.cgi",
		   ochusha_bulletin_board_get_server(board)) >= PATH_MAX)
	return NULL;
      break;

    case OCHUSHA_BBS_TYPE_MACHIBBS:
      if (snprintf(url, PATH_MAX, "http://%s/bbs/read.pl",
		   ochusha_bulletin_board_get_server(board)) >= PATH_MAX)
	return NULL;
      break;

    case OCHUSHA_BBS_TYPE_MITINOKU:
      if (snprintf(url, PATH_MAX, "http://%s/read.cgi",
		   ochusha_bulletin_board_get_server(board)) >= PATH_MAX)
	return NULL;
      break;

    default:
      return NULL;
    }

  return G_STRDUP(url);
}


static gboolean
ochusha_board_jbbs_is_new_thread_supported(OchushaBulletinBoard *board)
{
  g_return_val_if_fail(OCHUSHA_IS_BOARD_JBBS(board), FALSE);

  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_MACHIBBS:
    case OCHUSHA_BBS_TYPE_JBBS:
    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
    case OCHUSHA_BBS_TYPE_MITINOKU:
    default:
      return FALSE;
    }
}


static gboolean
ochusha_board_jbbs_create_new_thread(OchushaBulletinBoard *board,
				     OchushaNetworkBroker *broker,
				     const gchar *title_utf8,
				     const OchushaBBSResponse *response)
{
#if 0	/* Ǥ̤ݡ */
  OchushaBoard2ch *board_2ch;
  char *tmp_string = NULL;
  char *title = NULL;
  char *from = NULL;
  char *mail = NULL;
  char *message = NULL;
  const char *bbs;
  char *query = NULL;
  time_t time;
  iconv_t converter;
  OchushaUtils2chPostResult post_result;

  g_return_val_if_fail(OCHUSHA_IS_BOARD_2CH(board), FALSE);
  g_return_val_if_fail(title_utf8 != NULL && *title_utf8 != '\0', FALSE);
  g_return_val_if_fail(response->name != NULL, FALSE);
  g_return_val_if_fail(response->mailto != NULL, FALSE);
  g_return_val_if_fail(response->content != NULL, FALSE);

  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_2CH:
    case OCHUSHA_BBS_TYPE_2CH_COMPATIBLE:
      break;
    default:
      return FALSE;	/* ̤ݡ */
    }

  converter = iconv_open(ochusha_bulletin_board_get_response_character_encoding(board), "UTF-8");
  g_return_val_if_fail(converter != (iconv_t)-1, FALSE);

  board_2ch = OCHUSHA_BOARD_2CH(board);

  tmp_string = convert_string(converter, NULL, title_utf8, -1);
  title = ochusha_utils_url_encode_string(tmp_string);
  if (title == NULL)
    goto error_exit;
  G_FREE(tmp_string);

  tmp_string = convert_string(converter, NULL, response->name, -1);
  from = ochusha_utils_url_encode_string(tmp_string);
  if (from == NULL)
    goto error_exit;
  G_FREE(tmp_string);

  tmp_string = convert_string(converter, NULL, response->mailto, -1);
  mail = ochusha_utils_url_encode_string(tmp_string);
  if (mail == NULL)
    goto error_exit;
  G_FREE(tmp_string);

  tmp_string = convert_string(converter, NULL, response->content, -1);
  message = ochusha_utils_url_encode_string(tmp_string);
  if (message == NULL)
    goto error_exit;
  G_FREE(tmp_string);
  tmp_string = NULL;

  bbs = ochusha_bulletin_board_get_id(board);

  if (board_2ch->date != NULL)
    time = ochusha_utils_get_utc_time(board_2ch->date);
  else
    {
#if DEBUG_POST
      char *message[4096];
      snprintf(message, 4096, "board_2ch(%s)->date == NULL%s\n",
	       board->name);
      ochusha_network_broker_output_log(broker, message);
#endif
#if DEBUG_POST_MOST
      fprintf(stderr, "board_2ch->date == NULL!\n");
#endif
      time = ochusha_utils_get_utc_time(board_2ch->last_modified);
    }

  if (time == -1)
    goto error_exit;

  /* XXX: ϡɥǥ󥰤㤦ΤϤäʤġġ*/
  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_2CH:
    case OCHUSHA_BBS_TYPE_2CH_COMPATIBLE:
      query = g_strdup_printf("submit=%%90%%56%%8B%%4B%%83%%58%%83%%8C%%83%%62%%83%%68%%8D%%EC%%90%%AC&subject=%s&FROM=%s&mail=%s&MESSAGE=%s&bbs=%s&time=%ld", title, from, mail, message, bbs, time);
      break;

#if 0
    case OCHUSHA_BBS_TYPE_MACHIBBS:
      query = g_strdup_printf("submit=%%90%%56%%8B%%4B%%83%%58%%83%%8C%%83%%62%%83%%68%%8D%%EC%%90%%AC&NAME=%s&MAIL=%s&MESSAGE=%s&BBS=%s&KEY=%s&TIME=%ld", from, mail, message, bbs, key, time);
      break;
#endif
    default:
      goto error_exit;
    }

  post_result = ochusha_utils_2ch_try_post(broker, board, NULL, query, NULL);
  if (post_result == OCHUSHA_UTILS_2CH_POST_NO_COOKIE)
    {
      sleep(10);
      post_result = ochusha_utils_2ch_try_post(broker, board, NULL, query, NULL);
    }

#if 0
  if (post_result == OCHUSHA_UTILS_2CH_POST_MOCHITSUKE)
    {
      /* XXX: äݤΤȤͭʤΤ֡ */
      sleep(30);
      post_result = ochusha_utils_2ch_try_post(broker, board, NULL, query, NULL);
    }
#endif
  if (post_result != OCHUSHA_UTILS_2CH_POST_SUCCESS)
    goto error_exit;

  /* ｪλ */
  if (tmp_string != NULL)
    G_FREE(tmp_string);
  if (title != NULL)
    G_FREE(title);
  if (from != NULL)
    G_FREE(from);
  if (mail != NULL)
    G_FREE(mail);
  if (message != NULL)
    G_FREE(message);
  if (query != NULL)
    G_FREE(query);
  iconv_close(converter);

  return TRUE;

 error_exit:
#if DEBUG_POST_MOST
  fprintf(stderr, "Error happen\n");
#endif

  if (tmp_string != NULL)
    G_FREE(tmp_string);
  if (title != NULL)
    G_FREE(title);
  if (from != NULL)
    G_FREE(from);
  if (mail != NULL)
    G_FREE(mail);
  if (message != NULL)
    G_FREE(message);
  if (query != NULL)
    G_FREE(query);
  iconv_close(converter);
#endif

  return FALSE;
}


OchushaBulletinBoard *
ochusha_board_jbbs_new(const gchar *name, const char *base_url)
{
  g_assert(name != NULL && base_url != NULL);

  return OCHUSHA_BULLETIN_BOARD(g_object_new(OCHUSHA_TYPE_BOARD_JBBS,
					     "name", name,
					     "base_url", base_url,
					     NULL));
}


static OchushaAsyncBuffer *
ochusha_board_jbbs_get_threadlist_source(OchushaBulletinBoard *board,
					 OchushaNetworkBroker *broker,
					 OchushaAsyncBuffer *buffer,
					 OchushaNetworkBrokerCacheMode mode)
{
  OchushaBoard2ch *board_2ch;
  char url[PATH_MAX];

  g_return_val_if_fail(OCHUSHA_IS_BOARD_JBBS(board), NULL);

  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_MITINOKU:
      if (snprintf(url, PATH_MAX, "%s" OCHUSHA_SUBBACK_HTML, board->base_url)
	  >= PATH_MAX)
	return NULL;
      break;

    case OCHUSHA_BBS_TYPE_JBBS:
    case OCHUSHA_BBS_TYPE_MACHIBBS:
    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
    default:
      if (snprintf(url, PATH_MAX, "%s" OCHUSHA_SUBJECT_TXT, board->base_url)
	  >= PATH_MAX)
	return NULL;
    }

  board_2ch = OCHUSHA_BOARD_2CH(board);

  return ochusha_network_broker_read_from_url(broker, buffer, url,
					board_2ch->last_modified, mode,
					FALSE,
					broker->config->threadlist_chunksize);
}


static void
unmark_alive(gpointer data, gpointer user_data)
{
  OchushaThread2ch *thread = OCHUSHA_THREAD_2CH(data);
  thread->alive = FALSE;
}


typedef struct _CollectDroppedThreadArgs
{
  GSList *thread_list;
  EachThreadCallback *each_thread_cb;
  gpointer callback_data;
} CollectDroppedThreadArgs;


static void
collect_dropped_thread(gpointer data, gpointer user_data)
{
  CollectDroppedThreadArgs *args = (CollectDroppedThreadArgs *)user_data;
  OchushaThread2ch *thread_2ch = OCHUSHA_THREAD_2CH(data);

  if (!thread_2ch->alive)
    {
      /* threadDATäݤ*/
      OchushaBBSThread *thread = (OchushaBBSThread *)thread_2ch;
      if (thread->number_of_responses_read > 0)
	{
	  if (g_slist_find(args->thread_list, thread) == NULL)
	    {
	      thread->flags |= OCHUSHA_BBS_THREAD_DAT_DROPPED;
	      thread->number_of_responses_on_server = 0;
	      args->thread_list = g_slist_prepend(args->thread_list, thread);
	      if (args->each_thread_cb != NULL)
		if (!(*args->each_thread_cb)(thread, args->callback_data))
		  args->each_thread_cb = FALSE;
#if 0	/* debug */
	      fprintf(stderr, "Resurrect DAT dropped thread: %s\n",
		      ochusha_bbs_thread_get_url(thread));
#endif
	    }
	}
      else
	thread->board->dropped_list
	  = g_slist_prepend(thread->board->dropped_list, thread);
    }
  else
    OCHU_OBJECT_UNREF(data);	/* Ƥ륹thread_list
				 * Ѥ;פrefƤ롣
				 */
}


static void
undo_thread_ref(gpointer data, gpointer user_data)
{
  OchushaThread2ch *thread_2ch = OCHUSHA_THREAD_2CH(data);
  if (thread_2ch->alive)
    OCHU_OBJECT_UNREF(data);	/* Ƥ륹thread_list
				 * Ѥ;פrefƤ롣
				 */
}


static gboolean
advance_parsing(gpointer data)
{
  g_object_set_qdata(G_OBJECT(data), idle_checker_id, NULL);
  ochusha_async_buffer_broadcast((OchushaAsyncBuffer *)data);
  OCHU_OBJECT_UNREF(data);
  return FALSE;
}


/*
 * refresh_threadlist_by_subject_txt:
 *
 * Ϳ줿ХåեƤ2chĤsubject.txtȸʤƲϤͿ줿
 * OchushaBulletinBoard¤Τ򹹿롣
 * ĸĤ٤˻ꤵ줿ХåؿƤ֡
 *
 * ХåؿFALSE֤硢λǲϤλ롣
 *
 * ϤˤTRUE֤ԤFALSE֤
 */
static gboolean
refresh_threadlist_by_subject_txt(OchushaBulletinBoard *board,
				  OchushaAsyncBuffer *buffer,
				  EachThreadCallback *each_thread_cb,
				  StartParsingCallback *start_parsing_cb,
				  BeforeWaitCallback *before_wait_cb,
				  AfterWaitCallback *after_wait_cb,
				  EndParsingCallback *end_parsing_cb,
				  gpointer callback_data)
{
  char default_buffer[PATH_MAX];
  char scan_buffer[5];
  gboolean result = FALSE;
  GSList *thread_list = NULL;
  GSList *old_thread_list = board->thread_list;
  iconv_t converter;
  iconv_helper *converter_helper;
  OchushaNetworkBrokerBufferStatus *status
    = g_object_get_data(G_OBJECT(buffer),
			"OchushaNetworkBroker::BufferStatus");

  g_return_val_if_fail(OCHUSHA_IS_BOARD_2CH(board) && status != NULL, FALSE);

  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_MACHIBBS:
    case OCHUSHA_BBS_TYPE_JBBS:
      converter_helper = cp932_to_utf8_helper;
      break;

    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
    case OCHUSHA_BBS_TYPE_MITINOKU:
      converter_helper = eucjp_to_utf8_helper;
      break;

    default:
      converter_helper = NULL;
    }

  if (converter_helper != NULL)
    converter = iconv_open("UTF-8",
			   ochusha_board_jbbs_get_subject_txt_encoding(board));
  else
    converter = iconv_open("UTF-8//IGNORE",
			   ochusha_board_jbbs_get_subject_txt_encoding(board));
  g_return_val_if_fail(converter != (iconv_t)-1, FALSE);

  if (!ochusha_async_buffer_active_ref(buffer))
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "buffer has been terminated.\n");
#endif
      iconv_close(converter);
      return FALSE;
    }

  g_slist_foreach(old_thread_list, unmark_alive, NULL);

  ochusha_async_buffer_lock(buffer);
  {
    unsigned int offset = 0;
    gboolean buffer_fixed = FALSE;
    int interval = ANALYSIS_INTERVAL;

    if (start_parsing_cb != NULL)
      (*start_parsing_cb)(callback_data);

    while (TRUE)
      {
	char *buffer_top = (char *)buffer->buffer;
	char *cur_pos = buffer_top + offset;
	unsigned int rest_of_data = buffer->length - offset;
	char *eol_pos = NULL;

	while (rest_of_data > 0
	       && interval-- > 0
	       && (eol_pos = memchr(cur_pos, '\n', rest_of_data)) != NULL)
	  {
	    char *thread_id;
	    gchar *thread_title;
	    OchushaBBSThread *thread;
	    OchushaThread2ch *thread_2ch;
	    char *title_pos;
	    int n_responses_on_server = 0;
	    char *tmp_pos;
	    int title_len;
	    char *ext;

	    tmp_pos = memchr(cur_pos, ',', eol_pos - cur_pos);

	    if (tmp_pos == NULL)
	      {
#if DEBUG_LIBOCHUSHA
		fprintf(stderr, "Unknown format found in subject.txt.\n");
#endif
		goto prepare_next_line;
	      }
	    if ((tmp_pos - cur_pos) < PATH_MAX)
	      {
		thread_id = memcpy(default_buffer, cur_pos, tmp_pos - cur_pos);
		thread_id[tmp_pos - cur_pos] = '\0';
	      }
	    else
	      thread_id = G_STRNDUP(cur_pos, tmp_pos - cur_pos);

	    if ((ext = strstr(thread_id, ".cgi")) != NULL)
	      *ext = '\0';
	    title_pos = tmp_pos + 1;	/* skip "," */

	    title_len = eol_pos - title_pos;

	    for (tmp_pos = eol_pos - 1; tmp_pos > title_pos; tmp_pos--)
	      if (*tmp_pos == '(')
		break;

	    if (*tmp_pos == '(')
	      {
		char *close_paren = memchr(tmp_pos + 1, ')',
					   eol_pos - tmp_pos - 1);
		if (close_paren != NULL)
		  {
		    int tmp_len = close_paren - tmp_pos - 1;
		    if (tmp_len > 0 && tmp_len <= 4)
		      {
			memcpy(scan_buffer, tmp_pos + 1, tmp_len);
			scan_buffer[tmp_len] = '\0';
			sscanf(scan_buffer, "%d", &n_responses_on_server);
			title_len -= (tmp_len + 2);
		      }
		  }
	      }

	    thread = ochusha_bulletin_board_lookup_bbs_thread_by_id(board,
								    thread_id);
	    if (thread == NULL)
	      {
		thread_title = simple_string_canon(title_pos, title_len,
						   converter,
						   converter_helper);
		thread = ochusha_bulletin_board_bbs_thread_new(board,
							       thread_id,
							       thread_title);
		G_FREE(thread_title);
	      }
#if 1	/* 쥿äƤޤäĶ */
	    else
	      {
#if 0
		char *tmp_title = G_STRNDUP(title_pos, title_len);
		fprintf(stderr, "orig title: \"%s\"\n", tmp_title);
		G_FREE(tmp_title);
#endif
		thread_title = simple_string_canon(title_pos, title_len,
						   converter,
						   converter_helper);
#if 0
		fprintf(stderr, "title: \"%s\"\n", thread_title);
#endif
		ochusha_bbs_thread_set_title(thread, thread_title);
		G_FREE(thread_title);
	      }
#endif

	    if (thread_id != default_buffer)
	      G_FREE(thread_id);

	    thread_2ch = OCHUSHA_THREAD_2CH(thread);

	    if (!thread_2ch->alive)
	      {
		thread_2ch->alive = TRUE;

		thread->flags &= ~OCHUSHA_BBS_THREAD_DAT_DROPPED;
		thread->number_of_responses_on_server = n_responses_on_server;

		OCHU_OBJECT_REF(thread);
		thread_list = g_slist_prepend(thread_list, thread);

		if (each_thread_cb != NULL
		    && !(*each_thread_cb)(thread, callback_data))
		  {
		    goto terminated;
		  }
	      }

	  prepare_next_line:
	    offset = (eol_pos + 1) - buffer_top;
	    buffer_top = (char *)buffer->buffer;
	    cur_pos = buffer_top + offset;
	    rest_of_data = buffer->length - offset;
	    eol_pos = NULL;
	  }

	if (!buffer_fixed && buffer->fixed)
	  {
	    buffer_fixed = TRUE;
	    eol_pos = NULL;
	    continue;
	  }

	if (buffer->fixed && eol_pos == NULL && interval > 0)
	  {
	    result = TRUE;
	    goto terminated;
	  }

	if (interval > 0)
	  {
	    size_t old_len = buffer->length;

	    if (before_wait_cb != NULL)
	      (*before_wait_cb)(callback_data);

	    do
	      {
		if (!ochusha_async_buffer_wait(buffer))
		  {
#if DEBUG_ASYNC_BUFFER_MOST
		    fprintf(stderr, "buffer has been terminated.\n");
#endif
		    if (after_wait_cb != NULL)
		      (*after_wait_cb)(callback_data);
		    goto terminated;
		  }
	      } while (old_len >= buffer->length && !buffer->fixed);
	    if (after_wait_cb != NULL)
	      (*after_wait_cb)(callback_data);
	  }
	else
	  {
	    OCHU_OBJECT_REF(buffer);
	    g_object_set_qdata(G_OBJECT(buffer), idle_checker_id, buffer);
#if 0
	    g_idle_add_full(G_PRIORITY_HIGH_IDLE + 30,
			    advance_parsing, buffer, NULL);
#else
	    g_idle_add_full(G_PRIORITY_HIGH_IDLE + 15,
			    advance_parsing, buffer, NULL);
#endif
	    if (before_wait_cb != NULL)
	      (*before_wait_cb)(callback_data);
	    do
	      {
		if (!ochusha_async_buffer_wait(buffer))
		  {
		    if (after_wait_cb != NULL)
		      (*after_wait_cb)(callback_data);
		    goto terminated;
		  }
	      } while (g_object_get_qdata(G_OBJECT(buffer),
					  idle_checker_id) != NULL);
	    if (after_wait_cb != NULL)
	      (*after_wait_cb)(callback_data);
	    interval = ANALYSIS_INTERVAL;
	  }

	if (status->state == OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_DIRTY)
	  {
	    goto terminated;
	  }
      }
  }
 terminated:
  ochusha_async_buffer_unlock(buffer);

  if (result)
    {
      OchushaBoard2ch *board_2ch = OCHUSHA_BOARD_2CH(board);

      if (status->last_modified != NULL)
	{
	  if (board_2ch->last_modified != NULL)
	    G_FREE(board_2ch->last_modified);
	  board_2ch->last_modified = G_STRDUP(status->last_modified);
	}

      if (status->date != NULL)
	{
	  if (board_2ch->date != NULL)
	    G_FREE(board_2ch->date);
	  board_2ch->date = G_STRDUP(status->date);
	}
    }

  ochusha_async_buffer_active_unref(buffer);

  if (result)
    {
      CollectDroppedThreadArgs args =
	{
	  thread_list,
	  each_thread_cb,
	  callback_data
	};

      g_slist_foreach(old_thread_list, collect_dropped_thread, &args);
      board->thread_list = args.thread_list;
      g_slist_free(old_thread_list);
    }
  else
    {
      /* 顼ϸŤΤͥ */
      g_slist_foreach(old_thread_list, undo_thread_ref, NULL);
      g_slist_free(thread_list);
    }

  iconv_close(converter);

  if (end_parsing_cb != NULL)
    (*end_parsing_cb)(callback_data);

  return result;
}


/*
 * refresh_threadlist_by_subback_html:
 *
 * Ĥsubback.htmlȸʤƲϤͿ줿
 * OchushaBulletinBoard¤Τ򹹿롣
 * ĸĤ٤˻ꤵ줿ХåؿƤ֡
 *
 * ХåؿFALSE֤硢λǲϤλ롣
 *
 * ϤˤTRUE֤ԤFALSE֤
 */
static gboolean
refresh_threadlist_by_subback_html(OchushaBulletinBoard *board,
				   OchushaAsyncBuffer *buffer,
				   EachThreadCallback *each_thread_cb,
				   StartParsingCallback *start_parsing_cb,
				   BeforeWaitCallback *before_wait_cb,
				   AfterWaitCallback *after_wait_cb,
				   EndParsingCallback *end_parsing_cb,
				   gpointer callback_data)
{
  gboolean result = FALSE;
  GSList *thread_list = NULL;
  GSList *old_thread_list = board->thread_list;
  iconv_t converter;
  iconv_helper *converter_helper;
  regex_t threadlist_entry_pattern;
  OchushaNetworkBrokerBufferStatus *status
    = g_object_get_data(G_OBJECT(buffer),
			"OchushaNetworkBroker::BufferStatus");

  g_return_val_if_fail(OCHUSHA_IS_BOARD_2CH(board) && status != NULL, FALSE);

  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_MITINOKU:
      converter_helper = eucjp_to_utf8_helper;
      if (regcomp(&threadlist_entry_pattern,
		  ".*</a>:<a href=.*key=([1-9][0-9]*)[^>]*>(.*)&lt;([1-9][0-9]*)&gt;</a><br>$", REG_EXTENDED))
	{
	  fprintf(stderr, "invalid regular expression\n");
	  return FALSE;
	}
      break;

    case OCHUSHA_BBS_TYPE_MACHIBBS:
    case OCHUSHA_BBS_TYPE_JBBS:
    default:
      converter_helper = cp932_to_utf8_helper;
      return FALSE;	/* ̤ݡ */

    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
      converter_helper = eucjp_to_utf8_helper;
      return FALSE;	/* ̤ݡ */
    }

  converter = iconv_open("UTF-8//IGNORE",
			 ochusha_board_jbbs_get_subback_html_encoding(board));
  if (converter == (iconv_t)-1)
    regfree(&threadlist_entry_pattern);
  g_return_val_if_fail(converter != (iconv_t)-1, FALSE);

  if (!ochusha_async_buffer_active_ref(buffer))
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "buffer has been terminated.\n");
#endif
      iconv_close(converter);
      regfree(&threadlist_entry_pattern);
      return FALSE;
    }

  g_slist_foreach(old_thread_list, unmark_alive, NULL);

  ochusha_async_buffer_lock(buffer);
  {
    unsigned int offset = 0;
    gboolean buffer_fixed = FALSE;

    if (start_parsing_cb != NULL)
      (*start_parsing_cb)(callback_data);

    while (TRUE)
      {
	char *buffer_top = (char *)buffer->buffer;
	char *cur_pos = buffer_top + offset;
	unsigned int rest_of_data = buffer->length - offset;
	char *eol_pos = NULL;
	int interval = ANALYSIS_INTERVAL;

	while (rest_of_data > 0
	       && interval-- > 0
	       && (eol_pos = memchr(cur_pos, '\n', rest_of_data)) != NULL)
	  {
	    char *thread_id;
	    gchar *thread_title;
	    OchushaBBSThread *thread;
	    OchushaThread2ch *thread_2ch;
	    gchar *html_line;
	    int n_responses_on_server = 0;
	    regmatch_t match[4];

	    *eol_pos = '\0';
	    html_line = convert_string(converter, converter_helper,
				       cur_pos, -1);
	    *eol_pos = '\n';	/* ̣ʤ̣ġ */

	    if (html_line == NULL
		|| regexec(&threadlist_entry_pattern,
			   html_line, 4, match, 0) != 0)
	      {
#if 0
		fprintf(stderr, "html_line: \"%s\"\n", html_line);
#endif
		goto prepare_next_line;
	      }

	    thread_id = html_line + match[1].rm_so;
	    thread_id[match[1].rm_eo - match[1].rm_so] = '\0';

	    thread = ochusha_bulletin_board_lookup_bbs_thread_by_id(board,
								    thread_id);
	    if (thread == NULL)
	      {
		thread_title
		  = simple_string_canon(html_line + match[2].rm_so,
					match[2].rm_eo - match[2].rm_so,
					NULL, NULL);
		thread = ochusha_bulletin_board_bbs_thread_new(board,
							       thread_id,
							       thread_title);
		G_FREE(thread_title);
	      }

	    html_line[match[3].rm_eo] = '\0';
	    sscanf(html_line + match[3].rm_so, "%d", &n_responses_on_server);

	    thread_2ch = OCHUSHA_THREAD_2CH(thread);

	    if (!thread_2ch->alive)
	      {
		thread_2ch->alive = TRUE;

		thread->flags &= ~OCHUSHA_BBS_THREAD_DAT_DROPPED;
		thread->number_of_responses_on_server = n_responses_on_server;

		OCHU_OBJECT_REF(thread);
		thread_list = g_slist_prepend(thread_list, thread);

		if (each_thread_cb != NULL
		    && !(*each_thread_cb)(thread, callback_data))
		  {
		    goto terminated;
		  }
	      }

	  prepare_next_line:
	    if (html_line != NULL)
	      G_FREE(html_line);
	    offset = (eol_pos + 1) - buffer_top;
	    buffer_top = (char *)buffer->buffer;
	    cur_pos = buffer_top + offset;
	    rest_of_data = buffer->length - offset;
	  }

	if (!buffer_fixed && buffer->fixed)
	  {
	    buffer_fixed = TRUE;
	    eol_pos = NULL;
	    continue;
	  }

	if (buffer->fixed && eol_pos == NULL && interval > 0)
	  {
	    result = TRUE;
	    goto terminated;
	  }

	if (interval > 0)
	  {
	    size_t old_len = buffer->length;

	    if (before_wait_cb != NULL)
	      (*before_wait_cb)(callback_data);

	    do
	      {
		if (!ochusha_async_buffer_wait(buffer))
		  {
#if DEBUG_ASYNC_BUFFER_MOST
		    fprintf(stderr, "buffer has been terminated.\n");
#endif
		    if (after_wait_cb != NULL)
		      (*after_wait_cb)(callback_data);
		    goto terminated;
		  }
	      } while (old_len >= buffer->length && !buffer->fixed);
	    if (after_wait_cb != NULL)
	      (*after_wait_cb)(callback_data);
	  }
	else
	  {
	    OCHU_OBJECT_REF(buffer);
	    g_object_set_qdata(G_OBJECT(buffer), idle_checker_id, buffer);
	    g_idle_add_full(G_PRIORITY_HIGH_IDLE + 15,
			    advance_parsing, buffer, NULL);
	    if (before_wait_cb != NULL)
	      (*before_wait_cb)(callback_data);
	    do
	      {
		if (!ochusha_async_buffer_wait(buffer))
		  {
		    if (after_wait_cb != NULL)
		      (*after_wait_cb)(callback_data);
		    goto terminated;
		  }
	      } while (g_object_get_qdata(G_OBJECT(buffer),
					  idle_checker_id) != NULL);
	    if (after_wait_cb != NULL)
	      (*after_wait_cb)(callback_data);
	    interval = ANALYSIS_INTERVAL;
	  }

	if (status->state == OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_DIRTY)
	  {
	    goto terminated;
	  }
      }
  }
 terminated:
  ochusha_async_buffer_unlock(buffer);

  if (result)
    {
      OchushaBoard2ch *board_2ch = OCHUSHA_BOARD_2CH(board);

      if (status->last_modified != NULL)
	{
	  if (board_2ch->last_modified != NULL)
	    G_FREE(board_2ch->last_modified);
	  board_2ch->last_modified = G_STRDUP(status->last_modified);
	}

      if (status->date != NULL)
	{
	  if (board_2ch->date != NULL)
	    G_FREE(board_2ch->date);
	  board_2ch->date = G_STRDUP(status->date);
	}
    }

  ochusha_async_buffer_active_unref(buffer);

  if (result)
    {
      CollectDroppedThreadArgs args =
	{
	  thread_list,
	  each_thread_cb,
	  callback_data
	};

      g_slist_foreach(old_thread_list, collect_dropped_thread, &args);
      board->thread_list = args.thread_list;
      g_slist_free(old_thread_list);
    }
  else
    {
      /* 顼ϸŤΤͥ */
      g_slist_foreach(old_thread_list, undo_thread_ref, NULL);
      g_slist_free(thread_list);
    }

  regfree(&threadlist_entry_pattern);
  iconv_close(converter);

  if (end_parsing_cb != NULL)
    (*end_parsing_cb)(callback_data);

  return result;
}


/*
 * ochusha_board_jbbs_refresh_threadlist:
 *
 * Ϳ줿ХåեƤĤΥΥȸʤƲϤ
 * Ϳ줿OchushaBulletinBoard¤Τ򹹿롣
 * ĸĤ٤˻ꤵ줿ХåؿƤ֡
 *
 * ХåؿFALSE֤硢λǲϤλ롣
 *
 * ϤˤTRUE֤ԤFALSE֤
 */
static gboolean
ochusha_board_jbbs_refresh_threadlist(OchushaBulletinBoard *board,
				      OchushaAsyncBuffer *buffer,
				      EachThreadCallback *each_thread_cb,
				      StartParsingCallback *start_parsing_cb,
				      BeforeWaitCallback *before_wait_cb,
				      AfterWaitCallback *after_wait_cb,
				      EndParsingCallback *end_parsing_cb,
				      gpointer callback_data)
{
  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_MITINOKU:
      return refresh_threadlist_by_subback_html(board, buffer,
						each_thread_cb,
						start_parsing_cb,
						before_wait_cb, after_wait_cb,
						end_parsing_cb, callback_data);

    default:
      return refresh_threadlist_by_subject_txt(board, buffer,
					       each_thread_cb,
					       start_parsing_cb,
					       before_wait_cb, after_wait_cb,
					       end_parsing_cb, callback_data);

    }
}
