/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: monitor.c,v 1.1.2.11 2004/11/25 08:58:05 fuyu Exp $
 */

#include "config.h"

#include "monitor.h"

#include <errno.h>
#include <pthread.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#if defined(HAVE_CLOCK_GETTIME) || defined(HAVE_GETTIMEOFDAY)
# ifdef HAVE_SYS_TIME_H
#  include <sys/time.h>
# endif
#endif

#define CHECK_LOCK_ORDER	0


Monitor *
ochusha_monitor_new(Monitor *other)
{
  Monitor *monitor = (Monitor *)calloc(1, sizeof(Monitor));
  if (monitor == NULL)
    {
      fprintf(stderr, "Couldn't allocate a Monitor.\n");
      abort();
    }

  if (pthread_mutex_init(&monitor->mutex, NULL) != 0)
    {
      fprintf(stderr, "Couldn't initialize a mutex.\n");
      abort();
    }
  if (pthread_cond_init(&monitor->cond, NULL) != 0)
    {
      fprintf(stderr, "Couldn't initialize a condition.\n");
      abort();
    }
  monitor->other = other;
#if 0
  fprintf(stderr, "Monitor(@%p) constructed.\n", monitor);
#endif
  return monitor;
}


void
ochusha_monitor_free(Monitor *monitor)
{
  if (pthread_cond_destroy(&monitor->cond) != 0)
    {
      fprintf(stderr, "Couldn't destroy a condition.\n");
      abort();
    }
  if (pthread_mutex_destroy(&monitor->mutex) != 0)
    {
      fprintf(stderr, "Couldn't destroy a mutex.\n");
      abort();
    }
  free(monitor);
}


void
ochusha_monitor_enter(Monitor *monitor)
{
  int result;
  if (monitor->count > 0 && pthread_equal(monitor->locker, pthread_self()))
    {
      ++monitor->count;
#if 0
      fprintf(stderr, "Monitor(@%p) recursively entered.\n", monitor);
#endif
      return;
    }

#if CHECK_LOCK_ORDER
  if (monitor->other != NULL && monitor->other->count > 0
      && pthread_equal(monitor->other->locker, pthread_self()))
    {
      fprintf(stderr, "Lock Order Reversal: %p should follow %p\n",
	      monitor->other, monitor);
      abort();
    }
#endif

  if ((result = pthread_mutex_lock(&monitor->mutex)) == 0)
    {
      monitor->locker = pthread_self();
      monitor->count = 1;
      return;
    }

  fprintf(stderr, "Couldn't lock a mutex: %s(%d)\n", strerror(result), result);
  abort();
}


int
ochusha_monitor_try_enter(Monitor *monitor)
{
  int result;
  if (monitor->count > 0 && pthread_equal(monitor->locker, pthread_self()))
    {
      ++monitor->count;
#if 0
      fprintf(stderr, "Monitor(@%p) recursively entered.\n", monitor);
#endif
      return 1;
    }

#if CHECK_LOCK_ORDER
  if (monitor->other != NULL && monitor->other->count > 0
      && pthread_equal(monitor->other->locker, pthread_self()))
    {
      fprintf(stderr, "Lock Order Reversal: %p should follow %p\n",
	      monitor->other, monitor);
      abort();
    }
#endif

  if ((result = pthread_mutex_trylock(&monitor->mutex)) == 0)
    {
      monitor->locker = pthread_self();
      monitor->count = 1;
      return 1;
    }

  if (result == EBUSY)
    return 0;

  /* EBUSYʳϲ */
  fprintf(stderr, "Couldn't trylock a mutex: %s(%d)\n",
	  strerror(result), result);
  abort();
}


void
ochusha_monitor_exit(Monitor *monitor)
{
  int result;
  if (monitor->count < 1 || !pthread_equal(monitor->locker, pthread_self()))
    {
      fprintf(stderr, "Couldn't unlock a mutex held by another thread!\n");
#if 0
      fprintf(stderr,
	      "Couldn't unlock a mutex: locker=%p, self=%p, count=%d\n",
	      monitor->locker, pthread_self(), monitor->count);
#endif
      abort();
    }

  if (--monitor->count > 0)
    return;

  if ((result = pthread_mutex_unlock(&monitor->mutex)) != 0)
    {
      fprintf(stderr, "Couldn't unlock a mutex: %s(%d)\n",
	      strerror(result), result);
      abort();
    }
  return;
}


void
ochusha_monitor_wait(Monitor *monitor)
{
  int prev_count;
  int result;
  ochusha_monitor_enter(monitor);
  prev_count = monitor->count;
  monitor->count = 0;

#if CHECK_LOCK_ORDER
  if (monitor->other != NULL)
    {
      if (monitor->other->count > 0
	  && pthread_equal(monitor->other->locker, pthread_self()))
	{
	  fprintf(stderr, "Lock Order Reversal: attemplt to wait on %p during %p held.\n", monitor, monitor->other);
	  abort();
	}
    }
#endif

  if ((result = pthread_cond_wait(&monitor->cond, &monitor->mutex)) != 0)
    {
      fprintf(stderr, "Couldn't wait a condition: %s(%d)\n",
	      strerror(result), result);
      abort();
    }

  if (monitor->count != 0)
    {
      fprintf(stderr, "Monitor protocol error.\n");
      abort();
    }

  monitor->locker = pthread_self();
  monitor->count = prev_count;
  ochusha_monitor_exit(monitor);
}


int
ochusha_monitor_timedwait(Monitor *monitor, long ms)
{
#if defined(HAVE_CLOCK_GETTIME) || defined(HAVE_GETTIMEOFDAY)
  struct timespec abstime;
  int prev_count;
  int result;

  if (ms > 0)
    {
# ifdef HAVE_CLOCKGETTIME
      if (clock_gettime(CLOCK_REALTIME, &abstime) == 0)
	{
	  long nsec = abstime.tv_nsec + (ms % 1000) * 1000000;
	  abstime.tv_sec += (ms / 1000) + (nsec / 1000000000);
	  abstime.tv_nsec = nsec % 1000000000;
	}
      else
	ms = 0;
# else
      struct timeval abstime_now;
      if (gettimeofday(&abstime_now, NULL) == 0)
	{
	  long nsec;
	  abstime.tv_sec = abstime_now.tv_sec;
	  abstime.tv_nsec = abstime_now.tv_usec * 1000;

	  nsec = abstime.tv_nsec + (ms % 1000) * 1000000;
	  abstime.tv_sec += (ms / 1000) + (nsec / 1000000000);
	  abstime.tv_nsec = nsec % 1000000000;
	}
      else
	ms = 0;
# endif
    }

  if (ms == 0)
    {
      ochusha_monitor_wait(monitor);
      return 1;
    }

  ochusha_monitor_enter(monitor);
  prev_count = monitor->count;
  monitor->count = 0;

# if CHECK_LOCK_ORDER
  if (monitor->other != NULL)
    {
      if (monitor->other->count > 0
	  && pthread_equal(monitor->other->locker, pthread_self()))
	{
	  fprintf(stderr, "Lock Order Reversal: attemplt to wait on %p during %p held.\n", monitor, monitor->other);
	  abort();
	}
    }
# endif

  result = pthread_cond_timedwait(&monitor->cond, &monitor->mutex, &abstime);
  if (result != 0 && result != ETIMEDOUT)
    {
      fprintf(stderr, "Couldn't wait a condition: %s(%d)\n",
	      strerror(result), result);
      abort();
    }

  if (monitor->count != 0)
    {
      fprintf(stderr, "Monitor protocol error.\n");
      abort();
    }

  monitor->locker = pthread_self();
  monitor->count = prev_count;
  ochusha_monitor_exit(monitor);

  return result == 0;
#else
  ochusha_monitor_wait(monitor);
  return 1;
#endif
}


void
ochusha_monitor_notify(Monitor *monitor)
{
  int result;
  if ((result = pthread_cond_signal(&monitor->cond)) != 0)
    {
      fprintf(stderr, "Couldn't signal a condition: %s(%d)\n",
	      strerror(result), result);
      abort();
    }
}


void
ochusha_monitor_notify_all(Monitor *monitor)
{
  int result;
  if ((result = pthread_cond_broadcast(&monitor->cond)) != 0)
    {
      fprintf(stderr, "Couldn't broadcast a condition: %s(%d)\n",
	      strerror(result), result);
      abort();
    }
}
