#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>

#include <math.h>

#include "mgwtypes.h"
#include "interface.h"
#include "image.h"
#include "sort.h"
#include "utils.h"

/*
****************************************************************************
Sort Algorithms are from...

http://www.cs.ubc.ca/spider/harrison/Java/sorting-demo.html
http://www.ics.kagoshima-u.ac.jp/~fuchida/edu/algorithm/sort-algorithm/
****************************************************************************
*/

static GdkPixbuf *pixbuf;
static guchar *original;
static gint *sequence;
static gint width, height, rowstride, n_channels;

static void
games_still_in_progress (gdouble progress)
{
  gint i, j, k;
  guchar *p0 = gdk_pixbuf_get_pixels (pixbuf);

  show_progress (progress, pixbuf);

  for (i = 0; i < height; i++)
    for (j = 0; j < width; j++)
      for (k = 0; k < 3; k++)
	*(p0 + i * rowstride + j * n_channels + k)
	  = original[sequence[(i * width + j)] * 3 + k];

  redraw_image (pixbuf);
}

static gint
quick_sort_pivot (gint *a, gint i, gint j)
{
  gint k;

  k = i + 1;
  while (k <= j && a[i] == a[k])
    k++;
  if (k > j)
    return -1;
  if (a[i] >= a[k])
    return i;
  return k;
}

static gint
quick_sort_partition (gint *a, gint i, gint j, gint x)
{
  gint l, r;

  l = i;
  r = j;

  while (l <= r) {
    while (l <= j && a[l] < x)
      l++;
    while (r >= i && a[r] >= x)
      r--;

    if (l > r)
      break;
    swap (a + l, a + r);

    l++;
    r--;
  }
  return l;
}

static void
quick_sort (gint *a, gint i, gint j)
{
  static gint count = 0;
  gint p;

  if (i == j)
    return;

  p = quick_sort_pivot (a, i, j);
  if (p != -1) {
    gint k = quick_sort_partition (a, i, j, a[p]);
    quick_sort (a, i, k - 1);
    quick_sort (a, k,     j);
  }

  if (count % 10000 == 0)
    games_still_in_progress ((gdouble) (count % 500000) / 500000);

  if (++count == UINT_MAX)
    count = 0;
}

static void
kr_sort (gint *a, gint n)
{
  gint gap, i, j;
  
  for (gap = n / 2; gap > 0; gap /= 2) {
    for (i = gap; i < n; i++) {
      for (j = i - gap; j >= 0 && a[j] > a[j + gap]; j -= gap) {
	swap (a + j, a + j + gap);
      }
    }

    games_still_in_progress (1.0 - pow ((gdouble) gap / (n / 2), 0.333));
  }
}

static void
comb_sort (gint *a, gint n)
{
  gboolean flipped = FALSE;
  gint gap, top;
  gint i, j;

  gap = n;
  do {
    gap = (gint) ((gdouble) gap / 1.3);
    switch (gap) {
    case 0:
      gap = 1;
      break;
    case 9:
    case 10:
      gap = 11;
      break;
    defatult:
      break;
    }
    flipped = FALSE;
    top = n - gap;
    for (i = 0; i < top; i++) {
      j = i + gap;
      if (a[i] > a[j]) {
	swap (a + i, a + j);
	flipped = TRUE;
      }
    }

    games_still_in_progress (1.0 - pow ((gdouble) gap / n, 0.333));
  } while (flipped || (gap > 1));
}

static void
shell_sort (gint *a, gint n)
{
  gint h = 1, h0;
  gint i, j, B;

  while ((h * 3 + 1) < n) {
    h = 3 * h + 1;
  }

  h0 = h;

  while (h > 0) {
    for (i = h - 1; i < n; i++) {
      B = a[i];
      j = i;
      for (j = i; (j >= h) && (a[j - h] > B); j -= h)
	a[j] = a[j - h];
      a[j] = B;
    }
    h /= 3;

    games_still_in_progress (1.0 - pow ((gdouble) h / h0, 0.333));
  }
}

static void
heap_sort_down (gint *a, gint k, gint n)
{
  gint j, temp;

  temp = a[k - 1];

  while (k <= n / 2) {
    j = k + k;
    if ((j < n) && (a[j - 1] < a[j]))
      j++;
    if (temp >= a[j - 1])
      break;
    else {
      a[k - 1] = a[j - 1];
      k = j;
    }
  }
  a[k - 1] = temp;
}

static void
heap_sort (gint *a, gint n)
{
  gint nnn = n;
  gint k;

  for (k = nnn / 2; k > 0; k--)
    heap_sort_down (a, k, nnn);

  do {
    swap (a, a + nnn - 1);
    nnn--;
    heap_sort_down (a, 1, nnn);

    if (((n - nnn) % 10000) == 0)
      games_still_in_progress (1.0 - (gdouble) nnn / n);
  } while (nnn > 1);
}

static void
insertion_sort (gint *a, gint lo0, gint hi0)
{
  gint i, j, v;

  for (i = lo0 + 1; i <= hi0; i++) {
    v = a[i];
    j = i;
    while ((j > lo0) && (a[j - 1] > v)) {
      a[j] = a[j - 1];
      j--;
    }
    a[j] = v;
  }
}

static void
fast_quick_sort (gint *a, gint l, gint r)
{
  static gint count = 0;
  gint M = 4;
  gint i, j, v;

  if ((r - l) > M) {
    i = (r + l) / 2;
    if (a[l] > a[i]) swap (a + l, a + i);
    if (a[l] > a[r]) swap (a + l, a + r);
    if (a[i] > a[r]) swap (a + i, a + r);

    j = r - 1;
    swap (a + i, a + j);
    i = l;
    v = a[j];
    for (;;) {
      while (a[++i] < v);
      while (a[--j] > v);
      if (j < i) break;
      swap (a + i, a + j);
    }
    swap (a + i, a + r - 1);
    fast_quick_sort (a,     l, j);
    fast_quick_sort (a, i + 1, r);

    if (count % 10000 == 0)
      games_still_in_progress ((gdouble) (count % 500000) / 500000);
    if (++count == UINT_MAX)
      count = 0;
  }
}

static void
sort_process (GdkPixbuf *pbuf, gchar *flag)
{
  MgwSizeInfo si;
  gint i, j, n;

  get_size_info (&si, pbuf);

  width      = si.w;
  height     = si.h;
  rowstride  = si.rs;
  n_channels = si.nc;

  redraw_image (pbuf);
  n = width * height;
  if (!(sequence = (gint *) g_malloc (sizeof (gint) * n))) {
    g_warning ("sort_process(sort.c): memory allocation failed(a)");
    goto FUNC_END;
  }
  for (i = 0; i < n; i++)
    sequence[i] = i;
  for (i = 0; i < n; i++) {
    j = n * ((gdouble) random () / RAND_MAX);
    swap (sequence + i, sequence + j);
  }

  if (!(original = stock_rgb_data (pbuf, si))) {
    g_warning ("sort_process(sort.c): memory allocation failed(b)");
    goto FUNC_END;
  }

  if (!strcmp (flag, "K & R")) {
    kr_sort (sequence, n);
  } else if (!strcmp (flag, "Quick Sort")) {
    quick_sort (sequence, 0, n - 1);
    games_still_in_progress (1);
  } else if (!strcmp (flag, "Comb Sort")) {
    comb_sort (sequence, n);
  } else if (!strcmp (flag, "Shell Sort")) {
    shell_sort (sequence, n);
  } else if (!strcmp (flag, "Heap Sort")) {
    heap_sort (sequence, n);
    games_still_in_progress (1);
  } else if (!strcmp (flag, "Fast Quick Sort")) {
    fast_quick_sort (sequence, 0, n - 1);
    insertion_sort (sequence, 0, n - 1);
    games_still_in_progress (1);
  } else {
    g_warning ("sort_process: FALIED(Internal error): %s", flag);
  }
  show_progress (0, pixbuf);

 FUNC_END:
  g_free (sequence);
  g_free (original);
}

/* Entry point */
void
sort_exec (GtkWidget *dummy, gchar *algorithm)
{
  MgwClipRect c;
  GdkPixbuf *pbuf, *pbuf_org;
  gchar buff[64];
  gboolean has_clip;

  if (!(pbuf_org = get_pbuf ()))
    return;

  menubar_hilite_control (LOCK);
  set_cursor (GDK_WATCH);
  sprintf (buff, "%s: DEMONSTRATION...", algorithm);
  put_string_to_appbar (buff);

  has_clip = get_clip_rect_area (&c);

  if (has_clip) {
    pbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, c.x1, c.y1);
    gdk_pixbuf_copy_area (pbuf_org, c.x0, c.y0, c.x1, c.y1, pbuf, 0, 0);
  } else {
    pbuf = pbuf_org;
  }

  pixbuf = pbuf;

  sort_process (pbuf, algorithm);

  if (has_clip) {
    redraw_image (pbuf_org);
    draw_clip_rect_area ();
  }

  sprintf (buff, "%s: DONE", algorithm);
  put_string_to_appbar (buff);

  set_cursor (GDK_TOP_LEFT_ARROW);
  menubar_hilite_control (UNLOCK);
}
