/*  Window monitor / icon theme manager
 *  Copyright (C) 2005 UCHINO Satoshi.  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

#include "iconmgr.h"
#include <iostream>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <dirent.h>

IconMgr::IconMgr(ClientMgr& _clientMgr) :
  clientMgr(_clientMgr)
{
  // XXX for now, use these three themes
  setIconTheme("default");	// for GNOME
  addIconTheme("default.kde");	// for KDE
  addIconTheme("hicolor");
}

IconMgr::~IconMgr()
{
}

void IconMgr::addIconTheme(const char *theme)
{
  //cout << "+" << __FUNCTION__ << "(" << theme << ")" << endl;
  string themeDir;

  /* for each directories in data_dirs ... */
  vector<string>* dirList = ConfigFile::getXdgDataDirs();
  for (int i = 0; i < dirList->size(); i++) {
    string dir = (*dirList)[i] + "icons/" + theme;
    //cout << __FUNCTION__ << ": chdir to '" << dir << "'" << endl;
    /* check if theme dir exists */
    if (chdir(dir.c_str()) != 0) {
      //cerr << __FUNCTION__ << ": failed to chdir to '" << dir << "'" << endl;
      continue;
    }

    /* check if an index.theme exists */
    string themeIndex = dir + "/index.theme";
    int fd = open(dir.c_str(), O_RDONLY);
    if (fd != -1) { /* found */
      close(fd);
      themeDir = dir;
    }
  }
  delete dirList;
  if (themeDir.empty()) {
    cerr << __FUNCTION__ << ": could not find icon directory for " << theme << endl;
    return;
  }

  /* now we have an icon dir */
  //cout << __FUNCTION__ << ": icon dir = " << themeDir << endl;

  set<string> sectionSet;
  set<string> keySet;
  keySet.insert("Inherits");	/* for "Icon Theme" section */
  keySet.insert("Directories");	/* for "Icon Theme" section */
  keySet.insert("Size");
  keySet.insert("Type");
  keySet.insert("MaxSize");
  keySet.insert("MinSize");
  keySet.insert("Threshold");

  /* read an index.theme */
  string filename = themeDir + "/index.theme";
  string dummylocale; // dummy
  ConfigFile::SpItemMapMap itemMapMap = ConfigFile::readSectionSet(filename, sectionSet, dummylocale, keySet);
  if (!itemMapMap) {
    cerr << __FUNCTION__ << ": failed to read " << filename << endl;
    return;
  }

  if (itemMapMap->find("Icon Theme") == itemMapMap->end()) {
    cerr << __FUNCTION__ << ": could not find \"Icon Theme\" section." << endl;
    return;
  }

  ConfigFile::SpItemMap itemMap = (*itemMapMap)["Icon Theme"];
  if (itemMap->find("Directories") != itemMap->end()) {
    // add this map
    Item item;
    item.dir   = themeDir;
    item.index = itemMapMap;
    themeList.push_back(item);
  }

  if (itemMap->find("Inherits") != itemMap->end()) {
    // has an "Inherits"
    string inherits((*itemMap)["Inherits"].body);
    string::size_type pos;
    while (!inherits.empty()) {
      pos = inherits.find(',');
      string itheme;
      if (pos == string::npos) {
	itheme   = inherits;
	inherits = "";
      } else {
	itheme   = inherits.substr(0, pos);
	inherits = inherits.substr(pos + 1);
      }
      addIconTheme(itheme);
    };
  }
  //cout << "-" << __FUNCTION__ << "(" << theme << ")" << endl;
}

void IconMgr::setIconTheme(const char *theme)
{
  themeList.clear();
  addIconTheme(theme);
}

bool IconMgr::lookupIcon(string& iconname, string& retPath, ClientMgr::IconPref& iconInfo, Item item)
{
  ConfigFile::SpItemMap itemMap;
  if (item.index->find("Icon Theme") == item.index->end()) {
    cerr << __FUNCTION__ << ": could not find \"Icon Theme\" section." << endl;
    return false;
  }

  itemMap = (*item.index)["Icon Theme"];
  if (itemMap->find("Directories") == itemMap->end()) {
    cerr << __FUNCTION__ << ": could not find \"Directories\"" << endl;
    return false;
  }

  string directories = (*itemMap)["Directories"].body;
  string::size_type pos;
  while (!directories.empty()) {
    string directory;
    pos = directories.find(',');
    if (pos == string::npos) {
      directory   = directories;
      directories = "";
    } else {
      directory   = directories.substr(0, pos);
      directories = directories.substr(pos + 1);
    }
    if (item.index->find(directory) == item.index->end()) {
      cerr << __FUNCTION__ << ": could not find \"" << directory << "\" section." << endl;
      continue;
    }

    ConfigFile::SpItemMap itemMap = (*item.index)[directory];

    if (itemMap->find("Size") == itemMap->end())
      continue;

    int size = atoi((*itemMap)["Size"].body.c_str());

    if (itemMap->find("Type") == itemMap->end() ||
	(*itemMap)["Type"].body == "Threshold") {
      int threshold = 2;
      if (itemMap->find("Threshold") != itemMap->end())
	threshold = atoi((*itemMap)["Threshold"].body.c_str());
      if (iconInfo.size < size - threshold ||
	  size + threshold < iconInfo.size)
	continue;
    } else if ((*itemMap)["Type"].body == "Fixed") {
      if (iconInfo.size != size)
	continue;
    } else if ((*itemMap)["Type"].body == "Scalable") {
      int maxsize = size, minsize = size;
      if (itemMap->find("MaxSize") != itemMap->end())
	maxsize = atoi((*itemMap)["MaxSize"].body.c_str());
      if (itemMap->find("MinSize") != itemMap->end())
	minsize = atoi((*itemMap)["MinSize"].body.c_str());
      if (iconInfo.size < minsize || maxsize < iconInfo.size)
	continue;
    } else {
      /* invalid type */
      continue;
    }

    /* check if iconname exists */
    string base, extension;
    string::size_type pos = iconname.rfind('.');
    if (pos == string::npos) {
      base = iconname;
    } else {
      base = iconname.substr(0, pos);      
      extension = iconname.substr(pos + 1);
    }
    /* XXX just try PNG */
    retPath = item.dir + "/" + directory + "/" + base + ".png";

    struct stat file_stat;
    if (stat(retPath.c_str(), &file_stat) == 0 &&
	S_ISREG(file_stat.st_mode) &&
	((file_stat.st_mode & 0444) == 0444)) {
      iconInfo.bpp  = 0;
      iconInfo.size = size;
      iconInfo.formatType = rfbIconFormatTypePng;
      return true;
    }
  }
  retPath.clear();
  return false;
}

bool IconMgr::lookupIcon(string& iconname, unsigned long clientPtr, string& retPath, ClientMgr::IconPref& retIconInfo)
{
  const ClientMgr::IconPref* iconPref = clientMgr.getIconPref(clientPtr);

  if (!iconPref) {
    cerr << __FUNCTION__ << ": no IconPref specified." << endl;
    return false;
  }

  retIconInfo = *iconPref;
  for (ThemeList::iterator i = themeList.begin(); i != themeList.end(); i++) {
    if (lookupIcon(iconname, retPath, retIconInfo, *i))
      return true;
  }

  /* finally, check /usr/share/pixmaps */
  string base, extension;
  string::size_type pos = iconname.rfind('.');
  if (pos == string::npos) {
    base = iconname;
  } else {
    base = iconname.substr(0, pos);      
    extension = iconname.substr(pos + 1);
  }
  /* XXX just try PNG */
  retPath = "/usr/share/pixmaps/" + base + ".png";
  int fd = open(retPath.c_str(), O_RDONLY);
  if (fd != -1) {
    close(fd);
    retIconInfo.bpp  = 0;
    retIconInfo.size = 0;
    retIconInfo.formatType = rfbIconFormatTypePng;
    return true;
  }

  retPath.clear();
  return false;
}
