/*
 * libxend/xend.h -- Xend library
 *
 * Copyright (C) 2005,2006
 *
 *      Anthony Liguori <aliguori@us.ibm.com>
 *	Daniel Veillard <veillard@redhat.com>
 *
 *  This file is subject to the terms and conditions of the GNU Lesser General
 *  Public License. See the file COPYING in the main directory of this archive
 *  for more details.
 */

#ifndef _LIBXEND_XEND_H_
#define _LIBXEND_XEND_H_

#include <sys/types.h>
#include <stdint.h>
#include <stdbool.h>

#include "libvirt/libvirt.h"
#include "capabilities.h"
#include "buf.h"

#ifdef __cplusplus
extern "C" {
#endif

/**
 * \brief Setup the connection to a xend instance via TCP
 * \param host The host name to connect to
 * \param port The port number to connect to
 * \return 0 in case of success, -1 in case of error
 *
 * This method creates a new Xend instance via TCP.
 *
 * This function may not fail if Xend is not running.
 *
 * Make sure to call xenDaemonClose().
 */
int xenDaemonOpen_tcp(virConnectPtr xend, const char *host, int port);

/**
 * \brief Setup the connection to xend instance via a Unix domain socket
 * \param path The path to the domain socket
 * \return 0 in case of success, -1 in case of error
 *
 * This method creates a new xend instance via a Unix domain socket.
 *
 * This function may not fail if Xend is not running.
 *
 * Make sure to call xenDaemonClose().
 */
int xenDaemonOpen_unix(virConnectPtr xend, const char *path);


/**
 * \brief Blocks until a domain's devices are initialized
 * \param xend A xend instance
 * \param name The domain's name
 * \return 0 for success; -1 (with errno) on error
 *
 * xen_create() returns after a domain has been allocated including
 * its memory.  This does not guarentee, though, that the devices
 * have come up properly.  For instance, if you create a VBD with an
 * invalid filename, the error won't occur until after this function
 * returns.
 */
    int xend_wait_for_devices(virConnectPtr xend, const char *name);


/**
 * \brief Create a new domain
 * \param xend A xend instance
 * \param sexpr An S-Expr defining the domain
 * \return 0 for success; -1 (with errno) on error
 *
 * This method will create a domain based the passed in description.  The
 * domain will be paused after creation and must be unpaused with
 * xenDaemonResumeDomain() to begin execution.
 */
    int xenDaemonDomainCreateLinux(virConnectPtr xend, const char *sexpr);

/**
 * \brief Lookup the id of a domain
 * \param xend A xend instance
 * \param name The name of the domain
 * \param uuid pointer to store a copy of the uuid
 * \return the id number on success; -1 (with errno) on error
 *
 * This method looks up the ids of a domain
 */
int xenDaemonDomainLookupByName_ids(virConnectPtr xend,
                            const char *name, unsigned char *uuid);


/**
 * \brief Lookup the name of a domain
 * \param xend A xend instance
 * \param id The id of the domain
 * \param name pointer to store a copy of the name
 * \param uuid pointer to store a copy of the uuid
 *
 * This method looks up the name/uuid of a domain
 */
int xenDaemonDomainLookupByID(virConnectPtr xend,
                              int id,
                              char **name, unsigned char *uuid);


char *xenDaemonDomainDumpXMLByID(virConnectPtr xend,
                                 int domid,
                                 int flags,
                                 const char *cpus);

char *xenDaemonDomainDumpXMLByName(virConnectPtr xend,
                                   const char *name,
                                   int flags,
                                   const char *cpus);

/**
 * \brief Lookup information about the host machine
 * \param xend A xend instance
 * \return node info on success; NULL (with errno) on error
 *
 * This method returns information about the physical host
 * machine running Xen.
 */
    struct xend_node *xend_get_node(virConnectPtr xend);

/**
 * \brief Shutdown physical host machine
 * \param xend A xend instance
 * \return 0 on success; -1 (with errno) on error
 *
 * This method shuts down the physical machine running Xen.
 */
    int xend_node_shutdown(virConnectPtr xend);

/**
 * \brief Restarts physical host machine
 * \param xend A xend instance
 * \return 0 on success; -1 (with errno) on error
 *
 * This method restarts the physical machine running Xen.
 */
    int xend_node_restart(virConnectPtr xend);

/**
 * \brief Return hypervisor debugging messages
 * \param xend A xend instance
 * \param buffer A buffer to hold the messages
 * \param n_buffer Size of buffer (including null terminator)
 * \return 0 on success; -1 (with errno) on error
 *
 * This function will place the debugging messages from the
 * hypervisor into a buffer with a null terminator.
 */
    int xend_dmesg(virConnectPtr xend, char *buffer, size_t n_buffer);

/**
 * \brief Clear the hypervisor debugging messages
 * \param xend A xend instance
 * \return 0 on success; -1 (with errno) on error
 *
 * This function will clear the debugging message ring queue
 * in the hypervisor.
 */
    int xend_dmesg_clear(virConnectPtr xend);

/**
 * \brief Obtain the Xend log messages
 * \param xend A xend instance
 * \param buffer The buffer to hold the messages
 * \param n_buffer Size of buffer (including null terminator)
 * \return 0 on success; -1 (with errno) on error
 *
 * This function will place the Xend debugging messages into
 * a buffer with a null terminator.
 */
    int xend_log(virConnectPtr xend, char *buffer, size_t n_buffer);

    int xend_parse_sexp_desc_char(virConnectPtr conn,
                                  virBufferPtr buf,
                                  const char *devtype,
                                  int portNum,
                                  const char *value,
                                  const char *tty);

  char *xend_parse_domain_sexp(virConnectPtr conn,  char *root, int xendConfigVersion);

  int is_sound_model_valid(const char *model);
  int is_sound_model_conflict(const char *model, const char *soundstr);
  char *sound_string_to_xml(const char *sound);


/* refactored ones */
int xenDaemonOpen(virConnectPtr conn, xmlURIPtr uri, virConnectAuthPtr auth, int flags);
int xenDaemonClose(virConnectPtr conn);
int xenDaemonGetVersion(virConnectPtr conn, unsigned long *hvVer);
int xenDaemonNodeGetInfo(virConnectPtr conn, virNodeInfoPtr info);
int xenDaemonNodeGetTopology(virConnectPtr conn, virCapsPtr caps);
int xenDaemonDomainSuspend(virDomainPtr domain);
int xenDaemonDomainResume(virDomainPtr domain);
int xenDaemonDomainShutdown(virDomainPtr domain);
int xenDaemonDomainReboot(virDomainPtr domain, unsigned int flags);
int xenDaemonDomainDestroy(virDomainPtr domain);
int xenDaemonDomainSave(virDomainPtr domain, const char *filename);
int xenDaemonDomainRestore(virConnectPtr conn, const char *filename);
int xenDaemonDomainSetMemory(virDomainPtr domain, unsigned long memory);
int xenDaemonDomainSetMaxMemory(virDomainPtr domain, unsigned long memory);
int xenDaemonDomainGetInfo(virDomainPtr domain, virDomainInfoPtr info);
char *xenDaemonDomainDumpXML(virDomainPtr domain, int flags, const char *cpus);
unsigned long xenDaemonDomainGetMaxMemory(virDomainPtr domain);
char **xenDaemonListDomainsOld(virConnectPtr xend);

virDomainPtr xenDaemonDomainDefineXML(virConnectPtr xend, const char *sexpr);
int xenDaemonDomainCreate(virDomainPtr domain);
int xenDaemonDomainUndefine(virDomainPtr domain);

int	xenDaemonDomainSetVcpus		(virDomainPtr domain,
                                         unsigned int vcpus);
int	xenDaemonDomainPinVcpu		(virDomainPtr domain,
                                         unsigned int vcpu,
                                         unsigned char *cpumap,
                                         int maplen);
int	xenDaemonDomainGetVcpus		(virDomainPtr domain,
                                         virVcpuInfoPtr info,
                                         int maxinfo,
                                         unsigned char *cpumaps,
                                         int maplen);
int xenDaemonDomainGetAutostart          (virDomainPtr dom,
                                          int *autostart);
int xenDaemonDomainSetAutostart          (virDomainPtr domain,
                                          int autostart);

/* xen_unified calls through here. */
extern struct xenUnifiedDriver xenDaemonDriver;
int xenDaemonInit (void);

virDomainPtr xenDaemonLookupByID(virConnectPtr conn, int id);
virDomainPtr xenDaemonLookupByUUID(virConnectPtr conn, const unsigned char *uuid);
virDomainPtr xenDaemonLookupByName(virConnectPtr conn, const char *domname);
int xenDaemonDomainMigratePrepare (virConnectPtr dconn, char **cookie, int *cookielen, const char *uri_in, char **uri_out, unsigned long flags, const char *dname, unsigned long resource);
int xenDaemonDomainMigratePerform (virDomainPtr domain, const char *cookie, int cookielen, const char *uri, unsigned long flags, const char *dname, unsigned long resource);

int xenDaemonDomainBlockPeek (virDomainPtr domain, const char *path, unsigned long long offset, size_t size, void *buffer);

#ifdef __cplusplus
}
#endif
#endif
