/***************************************************************************
 *
 *   KYum - a KDE GUI for yum
 *
 *   Copyright (C) 2005 by Steffen Offermann
 *   steffen_ac@yahoo.com
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <qsplitter.h>
#include <qtable.h>
#include <qlayout.h>
#include <qscrollview.h>

#include <klocale.h>
#include <qtextbrowser.h>

#include "CfgFile.h"
#include "OptionsWidget.h"
#include "OptionsView.h"

static const char * c_cachedir            = "cachedir",
                  * c_reposdir            = "reposdir",
                  * c_debuglevel          = "debuglevel",
                  * c_errorlevel          = "errorlevel",
                  * c_logfile             = "logfile",
                  * c_gpgcheck            = "gpgcheck",
                  * c_assumeyes           = "assumeyes",
                  * c_tolerant            = "tolerant",
                  * c_exclude             = "exclude",
                  * c_exactarch           = "exactarch",
                  * c_installonlypkgs     = "installonlypkgs",
                  * c_kernelpkgnames      = "kernelpkgnames",
                  * c_showdupesfromrepos  = "showdupesfromrepos",
                  * c_obsoletes           = "obsoletes",
                  * c_overwrite_groups    = "overwrite_groups",
                  * c_installroot         = "installroot",
                  * c_rss_filename        = "rss-filename",
                  * c_distroverpkg        = "distroverpkg",
                  * c_diskspacecheck      = "diskspacecheck",
                  * c_tsflags             = "tsflags",
                  * c_recent              = "recent",
                  * c_retries             = "retries",
                  * c_keepalive           = "keepalive",
                  * c_throttle            = "throttle",
                  * c_bandwidth           = "bandwidth",
                  * c_commands            = "commands",
                  * c_pkgpolicy           = "pkgpolicy",
                  * c_proxy               = "proxy",
                  * c_proxy_username      = "proxy_username",
                  * c_proxy_password      = "proxy_password";


static struct OptionInfo
  {
    const char  * m_pKey,
                * m_pDefault,
                * m_pHelp;
  }
  g_optionTab [] =
    {
      { c_cachedir,
        "/var/cache/yum",
        "Directory where yum should store its cache and db files. "
        "The default is ‘/var/cache/yum’." },

      { c_reposdir,
        "/etc/yum.repos.d",
        "Directory where yum should look for .repo files for its "
        "configuration of repositories. Default is ‘/etc/yum.repos.d’." },

      { c_debuglevel,
        "2",
        "Debug message output level. Practical range is 0-10. Default "
        "is ‘2’." },

      { c_errorlevel,
        "2",
        "Error message output level. Practical range is 0-10. Default "
        "is ‘2’." },

      { c_logfile,
        "/var/log/yum.log",
        "Full directory and file name for where yum should write its "
        "log file." },

      { c_gpgcheck,
        "1",
        "Either ‘1’ or ‘0’. This tells yum whether or not it should "
        "perform a GPG signature check on packages. The value entered "
        "here will be used as the default for all repositories. This "
        "option also determines whether or not an install of a package "
        "from a local RPM file will be GPG signature checked. "
        "The default is ‘0’." },

      { c_assumeyes,
        "0",
        "Either ‘1’ or ‘0’. Determines whether or not yum prompts for "
        "confirmation of critical actions. Default is ‘0’ (do prompt). "
        "Commmand-line option: -y. "
        "This option is useless when using kyum, since kyum always "
        "uses -y explicitly, but you may want to configure it here if "
        "you want to start yum from the command line, too." },

      { c_tolerant,
        "1",
        "Either ‘1’ or ‘0’. If enabled, then yum will be tolerant of "
        "errors on the command line with regard to packages. For "
        "example: if you request to install foo, bar and baz and baz "
        "is installed; yum won’t error out complaining that baz is  "
        "already installed. Default to ‘0’ (not tolerant). "
        "Commmand-line option: -t." },

      { c_exclude,
        "",
        "List of packages to exclude from updates or installs. This "
        "should be a space separated list. Shell globs using wildcards "
        "(eg. * and ?) are allowed." },

      { c_exactarch,
        "1",
        "Either ‘1’ or ‘0’. Set to ‘1’ to make yum update only update "
        "the architectures of packages that you have installed. ie: with "
        "this enabled yum will not install an i686 package to update an "
        "i386 package. Default is ‘1’." },

      { c_installonlypkgs,
#if 0
        "kernel, kernel-smp, kernel-bigmem, kernel-enterprise, "
        "kernel-debug, kernel-unsupported",
#else
        "",
#endif
        "List of packages that should only ever be installed, never "
        "updated. Kernels in particular fall into this category. Defaults "
        "to ‘kernel, kernel-smp,  kernel-bigmem,  kernel-enterprise, "
        "kernel-debug, kernel-unsupported’." },

      { c_kernelpkgnames,
        "",
        "List of package names that are kernels. This is really only "
        "here for the updating of kernel packages and should be removed "
        "out in the yum 2.1 series." },

      { c_showdupesfromrepos,
        "0",
        "Either ‘0’ or ‘1’. Set to ‘1’ if you wish to show any duplicate "
        "packages from any repository. Set to ‘0’ if you want only to "
        "see the newest packages from any repository. Default is ‘0’. "
        "Currently this has no effect for kyum, since kyum itself "
        "sorts out older packages." },

      { c_obsoletes,
        "0",
        "This option only has effect during an update. It enables yum’s "
        "obsoletes processing logic. Useful when doing distribution "
        "level upgrades. "
        "Commmand-line option: --obsoletes." },

      { c_overwrite_groups,
        "0",
        "Either ‘0’ or ‘1’. Used to determine yum’s behaviour if two or "
        "more repositories offer the package groups with the same name. "
        "If overwrite_groups is ‘1’ then the group packages of the last "
        "matching repository will be used. If overwrite_groups is ‘0’ "
        "then the groups from all matching repositories will be merged "
        "together as one large group." },

      { c_installroot,
        "",
        "Specifies an alternative installroot, relative to which all "
        "packages will be installed. "
        "Commmand-line option: --installroot." },

      { c_rss_filename,
        "",
        "Sets the RSS output file when the generate-rss command is used. "
        "Commmand-line option: --rss-filename." },

      { c_distroverpkg,
        "redhat-release",
        "The package used by yum to determine the ‘version’ of the "
        "distribution. This can be any installed package. Default is "
        "‘redhat-release’." },

      { c_diskspacecheck,
        "1",
        "Either  ‘0’ or ‘1’. Set this to ‘0’ to disable the checking for "
        "sufficient diskspace before a RPM transaction is run. Default "
        "is ‘1’ (perform the check)." },

      { c_tsflags,
        "",
        "Comma or space separated list of transaction flags to pass "
        "to the rpm transaction set. These include ’noscripts’,  "
        "’notriggers’, ’nodocs’, ’test’. You can set all/any of them. "
        "However, if you don’t know what these do in the context of an "
        "rpm transaction set you’re best leaving it alone. Default is "
        "an empty list." },

      { c_recent,
        "7",
        "Number of days back to look for ‘recent’ packages added to a "
        "repository. Used by the list recent and generate-rss commands. "
        "Default is ‘7’." },

      { c_retries,
        "6",
        "Set the number of times any attempt to retrieve a file should "
        "retry before returning an error. Setting this to ‘0’ makes yum "
        "try forever. Default is ‘6’." },

      { c_keepalive,
        "1",
        "Either ‘0’ or ‘1’. Set whether HTTP keepalive should be used "
        "for HTTP/1.1 servers that support it. This can improve transfer "
        "speeds by using one connection when downloading multiple files "
        "from a repository. Default is ‘1’." },

      { c_throttle,
        "0",
        "Enable bandwidth throttling for downloads. This option can be "
        "expressed as a absolute data rate in bytes/sec.  An SI prefix "
        "(k, M or G) may be appended to the bandwidth value (eg. ‘5.5k’ "
        "is 5.5 kilobytes/sec, ‘2M’ is 2 Megabytes/sec)."
        "</P><P>"
        "Alternatively, this option can specify the percentage of total "
        "bandwidth to use (eg. ‘60%’). In this case the bandwidth option "
        "should be used to specify the maximum available bandwidth. "
        "</P><P>"
        "Set to ‘0’ to disable bandwidth throttling. This is the default." },

      { c_bandwidth,
        "0",
        "Use to specify the maximum available network bandwidth in "
        "bytes/second. Used with the throttle option (above). If throttle "
        "is a percentage and bandwidth is ‘0’ then bandwidth throttling "
        "will be disabled. If throttle is expressed as a data rate "
        "(bytes/sec) then this option is ignored. Default is ‘0’ (no "
        "bandwidth throttling)." },

      { c_commands,
        "",
        "List of functional commands to run if no functional commands "
        "are specified on the command  line (eg. ‘update foo bar baz "
        "quux‘). None of the short options (eg. -y,  -e,  -d) are "
        "accepted for this option." },

      { c_pkgpolicy, "", "" },

      { c_proxy, "", "URL to the proxy server that yum should use." },

      { c_proxy_username, "", "Username to use for proxy." },

      { c_proxy_password, "", "Password for this proxy" },

      { 0, 0, 0 }
    };


class OptionMap
  {
    private:
      typedef std::map<const char *, const OptionInfo *> Map;

    private:
      Map   m_map;

    public:
                        OptionMap   (const OptionInfo *);

     const OptionInfo * operator[]  (const char * pKey) const;

     const char       * getDefault  (const char * pKey) const;
     const char       * getHelp     (const char * pKey) const;
  };


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

OptionMap::OptionMap(const OptionInfo * infoTab)
{
    for ( const OptionInfo * pInfo = infoTab;
          pInfo && pInfo->m_pKey;
          pInfo++ )
    {
        m_map[pInfo->m_pKey] = pInfo;
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

const OptionInfo * OptionMap::operator [] (const char * pKey) const
{
    Map::const_iterator it = m_map.find(pKey);
    return (m_map.end() == it ?  0 : it->second);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

const char * OptionMap::getDefault(const char * pKey) const
{
    const OptionInfo * pInfo = (*this)[pKey];
    return pInfo ? pInfo->m_pDefault : "";
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

const char * OptionMap::getHelp(const char * pKey) const
{
    const OptionInfo * pInfo = (*this)[pKey];
    return pInfo ? pInfo->m_pHelp : "";
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

OptionsView::OptionsView(QWidget * pParent, const char * pName)

  : QWidget(pParent, pName)

{
    QHBoxLayout * pTopLayout = new QHBoxLayout(this, 0, 6, "TopLayout");
    pTopLayout->setAutoAdd(true);

    //
    // Splitter between options and help window...
    //
    QSplitter * pVSplitter = new QSplitter(this);
    pVSplitter->setOrientation(QSplitter::Vertical);


    OptionMap     optionMap(g_optionTab);
    Option::List  options;

    options.push_back(new DirOption (c_reposdir,
                                     optionMap.getDefault(c_reposdir),
                                     optionMap.getHelp(c_reposdir)));

    options.push_back(new DirOption (c_cachedir,
                                     optionMap.getDefault(c_cachedir),
                                     optionMap.getHelp(c_cachedir)));

    options.push_back(new FileOption(c_logfile,
                                     optionMap.getDefault(c_logfile),
                                     optionMap.getHelp(c_logfile)));

    options.push_back(new IntOption (c_debuglevel,
                                     optionMap.getDefault(c_debuglevel),
                                     "99",
                                     optionMap.getHelp(c_debuglevel)));

    options.push_back(new IntOption (c_errorlevel,
                                     optionMap.getDefault(c_errorlevel),
                                     "99",
                                     optionMap.getHelp(c_errorlevel)));

    options.push_back(new IntOption (c_retries,
                                     optionMap.getDefault(c_retries),
                                     "900",
                                     optionMap.getHelp(c_retries)));

    options.push_back(new IntOption (c_recent,
                                     optionMap.getDefault(0),
                                     "900",
                                     optionMap.getHelp(c_recent)));

    options.push_back(new BoolOption(c_diskspacecheck,
                                     optionMap.getDefault(c_diskspacecheck),
                                     optionMap.getHelp(c_diskspacecheck)));

    options.push_back(new BoolOption(c_tolerant,
                                     optionMap.getDefault(c_tolerant),
                                     optionMap.getHelp(c_tolerant)));

    options.push_back(new BoolOption(c_exactarch,
                                     optionMap.getDefault(c_exactarch),
                                     optionMap.getHelp(c_exactarch)));

    options.push_back(new BoolOption(c_obsoletes,
                                     optionMap.getDefault(c_obsoletes),
                                     optionMap.getHelp(c_obsoletes)));

    options.push_back(new BoolOption(c_gpgcheck,
                                     optionMap.getDefault(c_gpgcheck),
                                     optionMap.getHelp(c_gpgcheck)));

    options.push_back(new BoolOption(c_keepalive,
                                     optionMap.getDefault(c_keepalive),
                                     optionMap.getHelp(c_keepalive)));

    options.push_back(new BoolOption(c_overwrite_groups,
                                     optionMap.getDefault(c_overwrite_groups),
                                     optionMap.getHelp(c_overwrite_groups)));

    options.push_back(new StringOption(c_throttle,
                                       optionMap.getDefault(c_throttle),
                                       "",
                                       optionMap.getHelp(c_throttle)));

    options.push_back(new IntOption (c_bandwidth,
                                     optionMap.getDefault(c_bandwidth),
                                     "",
                                     optionMap.getHelp(c_bandwidth)));

    options.push_back(new StringOption(c_pkgpolicy,
                                       optionMap.getDefault(c_pkgpolicy),
                                       "",
                                       optionMap.getHelp(c_pkgpolicy)));

    options.push_back(new StringOption(c_distroverpkg,
                                       optionMap.getDefault(c_distroverpkg),
                                       "",
                                       optionMap.getHelp(c_distroverpkg)));

    options.push_back(new StringOption(c_proxy,
                                       optionMap.getDefault(c_proxy),
                                       "",
                                       optionMap.getHelp(c_proxy)));

    options.push_back(new StringOption(c_proxy_username,
                                       optionMap.getDefault(c_proxy_username),
                                       "",
                                       optionMap.getHelp(c_proxy_username)));

    options.push_back(new StringOption(c_proxy_password,
                                       optionMap.getDefault(c_proxy_password),
                                       "",
                                       optionMap.getHelp(c_proxy_password)));

    options.push_back(new StringOption(c_installonlypkgs,
                                       optionMap.getDefault(c_installonlypkgs),
                                       "",
                                       optionMap.getHelp(c_installonlypkgs)));

    options.push_back(new StringOption(c_exclude,
                                       optionMap.getDefault(c_exclude),
                                       "",
                                       optionMap.getHelp(c_exclude)));

    options.push_back(new StringOption(c_installroot,
                                       optionMap.getDefault(c_installroot),
                                       "",
                                       optionMap.getHelp(c_installroot)));


    QScrollView * pScrollView = new QScrollView(pVSplitter);

    m_pOptionsWidget = new OptionsWidget(options, pScrollView->viewport());
    m_pHelpView      = new QTextBrowser(pVSplitter);

    QVBoxLayout * pHelpViewLayout = new QVBoxLayout(m_pHelpView);
    (void) pHelpViewLayout;

    QFont helpFont(m_pHelpView->font());
//    helpFont.setFamily("System");
//    helpFont.setBold(true);
    helpFont.setPointSize(helpFont.pointSize() + 2);
    m_pHelpView->setFont(helpFont);

    m_pHelpView->setMinimumSize(400, 80);
    m_pOptionsWidget->setMinimumSize(600, m_pOptionsWidget->height());
    m_pOptionsWidget->resize(m_pHelpView->width(), m_pOptionsWidget->height());

    pScrollView->addChild(m_pOptionsWidget);

    connect(m_pOptionsWidget, SIGNAL(optionChanged(const QString &, const QString &)),
            this,             SLOT(slotOptionChanged(const QString &, const QString &)));

    connect(m_pOptionsWidget, SIGNAL(optionHelp(const QString &)),
            this,             SLOT(slotOptionHelp(const QString &)));
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

OptionsView::~OptionsView()
{
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void OptionsView::setOptions(const CfgGroup * pGroup)
{
    m_pOptionsWidget->setOptions(pGroup);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void OptionsView::setDefaults()
{
    m_pOptionsWidget->setDefaultOptions();
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void OptionsView::slotOptionChanged(const QString & strKey, const QString & strValue)
{
    emit optionChanged(strKey, strValue);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void OptionsView::slotOptionHelp(const QString & strHelp)
{
    QString strText;

    strText  = "<P>";
    strText += QString::fromUtf8(strHelp);
    strText += "</P>";

    m_pHelpView->setText(strText);
}

#include "OptionsView.moc"
