/***************************************************************************
 *
 *   KYum - a KDE GUI for yum
 *
 *   Copyright (C) 2005 by Steffen Offermann
 *   steffen_ac@yahoo.com
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <qlayout.h>
#include <qpushbutton.h>
#include <qlistview.h>
#include <qmutex.h>
#include <qcheckbox.h>
#include <qmessagebox.h>

#include <klocale.h>

#include "Process.h"
#include "DlgStopAction.h"
#include "GroupsView.h"

enum
  {
    c_idxName,
    c_idxDescription
  };

static struct
  {
    const char  * m_pTitle;
    int           m_cx;
  }
  g_columns[] =
    {
      { "Name",        -1 }/*,
      { "Description", -1 }*/   // Descriptions are not yet supported by yum
    };


#define ArraySize(a)  (sizeof(a) / sizeof((a)[0]))


/***************************************************************************/
/**
 * Adds an item to a section.
 *
 ***************************************************************************/

void GroupsView::Section::addItem(QString strItem)
{
    const Map::iterator it = m_itemMap.find(strItem);

    if ( m_itemMap.end() == it )
    {
        m_itemList.push_back(strItem);
        m_itemMap[strItem] = true;
    }
}


/***************************************************************************/
/**
 * Returns the pointer to the specified section.
 *
 ***************************************************************************/

GroupsView::GroupInfo::SectionPtr
  GroupsView::GroupInfo::getSection(QString strSection) const
{
    SectionMap::const_iterator  it = m_sectionMap.find(strSection);
    SectionPtr                  ptrSection;

    if ( it != m_sectionMap.end() )
        ptrSection = it->second;

    return ptrSection;
}


/***************************************************************************/
/**
 * Adds an entry (i.e. a package or a required group) to a section. If the
 * specified section does not yet exist, it will be created.
 *
 ***************************************************************************/

void GroupsView::GroupInfo::addItemToSection(QString strSection, QString strName)
{
    SectionPtr ptrSection = getSection(strSection);

    if ( !ptrSection.isValid() )
    {
        m_sectionList.push_back(ptrSection = new Section(strSection));
        m_sectionMap[strSection]           = ptrSection;
    }

    if ( ptrSection.isValid() )
        ptrSection->addItem(strName);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

GroupsView::GroupsView(QWidget * pParent)

  : QWidget           (pParent),
    m_pProcess        (0),
    m_pMutex          (new QMutex),
    m_currentAction   (c_NoAction),
    m_currentList     (c_NoList),
    m_currentStep     (c_Step_1),
    m_currentSection  (c_NoSection),
    m_numCheckedItems (0)

{
    QHBoxLayout * pTopLayout    = new QHBoxLayout(this, 0, 6, "TopLayout");
    QWidget     * pVButtonPanel = new QWidget(this, "VButtonPanel");
    QVBoxLayout * pVLayout      = new QVBoxLayout(pVButtonPanel, 6);

    m_pButtonListAvail = new QPushButton(pVButtonPanel, "buttonListAvail");
    m_pButtonListAvail->setText(i18n("List &Available"));

    m_pButtonListInstalled = new QPushButton(pVButtonPanel, "buttonListInstalled");
    m_pButtonListInstalled->setText(i18n("&List Installed"));

    m_pButtonRemove = new QPushButton(pVButtonPanel, "buttonRemove");
    m_pButtonRemove->setText(i18n("&Remove Selected"));
    m_pButtonRemove->setEnabled(false); // No list yet

    m_pButtonInstall = new QPushButton(pVButtonPanel, "buttonInstall");
    m_pButtonInstall->setText(i18n("&Install Selected"));

    m_pCheckCache = new QCheckBox(pVButtonPanel, "checkCache");
    m_pCheckCache->setText(i18n("Use &Cache"));
    //
    // Left button panel...
    //
    pVLayout->addWidget(m_pButtonListAvail);
    pVLayout->addWidget(m_pButtonListInstalled);

    QSpacerItem * pVSpacer = new QSpacerItem(60, 20, QSizePolicy::Minimum, QSizePolicy::Expanding);
    pVLayout->addItem(pVSpacer);

    pVLayout->addWidget(m_pButtonRemove);
    pVLayout->addWidget(m_pButtonInstall);
    pVLayout->addWidget(m_pCheckCache);

    //
    // List view...
    //
    m_pListView = new QListView(this, "ListView");

    for ( int idx=0; idx < (int) ArraySize(g_columns); idx++ )
        m_pListView->addColumn(g_columns[idx].m_pTitle, g_columns[idx].m_cx);

    m_pListView->setRootIsDecorated(true);

    //
    // Add all to top layout...
    //
    pTopLayout->addWidget(pVButtonPanel);
    pTopLayout->addWidget(m_pListView);


    enableActions(true);

    connect(m_pButtonListAvail,     SIGNAL(clicked()), this, SLOT(slotListAvailable()));
    connect(m_pButtonListInstalled, SIGNAL(clicked()), this, SLOT(slotListInstalled()));

    connect(m_pButtonRemove,  SIGNAL(clicked()), this, SLOT(slotRemoveGroup()));
    connect(m_pButtonInstall, SIGNAL(clicked()), this, SLOT(slotInstallGroup()));

    connect(m_pListView,  SIGNAL(selectionChanged(QListViewItem *)),
            this,         SLOT(slotListItemSelected(QListViewItem *)));

    connect(m_pListView,  SIGNAL(clicked(QListViewItem *)),
            this,         SLOT(slotListItemSelected(QListViewItem *)));
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

GroupsView::~GroupsView()
{
    delete m_pProcess;
    delete m_pMutex;
}


/***************************************************************************/
/**
 * Enables or disables action buttons...
 *
 ***************************************************************************/

void GroupsView::enableActions(bool bEnable)
{
    m_pButtonListAvail->setEnabled(bEnable);
    m_pButtonListInstalled->setEnabled(bEnable);

    m_pCheckCache->setEnabled(bEnable);

    if ( bEnable && 0 < m_numCheckedItems )
    {
        m_pButtonRemove->setEnabled(m_currentList == c_Installed);
        m_pButtonInstall->setEnabled(m_currentList != c_Installed);
    }
    else
    {
        m_pButtonRemove->setEnabled(false);
        m_pButtonInstall->setEnabled(false);
    }

    m_bActionsEnabled = bEnable;
}


/***************************************************************************/
/**
 * Gets group information for a specific group from our map.
 *
 ***************************************************************************/

GroupsView::GroupInfo::Ptr GroupsView::findGroupInfo(QString strGroup) const
{
    QString                       strKey = strGroup.lower();
    GroupInfo::Ptr                ptrGroupInfo;
    GroupInfoMap::const_iterator  it = m_groupInfoMap.find(strKey);

    if ( it != m_groupInfoMap.end() )
        ptrGroupInfo = it->second;

    return ptrGroupInfo;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void GroupsView::addGroup(QString strGroup)
{
    GroupInfo::Ptr  ptrGroupInfo = findGroupInfo(strGroup);

    if ( ptrGroupInfo.isNull() )
    {
        ptrGroupInfo = new GroupInfo(strGroup);
        m_groupInfoMap[strGroup.lower()] = ptrGroupInfo;

        /*QCheckListItem * pNewItem = */new QCheckListItem(m_pListView,
                                                       strGroup,
                                                       QCheckListItem::CheckBox);
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void GroupsView::addSubItems()
{
    QListViewItemIterator it(m_pListView);

    //
    // Step through the list of groups...
    //
    while ( it.current() )
    {
        QCheckListItem *  pGroupItem = dynamic_cast<QCheckListItem *>(it.current());
        GroupInfo::Ptr    ptrGroupInfo;

         if ( pGroupItem )
            ptrGroupInfo = getGroupInfo(pGroupItem);

        //
        // Get information for this group...
        //
        if ( ptrGroupInfo.isValid() )
        {
            Section::List::const_iterator itSections;
            const Section::List &         sections = ptrGroupInfo->m_sectionList;

            //
            // Run through all sections, e.g. "Required Groups", "Mandatory Packages", etc...
            //
            for ( itSections = sections.begin(); itSections != sections.end(); itSections++ )
            {
                QString                     strSection = (*itSections)->m_strName;
                const QStringList &         names      = (*itSections)->m_itemList;
                QStringList::ConstIterator  itNames;
                QListViewItem *             pSubItem;

/*
                if ( !strSection.startsWith("Required") && !strSection.startsWith("Mandatory") )
                {
                    pSubItem = new QCheckListItem(pGroupItem,
                                                  strSection,
                                                  QCheckListItem::CheckBox);
                }
                else
*/
                {
                    pSubItem = new QListViewItem(pGroupItem, strSection);
                }

                //
                // List packages (or required groups) for the current section...
                //
                for ( itNames = names.begin(); itNames != names.end(); itNames++)
                {
                    new QListViewItem(pSubItem, *itNames);
                }
            }
        }

        ++it;
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void GroupsView::slotListInstalled()
{
    delete m_pProcess;
    m_pProcess = new Process(this);

    m_pProcess->addArgument("yum");

    if ( m_pCheckCache->isChecked() )
        m_pProcess->addArgument("-C");

    m_pProcess->addArgument("grouplist");

    m_currentAction  = c_ListInstalled;
    m_currentStep    = c_Step_1;
    m_currentSection = c_NoSection;
    m_currentList    = c_Installed;

    m_pProcess->addArgument("installed");

    m_pListView->clear();
    m_groupInfoMap.clear();

    m_numCheckedItems = 0;

    if ( !startAction() )
    {
        m_currentAction = c_NoAction;

        delete m_pProcess;
        m_pProcess = 0;

        QMessageBox::warning(this, "KYum", i18n("Could not start yum."));
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void GroupsView::slotListAvailable()
{
    delete m_pProcess;
    m_pProcess = new Process(this);

    m_pProcess->addArgument("yum");

    if ( m_pCheckCache->isChecked() )
        m_pProcess->addArgument("-C");

    m_pProcess->addArgument("grouplist");

    m_currentAction  = c_ListAvailable;
    m_currentStep    = c_Step_1;
    m_currentSection = c_NoSection;
    m_currentList    = c_Available;

    m_pProcess->addArgument("available");

    m_pListView->clear();
    m_groupInfoMap.clear();

    m_numCheckedItems = 0;

    if ( !startAction() )
    {
        m_currentAction = c_NoAction;

        delete m_pProcess;
        m_pProcess = 0;

        QMessageBox::warning(this, "KYum", i18n("Could not start yum."));
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void GroupsView::slotInstallGroup()
{
    delete m_pProcess;
    m_pProcess = new Process(this);

    m_pProcess->addArgument("yum");
    m_pProcess->addArgument("-y");

    if ( m_pCheckCache->isChecked() )
        m_pProcess->addArgument("-C");

    if ( c_Installed == m_currentList )
        m_pProcess->addArgument("groupupdate");
    else
        m_pProcess->addArgument("groupinstall");

    QListViewItemIterator it(m_pListView);

    while ( it.current() )
    {
        QCheckListItem * pItem = dynamic_cast<QCheckListItem *>(it.current());

        if ( pItem && pItem->isOn() )
            m_pProcess->addArgument(getGroupName(pItem));

        ++it;
    }

    m_currentAction  = c_Install;
    m_currentStep    = c_Step_1;

    if ( !startAction() )
    {
        m_currentAction = c_NoAction;

        delete m_pProcess;
        m_pProcess = 0;

        QMessageBox::warning(this, "KYum", i18n("Could not start yum."));
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void GroupsView::slotRemoveGroup()
{
    QString strMsg;

    strMsg.sprintf(i18n("About to remove %d installed group(s). Are you REALLY sure?"),
                   m_numCheckedItems);

    if ( 0 == QMessageBox::question(this, "KYum", strMsg, i18n("Yes"), i18n("No")) )
    {
        delete m_pProcess;
        m_pProcess = new Process(this);

        m_pProcess->addArgument("yum");
        m_pProcess->addArgument("-y");

        if ( m_pCheckCache->isChecked() )
            m_pProcess->addArgument("-C");

        m_pProcess->addArgument("groupremove");

        QListViewItemIterator it(m_pListView);

        while ( it.current() )
        {
            QCheckListItem * pItem = dynamic_cast<QCheckListItem *>(it.current());

            if ( pItem && pItem->isOn() )
                m_pProcess->addArgument(getGroupName(pItem));

            ++it;
        }

        m_currentAction = c_Remove;
        m_currentStep   = c_Step_1;

        if ( !startAction() )
        {
            m_currentAction = c_NoAction;

            delete m_pProcess;
            m_pProcess = 0;

            QMessageBox::warning(this, "KYum", i18n("Could not start yum."));
        }
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

bool GroupsView::startAction()
{
    bool bSuccess = false;
    //QMutexLocker locker(m_pMutex);

    if ( !m_pProcess->isRunning() )
    {
        // This is now done in KYum::slotActionStarted()
        //enableActions(false);

        connect(m_pProcess, SIGNAL(launchFinished()),  this, SLOT(slotLaunchFinished()));
        connect(m_pProcess, SIGNAL(processExited()),   this, SLOT(slotProcessExited()));
        connect(m_pProcess, SIGNAL(readyReadStdout()), this, SLOT(slotReadyStdout()));
        connect(m_pProcess, SIGNAL(readyReadStderr()), this, SLOT(slotReadyStderr()));

        if ( m_pProcess->start() )
        {
            //
            // Don't emit actionStarted() for the second step to prevent
            // the output window from being cleared().
            //
            if ( c_Step_1 == m_currentStep )
                emit actionStarted(m_pProcess);
            else
            {
                QString strMsg;

                strMsg.sprintf(i18n("Started process %d for step 2."),
                               m_pProcess->processIdentifier());

                emit yumInfo(strMsg);
            }

            bSuccess = true;
        }
    }

    return bSuccess;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void GroupsView::stopAction()
{
    if ( c_NoAction != m_currentAction )
    {
        DlgStopAction dlg(this);
        QProcess::PID pid = m_pProcess->processIdentifier();

        if ( DlgStopAction::Accepted == dlg.exec() )
        {
            QMutexLocker locker(m_pMutex);

            //
            // The process may have terminated already
            //
            if ( m_pProcess && m_pProcess->processIdentifier() == pid )
            {
                if ( dlg.graceful() )
                {
                    emit yumError(i18n("Sending terminate signal to process..."));
                    m_pProcess->tryTerminate();
                }
                else
                {
                    emit yumError(i18n("Killing process..."));
                    m_pProcess->kill();
                }
            }
        }
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void GroupsView::startStep2()
{
    if ( 0 < m_pListView->childCount() )
    {
        delete m_pProcess;
        m_pProcess = new Process(this);

        m_pProcess->addArgument("yum");

        if ( m_pCheckCache->isChecked() )
            m_pProcess->addArgument("-C");

        m_pProcess->addArgument("groupinfo");

        QListViewItemIterator it(m_pListView);

        while ( it.current() )
        {
            QCheckListItem *  pItem = dynamic_cast<QCheckListItem *>(it.current());

            m_pProcess->addArgument(getGroupName(pItem));
            ++it;
        }

        m_currentAction  = c_GetInfo;
        m_currentSection = c_NoSection;
        m_currentStep    = c_Step_2;

        m_strCurrentInfoSection = "";
        m_ptrCurrentGroup       = 0;

        if ( !startAction() )
        {
            m_currentAction = c_NoAction;

            delete m_pProcess;
            m_pProcess = 0;

            QMessageBox::warning(this, "KYum", i18n("Could not start yum."));
        }
    }
    else
        m_currentList = c_NoList;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void GroupsView::slotLaunchFinished()
{
    //QMutexLocker locker(m_pMutex);
    emit actionStarted(m_pProcess);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void GroupsView::slotProcessExited()
{
    bool bNeedsStep2 = c_Step_1 == m_currentStep
                         && 0 < m_pListView->childCount()
                         && (c_ListInstalled == m_currentAction ||
                             c_ListAvailable == m_currentAction),

         bStatusOK   = m_pProcess->normalExit() && 0 == m_pProcess->exitStatus();


    if ( bNeedsStep2 && bStatusOK )
    {
        emit yumInfo(i18n("Now getting group details..."));
        startStep2();
    }
    else
    {
        if ( c_Step_2 == m_currentStep /* && bStatusOK */)
            addSubItems();

        emit actionTerminated(m_pProcess);

        m_currentAction = c_NoAction;

        m_pMutex->lock();

        delete m_pProcess;
        m_pProcess = 0;

        m_pMutex->unlock();

        // This is now done in KYum::slotActionTerminated()
        //enableActions(true);
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void GroupsView::slotReadyStdout()
{
    //QMutexLocker locker(m_pMutex);

    while ( m_pProcess->canReadLineStdout() )
    {
        QString strLine = m_pProcess->readLineStdout();

        //
        // Step 2: Get group information...
        //
        if ( c_Step_2 == m_currentStep )
        {
            if ( strLine.startsWith("Group: ") )
            {
                m_ptrCurrentGroup = findGroupInfo(strLine.mid(7).stripWhiteSpace());
            }
            else
            {
                if ( strLine.startsWith(" ") && strLine[1] != ' ' )
                {
                    m_strCurrentInfoSection = strLine.stripWhiteSpace();

                    if ( m_strCurrentInfoSection.endsWith(":") )
                        m_strCurrentInfoSection.truncate(m_strCurrentInfoSection.length() - 1);
                }
                else if ( !strLine.startsWith("  ") )
                {
                    emit yumInfo(strLine);
                    m_strCurrentInfoSection = "";
                }
                else if ( m_ptrCurrentGroup.isValid() )
                {
                    m_ptrCurrentGroup->addItemToSection(m_strCurrentInfoSection,
                                                        strLine.stripWhiteSpace());
                }
            }
        }
        //
        // Step 1: Gather group names...
        //
        else if ( c_ListAvailable == m_currentAction || c_ListInstalled == m_currentAction )
        {
            if ( strLine.startsWith("Installed Groups:") )
            {
                m_currentSection = c_SectionInstalled;
            }
            else if ( strLine.startsWith("Available Groups:") )
            {
                m_currentSection = c_SectionAvailable;
            }
            else if ( strLine.startsWith("  ") || strLine.startsWith("\t") )
            {
                strLine = strLine.stripWhiteSpace();

                if ( !strLine.isEmpty() )
                {
                    if ( (c_SectionInstalled == m_currentSection
                            && c_Installed == m_currentList)
                            ||
                          (c_SectionAvailable == m_currentSection
                            && c_Available == m_currentList) )
                    {
                        addGroup(strLine);
                    }
                }
            }
            else
                emit yumInfo(strLine);
        }
        else
            emit yumInfo(strLine);
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void GroupsView::slotReadyStderr()
{
    //QMutexLocker locker(m_pMutex);

    while ( m_pProcess->canReadLineStderr() )
    {
        QString strLine = m_pProcess->readLineStderr();
        emit yumError(strLine);
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

QString GroupsView::getGroupName(const QListViewItem * pItem) const
{
    return pItem->text(0);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void GroupsView::slotListItemSelected(QListViewItem * pItem)
{
    //
    // We are only interested in first-level items...
    //
    if ( pItem && 0 == pItem->parent() )
    {
        GroupInfo::Ptr    ptrGroupInfo  = getGroupInfo(pItem);
        QCheckListItem *  pCheckItem    = dynamic_cast<QCheckListItem *>(pItem);

        if ( pCheckItem && ptrGroupInfo.isValid() ) // this should NEVER fail!
        {
            if ( pCheckItem->isOn() )
            {
                if ( !ptrGroupInfo->m_bChecked )
                {
                    m_numCheckedItems++;
                    ptrGroupInfo->m_bChecked = true;
                }
            }
            else
            {
                if ( ptrGroupInfo->m_bChecked )
                {
                    m_numCheckedItems--;
                    ptrGroupInfo->m_bChecked = false;
                }
            }
        }

        //
        // Enable/disable buttons appropriately
        //
        if ( m_bActionsEnabled )
            enableActions(true);
    }
}


#include "GroupsView.moc"
