/***************************************************************************
 *   Copyright (C) 2004 Nadeem Hasan <nhasan@kde.org>                      *
 *   Copyright (C) 2004 Stefan Kombrink <katakombi@web.de>                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "config.h"

#include <sys/types.h>

# include <sys/time.h>
# include <time.h>

#include <sys/shm.h>

#include <kdebug.h>
#include <kstaticdeleter.h>
#include <synparam.h>

#include "synconfig.h"
#include "touchpad.h"

using namespace Synaptics;

TouchPad *TouchPad::m_self = 0;
static Pad* sm_Pad = 0;

static KStaticDeleter<TouchPad> staticTouchPadDeleter;

static int finger_low[] = { 53, 38, 25, 18, 11 };

TouchPad *TouchPad::self()
{
  if ( !m_self )
  {
    staticTouchPadDeleter.setObject( m_self, new TouchPad() );
  }

  return m_self;
}

TouchPad::TouchPad()
{
  sm_Pad = Pad::getInstance();
}

TouchPad::~TouchPad()
{
    if (  m_self == this )
        staticTouchPadDeleter.setObject( m_self, 0, false );
}

bool TouchPad::isValid()
{
    return sm_Pad->hasDriver();
}

int TouchPad::isEnabled()
{
    return isValid()? ( !( sm_Pad->getParam( TOUCHPADOFF ) == 0 ) ) : false;
}

void TouchPad::setEnabled( int enable )
{
    sm_Pad->setParam( TOUCHPADOFF, (double) enable );
}

QRect TouchPad::edges()
{
    if ( !isValid() )
        return QRect();

    return QRect( QPoint( (int)sm_Pad->getParam( TOPEDGE ), 
                          (int)sm_Pad->getParam( LEFTEDGE) ),
                  QPoint( (int)sm_Pad->getParam( BOTTOMEDGE ), 
                          (int)sm_Pad->getParam( RIGHTEDGE ) ) );
}

void TouchPad::setEdges( const QRect &edge )
{
    if ( !isValid() )
      return;

    sm_Pad->setParam( TOPEDGE, edge.top() );
    sm_Pad->setParam( LEFTEDGE, edge.left() );
    sm_Pad->setParam( BOTTOMEDGE, edge.bottom() );
    sm_Pad->setParam( RIGHTEDGE, edge.right() );
}

bool TouchPad::isTappingEnabled()
{
    return isValid()? sm_Pad->getParam( MAXTAPTIME ) > 0 : false;
}

int TouchPad::tapTime()
{
    return isValid()? (int)sm_Pad->getParam( MAXTAPTIME ) : 0;
}

void TouchPad::setTapTime( int time )
{
    sm_Pad->setParam( MAXTAPTIME, (double) time );
}

int TouchPad::sensitivity()
{
    if ( !isValid() )
        return 0;

    int i;

    for ( i = 0; i < 5; ++i )
    {
        if ( sm_Pad->getParam( FINGERLOW ) >= finger_low[ i ] )
            return i;
    }

    return i - 1;
}

void TouchPad::setSensitivity( int i )
{
    if ( !isValid() )
        return;

    if ( i< 0 && i>4 )
        return;

    if ( !SynConfig::treatAsALPS() )
    {
        sm_Pad->setParam( FINGERLOW, finger_low[ i ] );
        sm_Pad->setParam( FINGERHIGH, finger_low[ i ] + 5 );
    }
    else
    {
        sm_Pad->setParam( FINGERLOW, finger_low[ i ] - 11 );
        sm_Pad->setParam( FINGERHIGH, finger_low[ i ] - 10 );
    }
}

bool TouchPad::isHorizontalScrollEnabled()
{
  return isValid()? sm_Pad->getParam( HORIZSCROLLDELTA ) > 0 : false;
}

int TouchPad::horizontalScrollDelta()
{
  return isValid()? (int)sm_Pad->getParam( HORIZSCROLLDELTA ) : 0;
}

void TouchPad::setHorizontalScrollDelta( int delta )
{
    sm_Pad->setParam( HORIZSCROLLDELTA, delta );
}

bool TouchPad::isVerticalScrollEnabled()
{
  return isValid()? sm_Pad->getParam( VERTSCROLLDELTA ) > 0 : false;
}

int TouchPad::verticalScrollDelta()
{
  return isValid()? (int)sm_Pad->getParam( HORIZSCROLLDELTA ) : 0;
}

void TouchPad::setVerticalScrollDelta( int delta )
{
    sm_Pad->setParam( VERTSCROLLDELTA, delta );
}

bool TouchPad::isEdgeMotionEnabled()
{
  if ( isValid() ) 
    return sm_Pad->getParam( EDGEMOTIONUSEALWAYS );
  else 
    return false;
}

bool TouchPad::isCoastingEnabled()
{
  if ( isValid() )
    return ( sm_Pad->getParam( COASTINGSPEED ) < 0.1 );
  else
    return false;
}

void TouchPad::setEdgeMotionEnabled( bool enable )
{
    sm_Pad->setParam( EDGEMOTIONUSEALWAYS, enable? 1:0 );
}

void TouchPad::setCoastingEnabled( bool enable )
{
  if ( enable ) 
      sm_Pad->setParam( COASTINGSPEED, 38 );
  else
      sm_Pad->setParam( COASTINGSPEED, 0 ); // disable it completely
}

bool TouchPad::isCircularScrollEnabled()
{
  return isValid()? (bool)sm_Pad->getParam( CIRCULARSCROLLING ) : false;
}

bool TouchPad::isTFHorizScrollEnabled()
{
  return isValid()? (bool)sm_Pad->getParam( HORIZTWOFINGERSCROLL ) : false;
}

bool TouchPad::isTFVertScrollEnabled()
{
  return isValid()? (bool)sm_Pad->getParam( VERTTWOFINGERSCROLL ) : false;
}

void TouchPad::setCircularScrollEnabled( bool enable )
{
    sm_Pad->setParam( CIRCULARSCROLLING, enable? 1:0 );
}

void TouchPad::setTFHorizScrollEnabled( bool enable )
{
    sm_Pad->setParam( HORIZTWOFINGERSCROLL, enable? 1:0 );
}

void TouchPad::setTFVertScrollEnabled( bool enable )
{
    sm_Pad->setParam( VERTTWOFINGERSCROLL, enable? 1:0 );
}

int TouchPad::circularScrollDelta()
{
  return isValid()? ( int )( sm_Pad->getParam( CIRCSCROLLDELTA ) * 1000 ) : 0;
}

void TouchPad::setCircularScrollDelta( int delta )
{
    sm_Pad->setParam( CIRCSCROLLDELTA, delta * 0.001 );
}

ScrollTrigger TouchPad::circularScrollTrigger()
{
  return isValid()? ( ScrollTrigger )sm_Pad->getParam( CIRCSCROLLTRIGGER ) : NoTrigger;
}

void TouchPad::setCircularScrollTrigger( ScrollTrigger t )
{
    sm_Pad->setParam( CIRCSCROLLTRIGGER, (double)t );
}

Button TouchPad::buttonForTap( TapType tap )
{
    if ( !isValid() ) return NoButton;

    Button b;

    switch ( tap )
    {
        case RightTop:
            b = (Button) sm_Pad->getParam( RTCORNERBUTTON );
            break;
        case RightBottom:
            b = (Button) sm_Pad->getParam( RBCORNERBUTTON );
            break;
        case LeftTop:
            b = (Button) sm_Pad->getParam( LTCORNERBUTTON );
            break;
        case LeftBottom:
            b = (Button) sm_Pad->getParam( LBCORNERBUTTON );
            break;
        case OneFinger:
            b = (Button) sm_Pad->getParam( TAPBUTTON1 );
            break;
        case TwoFingers:
            b = (Button) sm_Pad->getParam( TAPBUTTON2 );
            break;
        case ThreeFingers:
            b = (Button) sm_Pad->getParam( TAPBUTTON3 );
            break;
        default:
            // must not be the case!
            b = NoButton;
            break;
    }

    return b;
}

bool TouchPad::areFastTapsEnabled()
{
  return isValid()? (bool)sm_Pad->getParam( FASTTAPS ) : false;
}

void TouchPad::setFastTaps( bool enable )
{
    sm_Pad->setParam( FASTTAPS, enable? 1:0 );
}

void TouchPad::setButtonForTap( TapType tap, Button button )
{
    if ( !isValid() )
        return;

    switch ( tap )
    {
        case RightTop:
            sm_Pad->setParam( RTCORNERBUTTON, button );
            break;
        case RightBottom:
            sm_Pad->setParam( RBCORNERBUTTON, button );
            break;
        case LeftTop:
            sm_Pad->setParam( LTCORNERBUTTON, button );
            break;
        case LeftBottom:
            sm_Pad->setParam( LBCORNERBUTTON, button );
            break;
        case OneFinger:
            sm_Pad->setParam( TAPBUTTON1, button );
            break;
        case TwoFingers:
            sm_Pad->setParam( TAPBUTTON2, button );
            break;
        case ThreeFingers:
            sm_Pad->setParam( TAPBUTTON3, button );
            break;
        default:
            // must not be the case!
            break;
    }

}

int TouchPad::absCoordX()
{
    return isValid()? (int)sm_Pad->getParam( ABSCOORDX ) : 0;
}

int TouchPad::absCoordY()
{
    return isValid()? (int)sm_Pad->getParam( ABSCOORDY ) : 0;
}

void TouchPad::applyConfig()
{
    setEnabled( SynConfig::enableTouchPad() );
    setSensitivity( SynConfig::sensitivity() );
    setTapTime( SynConfig::enableTapping()? SynConfig::tapTime() : 0 );
    setButtonForTap( OneFinger, ( Button )SynConfig::tapOneFinger() );
    setButtonForTap( TwoFingers, ( Button )SynConfig::tapTwoFingers() );
    setButtonForTap( ThreeFingers, ( Button )SynConfig::tapThreeFingers() );
    setButtonForTap( RightTop, ( Button )SynConfig::tapRightTop() );
    setButtonForTap( RightBottom, ( Button )SynConfig::tapRightBottom() );
    setButtonForTap( LeftTop, ( Button )SynConfig::tapLeftTop() );
    setButtonForTap( LeftBottom, ( Button )SynConfig::tapLeftBottom() );  
    setHorizontalScrollDelta( SynConfig::enableHorizontalScrolling()? SynConfig::horizontalScrollDelta() : 0 );
    setVerticalScrollDelta( SynConfig::enableVerticalScrolling()? SynConfig::verticalScrollDelta() : 0 );
    setEdgeMotionEnabled( SynConfig::enableEdgeMotion() );
    setCoastingEnabled( SynConfig::enableCoasting() );
    setCircularScrollEnabled( SynConfig::enableCircularScrolling() );
    setCircularScrollDelta( SynConfig::circularScrollDelta() );
    setCircularScrollTrigger( ( ScrollTrigger )SynConfig::circularScrollTrigger() );
    
    setTFVertScrollEnabled( SynConfig::enableTFVerticalScrolling() );
    setTFHorizScrollEnabled( SynConfig::enableTFHorizontalScrolling() );

    setEdges( SynConfig::edges() );
    setFastTaps( SynConfig::fastTaps() );
}

